/* TaRT 2.50 Utility to generate taglines automatically
   Copyright (C) 1999  Mark Veinot

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* file: main.c
 * ------------
 * Programming and concept: Mark Veinot
 * Creates custom e-mail signatures
 * --------------------------------
 * This program represents my first major undertaking in C programming.
 * I have taken care to write functions that are 'safe' in that they
 * should not crash the system or overwrite files.
 *
 * For a version of TaRT for the Microsoft Windows 95/NT
 * environments, see my web page: http://dragon.acadiau.ca/~031233v
 *
 * Feel free to e-mail me at: 031233v@dragon.acadiau.ca
 * with any questions, comments, suggestions, or (yes, even)
 * complaints.
 */

#ifdef __riscos__
#include "nocolors.h"
#else
#ifdef NOCOLOR
#include "nocolors.h"
#else
#include "colors.h"
#endif
#endif
#include "ibeat.h"
#include "tartutil.h"
#include "stringutil.h"
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <ctype.h>

#define VER 2.50
#define HLINE "----------------------------------------------------------------"

/* global variables */

/* a bunch of defaults */

int quiet = 0;			/*  default to verbose operation */
int date = 1;			/*  display the date by default */
int version = 1;		/*  display version info by default */
int center = 0;			/*  default to no centering */
int custom = 0;			/*  use standard signature by default */
int special = 1;		/*  use special dates by default */

/* function declarations */

void Help (void);
int MakeTag (char custom1[], char custom2[], char filename[], char tagfile[], char datefile[]);
int MakeTagCustom (char custom1[], char custom2[], char filename[], char tagfile[], char customfile[], char datefile[]);
void TaRTSetup (char custom1[], char custom2[], char customfile[], char sigfile[], char tagfile[], char datefile[]);

/* main program */

int main (int argc, char *argv[])
{
	int i;
	char custom1[256], custom2[256], tagfile[256], sigfile[256],
	 customfile[256], datefile[256];

	srand ((unsigned) time (NULL));		/*  seed the random number generator */

	/* parse all command line params */

 	if (IsSwitch (argv, argc, "--help") != -1 || IsSwitch (argv, argc, "-h") != -1)
		Help ();
	if (IsSwitch (argv, argc, "--setup") != -1 || IsSwitch (argv, argc, "-s") != -1)
		TaRTSetup (custom1, custom2, customfile, sigfile, tagfile, datefile);
	if (ReadINIFile (custom1, custom2, customfile, sigfile, tagfile, datefile) == 1) {
		fprintf (stderr, "error reading .TaRTrc (First time running TaRT? Try `TaRT --setup')\n");
		exit (1);
	}
	if (IsSwitch (argv, argc, "--add") != -1 || IsSwitch (argv, argc, "-a") != -1)
		AddLine (tagfile);
	if (IsSwitch (argv, argc, "-cust") != -1)
		custom = 0;
	if (IsSwitch (argv, argc, "+cust") != -1)
		custom = 1;
	if (IsSwitch (argv, argc, "-n") != -1)
		date = 0;
	if (IsSwitch (argv, argc, "+n") != -1)
		date = 1;
	if (IsSwitch (argv, argc, "-v") != -1)
		version = 0;
	if (IsSwitch (argv, argc, "+v") != -1)
		version = 1;
	if (IsSwitch (argv, argc, "-sd") != -1)
		special = 0;
	if (IsSwitch (argv, argc, "+sd") != -1)
		special = 1;
	if (IsSwitch (argv, argc, "-c") != -1)
		center = 0;
	if (IsSwitch (argv, argc, "+c") != -1)
		center = 1;
	if (IsSwitch (argv, argc, "-q") != -1)
		quiet = 0;
	if (IsSwitch (argv, argc, "+q") != -1)
		quiet = 1;
	if (IsSwitch (argv, argc, "--display") != -1 || IsSwitch (argv, argc, "-d") != -1) {
		DisplaySignature (sigfile);
		return (0);
	}
	/* end command line parsing */

	if (center && !custom)
		CenterText (custom1, 64);
	if (center && !custom)
		CenterText (custom2, 64);

	/* generate the signatures */

	switch (custom) {
	case 0:
		if (MakeTag (custom1, custom2, sigfile, tagfile, datefile)) {
			fprintf (stderr, "error creating signature\n");
			exit (1);
		}
		break;
	case 1:
		i = MakeTagCustom (custom1, custom2, sigfile, tagfile, customfile, datefile);
		switch (i) {
		case 1:
			fprintf (stderr, "error creating signature\n");
			exit (1);
		case 2:
			if (!quiet)
				fprintf (stderr, "custom file not present, using standard layout\n");
			if (MakeTag (custom1, custom2, sigfile, tagfile, datefile)) {
				fprintf (stderr, "error creating signature\n");
				exit (1);
			}
		}
	}
	return (0);
}

/* function definitions */

/* function: Help
 * usage: Help();
 * --------------
 * This procedure sends the help message to stdout.
 */

void Help (void)
{
#ifdef __riscos__
        printf (BLUE "\nRiscosTaRT" GREEN " version %.2f " WHITE " (%s)\n" NORMAL, VER, __DATE__);
#else
        printf (BLUE "\nLinuxTaRT" GREEN " version %.2f " WHITE " (%s)\n" NORMAL, VER, __DATE__);
#endif
	HLine (74);
	printf ("Command line e-mail signature creator\n");
	printf ("\n");
	printf ("Options:\n");
	printf (" -a, --add\tprompt for a new tagline and add it to the database\n");
	printf (" -d, --display\tdisplays current signature, doesn't change it\n");
	printf (" -h, --help\tdisplays this help\n");
	printf (" -s, --setup\trun the internal setup program\n");
	printf (" [+/-]c\t\tenable/disable signature centering\t[default = off]\n");
	printf (" [+/-]cust\tbuild signature based on custom file\t[default = off]\n");
	printf (" [+/-]n\t\tenable/disable date in signature\t[default = on]\n");
	printf (" [+/-]q\t\tenable/disable quiet mode\t\t[default = off]\n");
	printf (" [+/-]sd\tenable/disable special dates\t\t[default = on]\n");
	printf (" [+/-]v\t\tenable/disable TaRT version info\t[default = on]\n");
	HLine (74);
#ifdef __riscos__
        printf (BLUE "\nPorted to RISC OS by iain truskett " NORMAL " -- " BLUE "koschei@eh.org\n" NORMAL);
#endif
	printf (BLUE "\nWritten by Mark Veinot" NORMAL "  --  " BLUE "031233v@dragon.acadiau.ca\n\n" NORMAL);
	exit (0);
}

/* function: MakeTag
 * usage: i = MakeTag(custom1, custom2, filename, tagfile, datefile);
 * --------------------------------------
 * This function creates a standard TaRT type signature
 * in file 'filename' using a tagline from 'tagfile'.
 * it returns 1 if there was an error, 0 otherwise.
 */

int MakeTag (char custom1[], char custom2[], char filename[], char tagfile[], char datefile[])
{
	FILE *outfile;
	struct tm *l_time;
//	struct tm *gm_time;
	char dateToday[256];
	char tagline[256];
	char shortDate[256];
	char VersionInfo[256];
	time_t now;
	int taglines, tagnumber;

	now = time ((time_t *) NULL);	/*  a lot of obscene code */
	time (&now);		/*  for retrieving the date */
	l_time = localtime (&now);	/*  in a useable manner :) */
	
	strftime (shortDate, 64, "%m/%d\n", l_time);
	strftime (dateToday, 64, "Tagline for %A, %B %d, %Y\n", l_time);
	if (center)
		CenterText (dateToday, 64);
	taglines = CountTagLines (tagfile);	/*  count the taglines in the database */
	if (taglines == -1) {
		fprintf (stderr, "error reading taglines\n");
		return (1);
	}
	tagnumber = SelectRandom (1, taglines);
	GetTagLine (tagfile, tagline, tagnumber, shortDate, datefile, special);		/*  get a tagline */
	if (center)
		CenterText (tagline, 64);

#ifdef __riscos__
	sprintf (VersionInfo, "RiscosTaRT version %.2f\n", VER);
#else
	sprintf (VersionInfo, "LinuxTaRT version %.2f\n", VER);
#endif
	if (center)
		CenterText (VersionInfo, 64);

	outfile = fopen (filename, "w");	/*  open signature file for writing */
	if (outfile == NULL)
		return (1);

	fprintf (outfile, "%s\n", HLINE);	/*  write the signature out */
	fprintf (outfile, "%s\n", custom1);	/*  to the file, line by line */
	fprintf (outfile, "%s\n", HLINE);
	if (date)
		fprintf (outfile, dateToday);
	if (date)
		fprintf (outfile, "\n");
	fprintf (outfile, "%s", tagline);
	if (version)
		fprintf (outfile, "\n");
	if (version)
		fprintf (outfile, VersionInfo);
	fprintf (outfile, "%s\n", HLINE);
	fprintf (outfile, "%s\n", custom2);
	fprintf (outfile, "%s", HLINE);
	if (fclose (outfile))
		return (1);

	if (!quiet)
		DisplaySignature (filename);	/*  display the new signature if not quiet */

	return (0);
}

/* function: MakeTagCustom
 * usage: i = MakeTagCustom(custom1, custom2, filename, tagfile, customfile, datefile);
 * --------------------------------------
 * This function creates a custom TaRT signature
 * in file 'filename' using the string in 'tagline' and a custom layout
 * from the file 'customfile'.
 * it returns 1 if there was an error, 0 otherwise.
 */

int MakeTagCustom (char custom1[], char custom2[], char filename[], char tagfile[], char customfile[], char datefile[])
{
#ifdef PROC
	FILE *procuptime;
	double uptm;
	int uptime;
#endif
#ifdef __riscos__
        int uptime;
#endif
        FILE *outfile;
	FILE *custfile;
	struct tm *l_time;
	struct tm *gm_time;
	signed char c = '\0', d = '\0';
	char fmt[3], dateToday[256], VersionInfo[256], shortDate[256];
	char Hour12[3], Hour24[3], Minute[3], AMPM[3];
	char tagline[256];
	char Ibeats[5];
	time_t now;
	int taglines, tagnumber;

	custfile = fopen (customfile, "r");	/*  open custom definition file for parsing */
	if (custfile == NULL)
		return (2);

	StripCR (custom1);
	StripCR (custom2);

	taglines = CountTagLines (tagfile);
	if (taglines == -1) {
		fprintf (stderr, "error reading taglines\n");
		return (1);
	}
#ifdef __riscos__
        sprintf (VersionInfo, "RiscosTaRT version %.2f", VER);
#else
	sprintf (VersionInfo, "LinuxTaRT version %.2f", VER);
#endif

#ifdef __riscos__
        uptime = clock() / CLOCKS_PER_SEC / 3600;
#endif
#ifdef PROC
	procuptime = fopen ("/proc/uptime", "r");
	if (procuptime == NULL) {
		fprintf (stderr, "error determining uptime - do you have /proc?\n");
		return (1);
	}
	fscanf (procuptime, "%lf", &uptm);
	uptime = uptm / 3600;
#endif

	now = time ((time_t *) NULL);	/*  a lot of obscene code */
	time (&now);		/*  for retrieving the date */
	l_time = localtime (&now);	/*  in a useable manner :) */	
	gm_time = gmtime (&now);	/* since ibeats works from GMT+1, may as well use GMT as a base for dates  */

	strftime (shortDate, 64, "%m/%d\n", l_time);
	strftime (dateToday, 64, "%A, %B %d, %Y", l_time);
	strftime (Hour12, 3, "%I", l_time);
	strftime (Hour24, 3, "%H", l_time);
	strftime (Minute, 3, "%M", l_time);
	strftime (AMPM, 3, "%p", l_time);
	sprintf  (Ibeats, "@%03i", time_ibeats(&now));

	outfile = fopen (filename, "w");	/*  open signature file for writing */
	if (outfile == NULL)
		return (1);

	while (!feof (custfile)) {	/*  basically, read through the custom layout char by */
		c = fgetc (custfile);	/*  char and parse it for formatting characters. */
		if (c == '%') {
			c = fgetc (custfile);
			if (c == '%')
				fputc (c, outfile);
			else {
				fmt[0] = c;
				fmt[1] = fgetc (custfile);
				fmt[2] = '\0';
#ifdef PROC
				if (strncmp (fmt, "ut", 2) == 0)
					fprintf (outfile, "%d", uptime);
#endif
#ifdef __riscos__
				if (strncmp (fmt, "ut", 2) == 0)
					fprintf (outfile, "%d", uptime);
#endif
				if (strncmp (fmt, "ib", 2) == 0)
					fprintf (outfile, Ibeats);
				if (strncmp (fmt, "c1", 2) == 0)
					fprintf (outfile, custom1);
				if (strncmp (fmt, "c2", 2) == 0)
					fprintf (outfile, custom2);
				if (strncmp (fmt, "dt", 2) == 0)
					fprintf (outfile, dateToday);
				if (strncmp (fmt, "12", 2) == 0)
					fprintf (outfile, Hour12);
				if (strncmp (fmt, "24", 2) == 0)
					fprintf (outfile, Hour24);
				if (strncmp (fmt, "mn", 2) == 0)
					fprintf (outfile, Minute);
				if (strncmp (fmt, "ap", 2) == 0)
					fprintf (outfile, AMPM);
				if (strncmp (fmt, "hr", 2) == 0)
					fprintf (outfile, HLINE);
				if (strncmp (fmt, "tg", 2) == 0) {
					tagnumber = SelectRandom (1, taglines);
					GetTagLine (tagfile, tagline, tagnumber, shortDate, datefile, special);
					StripCR (tagline);
					fprintf (outfile, "%s", tagline);
					special = 0;
				}
				if (strncmp (fmt, "vi", 2) == 0)
					fprintf (outfile, VersionInfo);
				strcpy (fmt, "   ");
				c = '\0';
			}
		} else {
			d = fgetc (custfile);
			if ((isascii (c) || iscntrl (c)) && d != EOF)
				fputc (c, outfile);
			ungetc (d, custfile);
			d = '\0';
		}
	}
	if (fclose (custfile))
		return (1);
	if (fclose (outfile))
		return (1);

	if (!quiet)
		DisplaySignature (filename);	/*  display the new signature if not quiet */

	return (0);
}

/* procedure: TaRTSetup
 * usage: TaRTSetup();
 * ---------------
 * This procedure prompts the user for different settings and
 * creates a new .TaRTrc file in the users home directory.
 * Terminates program execution.
 */

void TaRTSetup (char custom1[], char custom2[], char customfile[], char sigfile[], char tagfile[], char datefile[])
{
	char tmp[256];

	printf (BLUE "\n-- TaRT %.2f" NORMAL " Interactive Setup --\n", VER);

	if (ReadINIFile (custom1, custom2, customfile, sigfile, tagfile, datefile)) {
		strcpy (custom1, "TaRT - The Automatic Random Tagline\n");	/*  set some defaults */
		strcpy (custom2, "My webpage: http://www.apage.com/mypage\n");
#ifdef __riscos__
		strcpy (customfile, "Choices:TaRT.Layout");
		strcpy (sigfile, "<Choices$Write>.mail./signature");
		strcpy (tagfile, "<Choices$Write>.TaRT.taglines");
		strcpy (datefile, "Choices:TaRT.SpecialDates");
#else
		strcpy (customfile, "~/.custom");
		strcpy (sigfile, "~/.signature");
		strcpy (tagfile, "/etc/tag.trt");
		strcpy (datefile, "~/.specialdates");
#endif
	}
	StripCR (custom1);
	printf (YELLOW "\nEnter text for first custom line (64 chars max): ----------------|" GREEN "\n[%s]\n" NORMAL "> ", custom1);
	fgets (tmp, 64, stdin);
	if (tmp[0] != '\n')
		strcpy (custom1, tmp);

	StripCR (custom2);
	printf (YELLOW "\nEnter text for second custom line (64 chars max): ---------------|" GREEN "\n[%s]\n" NORMAL "> ", custom2);
	fgets (tmp, 64, stdin);
	if (tmp[0] != '\n')
		strcpy (custom2, tmp);

	printf (YELLOW "\nEnter the full path to your signature file:" GREEN "\n[%s]\n" NORMAL "> ", sigfile);
	fgets (tmp, 256, stdin);
	if (tmp[0] != '\n')
		strcpy (sigfile, tmp);

	printf (YELLOW "\nEnter the full path to your custom layout file:" GREEN "\n[%s]\n" NORMAL "> ", customfile);
	fgets (tmp, 256, stdin);
	if (tmp[0] != '\n')
		strcpy (customfile, tmp);

	printf (YELLOW "\nEnter the full path to your special dates file:" GREEN "\n[%s]\n" NORMAL "> ", datefile);
	fgets (tmp, 256, stdin);
	if (tmp[0] != '\n')
		strcpy (datefile, tmp);

	printf (YELLOW "\nEnter the full path to the tagline database:" GREEN "\n[%s]\n" NORMAL "> ", tagfile);
	fgets (tmp, 256, stdin);
	if (tmp[0] != '\n')
		strcpy (tagfile, tmp);

	printf ("\n");

	if (WriteINIFile (custom1, custom2, customfile, sigfile, tagfile, datefile)) {
		fprintf (stderr, "error creating .TaRTrc\n");
		exit (1);
	}
	exit (0);
}
