/* TI-99/4A Emulator.

   VDP Emulator part.

   The adresses through which the VDP is addressed are:

   >8800:	VDP read data
   >8802:	VDP read status
   >8C00:	VDP write data
   >8C02:	VDP write address

   These addresses are not fully decoded, so the same adressing is achieved
   throughout the >8800..>8FFF area. (e.g. >8800=>8804=>8808=... etc.)
*/

void init_vdp(void)
{
	vdp=(byte *)calloc(VDP_MEMSIZE,1);
	if (vdp==NULL)
	{
		printf("Not enough memory for VDP.\n");
		exit(0);
	}
	VdpAddressCounter=0;
	CRT_ROWS=24;
	CRT_COLS=32;
	CRT_SIZE=CRT_ROWS*CRT_COLS;
}


word vdp_read(word address)
{
	if(address&1)  /* VDP read status. */
	return(DEFAULT_VDP_STATUS<<8);


	else		/* VDP read data */
	return(word)(*(vdp+VdpAddressCounter++)<<8);

}

void vdp_write(word address,word data)
{
	static int VACparity=0;	/* retains the parity of the VDPWA actions */

	if (address&1)	/* write address (>8Cx2 addressed) */
	{
		VACparity=!VACparity;
		VdpAddressCounter>>=8;
		VdpAddressCounter|=(data&0xFF00);
		if (!VACparity)
		{
			if (data&0x8000)
				vdp_write_to_register(VdpAddressCounter);
			VdpAddressCounter&=0x3FFF;
		}
	}
	else
	{
		byte vdpbyte;
		vdpbyte=data>>8;
		if ((VdpAddressCounter<CRT_SIZE)&&
		  (vdpbyte!=*(vdp+VdpAddressCounter))) update_crt(vdpbyte);
		*(vdp+VdpAddressCounter++)=vdpbyte;
	}
}

void update_crt(byte data)
{
	byte ccode;

	ccode=data-((data>=126)? 96:0);	/* TI-BASIC offset */
	if (ccode<31) ccode=219;
	if ((ccode==0)||(ccode==31)) ccode=32;
	print_at(	1+VdpAddressCounter%CRT_COLS,
			1+(int)VdpAddressCounter/CRT_COLS,
			ccode);
}

void vdp_write_to_register(word regdata)
{
	int vdpregister,value;
	vdpregister=(regdata>>8)&0x3F;
	value=regdata&0xFF;
	if (vdpregister==1)
	{
		value&=0x10;
		if (value==0x10) CRT_COLS=40;
		if (value==0)	 CRT_COLS=32;
		if (CRT_SIZE!=CRT_ROWS*CRT_COLS)
		{
			CRT_SIZE=CRT_ROWS*CRT_COLS;
			clear_screen();
			for(	VdpAddressCounter=0;
				VdpAddressCounter<CRT_SIZE;
				VdpAddressCounter++)
				update_crt(*(vdp+VdpAddressCounter));
		}

	}
}
