/*
 * usage log generator
 * Copr. 1997 Chris Nott (c.nott@student.canberra.edu.au)
 *
 * excuse the c++ comments.. I get lazy
 */

/*
 * design:
 *	- read entries from the wtmp
 *	- interpret entries as entries of the form tty:login - logout
 *	- split the entries into minutes per hour for each day the record
 *	  spans
 *	- place the entry in a hash table for that tty
 *	- write the hash table for each tty to a file
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <sys/stat.h>
#include <utmp.h>
#include <time.h>
#include <string.h>

#ifndef min
#define min(x,y) ((x < y) ? (x) : (y))
#endif
#define LOGDIR "/usr/local/etc/usagelogs"
#define PREFIX "usage."

typedef struct {
	time_t login;
	pid_t pid;
	char user[20];
} logrec;
char **ttys;
int numttys;


/*
 * hash table code.. bound to be disgusting and horrible
 */

#define HASHSIZE 367

typedef int Key;	/* <-- change these to reuse hash code */
typedef int *Data;	/* <-- ... */
typedef struct Item {
	Key	key;
	Data	data;
	struct Item	*next;
} Item;
typedef Item **Hash;

/* usage 'hash` tables */
Hash *htabs;

/* hash internals */

/* hash integers (days in this case) */
inline int iHash(Key k) { return(k % HASHSIZE); }

Item *allochashent(Key k,Data d) {
	Item *it;

	it = (Item *)malloc(sizeof(Item));
	if(!it) { perror("malloc"); exit(-1); }
	it->key = k;
	it->data = d;
	it->next = 0;
	return(it);
}

/* hash 'exported' functions */

void inithash(Hash *h) {
	int i;
	*h = (Hash)malloc(sizeof(int) * HASHSIZE);
	if(!*h) { perror("malloc"); exit(-1); }
	for(i=0;i<HASHSIZE;i++) (*h)[i] = 0;
}

/* add a new hash entry */
void addhashent(Hash *h,Key k,Data d) {
	int hash = iHash(k);
	Item *it;
	
	it = (*h)[hash];
	if(!it) {
		(*h)[hash] = allochashent(k,d);
		return;
	}
	
	do {
		if(it->key == k) {
/*
 * user defined - replace data if needed .. free malloced stuff etc
 */
 			if(it->data != d) {
// don't replace the data for usage log
//				free(it->data); /* <--- evil... */
//				it->data = d;
			}
			return;
		}
		it = it->next;
	} while(it->next); 
	it->next = allochashent(k,d);
}

/* get a hash entry by key */
Data gethashent(Hash *h,Key k) {
	Item *it;

	it = (*h)[iHash(k)];
	while(it) {
		if(it->key == k) return(it->data);
		it = it->next;
	}
	return(0);	/* ? what if 0 is a valid value for Data ? */
}

/* end hash table bits */


inline int matchtty(char *tty) {
	int i;

	for(i=0;i<numttys;i++) if(!strcmp(ttys[i],tty)) return(i);
	return(-1);
}


inline int daysbetween(int yr1,int yr2) {
	int t,year,days;
	if(yr1>yr2) {
		t = yr1;	
		yr1 = yr2;
		yr2 = t;
	}

	for(days=0,year=yr1; year<yr2; year++) {
		/* is __isleap a standard function? */
		days += (__isleap(year) ? 366 : 365);
	}
	return(days);
}


void logtime(int line, time_t login, time_t logout) {
	struct tm *ttm, itm, otm;
	int days,minutes,day,hour,starthour,endhour,startmin,endmin;
	int *usage;
	Hash *h;

	h = &htabs[line];

	ttm = localtime(&login);
	memcpy(&itm,ttm,sizeof(struct tm));
	ttm = localtime(&logout);
	memcpy(&otm,ttm,sizeof(struct tm));

	starthour = itm.tm_hour;
	startmin = itm.tm_min;
	day = itm.tm_yday + daysbetween(0,itm.tm_year);
	while(days = daysbetween(0,otm.tm_year) 
		     + otm.tm_yday - day,
		     days >= 0) {
		int minsthishour;

		if(days>0) {
			endhour = 24;
			endmin = 60;
		} else {
			endhour = otm.tm_hour;
			endmin = otm.tm_min;
		}

		/* watch that return value of gethashent.. */
		usage = gethashent(h,day);
		if(!usage) {
			usage = (int *)malloc(sizeof(int) * 24);
			if(!usage) { perror("malloc"); exit(-1); }
			for(hour=0;hour<24;hour++) usage[hour]=0;
		}

		hour = starthour;
		while(minutes = (endhour - hour) * 60 
				+ endmin - startmin,
				minutes > 0) {
			minsthishour = min(60 - startmin, minutes);
			if(startmin) startmin = 0; /* reset mins offset 
							  after first run */
			usage[hour] += minsthishour;
//			printf("%i(%i): %i\n",day,hour,minsthishour);
			hour++;
		}
		addhashent(h,day,usage);
		if(starthour) starthour = 0;
		day++;
	}
}


FILE *openlog(char *tty,char *mode) {
	char fname[NAME_MAX];
	struct stat s;
	FILE *f;

	if(stat(LOGDIR,&s)) {
		if(!ENOENT) { perror(LOGDIR); exit(-1); }
		if(mkdir(LOGDIR,0755)) { perror(LOGDIR); exit(-1); }
	}
	if(!S_ISDIR(s.st_mode)) {
		printf(LOGDIR ": is not a directory.\n");
		exit(-1);
	}

	sprintf(fname,LOGDIR "/" PREFIX "%s",tty); /* buffer overflow? */
	f = fopen(fname,mode);
	if(!f && mode[0]!='r') {
		perror(fname);
	 	return(0);
	}
	return(f);
}


void readlog(char *fname,Hash *h) {
	char buff[8192];
	int i,c,c2,hour,day,*usage,eof=0,lno=0;
	FILE *logf;

	logf = openlog(fname,"r");
	while(!eof) {
		for(i=0;i<8192;i++) {
			c = fgetc(logf);
			if(c == EOF) {
				eof = 1;
			}
			if(c == EOF || c == '\n') {
				buff[i] = 0;
				lno++;
				break;
			}
			buff[i] = c;
		}
		if(!i) {
			continue;
		}
		
		usage = (int *)malloc(sizeof(int) * 24);
		if(!usage) { perror("malloc"); exit(-1); }
		
		/* find the day number */
		for(c=0;c<i;c++) {
			if(buff[c] == ':') {
				buff[c++] = 0;
				break;
			}
		}

		day = atoi(buff);

		/* extract the usage */
		for(hour=0;hour<24;hour++) {
			/* eat whitespace */
			for(; c<i && buff[c] == ' '; c++);

			/* find next space and mark it */
			for(c2=c; c2<i && buff[c2] != ' '; c2++);

			if(c2 >= i && hour != 23) {	/* no space found.. */
				printf(PREFIX "%s: invalid log format on line %i\n",fname,lno);
				exit(-1);
			}
			buff[c2++] = 0;
			usage[hour] = atoi(&buff[c]);
			c = c2;
		}
		if(c<i) {
			printf("warning: '" PREFIX "%s' possible format error on line %i\n",fname,lno);
		}
		addhashent(h,day,usage);
	}
	fclose(logf);
}


int main(int argc,char *argv[]) {
	struct utmp *ut;
	logrec *lr;
	int line,i;
	time_t current_time;

	if(argc<2) {
		printf("call with: %s <tty1> [tty2] ..\n",argv[0]);
		exit(-1);
	}
	numttys = argc - 1;
	ttys = &(argv[1]);

	current_time = time(NULL);

	lr = (logrec *)malloc(numttys * sizeof(logrec));
	if(!lr) { perror("malloc"); exit(-1); }
	htabs = (Hash *)malloc(numttys * sizeof(int));
	if(!htabs) { perror("malloc"); exit(-1); }
	
	utmpname(WTMP_FILE);
	setutent();

	for(i=0;i<numttys;i++) {
		lr[i].pid=0;
		inithash(&htabs[i]);
	}
	while( (ut = getutent()) ) {
		line = matchtty(ut->ut_line);
		if(line != -1) {
			if(ut->ut_type == USER_PROCESS) {
				if(lr[line].pid) {
/*
 * apparently if you don't actually log in but use mgetty in ppp mode
 * it doesn't put a proper logout entry but puts a null entry with the 
 * same pid as the login entry (I think)
 */
					if(ut->ut_pid == lr[line].pid && !ut->ut_user[0]) {
						logtime(line,lr[line].login,ut->ut_time);
						lr[line].login = 0;
						lr[line].pid = 0;
						continue;
					}
/*
 * if you use pppd as above it mgetty can log it in as a generic ppp
 * login and then pppd will put a wtmp entry for authenticated logins
 * resulting in two logins and one logout
 */
//					printf("warning: %s already logged in on line %s\n",
//						ut->ut_user,ttys[line]);
					logtime(line,lr[line].login,ut->ut_time);
				}
				lr[line].pid = ut->ut_pid;
				lr[line].login = ut->ut_time;
				strcpy(lr[line].user,ut->ut_user);
				continue;
			}
			if(ut->ut_type == DEAD_PROCESS) {
				if(!lr[line].pid) {
					continue;
				}
				if(lr[line].pid != ut->ut_pid) {
					printf("logout error - pid of login (%i) is different to that of logout (%i)\n",lr[line].pid,ut->ut_pid);
				}
				logtime(line,lr[line].login,ut->ut_time);
//				printf("%s %s-",lr[line].user,ctime(&lr[line].login));
//				printf("%s (%i)\n",ctime(&ut->ut_time),(int)(ut->ut_time-lr[line].login));
				lr[line].login = 0;
				lr[line].pid = 0;
				continue;
			}
		}
		if(ut->ut_type == BOOT_TIME) {
			for(line=0;line<numttys;line++) {
				if(lr[line].pid) {
					logtime(line,lr[line].login,ut->ut_time);
//					printf("%s %s %s-crash (%i)\n",lr[line].user,ttys[line],ctime(&lr[line].login),(int)(ut->ut_time-lr[line].login));
					lr[line].login = 0;
					lr[line].pid = 0;
				}
				continue;
			}
		}
	}

	for(line=0;line<numttys;line++) {
		Hash h;
		FILE *logf;

		if(lr[line].pid) logtime(line,lr[line].login,current_time);

		h = htabs[line];
		readlog(ttys[line],&h);

		logf = openlog(ttys[line],"w");
		for(i=0;i<HASHSIZE;i++) {
			Item *it;
			int day,*usage;
			
			it = h[i];
			while(it) {
				int hour;
				
				day = it->key;
				usage = it->data;
				if(logf) {
					fprintf(logf,"%i:",day);
					for(hour=0;hour<24;hour++) 
						fprintf(logf," %i",usage[hour]);
					fprintf(logf,"\n");
				}
	
//				printf("day: %i\n",day);
//				for(hour=0;hour<24;hour++) printf("  %i:%i\n",hour,usage[hour]);
				it = it->next;
			}
		}
		fclose(logf);
	}
	return(0);
}
