/* #Specification: netconf / process management
	All the rc's script in /etc/rc.d are week. They simply
	start a bunch of daemon. If something goes wrong, they
	generally continue to fire daemon after daemon. All this
	is fragile.

	Instead, all this management is done using netconf. Netconf
	will probe around to find out if a daemon must be started, if
	there is enough configuration information for this daemon available.

	One goal is to allow netconf to "update" the daemon after
	making some changes to the network configuration.

	For example, you edit /etc/exports and run netconf -update
	after. It will kill mountd and start it again.
*/
#include <stdlib.h>
#include "netconf.h"
#include <dialog.h>
#include <dnsconf.h>
#include <fstab.h>
#include "../askrunlevel/askrunlevel.h"
#include "netconf.m"

/*
	Start a daemon if it is not already started
	Return -1 if any error.
*/
int netconf_startstop(const char *name, int go)
{
	int ret = -1;
	DAEMON *dae = daemon_find (name);
	if (dae != NULL){
		if(!dae->is_managed()){
			/* #Spcification: netconf / daemons and commands / not managed
				The user may choose (advance feature) not to let netconf
				managed some command. This may severly break a system.
				Off course a user who do so is on his own. For example
				a user who decided to disable the management of ifconfig
				and make a mistake will have no network and a bundle
				or error. Back to square one. The purpose of netconf
				is to make sure  everything works...

				At least, netconf will print a message
				in the log as a remainder that if it was
				allowed, netconf would have done this
				and this.
			*/
			ret = 0;
			net_prtlog (NETLOG_VERB
				,MSG_U(X_WOULDHAVE,"Would have %s %s\n")
				,go ? MSG_U(X_STARTED,"started")
					: MSG_U(X_STOPPED,"stopped")
				,name);
		}else{
			if (go){
				ret = dae->startif();
			}else{
				ret = dae->stop();
			}
		}
	}
	return ret;
}
/*
	Establish a run level by starting, restarting or stopping some
	daemon.
*/
static void netconf_runlevel0(
	int level)		// 0 = Minimal local service (loopback)
					// 1 = Basic client mode
					// 2 = full client/server mode
					// -1 == use the same as the last time
					// This is used by netconf --update
{
	daemon_setsession(1);
	simul_setdisable(0);
	process_flushcache();
	/* #Specification: crond
		netconf make sure that crond is active (unless told
		no to do so). This is not really related to networking
		but since netconf manage almost all other daemons
		in the system, why not doing it there.
		
		Currently, it checks that crond is running and that's all.
		I am not aware of any reason (beside maintenance mode maybe)
		why crond would have to be shutdown or restart or signaled.
		So netconf only check it is active.
	*/
	net_section (MSG_U(S_SECTBASE,"Checking base configuration"));
	net_title (MSG_U(S_KMODULES,"Checking kernel's modules"));
	modules_check();
	net_title (MSG_U(S_CROND,"Cron daemon"));
	netconf_startstop ("crond",1);
	/* #Specification: hostname / must be set
		If the hostname can't be set, the rest of the networking won't
		be activated. A message is printed, but nothing can be started.
	*/
	net_title (MSG_U(S_MOUNTALL,"Mounting local volumes"));
	fstab_checkmount (1);
	net_title (MSG_U(S_FIXPERM,"Checking files permissions"));
	fixperm_check();
	net_title (MSG_U(S_LILO,"Checking LILO"));
	lilo_update();
	char msg[10000];
	if (netconf_netok(msg)==NULL
		|| netconf_sethostname() == -1){
		xconf_error (MSG_U(E_IVLBASIC
			,"Invalid basic configuration of the host\n%s\n")
			,msg);
	}else{
		if (level != -1){
			netconf_setnetlevel(level);
		}else{
			level = netconf_getnetlevel();
		}
		net_title (MSG_U(S_LOOPBACK,"Setting network loopback"));
		netconf_setloopback();
		net_title (MSG_U(S_ALIASLOOP,"Setting IP aliases on network loopback"));
		alias_setup("lo");	// See below
		net_title (MSG_U(S_PORTMAP,"Starting the RPC portmapper"));
		netconf_startstop ("rpc.portmap",1);
		net_title (MSG_U(S_INETD,"Starting inetd"));
		netconf_startstop ("inetd",1);
		net_title (MSG_U(S_SYSLOG,"Starting system loggers"));
		netconf_startstop ("syslogd",1);
		netconf_startstop ("klogd",1);
		net_title (MSG_U(S_PRTSPOOL,"Starting printer spooler"));
		netconf_startstop ("lpd",1);
		/* #Specification: module / probing
			We let the modules do something at 4 different places
			during the probing for configuration changes.

			#
			At the end of the probing for local mode
			At the end of the client mode
			At the end of the server mode
			#
		*/
		module_probe (0,level);
		if (level > 0){
			net_section (MSG_U(S_SECTCLIENT,"Setting client networking"));
			net_title (MSG_U(S_IPDEVICES,"Configure network IP devices"));
			netconf_setdevices();
			net_title (MSG_U(S_IPXDEVICES,"Configure network IPX devices"));
			ipx_set();
			net_title (MSG_U(S_IPROUTES,"Configure IP routes"));
			route_install();
			net_title (MSG_U(S_ROUTEDS,"Start routing daemons"));
			netconf_startstop ("routed",1);
			netconf_startstop ("gated",1);
			net_title (MSG_U(S_NAMED,"Start DNS server"));
			netconf_startstop ("named",1);
			if (dns_ping()!=-1){
				net_title (MSG_U(S_NIS,"Starting NIS"));
				netconf_startstop ("ypbind",1);
				/* #Specification: netconf / datetime / updating
					if configured, datetime_getfromnet() always
					perform an action. We avoid doing it while
					in simulation mode as it gives
					the impression the system is never in sync
					with its configuration.
					
					"netconf --status" would always complain.
				*/
				if (!simul_ison()) datetime_getfromnet();
				net_title (MSG_U(S_AMD,"Starting automounter"));
				netconf_startstop ("amd",1);
				net_title (MSG_U(S_MOUNTNET,"Mounting network volumes"));
				fstab_checkmount (0);
				net_title (MSG_U(S_SENDMAIL,"Starting sendmail"));
				netconf_startstop ("sendmail",1);
				module_probe (1,level);
				if (level > 1){
					net_section (MSG_U(S_SECTSERVER,"Setting server networking"));
					/* #Specification: netconf / aliases / activating
						IP aliases are only activated in server mode.
						I don't see much usage for it in another mode.
						They are not disactivated when going back in client
						mode though.

						There is a small exception. The alias for
						the loopback device are always activated. If someone
						setup an alias on "lo", better activate it when "lo"
						is.
					*/
					net_title (MSG_U(S_ALIAS,"Setting IP aliases on net devices"));
					alias_setup();
					net_title (MSG_U(S_NFS,"Starting NFS service"));
					netconf_startstop ("rpc.nfsd",1);
					netconf_startstop ("rpc.mountd",1);
					net_title (MSG_U(S_SSHD,"Starting sshd"));
					netconf_startstop ("sshd",1);
					net_title (MSG_U(S_RARP,"Setting RARP service"));
					rarp_activate();
					module_probe (2,level);
				}else{
					net_section (MSG_U(S_UNSETSERVER,"Unsetting server networking"));
					net_title (MSG_U(S_STOPNFS,"Stopping NFS service"));
					netconf_startstop ("rpc.nfsd",0);
					netconf_startstop ("rpc.mountd",0);
				}
				net_title (MSG_U(S_FIREWALL,"Setting IP firewall"));
				firewall_setup();
			}
		}else{
			net_section (MSG_U(S_UNSETCLIENT,"Unsetting networking"));
			net_title (MSG_U(S_KSENDMAIL,"Stopping sendmail"));
			netconf_startstop ("sendmail",0);
			net_title (MSG_U(S_KROUTEDS,"Stopping routing daemons"));
			netconf_startstop ("gated",0);
			netconf_startstop ("routed",0);
			net_title (MSG_U(S_KAMD,"Stopping automounter"));
			netconf_startstop ("amd",0);
			net_title (MSG_R(S_STOPNFS));
			netconf_startstop ("rpc.nfsd",0);
			netconf_startstop ("rpc.mountd",0);
		}
	}
}

/*
	Establish a run level by starting, restarting or stopping some
	daemon.
*/
void netconf_runlevel(
	int level)		// 0 = Minimal local service (loopback)
					// 1 = Basic client mode
					// 2 = full client/server mode
					// -1 == use the same as the last time
					// This is used by netconf --update
{
	/*# Specification: netconf / setting runlevel / timeout on msgs
		When netconf activate the different networking services
		and other, it may generate different error message.
		A timeout of 15 seconds is established. If there is
		no operator, netconf will continue by itself.

		This avoid to have a server with a small configuration
		problem failing to reboot because it wait for a
		single <enter>
	*/
	dialog_settimeout (15,MENU_ESCAPE,true);
	int is_simul = simul_ison();
	while (1){
		net_resetnberr();
		daemon_setsession(1);
		simul_setdisable(0);
		netconf_runlevel0(level);
		if (!daemon_wasconfig() || is_simul) break;
	}
	simul_setdisable(0);
	daemon_setsession(0);
	if (net_getnberr()
		&& dialog_yesno (MSG_U(Q_SOMEERRORS,"There were some errors")
			,MSG_U(Q_SEELOGS
				,"Some errors were reported\n"
			 	 "Do you want to examine the logs")
			,help_nil)
			 == MENU_YES){
		net_showlog();
	}
	dialog_settimeout (0,MENU_ESCAPE,false);
}

