#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include "netconf.h"
#include <misc.h>
#include <userconf.h>
#include "netconf.m"

static const char NIS[] = "nis";
static const char NIS_DOMAIN[] = "domain";
static const char NIS_SERVER[] = "server";

static NETCONF_HELP_FILE help_nis ("nis");

/*
	Manage the file /etc/conf.nis
	The construtor read the file. A missing file means that NIS
	is not configured.
*/
PUBLIC NIS_CONF::NIS_CONF()
{
	domain.setfrom (linuxconf_getval(NIS,NIS_DOMAIN));
	server.setfrom (linuxconf_getval(NIS,NIS_SERVER));
}

/*
	Check that the server is valid.
	The NIS server specification is optionnal: If the field is
	empty, this is ok.

	Signal an error message if not ok.

	Return != 0 if ok.
*/
PUBLIC int NIS_CONF::valid_server()
{
	int ret = 0;
	if (!server.is_empty()){
		const char *serverp = server.get();
		if (net_isipnum(serverp)){
			ret = 1;
		}else if (gethostbyname(serverp) == NULL){
			xconf_error (MSG_U(E_NISSERV,"Unknown NIS server %s\n"),serverp);
		}else{
			ret = 1;
		}
	}else{
		ret = 1;
	}
	return ret;
}

/*
	Tells if NIS is properly configured
*/
PUBLIC int NIS_CONF::configok()
{
	int ret = 0;
	if (!domain.is_empty()){
		ret = valid_server();
	}
	return ret;
}

/*
	Return the name of the optionnal NIS server or NULL.
*/
PUBLIC const char *NIS_CONF::getserver()
{
	return server.is_empty() ? (const char *)NULL : server.get();
}

/*
	Return the NIS domain name (Unrelated to the BIND domain name).
*/
PUBLIC const char *NIS_CONF::getdomain()
{
	return domain.get();
}

PUBLIC void NIS_CONF::write ()
{
	linuxconf_replace(NIS,NIS_DOMAIN,domain);
	linuxconf_replace(NIS,NIS_SERVER,server);
	linuxconf_save();
}

/*
	Edit the configuration of the NIS client
	Return 0 if the edition was successful.
*/
PUBLIC int NIS_CONF::edit ()
{
	DIALOG dia;
	dia.newf_str (MSG_U(F_NISDOM,"NIS domain"),domain);
	dia.newf_str (MSG_U(F_NISSERV,"NIS server(optionnal)"),server);
	int ret = -1;
	while (1){
		if (dia.edit(MSG_U(T_NISCONF,"NIS client configuration")
			,MSG_U(I_NISCONF,"You must enter the NIS domain\n"
			 "A server must be specified\n"
			 "if it can't be probed by a broadcast\n"
			 "The server is either specified as an IP number\n"
			 "or as a name.\n")
			,help_nis) == MENU_ACCEPT){
			if (valid_server()){
				ret = 0;
				break;
			}
		}else{
			break;
		}
	}
	if (ret != 0) dia.restore();
	return ret;
}

/*
	Edite the file /etc/hosts
*/
void netconf_editnis()
{
	NIS_CONF nis;
	if (nis.edit() == 0) nis.write();
}
/*
	Clone of the /bin/domainname command.
*/
int netconf_domainname (int argc, char *argv[])
{
	/* #Specification: netconf / aliases / domainname
		netconf emulate the command /bin/domainname. Without argument
		it print the domainname (used by NIS). With one, it set the
		domain name.
	*/
	int ret = -1;
	if (argc == 1){
		char buf[100];
		if (getdomainname(buf,sizeof(buf)-1)==-1) strcpy (buf,"none");
		ret = 0;
		printf ("%s\n",buf);
	}else if (argc == 2){
		if (perm_rootaccess (MSG_U(P_SETTINGNIS
			,"Setting the NIS domain name"))){
			ret = setdomainname(argv[1],strlen(argv[1])+1);
		}
	}else{
		fprintf (stderr,MSG_U(E_USAGEDOM
			,"usage: domainname [ nisdomain ]\n"));
	}
	return ret;
}

