#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <string.h>
#include "netconf.h"

static int ifconfig_ioctl(
	int fd,
	const char *ifname,
	int cmd,
	struct ifreq &ifr)
{
	strcpy(ifr.ifr_name, ifname);
	return ioctl(fd, cmd,&ifr);
}


static void ifconfig_format (struct sockaddr &adr, char *buf)
{
	struct sockaddr_in *sin = (struct sockaddr_in*)&adr;
	ipnum_ip2a (htonl(sin->sin_addr.s_addr),buf);
}

/*
	Fetch the inteface configuration from the kernel.
	Return -1 if any error.
*/
int ifconfig_getinfo (
	const char *ifname,
	IFCONFIG_INFO &info)
{
	int ret = -1;
	memset (&info,0,sizeof(IFCONFIG_INFO));
	if (devlist_devexist(ifname)){
		int skfd = socket(AF_INET, SOCK_DGRAM, 0);
		if (skfd != -1){
			struct ifreq ifr;
			if (ifconfig_ioctl(skfd, ifname, SIOCGIFFLAGS, ifr) >= 0){
				info.flags = ifr.ifr_flags;
				if (ifconfig_ioctl(skfd,ifname,SIOCGIFADDR, ifr) >= 0){
					ifconfig_format (ifr.ifr_addr,info.ip_addr);
					if (ifconfig_ioctl(skfd,ifname, SIOCGIFDSTADDR, ifr) >= 0) {
						ifconfig_format (ifr.ifr_dstaddr,info.dst_addr);
						if (ifconfig_ioctl(skfd, ifname,SIOCGIFNETMASK, ifr) >= 0) {
							ifconfig_format (ifr.ifr_netmask,info.netmask);
							if (ifconfig_ioctl(skfd, ifname,SIOCGIFBRDADDR, ifr) >= 0) {
								ifconfig_format (ifr.ifr_broadaddr,info.bcast);
								ret = 0;
							}
						}
					}
				}
			}
		}
	}
	return ret;
}

#ifdef TEST

int main (int, char *argv[])
{
	IFCONFIG_INFO info;
	if (ifconfig_getinfo(argv[1],info) == -1){
		fprintf (stderr,"No info on network device %s\n",argv[1]);
	}else{
		printf ("Network devices %s\n",argv[1]);
		printf ("flags    : %04x\n",info.flags);
		printf ("addr     : %s\n",info.ip_addr);
		printf ("dst addr : %s\n",info.dst_addr);
		printf ("netmask  : %s\n",info.netmask);
	}
	return 0;
}

#endif


