# Lams.tcl Library
#
# Copyright (c) 1997 Mark Black
#
# Rebuild index using "auto_mkindex /home/mark/mat/lib *.tcl"
#
# This library contains the following functions:
#   AutoProbe  - Ping the hosts to see if they respond
#   MatHost    - Add/Delete Hosts from the MAT hosts file
#   MatUser    - Add/Delete MAt users, and set the permissions
#   Sudo       - Sudo update GUI
#   Groups     - Group update GUI
#   Hosts      - Hosts update GUI
#   hostUpdate - Host IP check, and update procedure
#   User       - User update GUI
#   useropt    - User options GUI
#   Fstab      - Mount GUI
#   fstype     - Window reconfigerer for Fstab 
#   DnsClient  - DNS Clint GUI
#   -------------- Task Procedures ----------------

set DEBUG 1

#
# AutoProbe - Ping the hosts to see if they respond
#             The main Host display canvas uses the hostname
#             as the tag, so use the tag to modify the hosts
#             icon.
#             Set currentProbe to 9999 to stop it
proc AutoProbe { } {
    global currentProbe
    if { $currentProbe != 9999 } {
	# Get name of host to probe
	global active_hosts
	set probeHost [ lindex [split [lindex $active_hosts $currentProbe ] ":" ] 0 ]
	puts stdout "Probing $probeHost"
	set dataline [lindex $active_hosts $currentProbe ]
	set pingval [pingHost $probeHost ]
	if { $pingval == 0 } {
	    # Ping failed
	    if { [lindex [split $dataline ":" ] 4 ] != 0 } {
		set new [string trimright $dataline {1} ]
		append new "0"
		set dataline $new
		set newlist [ lreplace $active_hosts $currentProbe $currentProbe $dataline ]
		set active_hosts $newlist
	    } 
	    global level
	    if { $level == 10 } {
		set hostoff [ image create photo -file "../lib/host_off.gif" ]
		.tp.host.canvas itemconfigure $probeHost -image $hostoff
	    }
	} else {
	    set hosttype [lindex [split $pingval ] 2 ]
	    if { [lindex [split $dataline ":" ] 1 ] != $hosttype } {
		set tmp [split $dataline ":" ]
		set dataline "[lindex $tmp 0 ]:$hosttype:[lindex $tmp 2 ]:[lindex $tmp 3 ]:[lindex $tmp 4 ]"
		set active_hosts [ lreplace $active_hosts $currentProbe $currentProbe $dataline ]
	    }
	    if { [lindex [split $dataline ":" ] 4 ] != 1 } {
		set new [string trimright $dataline {0} ]
		append new "1"
		set dataline $new
		set active_hosts [ lreplace $active_hosts $currentProbe $currentProbe $dataline ]
	    }
	}
	# Start another
	after 1000 { AutoProbe }
	incr currentProbe 1
	if { $currentProbe > [expr [llength $active_hosts ] - 1] } {
	    set currentProbe 9999
	    set fid [ open "../hosts" w ]
	    foreach line $active_hosts {
		puts $fid "$line"
	    }
	    close $fid
	}
    }
}

#
# MatHost - Procedure for updating the MAT host file
#           If an argument is given then modify that host    
# The hosts file is resident on the management station
#
proc MatHost { {edithost "-" } args } {
    set host_list {}
    if { [string length $args ] != 0 } {
	append edithost " $args"
    }
    if { [winfo exists .mathost ] == 0 } {
	toplevel .mathost -class Dialog
	wm title .mathost "MAT Hosts Editor"
	wm iconname .mathost "MAT Hosts Editor"
	entry .mathost.hidden1
	.mathost.hidden1 insert end $edithost
	frame .mathost.f1 -bd 2 -relief "raised"
	label .mathost.f1.l1 -text "MAT Host Name" 
	pack .mathost.f1.l1 -fill x
	frame .mathost.f1.f1 -bd 2 -relief "flat"
	label .mathost.f1.f1.l1 -text "MAT Host:"
	entry .mathost.f1.f1.e1 -width 10
	pack .mathost.f1.f1.l1 .mathost.f1.f1.e1 -fill both -side left -expand 1
	pack .mathost.f1.f1 -side top -fill both -expand 1
	
	frame .mathost.f2 -bd 2 -relief "raised"
	frame .mathost.f2.f0 -bd 0 -relief "flat"
	frame .mathost.f2.f1 -bd 0 -relief "flat"
	frame .mathost.f2.f2 -bd 0 -relief "flat"
	frame .mathost.f2.f3 -bd 0 -relief "flat"
	label .mathost.f2.f0.l1 -text "Services offered by Host:"
	pack .mathost.f2.f0.l1 -fill x
	checkbutton .mathost.f2.f1.ck1 -text "DNS Master" -onvalue 1 -offvalue 0 -variable DNSyes -selectcolor #000080
	checkbutton .mathost.f2.f1.ck2 -text "NIS Master" -onvalue 1 -offvalue 0 -variable NISyes -selectcolor #000080
	checkbutton .mathost.f2.f2.ck3 -text "NFS Server" -onvalue 1 -offvalue 0 -variable NFSyes -selectcolor #000080
	checkbutton .mathost.f2.f2.ck4 -text "Samba Server" -onvalue 1 -offvalue 0 -variable SAMBAyes -selectcolor #000080
	checkbutton .mathost.f2.f3.ck5 -text "Backup Host" -onvalue 1 -offvalue 0 -variable TAPEyes -selectcolor #000080
	pack .mathost.f2.f1.ck1 .mathost.f2.f1.ck2 -side left -anchor w  -expand 1
	pack .mathost.f2.f2.ck3 .mathost.f2.f2.ck4 -side left -anchor w -expand 1
	pack .mathost.f2.f3.ck5 -side left -anchor w -expand 1
	pack .mathost.f2.f0 .mathost.f2.f1 .mathost.f2.f2 .mathost.f2.f3 -side top -fill x

	frame .mathost.f5 -bd 2 -relief "flat"
	button .mathost.f5.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [ .mathost.f1.f1.e1 get ] != {} } {
		set cmd "[ .mathost.f1.f1.e1 get ]:::"
		if { $DNSyes == 1 } { append cmd "DNS," }
		if { $NISyes == 1 } { append cmd "NIS," }
		if { $NFSyes == 1 } { append cmd "NFS," }
		if { $SAMBAyes == 1 } { append cmd "SAMBA," }
		if { $TAPEyes == 1 } { append cmd "TAPE" }
		append cmd ":"
		global mb_hosts
		if { [ .mathost.hidden1 get ] == "-" } {
		    set fid [ open $mb_hosts a ]
		    puts $fid $cmd
		    close $fid
		} else {
		    # Modify host
		    set host_list {}
		    set fid [ open $mb_hosts r ]
		    while { [ gets $fid fout ] > 0 } {
			lappend host_list $fout
		    }
		    close $fid
		    set edit_host [ lindex [split [ .mathost.hidden1 get ] ":" ] 0 ]
		    # Write out new file
		    set fid [ open $mb_hosts w ]
		    foreach data $host_list {
			if { $edit_host == [lindex [ split $data ":" ] 0 ] } {
			    # replace this line
			    puts $fid $cmd
			} else {
			    puts $fid $data
			}
		    }
		    close $fid
		}
		ReadHosts
		fillCan 10
		repack 90
		destroy .mathost
	    }
	}
	button .mathost.f5.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .mathost
	}
	button .mathost.f5.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help mathost
	}
	pack .mathost.f5.b1 .mathost.f5.b2 .mathost.f5.b3 -side left -fill x 
	pack .mathost.f1 .mathost.f2 .mathost.f5 -fill both -expand 1
    } else {
	# Window exists
	.mathost.hidden1 delete 0 end
	.mathost.hidden1 insert end $edithost
    }

    #------------------- Routine GUTS ------------------------------
    .mathost.f2.f1.ck1 deselect
    .mathost.f2.f1.ck2 deselect
    .mathost.f2.f2.ck3 deselect
    .mathost.f2.f2.ck4 deselect
    .mathost.f2.f3.ck5 deselect
    if { $edithost != "-" } {
	.mathost.f1.f1.e1 delete 0 end
	.mathost.f1.f1.e1 insert end [ lindex [split $edithost ":" ] 0 ]
	foreach data [split [lindex [ split $edithost ":" ] 3 ] "," ] {
	    if { $data == "DNS" } {
		.mathost.f2.f1.ck1 select
	    } elseif { $data == "NIS" } {
		.mathost.f2.f1.ck2 select
	    } elseif { $data == "NFS" } {
		.mathost.f2.f2.ck3 select
	    } elseif { $data == "SAMBA" } {
		.mathost.f2.f2.ck4 select
	    } elseif { $data == "TAPE" } {
		.mathost.f2.f3.ck5 select
	    } 
	}
    }
}

#
# MatUser - Procedure for updating the MAT user/password file
#     
# USE MatUser CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc MatUser { currenthost currentline line args } {
    set host $currenthost
    if { [winfo exists .matuser ] == 0 } {
	toplevel .matuser -class Dialog
	wm title .matuser "MAT User Editor: $currenthost"
	wm iconname .matuser "MAT User Editor"
	entry .matuser.hidden1 
	entry .matuser.hidden2
	entry .matuser.hidden3
	# NOT GOOD
	.matuser.hidden1 insert end $currenthost
	.matuser.hidden2 insert end $currentline
	.matuser.hidden3 insert end $line
	frame .matuser.f1 -bd 2 -relief "raised"
	label .matuser.f1.l1 -text "Login:" -width 16
	entry .matuser.f1.e1 -width 16
	pack .matuser.f1.l1 .matuser.f1.e1 -fill x -side left -expand 1
	frame .matuser.f2 -bd 2 -relief "raised"
	label .matuser.f2.l1 -text "Password:" -width 16
	entry .matuser.f2.e1 -width 16 -show "*"
	pack .matuser.f2.l1 .matuser.f2.e1 -fill x -side left -expand 1

	frame .matuser.f25 -bd 2 -relief "raised"
	label .matuser.f25.l1 -text "Confirm Password:" -width 16
	entry .matuser.f25.e1 -width 16 -show "*"
	pack .matuser.f25.l1 .matuser.f25.e1 -fill x -side left -expand 1

	frame .matuser.f3 -bd 2 -relief "raised"
	frame .matuser.f3.f1 -bd 2 -relief "flat"

	frame .matuser.f3.f1.f1 -bd 0 -relief "flat"
	label .matuser.f3.f1.f1.l1 -text "Available"
	pack .matuser.f3.f1.f1.l1 -fill x -side top
	scrollbar .matuser.f3.f1.f1.vscroll -orient vertical -command ".matuser.f3.f1.f1.text yview" 
	listbox .matuser.f3.f1.f1.text -selectforeground white -selectbackground #000080 -yscrollcommand {.matuser.f3.f1.f1.vscroll set } -height 5 -width 20
	.matuser.f3.f1.f1.text config -cursor hand2
	pack .matuser.f3.f1.f1.vscroll -side left -fill y
	pack .matuser.f3.f1.f1.text -side left -expand 1 -fill both

	frame .matuser.f3.f1.f2 -bd 0 -relief "flat"
	label .matuser.f3.f1.f2.l1 -text "Joined"
	pack .matuser.f3.f1.f2.l1 -fill x -side top
	scrollbar .matuser.f3.f1.f2.vscroll -orient vertical -command ".matuser.f3.f1.f2.text yview" 
	listbox .matuser.f3.f1.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.matuser.f3.f1.f2.vscroll set } -height 5 -width 20
	.matuser.f3.f1.f2.text config -cursor hand2
	pack .matuser.f3.f1.f2.vscroll -side left -fill y
	pack .matuser.f3.f1.f2.text -side left -expand 1 -fill both
	pack .matuser.f3.f1.f1 .matuser.f3.f1.f2 -fill both -side left -expand 1

	frame .matuser.f3.f2 -bd 2 -relief "flat"
	button .matuser.f3.f2.b1 -text "Add" -command {
	    if { [.matuser.f3.f1.f1.text curselection ] != "" } {
		.matuser.f3.f1.f2.text insert end [.matuser.f3.f1.f1.text get [.matuser.f3.f1.f1.text curselection ]]
		.matuser.f3.f1.f1.text delete [.matuser.f3.f1.f1.text curselection ]
	    }
	}
	button .matuser.f3.f2.b2 -text "Delete" -command {
	    if { [.matuser.f3.f1.f2.text curselection ] != "" } {
		.matuser.f3.f1.f1.text insert end [.matuser.f3.f1.f2.text get [.matuser.f3.f1.f2.text curselection ]]
		.matuser.f3.f1.f2.text delete [.matuser.f3.f1.f2.text curselection ]
	    }
	}
	pack .matuser.f3.f2.b1 .matuser.f3.f2.b2 -fill x -side left -expand 1
	pack .matuser.f3.f1 .matuser.f3.f2 -fill both -side top -expand 1

	frame .matuser.f4 -bd 2 -relief "raised"
	button .matuser.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.matuser.f1.e1 get] != "" } {
		set fail 0
		if { [.matuser.hidden2 get ] == 9999 } {
		    # New MAT password entry
		    set cmd "add matpass "
		    if { [.matuser.f2.e1 get] == {} } {
			error 341
			set fail 1
		    } else {
			if { [.matuser.f2.e1 get] == [.matuser.f25.e1 get] } {
			    # Build command arguements
			    append cmd "[ string tolower [.matuser.f1.e1 get]]:[.matuser.f2.e1 get]:"
			} else {
			    error 342
			    .matuser.f2.e1 delete 0 end
			    .matuser.f25.e1 delete 0 end
			    focus .matuser.f2.e1
			    set fail 1
			}
		    }
		} else {
		    # Modify an old entry
		    set cmd "mod matpass "

		    if { [.matuser.f2.e1 get] == {} } {
			# Do not change password
			append cmd "[ string tolower [.matuser.f1.e1 get]]::"
		    } else {
			# Change the password
			if { [.matuser.f2.e1 get] == [.matuser.f25.e1 get] } {
			    # Build command arguements
			    append cmd "[ string tolower [.matuser.f1.e1 get]]:[.matuser.f2.e1 get]:"
			} else {
			    error 342
			    .matuser.f2.e1 delete 0 end
			    .matuser.f25.e1 delete 0 end
			    set fail 1
			}
		    }
		}
		if {$fail == 0 } {
		    append cmd [ list2Perms .matuser.f3.f1.f2.text $feature_list ]
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
			error 91
			puts stdout "ERROR:  Unable to update the MAT password file on $currenthost\n       Check the files permissions"
		    } else {
			set returndata [queryHost $currenthost "get matpass" 0 ]
			filltext $returndata .general.text 91
			destroy .matuser
		    }
		}
	    } else {
		error 340
	    }
	}
	button .matuser.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    set returndata [queryHost $currenthost "get matpass" 0 ]
	    filltext $returndata .general.text 91
	    destroy .matuser
	}
	button .matuser.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help sudo
	}
	pack .matuser.f4.b1 .matuser.f4.b2 .matuser.f4.b3 -side left -fill x 
	pack .matuser.f1 .matuser.f2 .matuser.f25 .matuser.f3 .matuser.f4 -fill both -side top -expand 1
    } else {
	# Window Exists already

    }
    # Initialize windows
    global feature_list_file
    global feature_list
    set feature_list {}
    set fid [ open $feature_list_file r ]
    while { [ gets $fid fout ] > 0 } {
	lappend feature_list $fout
    }
    close $fid
    for { set position 0 } { $position < [llength $feature_list] } { incr position 1 } {
	set tmp [lindex [split [lindex $feature_list $position ] ":" ] 1 ]
	.matuser.f3.f1.f1.text insert end $tmp
    }
    .matuser.f1.e1 delete 0 end
    .matuser.f2.e1 delete 0 end
    .matuser.f25.e1 delete 0 end
    .matuser.f3.f1.f2.text delete 0 end
    if { $currentline == 9999 } {
	# New entry


    } else {
	# Remove permission groups from the available box
	.matuser.f1.e1 insert end [lindex [split [.matuser.hidden3 get] ":" ] 0 ]
	foreach data [ perms2list [lindex [split [.matuser.hidden3 get] ":" ] 2 ] $feature_list ] {
	    .matuser.f3.f1.f2.text insert end $data
	    # Delete lines from other list
	    for { set position 0 } { $position < [.matuser.f3.f1.f1.text index end ] } { incr position 1 } {
		set tmp [ .matuser.f3.f1.f1.text get $position ]
		if { $tmp == $data } {
		    .matuser.f3.f1.f1.text delete $position
		}
	    }
	}
    }
    # ------------------ Bindings ----------------------------
    bind .matuser.f1.e1 <Return> {
	focus .matuser.f2.e1
    }
    bind .matuser.f2.e1 <Return> {
	focus .matuser.f25.e1
    }
    bind .matuser.f3.f1.f1.text <Double-1> {
	.matuser.f3.f2.b1 invoke
    }
    bind .matuser.f3.f1.f2.text <Double-1> {
	.matuser.f3.f2.b2 invoke
    }
}

#
# Convert the entries in a listbox to their
# numerical permission numbers
# Returns a permission string for the password file
proc list2Perms { window features } {
    set perms {}
    foreach permLine [ $window get 0 end ] {
	for { set position 0 } { $position < [llength $features] } { incr position 1 } {
	    set tmp [lindex [split [lindex $features $position ] ":" ] 1 ]
	    if { $tmp == $permLine } {
		append perms [lindex [split [lindex $features $position ] ":" ] 0 ]
	    }
	}
    }
    return $perms
}

#
# Convert a password perms field into a string list
# Returns a list with the matching strings for the permissions
proc perms2list { perms features } {
    set retlist {}
    foreach permVal [ split $perms {} ] {
	for { set position 0 } { $position < [llength $features] } { incr position 1 } {
	    set tmp [lindex [split [lindex $features $position ] ":" ] 0 ]
	    if { $tmp == $permVal } {
		lappend retlist [lindex [split [lindex $features $position ] ":" ] 1 ]
	    }
	}
    }
    return $retlist
}
#
# Sudoers update procedure
#
# USE Sudo CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc Sudo { currenthost currentline line args } {
    set host $currenthost
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    if { [winfo exists .sudo ] == 0 } {
	toplevel .sudo -class Dialog
	wm title .sudo "Groups Editor: $currenthost"
	wm iconname .sudo "Groups Editor"
	entry .sudo.hidden1 
	entry .sudo.hidden2
	# NOT GOOD
	.sudo.hidden1 insert end $currenthost
	.sudo.hidden2 insert end $currentline
	frame .sudo.f1 -bd 2 -relief "raised"
	label .sudo.f1.l1 -text "User:" -width 6
	entry .sudo.f1.e1 -width 16
	pack .sudo.f1.l1 .sudo.f1.e1 -fill both -side left -expand 1
	frame .sudo.f2 -bd 2 -relief "raised"
	label .sudo.f2.l1 -text "Access:" -width 6
	entry .sudo.f2.e1 -width 16
	pack .sudo.f2.l1 .sudo.f2.e1 -fill both -side left -expand 1
	
	frame .sudo.f4 -bd 2 -relief "raised"
	button .sudo.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.sudo.f2.e1 get] != "" } {
		if { [.sudo.f1.e1 get] != "" } {
		    set newline "[ string tolower [.sudo.f1.e1 get]]    [ string tolower [.sudo.f2.e1 get]]:"
		    puts stdout "Newline = $newline"
		    set currentline [.sudo.hidden2 get]
		    if { $currentline == 9999 } {
			set cmd "add sudo $newline"
		    } else {
			set cmd "mod sudo $currentline $newline"
		    }
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
			error 91
			puts stdout "ERROR:  Unable to update the /etc/group file on $currenthost\n       Check the files permissions"
		    } else {
			set returndata [queryHost $currenthost "get sudo" 0 ]
			filltext $returndata .general.text 11
			destroy .sudo
		    }
		}
	    }
	}
	button .sudo.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    set returndata [queryHost $currenthost "get sudo" 0 ]
	    filltext $returndata .general.text 11
	    destroy .sudo
	}
	button .sudo.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help sudo
	}
	pack .sudo.f4.b1 .sudo.f4.b2 .sudo.f4.b3 -side left -fill x 
	pack .sudo.f1 .sudo.f2 .sudo.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .sudo.f1.e1 <Return> {
	    focus .sudo.f2.e1
	}
    }
}



#
# Group update procedure
#
# USE Group CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc Groups { currenthost currentline line } {
    set host $currenthost
    if { [winfo exists .group ] == 0 } {
	toplevel .group -class Dialog
	wm title .group "Groups Editor: $currenthost"
	wm iconname .group "Groups Editor"
	entry .group.hidden1 
	entry .group.hidden2
	# NOT GOOD
	.group.hidden1 insert end $currenthost
	.group.hidden2 insert end $currentline
	frame .group.f1 -bd 2 -relief "raised"
	label .group.f1.l1 -text "Group:" -width 6
	entry .group.f1.e1 -width 10
	pack .group.f1.l1 .group.f1.e1 -fill both -side left -expand 1
	frame .group.f2 -bd 2 -relief "raised"
	label .group.f2.l1 -text "GID #:" -width 6
	entry .group.f2.e1 -width 10
	pack .group.f2.l1 .group.f2.e1 -fill both -side left -expand 1
	frame .group.f3 -bd 2 -relief "raised"
	label .group.f3.l1 -text "Users" 
	pack .group.f3.l1 -fill x
	frame .group.f3.f1 -bd 2 -relief "flat"
	entry .group.f3.f1.e1 -width 10
	button .group.f3.f1.b1 -text "Add" -command {
	    if { [.group.f3.f1.e1 get] != "" } {
		.group.f3.f2.text insert end [.group.f3.f1.e1 get]
	    }
	    .group.f3.f1.e1 delete 0 end
	}
	pack .group.f3.f1.e1 .group.f3.f1.b1 -fill both -side left -expand 1
	frame .group.f3.f2 -bd 2 -relief "flat"
	scrollbar .group.f3.f2.vscroll -orient vertical -command ".group.f3.f2.text yview" 
	listbox .group.f3.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.group.f3.f2.vscroll set } -height 5 -width 10
	.group.f3.f2.text config -cursor hand2
	pack .group.f3.f2.vscroll -side left -fill y
	pack .group.f3.f2.text -side left -expand 1 -fill both
	button .group.f3.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.group.f3.f2.text curselection ] != "" } {
		.group.f3.f2.text delete [.group.f3.f2.text curselection ]
	    }
	}
	pack .group.f3.f2.b1 -side left
	pack .group.f3.f1 .group.f3.f2 -side top -fill both -expand 1
	frame .group.f4 -bd 2 -relief "flat"
	button .group.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.group.f2.e1 get] > -2 || [.group.f2.e1 get] < 65536 } {
		if { [.group.f1.e1 get] != "" } {
		    set tmpline "[ string tolower [.group.f1.e1 get]]::[ string tolower [.group.f2.e1 get]]:"
		    foreach user [.group.f3.f2.text get 0 end] {
			append tmpline "[string tolower [string trimright $user]],"
		    }
		    set newline [string trimright $tmpline "," ] 
		    set currentline [.group.hidden2 get]
		    if { $currentline == 9999 } {
			set cmd "add group $newline"
		    } else {
			set cmd "mod group $currentline $newline"
		    }
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
			error 91
			puts stdout "ERROR:  Unable to update the /etc/group file on $currenthost\n       Check the files permissions"
		    } else {
			set returndata [queryHost $currenthost "get group" 0 ]
			filltext $returndata .general.text 2
			destroy .group
		    }
		}
	    }
	}
	button .group.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    set returndata [queryHost $currenthost "get group" 0 ]
	    filltext $returndata .general.text 2
	    destroy .group
	}
	button .group.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help groups
	}
	pack .group.f4.b1 .group.f4.b2 .group.f4.b3 -side left -fill x 
	pack .group.f1 .group.f2 .group.f3 .group.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .group.f1.e1 <Return> {
	    focus .group.f2.e1
	}
	bind .group.f2.e1 <Return> {
	    focus .group.f3.f1.e1
	}
	bind .group.f3.f1.e1 <Return> {
	    if { [.group.f3.f1.e1 get] != "" } {
		.group.f3.f2.text insert end [.group.f3.f1.e1 get]
	    }
	    .group.f3.f1.e1 delete 0 end
	}
    } else {
	.group.hidden1 delete 0 end
	.group.hidden2 delete 0 end
	.group.hidden1 insert end $currenthost
	.group.hidden2 insert end $currentline
	wm title .group "Groups Editor: $currenthost"
    }

    # Fill boxes with data
    if { $currentline != 9999 } {
	.group.f1.e1 delete 0 end
	.group.f2.e1 delete 0 end
	.group.f3.f2.text delete 0 end
	.group.f1.e1 insert end [lindex [split $line ":"] 0]
	.group.f2.e1 insert end [lindex [split $line ":"] 2]
	set tmp [lindex [split $line ":"] 3]
	foreach user [split $tmp ","] {
	    .group.f3.f2.text insert end $user
	}
	    
    }	
}


#
# Hosts Update procedure
#
# USE:  Hosts CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc Hosts { currenthost currentline line args } {
    set host $currenthost
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    if { [winfo exists .host ] == 0 } {
	toplevel .host -class Dialog
	wm title .host "Hosts Editor: $currenthost"
	wm iconname .host "Hosts Editor"
	entry .host.hidden1 
	entry .host.hidden2
	# NOT GOOD
	.host.hidden1 insert end $currenthost
	.host.hidden2 insert end $currentline
	frame .host.f1 -bd 2 -relief "raised"
	label .host.f1.label -text "IP address:" -width 12
	entry .host.f1.e1 -width 3
	label .host.f1.l1 -text "."
	entry .host.f1.e2 -width 3
	label .host.f1.l2 -text "."
	entry .host.f1.e3 -width 3
	label .host.f1.l3 -text "."
	entry .host.f1.e4 -width 3
	
	pack .host.f1.label .host.f1.e1 .host.f1.l1 .host.f1.e2 .host.f1.l2 .host.f1.e3 .host.f1.l3 .host.f1.e4 -fill both -side left -expand 1
	frame .host.f2 -bd 2 -relief "raised"
	label .host.f2.label -text "Host name:" -width 12
	entry .host.f2.entry 
	pack .host.f2.label .host.f2.entry -fill both -side left -expand 1
	frame .host.f3 -bd 2 -relief "raised"
	label .host.f3.label -text "Alias(es):" -width 12
	entry .host.f3.entry 
	pack .host.f3.label .host.f3.entry -fill x -side left -expand 1
	pack .host.f1 .host.f2 .host.f3 -fill both
	button .host.ok -text "Okay" -foreground red -activeforeground #a00000 -command {
	    hostUpdate [.host.hidden1 get] [.host.hidden2 get] [ .host.f1.e1 get] [ .host.f1.e2 get] [ .host.f1.e3 get] [ .host.f1.e4 get] [ .host.f2.entry get] [ .host.f3.entry get]
	    set returndata [queryHost $currenthost "get hosts" 0 ]
	    filltext $returndata .general.text 1
	    destroy .host
	}
	button .host.app -text "Apply" -foreground red -activeforeground #a00000 -command {
	    hostUpdate [.host.hidden1 get] [.host.hidden2 get] [ .host.f1.e1 get] [ .host.f1.e2 get] [ .host.f1.e3 get] [ .host.f1.e4 get] [ .host.f2.entry get] [ .host.f3.entry get]
	    .host.f1.e1 delete 0 end
	    .host.f1.e2 delete 0 end
	    .host.f1.e3 delete 0 end
	    .host.f1.e4 delete 0 end
	    .host.f2.entry delete 0 end
	    .host.f3.entry delete 0 end
	    .host.hidden2 delete 0 end
	    .host.hidden2 insert end 9999
	    set returndata [queryHost $currenthost "get hosts" 0 ]
	    filltext $returndata .general.text 1
	}
	button .host.can -text "Cancel"  -foreground blue -activeforeground #000080 -command {
	    destroy .host
	}
	button .host.help -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help hosts
	}
	pack .host.ok .host.app .host.can .host.help -fill x -side left -expand 1
	
	# -----------  Bindings for Hosts  -------------------------
	bind .host.f1.e1 <Return> {
	    focus .host.f1.e2 
	}
	bind .host.f1.e1 . {
	    after 250 {
		.host.f1.e1 delete [expr [string length [.host.f1.e1 get]] -1] end
	    }
	    focus .host.f1.e2 
	}
	bind .host.f1.e2 <Return> {
	    focus .host.f1.e3 
	}
	bind .host.f1.e2 . {
	    after 250 {
		.host.f1.e2 delete [expr [string length [.host.f1.e2 get]] -1] end
	    }
	    focus .host.f1.e3 
	}
	bind .host.f1.e3 <Return> {
	    focus .host.f1.e4 
	}
	bind .host.f1.e3 . {
	    after 250 {
		.host.f1.e3 delete [expr [string length [.host.f1.e3 get]] -1] end
	    }
	    focus .host.f1.e4 
	}
	bind .host.f1.e4 <Return> {
	    focus .host.f2.entry 
	}
	bind .host.f2.entry <Return> {
	    focus .host.f3.entry 
	}
	bind .host.f3.entry <Return> {
	    #	focus .host.f1.entry 
	}
    } else {
	wm title .host "Hosts Editor: $currenthost"
	.host.hidden1 delete 0 end
	.host.hidden2 delete 0 end
	.host.hidden1 insert end $currenthost
	.host.hidden2 insert end $currentline
    }

    if { $currentline != "9999" && [lindex [split $line {} ] 0] != "#" } {
	set tmp [lindex [split $line " 	"] 0]
	.host.f1.e1 delete 0 end
	.host.f1.e2 delete 0 end
	.host.f1.e3 delete 0 end
	.host.f1.e4 delete 0 end
	.host.f2.entry delete 0 end
	.host.f3.entry delete 0 end
	.host.f1.e1 insert end [lindex [split $tmp "."] 0]
	.host.f1.e2 insert end [lindex [split $tmp "."] 1]
	.host.f1.e3 insert end [lindex [split $tmp "."] 2]
	.host.f1.e4 insert end [lindex [split $tmp "."] 3]
	.host.f2.entry insert end [lindex [split $line " 	"] 1]
#	.host.f3.entry insert end [lindex [split $line " 	"] 2]
	for { set position 2 } { $position < [llength [split $line " 	"] ] } { incr position 1 } {
	    .host.f3.entry insert end [lindex [split $line " 	"] $position ]
	    .host.f3.entry insert end { }
	}
    }
}


	    
#
# ---- Procedure to build the update line
#
proc hostUpdate { currenthost lineNo ip1 ip2 ip3 ip4  name alias } {
    if { $ip1 > 255 || $ip1 < 0 } {
	error 50
	return -1
    } elseif { $ip2 > 255 || $ip2 < 0 } {
	error 50
	return -1
    } elseif { $ip3 > 255 || $ip3 < 0 } {
	error 50
	return -1
    } elseif { $ip4 > 255 || $ip4 < 0 } {
	error 50
	return -1
    } else { 
	if { $lineNo != 9999 } {
	    set cmd "mod hosts $lineNo $ip1.$ip2.$ip3.$ip4	[string tolower $name ]	[string tolower $alias]"
	} else {
	    set cmd "add hosts $ip1.$ip2.$ip3.$ip4	[string tolower $name ]	[string tolower $alias]"
	}
	set returnval [queryHost $currenthost $cmd 1 ]
	if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
	    error 51
	    puts stdout "ERROR:  Unable to update the /etc/hosts file on $currenthost\n        Check the files permissions"
	}
    }
}


#
# User Update procedure
#
# USE:  User  CurrentUser CurrentLine SelectedData
#
#             CurrentUser = User to update
#             CurrentLine = Line number of selected line  (9999 = no data)
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc User { currenthost currentline line args } {
    global userkey
    # This is the minimum alowable gid.  It protects the more sensiteve groups from less sensitive administerators
    set mingid 10
    # This is the minimum allowable User ID
    set minuid 100
    set homedir "/home"
    # NOTE:   modify apply button to reflect any changes here
    set shell "/bin/sh"
    set key {}
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    set host $currenthost
    if { [winfo exists .opt ] != 0 } {
	destroy .opt
    }
    if { [winfo exists .user ] == 0 } {
	toplevel .user -class Dialog
	wm title .user "User Editor: $currenthost"
	wm iconname .user "User Editor"
	entry .user.hidden1 
	entry .user.hidden2
	entry .user.hidden3
	entry .user.hidpass
	entry .user.hiduid
	entry .user.hidgid
	entry .user.hidshell
	entry .user.hiddir
	listbox .user.hiddenlist
	# NOT GOOD
	.user.hidden1 insert end $currenthost
	.user.hidden2 insert end $currentline
	.user.hidden3 insert end $line
	.user.hidpass delete 0 end
	frame .user.f1 -bd 2 -relief "raised"
	label .user.f1.label -text "First Name" -width 10
	entry .user.f1.e1 -width 16
	pack .user.f1.label .user.f1.e1 -fill both -side left -expand 1
	frame .user.f2 -bd 2 -relief "raised"
	label .user.f2.label -text "Surname" -width 10
	entry .user.f2.entry -width 16
	pack .user.f2.label .user.f2.entry -fill both -side left -expand 1
	frame .user.f3 -bd 2 -relief "raised"
	label .user.f3.label -text "Login" -width 10
	entry .user.f3.entry -width 16
	pack .user.f3.label .user.f3.entry -fill both -side left -expand 1
	frame .user.f4 -bd 2 -relief "raised"
	label .user.f4.l1 -text "Primary Group"
	pack .user.f4.l1 -fill x
	scrollbar .user.f4.vscroll -orient vertical -command ".user.f4.text yview" 
	listbox .user.f4.text -selectforeground white -selectbackground #000080 -yscrollcommand {.user.f4.vscroll set } -height 5 -width 10
	.user.f4.text config -cursor hand2
	pack .user.f4.vscroll -side left -fill y
	pack .user.f4.text -side left -expand 1 -fill both

	frame .user.f5 -bd 2 -relief "raised"
	button .user.f5.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    global userkey
	    set flag 0
	    if { [.user.f4.text curselection ] != "" } {
		set gid [lindex [split [.user.hiddenlist get [lindex $userkey [ .user.f4.text curselection ]]] ":" ] 2]
	    } else {
		set gid [ .user.hidgid get ]
	    }
	    if { [.user.hiddir get] == "" } {
		.user.hiddir insert end "/home/[.user.f3.entry get]"
	    }
	    if { [.user.f2.entry get] == "" } {
#		set newline "[.user.f3.entry get]:[.user.hidpass get]:[.user.hiduid get]:[.user.hidgid get]:[.user.f1.e1 get]:[.user.hiddir get]:[.user.hidshell get]"
set newline "[.user.f3.entry get]:[.user.hidpass get]:[.user.hiduid get]:$gid:[.user.f1.e1 get]:[.user.hiddir get]:[.user.hidshell get]"
	    } else {
#		set newline "[.user.f3.entry get]:[.user.hidpass get]:[.user.hiduid get]:[.user.hidgid get]:[.user.f1.e1 get] [.user.f2.entry get]:[.user.hiddir get]:[.user.hidshell get]"
		set newline "[.user.f3.entry get]:[.user.hidpass get]:[.user.hiduid get]:$gid:[.user.f1.e1 get] [.user.f2.entry get]:[.user.hiddir get]:[.user.hidshell get]"
	    }
	    if { [.user.hidden2 get] == 9999 } {
		set cmd "add users $newline"
	    } else {
		set cmd "mod users [.user.hidden2 get] $newline"
	    }
	    set returnval [queryHost $currenthost $cmd 1 ]
	    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
		error 91
		puts stdout "ERROR:  Unable to update the /etc/passwd file on $currenthost\n       Check the file and directory permissions"
	    }
	    # Close window
	    repack 5
	    destroy .user
	}

	button .user.f5.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    if { [winfo exists .opt ] != 0 } {
		destroy .opt
	    }
	    destroy .user
	}
	button .user.f5.b3 -text "Options" -foreground #008000 -command {
	    useropt 
	}
	button .user.f5.b4 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help users
	}
	pack .user.f5.b1 .user.f5.b2 .user.f5.b3 .user.f5.b4 -fill x -side left -expand 1

	frame .user.f6 -bd 2 -relief "raised"
	button .user.f6.b1 -text "Lock Account" -foreground red -activeforeground #a00000 -command {
	    .user.hidpass delete 0 end
	    .user.hidpass insert end "LOCK"
	}
	button .user.f6.b2 -text "Unlock Account" -foreground red -activeforeground #a00000 -command {
	    .user.hidpass delete 0 end
	    .user.hidpass insert end "UNLOCK"
	}
	pack .user.f6.b1 .user.f6.b2 -fill x -side left -expand 1

	pack .user.f1 .user.f2 .user.f3 .user.f4 .user.f5 .user.f6 -fill x -side top -expand 1

	bind .user.f1.e1 <Return> {
	    focus .user.f2.entry 
	}
	bind .user.f2.entry <Return> {
	    focus .user.f3.entry
	    if { [ .user.hidden2 get ] != 9999 } {
		.user.f3.entry insert end [ string tolower [ lindex [ split [ .user.f1.e1 get ] "" ] 0 ] ]
		set splt_name [ split [ .user.f2.entry get ] "" ]
		set max [ llength $splt_name ]
		if { $max > 0 } {
		    if { $max >= 7 } {
			set max 7
		    }
		    set i 0
		    while { $i < $max } {
			.user.f3.entry delete 0 end
			.user.f3.entry insert end [ string tolower [ lindex $splt_name $i ] ]
			incr i
		    }
		}
	    }
	}
	bind .user.f3.entry <Return> {
	    focus .user.f1.e1 
	}
    } else {
	# --- Window open already ---
	wm title .user "User Editor: $currenthost"
	.user.hidden1 delete 0 end
	.user.hidden2 delete 0 end
	.user.hidden3 delete 0 end
	.user.hidden1 insert end $currenthost
	.user.hidden2 insert end $currentline
	.user.hidden3 insert end $line
    }
    .user.f4.text delete 0 end
    set userkey {}
    set cnt1 0
    .user.hidpass delete 0 end
    .user.hidpass insert end "PASSWORD"
    .user.hiduid delete 0 end
    .user.hiduid insert end "AUTO"
    .user.hidgid delete 0 end
    .user.hidshell delete 0 end
    .user.hidshell insert end $shell
    .user.hiddir delete 0 end
    
    set index 0
    set returndata [queryHost $currenthost "get group" 0 ]
    for { set position 0 } { $position < [llength $returndata] } { incr position 1 } {
	set tmp [string trimright [lindex $returndata $position ] ]
	# Maintain an actual list of the file in a hidden list
	.user.hiddenlist insert end $tmp
	set tmp2 [split $tmp ":" ]
	if { [lindex $tmp2 2 ] >= $mingid } {
	    set gidnum [lindex $tmp2 2 ]
	    set gid2index($gidnum) $index
	    incr index 1
	    .user.f4.text insert end [lindex $tmp2 0 ]
	    lappend userkey $cnt1
	}
	incr cnt1 1
    }
    if { $currentline != 9999 } {
	#####   Set values from the supplied list
	.user.hidpass delete 0 end
	.user.hidpass insert end [lindex [split $line ":"] 1]
	.user.hiduid delete 0 end
	.user.hiduid insert end [lindex [split [.user.hidden3 get] ":"] 2]
	.user.hidshell delete 0 end
	.user.hidshell insert end [lindex [split [.user.hidden3 get] ":"] 6]
	.user.hiddir delete 0 end
	.user.hiddir insert end [lindex [split [.user.hidden3 get] ":"] 5]
	.user.f1.e1 delete 0 end
	.user.f2.entry delete 0 end
	.user.f3.entry delete 0 end
	.user.f1.e1 insert end [lindex [split [lindex [split $line ":"] 4] " "] 0]
	.user.f2.entry insert end [lindex [split [lindex [split $line ":"] 4] " "] 1]
	.user.f3.entry insert end [lindex [split $line ":"] 0]
	#####   Find and select the appropriate gid
	set defgid [lindex [split $line ":"] 3]
	.user.hidgid delete 0 end
	.user.hidgid insert end $defgid
	focus .user.f4.text
	catch ".user.f4.text selection set $gid2index($defgid)" fail
	if { $fail == {} } {
	    .user.f4.text see $gid2index($defgid)
	    .user.f4.text activate $gid2index($defgid)
	} else {
	    # GID not listed in group file
	    puts stdout "GID not in group file.  Failed with: $fail"
	    error 310
	    
	}
    }
}

#
# ------- User Options Menu -----------
#
proc useropt {} {
    if { [winfo exists .opt ] == 0 } {
	toplevel .opt -class Dialog
	wm title .opt "Options for: [ .user.f3.entry get ]"
	wm iconname .opt "Options"
	frame .opt.f1 -bd 2 -relief "raised"
	label .opt.f1.label -text "Password" -width 9
	entry .opt.f1.e1 -width 16 -show "*"
	label .opt.f1.label2 -text "Confirm" -width 9
	entry .opt.f1.e2 -width 16 -show "*"
	pack .opt.f1.label .opt.f1.e1 .opt.f1.label2 .opt.f1.e2 -fill both -side left -expand 1
	frame .opt.f2 -bd 2 -relief "raised"
	label .opt.f2.label -text "UID" -width 8
	set uid "a"
	radiobutton .opt.f2.rb1 -text "Auto" -selectcolor #000080 -variable uid -value "1" -command {
	    .opt.f2.e1 delete 0 end
	    .user.hiduid delete 0 end
	    .opt.f2.e1 insert end "AUTO"
	    .user.hiduid insert end "AUTO"
	    .opt.f2.e1 configure -state disabled
	}
	radiobutton .opt.f2.rb2 -text "Select" -selectcolor #000080 -variable uid -value 0 -command {
	    .opt.f2.e1 configure -state normal
	    .user.hiduid delete 0 end
	    .opt.f2.e1 delete 0 end
	    focus .opt.f2.e1
	}
	entry .opt.f2.e1 -width 6
	pack .opt.f2.label .opt.f2.rb1 .opt.f2.rb2 .opt.f2.e1 -side left -fill both -expand 1
	frame .opt.f3 -bd 2 -relief "raised"
	label .opt.f3.label -text "Shell" -width 10
	entry .opt.f3.entry -width 24
	pack .opt.f3.label .opt.f3.entry -fill both -side left -expand 1
	frame .opt.f4 -bd 2 -relief "raised"
	label .opt.f4.label -text "Home Dir" -width 10
	entry .opt.f4.entry -width 24
	pack .opt.f4.label .opt.f4.entry -fill both -side left -expand 1
	frame .opt.f5 -bd 2 -relief "raised" 
	button .opt.f5.b1 -text "OK" -foreground red -activeforeground #a00000 -command {
	    if { [.user.hidden2 get] == 9999 } {
		#####   New Password
		if { [.opt.f1.e1 get] != "" } {
		    if { [.opt.f1.e1 get] != [.opt.f1.e2 get] } {
			.opt.f1.e1 delete 0 end
			.opt.f1.e2 delete 0 end
			focus .opt.f1.e1
			error 102
		    } else {
			.user.hidpass delete 0 end
			.user.hidpass insert end [.opt.f1.e1 get]
			.user.hiduid delete 0 end
			.user.hiduid insert end [.opt.f2.e1 get]
			.user.hiddir delete 0 end
			.user.hiddir insert end [.opt.f4.entry get]
			.user.hidshell delete 0 end
			.user.hidshell insert end [.opt.f3.entry get]
			destroy .opt
		    }
		} else {
		    .user.hidshell delete 0 end
		    .user.hidshell insert end [.opt.f3.entry get]
		    .user.hiduid delete 0 end
		    .user.hiduid insert end [.opt.f2.e1 get]
		    .user.hiddir delete 0 end
		    .user.hiddir insert end [.opt.f4.entry get]
		    destroy .opt
		}
	    } else {
		#####  Old Password
		if { [.opt.f1.e1 get] != "" } {
		    #####  Update password
		    if { [.opt.f1.e1 get] != [.opt.f1.e2 get] } {
			.opt.f1.e1 delete 0 end
			.opt.f1.e2 delete 0 end
			focus .opt.f1.e1
			error 102
		    } else {
			.user.hidpass delete 0 end
			.user.hidpass insert end [.opt.f1.e1 get]
			.user.hiduid delete 0 end
			.user.hiduid insert end [.opt.f2.e1 get]
			.user.hiddir delete 0 end
			.user.hiddir insert end [.opt.f4.entry get]
			.user.hidshell delete 0 end
			.user.hidshell insert end [.opt.f3.entry get]
			destroy .opt
		    }
		} else {
		    .user.hidshell delete 0 end
		    .user.hidshell insert end [.opt.f3.entry get]
		    .user.hiduid delete 0 end
		    .user.hiduid insert end [.opt.f2.e1 get]
		    .user.hiddir delete 0 end
		    .user.hiddir insert end [.opt.f4.entry get]
		    destroy .opt
		}  
	    }
	}
	button .opt.f5.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .opt
	}
	button .opt.f5.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help users
	}
	pack .opt.f5.b1 .opt.f5.b2 -side left -anchor w 
	pack .opt.f1 .opt.f2 .opt.f3 .opt.f4 .opt.f5 -side top -fill both -expand 1
	
    } else {
	for { set entry 0 } { $entry < $num } { incr entry 1 } {
	    destroy .opt.f3.$entry
	}
    }
    #####  Fill the Entrys with the appropriate values
    .opt.f4.entry delete 0 end
    .user.hiddir delete 0 end
    .user.hiduid delete 0 end
    .user.hidshell delete 0 end
    .opt.f2.e1 delete 0 end
    if { [.user.hidden2 get] != 9999 } {
	#####  Set the Appropriate values
	##  Home Dir
	.opt.f4.entry insert end [lindex [split [.user.hidden3 get] ":"] 5]
	.user.hiddir insert end [lindex [split [.user.hidden3 get] ":"] 5]
	##  UID
	.opt.f2.e1 insert end [lindex [split [.user.hidden3 get] ":"] 2]
	.user.hiduid insert end [lindex [split [.user.hidden3 get] ":"] 2]
	.opt.f2.rb2 select
	## SHell 
	.opt.f3.entry insert end [lindex [split [.user.hidden3 get] ":"] 6]
	.user.hidshell insert end [lindex [split [.user.hidden3 get] ":"] 6]
    } else {
	.opt.f4.entry delete 0 end
	.opt.f4.entry insert end "/home/[.user.f3.entry get]"
	.opt.f3.entry delete 0 end
	.opt.f3.entry insert end "/bin/sh"
	.opt.f2.rb1 invoke
    }
    if { [.opt.f2.e1 get] != "AUTO" } {
	set uid 1000
    }
#-------------  Support for the Shell's file -------------------------------
#    set returnval [queryHost [.user.hidden1 get] "get shells" 0 ]
#    set num 0 
#    set tmpline [lindex [split [.user.hidden3 get] ":"] 6]
#    foreach line $returnval {
#	if { [lindex [split $line {} ] 0 ] == "/" } {
#	    radiobutton .opt.f3.$num -text "$line" -selectcolor #000080 -variable shell -value $line -command "
#	        .user.hidshell delete 0 end
#		.user.hidshell insert end $line 
#	    "
#	    if {$line == $tmpline } {
#		set shell $line
#		.opt.f3.$num select
#		.user.hidshell delete 0 end
#		.user.hidshell insert end $line
#	    }
#	    pack .opt.f3.$num -side left
#	    incr num 1
#	}
#    }
    # -----------------  Bindings Option Menu  ----------------------------
    bind .opt.f1.e1 <Return> {
	focus .opt.f1.e2
    }
    bind .opt.f1.e2 <Return> {
	if { [.opt.f1.e1 get] != [.opt.f1.e2 get] } {
	    .opt.f1.e1 delete 0 end
	    .opt.f1.e2 delete 0 end
	    focus .opt.f1.e1
	    error 102
	} else {
	    focus .opt.f4.entry
	}
    }
}




#
# Fstab update procedure
#
# USE Fstab CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc Fstab { currenthost currentline line args } {
    global DEBUG
    set host $currenthost
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    if { [winfo exists .fstab ] == 0 } {
	toplevel .fstab -class Dialog
	wm title .fstab "Fstab Editor: $currenthost"
	wm iconname .fstab "Fstab Editor"
	entry .fstab.hidden1 
	entry .fstab.hidden2
	# NOT GOOD
	.fstab.hidden1 insert end $currenthost
	.fstab.hidden2 insert end $currentline
	frame .fstab.f0 -bd 2 -relief "raised"
	label .fstab.f0.l1 -text "NFS Server:" -width 16
	entry .fstab.f0.e1 -width 16
	pack .fstab.f0.l1 .fstab.f0.e1 -fill both -side left -expand 1
	frame .fstab.f1 -bd 2 -relief "raised"
	label .fstab.f1.l1 -text "Device:" -width 16
	entry .fstab.f1.e1 -width 16
	.fstab.f1.e1 insert end "/dev/"
	pack .fstab.f1.l1 .fstab.f1.e1 -fill both -side left -expand 1
	frame .fstab.f2 -bd 2 -relief "raised"
	label .fstab.f2.l1 -text "Mount point:" -width 16
	entry .fstab.f2.e1 -width 16
	pack .fstab.f2.l1 .fstab.f2.e1 -fill both -side left -expand 1
	frame .fstab.f3 -bd 2 -relief "raised"
	label .fstab.f3.l1 -text "Type of Filesystem" -width 16
	pack .fstab.f3.l1 -fill x -side top
	radiobutton .fstab.f3.rb1 -text "Ext2" -selectcolor #000080 -variable ftype -value "ext2" -command { fstype 0 }
	radiobutton .fstab.f3.rb2 -text "DOS" -selectcolor #000080 -variable ftype -value "msdos" -command { fstype 0 }
	radiobutton .fstab.f3.rb3 -text "Win95" -selectcolor #000080 -variable ftype -value "vfat" -command { fstype 0 }
	radiobutton .fstab.f3.rb4 -text "NFS" -selectcolor #000080 -variable ftype -value "nfs" -command { fstype "nfs" }
	radiobutton .fstab.f3.rb5 -text "CDROM" -selectcolor #000080 -variable ftype -value "iso9660" -command { 
	    fstype 0
	    .fstab.f4.f0.rb2 select
	}
	pack .fstab.f3.rb4 .fstab.f3.rb1 .fstab.f3.rb2 .fstab.f3.rb3 .fstab.f3.rb5 -fill x -side left
	frame .fstab.f4 -bd 2 -relief "raised"
	label .fstab.f4.l1 -text "Mount Options" -width 6
	pack .fstab.f4.l1 -fill x -side top
	frame .fstab.f4.f0 -bd 0
	checkbutton .fstab.f4.f0.ck1 -text "Defaults" -selectcolor #000080 -variable dopt -onvalue 1 -offvalue 0 -command {
	    if { $dopt == 1 } {
		# Default options selected
		pack forget .fstab.f4.f1
		pack forget .fstab.f4.f2
	    } else {
		# NON default options selected
		pack .fstab.f4.f1 .fstab.f4.f2 -fill x -side top
	    }
	}
	
	radiobutton .fstab.f4.f0.rb1 -text "rw" -selectcolor #a00000 -variable ropt -value rw
	radiobutton .fstab.f4.f0.rb2 -text "ro" -selectcolor #a00000 -variable ropt -value ro
	pack .fstab.f4.f0.ck1 .fstab.f4.f0.rb1 .fstab.f4.f0.rb2 -fill x -side left
	frame .fstab.f4.f1 -bd 0
	radiobutton .fstab.f4.f1.rb5 -text "suid" -selectcolor #008000 -variable sopt -value suid
	radiobutton .fstab.f4.f1.rb6 -text "nosuid" -selectcolor #008000 -variable sopt -value nosuid
	radiobutton .fstab.f4.f1.rb7 -text "exec" -selectcolor #a00000 -variable opt -value exec
	radiobutton .fstab.f4.f1.rb8 -text "noexec" -selectcolor #a00000 -variable opt -value noexec
	frame .fstab.f4.f2 -bd 2 -relief "flat"
	label .fstab.f4.f2.l1 -text "Other Options:" -width 16
	entry .fstab.f4.f2.e1 -width 16
	pack .fstab.f4.f2.l1 .fstab.f4.f2.e1 -fill both -side left -expand 1
	pack .fstab.f4.f1.rb5 .fstab.f4.f1.rb6 .fstab.f4.f1.rb7 .fstab.f4.f1.rb8 -fill x -side left
	pack .fstab.f4.f0 -fill x -side top
	frame .fstab.f5 -bd 2 -relief "raised"
	button .fstab.f5.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { $ftype == "nfs" } {
		if { [ .fstab.f0.e1 get ] != {} } {
		    set outline "[ .fstab.f0.e1 get ]:[.fstab.f1.e1 get]\t[.fstab.f2.e1 get]\t$ftype\t"
		} else {
		    error 181
		}
	    } else {
		set outline "[.fstab.f1.e1 get]\t[.fstab.f2.e1 get]\t$ftype\t"
	    }
	    if { $dopt == 1 } {
		append outline "defaults,$ropt"
	    } else {
		append outline "$ropt"
		if { $sopt == "suid" | $sopt == "nosuid" } {
		    append outline ",$sopt"
		}
		if { $opt == "exec" | $opt == "noexec" } {
		    append outline ",$opt"
		}
		if { [.fstab.f4.f2.e1 get] != "" } {
		    append outline ",[.fstab.f4.f2.e1 get]"
		}
	    }
	    if { $ftype == "nfs" } {
		append outline "\t0\t0"
	    } else {
		append outline "\t2\t2"
	    }
	    if { [.fstab.hidden2 get] == 9999 } {
		set cmd "add fstab $outline"
	    } else {
		set cmd "mod fstab [.fstab.hidden2 get] $outline"
	    }
	    set returnval [queryHost [.fstab.hidden1 get] $cmd 1 ]
	    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
		error 182
		puts stdout "ERROR:  Unable to update the /etc/fstab file on [.fstab.hidden1 get]\n       Check the files permissions"
	    } else {
		set returndata [queryHost $currenthost "get fstab" 0 ]
		filltext $returndata .general.text 3
		destroy .fstab
	    }
	}
	button .fstab.f5.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .fstab
	}
	button .fstab.f5.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help fstab
	}
	pack .fstab.f5.b1 .fstab.f5.b2 .fstab.f5.b3 -side left -anchor w -fill x
	pack .fstab.f3 .fstab.f1 .fstab.f2 .fstab.f4 .fstab.f5 -side top -expand 1 -fill both
    } else {
	wm title .fstab "Fstab Editor: $currenthost"
	.fstab.hidden1 delete 0 end
	.fstab.hidden2 delete 0 end
	.fstab.hidden1 insert end $currenthost
	.fstab.hidden2 insert end $currentline
    }
    # ------------- Bindings ------------------
    bind .fstab.f0.e1 <Return> {
	focus .fstab.f1.e1
    }
    bind .fstab.f1.e1 <Return> {
	focus .fstab.f2.e1
    }
    # ---------  Initialize  ----------------
    set dopt 1
    .fstab.f4.f0.ck1 invoke
    .fstab.f4.f0.rb2 select
    .fstab.f3.rb1 invoke
    if { $currentline != 9999 } {
	# regexp -nocase {^([a-z0-9:/]+)\ *\t*([a-z0-9/]+)\ *\t*([a-z]+)\ *\t*([a-z]+)\ *\t*(.+)\ *\t*(.+)} $line match mntWhat mntWhere mntType mntOpt mntPass mntPass2
	set mntWhat [lindex $line 0 ]
	set mntWhere [lindex $line 1 ]
	set mntType [lindex $line 2 ]
	set mntOpt [lindex $line 3 ]
	set mntPass [lindex $line 4 ]
	set mntPass2 [lindex $line 5 ]
	if $DEBUG { puts stdout "line = $line" } 
	if $DEBUG { puts stdout "mntWhat = $mntWhat" }
	if $DEBUG { puts stdout "mntWhere = $mntWhere" }
	if $DEBUG { puts stdout "mntType = $mntType" } 
	if $DEBUG { puts stdout "mntOpt = $mntOpt" }  
	if $DEBUG { puts stdout "mntPass = $mntPass" }  
	if $DEBUG { puts stdout "mntPass2 = $mntPass2" }  
	if { $mntType == "nfs" } {	
	    set ftype "nfs"
	    .fstab.f3.rb4 invoke
	} elseif { $mntType == "vfat" } {
	    set ftype "vfat"
	    .fstab.f3.rb3 invoke
	} elseif { $mntType == "msdos" } {
	    set ftype "msdos"
	    .fstab.f3.rb2 invoke
	} elseif { $mntType == "ext2" } {
	    set ftype "ext2"
	    .fstab.f3.rb1 invoke
	} elseif { $mntType == "iso9660" } {
	    set ftype "iso9660"
	    .fstab.f3.rb5 invoke
	} else {
	    error 180 
	}
	# Flag for default settings
	set flag 0
	# Flag indicating other options
	set oflag 0
	foreach val [ split $mntOpt {, } ] {
	    if { $val == "defaults" } {	
		set dopt 1
		.fstab.f4.f0.ck1 deselect
		.fstab.f4.f0.ck1 invoke
		set ropt $val
		.fstab.f4.f0.rb1 invoke
		# Use flag to indicate that this is non-default, so change view
		set flag 1
	    } elseif { $val == "rw" } {
		set ropt $val
		.fstab.f4.f0.rb1 invoke
	    } elseif { $val == "ro" } {	
		set ropt $val
		.fstab.f4.f0.rb2 invoke
	    } elseif { $val == "exec" } {
		set opt $val
		.fstab.f4.f1.rb7 invoke
	    } elseif { $val == "noexec" } {	
		set opt $val
		.fstab.f4.f1.rb8 invoke
	    } elseif { $val == "suid" } {
		set sopt $val
		.fstab.f4.f1.rb5 invoke
	    } elseif { $val == "nosuid" } {	
		set sopt $val
		.fstab.f4.f1.rb6 invoke
	    } else {
		if { $oflag == 0 } {
		    .fstab.f4.f2.e1 delete 0 end
		    .fstab.f4.f2.e1 insert end $val
		    set oflag 1
		} else {
		    .fstab.f4.f2.e1 insert end ",$val"
		}
	    }
	}
	if { $flag == 0 } {
	    .fstab.f4.f0.ck1 select
	    .fstab.f4.f0.ck1 invoke
	}
	.fstab.f2.e1 delete 0 end
	.fstab.f2.e1 insert end $mntWhere
	if [ regexp {^/} $mntWhat ] {
	    .fstab.f1.e1 delete 0 end
	    .fstab.f1.e1 insert end $mntWhat
	} else {
	    regexp -nocase {^([a-z0-9]+):(.+)} $mntWhat match mntServer tmpMntWhat
	    set mntWhat $tmpMntWhat
	    if $DEBUG { puts stdout "mntServer = $mntServer" }
	    if $DEBUG { puts stdout "mntWhat = $mntWhat" }
	    .fstab.f1.e1 delete 0 end
	    .fstab.f1.e1 insert end $mntWhat
	    .fstab.f0.e1 delete 0 end
	    .fstab.f0.e1 insert end $mntServer
	    
	}
    }

}

proc fstype { type } {
    if {$type == "nfs"} {
	.fstab.f1.l1 configure -text "Remote directory:"
	.fstab.f1.e1 delete 0 end
	pack forget .fstab.f1 
	pack forget .fstab.f2 
	pack forget .fstab.f4
	pack forget .fstab.f5
	pack .fstab.f0 .fstab.f1 .fstab.f2 .fstab.f4 .fstab.f5 -side top -expand 1 -fill both
    } else {
	.fstab.f1.l1 configure -text "Device:"
	.fstab.f1.e1 delete 0 end
	.fstab.f1.e1 insert end "/dev/"
	pack forget .fstab.f0
	pack forget .fstab.f1 
	pack forget .fstab.f2 
	pack forget .fstab.f4
	pack forget .fstab.f5
	pack .fstab.f1 .fstab.f2 .fstab.f4 .fstab.f5 -side top -expand 1 -fill both
    }
}

#
# DnsClient update procedure
#
# USE DnsClient CurrentHost CurrentLine SelectedData
#               CurrentHost = host to update
#               CurrentLine = Not Used
#               SelectedData = Not used
# NOTE: Data passed to this routine will be ignored.  Only the data from the client will be used
#
proc DnsClient { currenthost currentline line } {
    set host $currenthost
    if { [winfo exists .dnsC ] == 0 } {
	toplevel .dnsC -class Dialog
	wm title .dnsC "DNS Client Editor: $currenthost"
	wm iconname .dnsC "Groups Editor"
	entry .dnsC.hidden1 
	entry .dnsC.hidden2
	# NOT GOOD
	.dnsC.hidden1 insert end $currenthost
	frame .dnsC.f1 -bd 2 -relief "raised"
	label .dnsC.f1.l1 -text "Domain:" -width 6
	entry .dnsC.f1.e1 -width 20
	pack .dnsC.f1.l1 .dnsC.f1.e1 -fill both -side left -expand 1

	frame .dnsC.f2 -bd 2 -relief "raised"
	label .dnsC.f2.ltop -text "DNS Name Servers" 
	pack .dnsC.f2.ltop -fill x
	frame .dnsC.f2.f1 -bd 2 -relief "flat"
	entry .dnsC.f2.f1.e1 -width 3
	label .dnsC.f2.f1.l1 -text "."
	entry .dnsC.f2.f1.e2 -width 3
	label .dnsC.f2.f1.l2 -text "."
	entry .dnsC.f2.f1.e3 -width 3
	label .dnsC.f2.f1.l3 -text "."
	entry .dnsC.f2.f1.e4 -width 3
	button .dnsC.f2.f1.b1 -text "Add" -command {
	    if { [.dnsC.f2.f1.e1 get] < 256 && [.dnsC.f2.f1.e1 get] > -1  } {
		if { [.dnsC.f2.f1.e2 get] < 256 && [.dnsC.f2.f1.e2 get] > -1  } {
		    if { [.dnsC.f2.f1.e4 get] < 256 && [.dnsC.f2.f1.e3 get] > -1  } {
			if { [.dnsC.f2.f1.e4 get] < 255 && [.dnsC.f2.f1.e4 get] > 0  } {
			    .dnsC.f2.f2.text insert end "[.dnsC.f2.f1.e1 get].[.dnsC.f2.f1.e2 get].[.dnsC.f2.f1.e3 get].[.dnsC.f2.f1.e4 get]"
			    .dnsC.f2.f1.e1 delete 0 end
			    .dnsC.f2.f1.e2 delete 0 end
			    .dnsC.f2.f1.e3 delete 0 end
			    .dnsC.f2.f1.e4 delete 0 end
			    focus .dnsC.f2.f1.e1
			} else {
			    focus .dnsC.f2.f1.e4
			    if { [.dnsC.f2.f1.e4 get] < 255 } { 
				error 202
			    } elseif { [.dnsC.f2.f1.e4 get] > 0  } {
				error 201
			    } else { error 200 }
			}
		    } else {
			focus .dnsC.f2.f1.e3
			error 200
		    }
		} else {
		    focus .dnsC.f2.f1.e2
		    error 200
		}
	    } else {
		focus .dnsC.f2.f1.e1
		error 200
	    }
	    
	}
	pack .dnsC.f2.f1.e1 .dnsC.f2.f1.l1 .dnsC.f2.f1.e2 .dnsC.f2.f1.l2 .dnsC.f2.f1.e3 .dnsC.f2.f1.l3 .dnsC.f2.f1.e4 .dnsC.f2.f1.b1 -fill x -side left -expand 1
	frame .dnsC.f2.f2 -bd 2 -relief "flat"
	listbox .dnsC.f2.f2.text -selectforeground white -selectbackground #000080 -height 3 -width 16
	.dnsC.f2.f2.text config -cursor hand2
	pack .dnsC.f2.f2.text -side left -expand 1 -fill both
	button .dnsC.f2.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.dnsC.f2.f2.text curselection ] != "" } {
		.dnsC.f2.f2.text delete [.dnsC.f2.f2.text curselection ]
	    }
	}
	pack .dnsC.f2.f2.b1 -side left
	pack .dnsC.f2.f1 .dnsC.f2.f2 -side top -fill both -expand 1

	frame .dnsC.f3 -bd 2 -relief "raised"
	label .dnsC.f3.l1 -text "Additional Search Domains" 
	pack .dnsC.f3.l1 -fill x
	frame .dnsC.f3.f1 -bd 2 -relief "flat"
	entry .dnsC.f3.f1.e1 -width 10
	button .dnsC.f3.f1.b1 -text "Add" -command {
	    if { [.dnsC.f3.f1.e1 get] != "" } {
		.dnsC.f3.f2.text insert end [.dnsC.f3.f1.e1 get]
	    }
	    .dnsC.f3.f1.e1 delete 0 end
	}
	pack .dnsC.f3.f1.e1 .dnsC.f3.f1.b1 -fill both -side left -expand 1
	frame .dnsC.f3.f2 -bd 2 -relief "flat"
	scrollbar .dnsC.f3.f2.vscroll -orient vertical -command ".dnsC.f3.f2.text yview" 
	listbox .dnsC.f3.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.dnsC.f3.f2.vscroll set } -height 4 -width 16
	.dnsC.f3.f2.text config -cursor hand2
	pack .dnsC.f3.f2.vscroll -side left -fill y
	pack .dnsC.f3.f2.text -side left -expand 1 -fill both
	button .dnsC.f3.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.dnsC.f3.f2.text curselection ] != "" } {
		.dnsC.f3.f2.text delete [.dnsC.f3.f2.text curselection ]
	    }
	}
	pack .dnsC.f3.f2.b1 -side left
	pack .dnsC.f3.f1 .dnsC.f3.f2 -side top -fill both -expand 1
	frame .dnsC.f4 -bd 2 -relief "flat"
	button .dnsC.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [ .dnsC.f1.e1 get ] != "" } {
		# Check for at least one nameserver
		if { [ .dnsC.f2.f2.text size ] > 0 } {
		    # The arguemnets to the Perl script will include "!" and "+".  The "!" seperates fields,
		    # while the "+" seperates values field values
		    set cmd "dnscset [ .dnsC.f1.e1 get ]!"
		    for { set position 0 } { $position < [ .dnsC.f2.f2.text size ] } { incr position 1 } {
			append cmd "[ .dnsC.f2.f2.text get $position]+"
		    }
		    set cmd2 [string trimright $cmd "+" ]
		    append cmd2 "!"
		    if { [ .dnsC.f3.f2.text size ] > 0 } {
			for { set position 0 } { $position < [ .dnsC.f3.f2.text size ] } { incr position 1 } {
			    append cmd2 "[ .dnsC.f3.f2.text get $position]+"
			}
		    }
		    set cmd [string trimright $cmd2 "+" ]
		    if $DEBUG { puts stdout "Command = $cmd" } 
		    set returnval [queryHost [.dnsC.hidden1 get] $cmd 1 ]
		    if { [lindex [split $returnval " " ] 0 ] == "\{ERROR:" } {
			error 205
			puts stdout "ERROR:  Unable to update the /etc/resolv.conf file on [.dnsC.hidden1 get]\n       Check the files permissions"
		    } else {
			set returndata [queryHost $currenthost "get dnsc" 0 ]
			filltext $returndata .general.text 6
			destroy .dnsC
		    }
		} else {
		    error 204
		}
	    } else {
		error 203
	    }
	}
	button .dnsC.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    set returndata [queryHost $currenthost "get dnsc" 0 ]
	    filltext $returndata .general.text 6
	    destroy .dnsC
	}
	button .dnsC.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help dnsc
	}
	pack .dnsC.f4.b1 .dnsC.f4.b2 .dnsC.f4.b3 -side left -fill x 
	pack .dnsC.f1 .dnsC.f2 .dnsC.f3 .dnsC.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .dnsC.f2.f1.e1 <Return> {
	    focus .dnsC.f2.f1.e2
	}
	bind .dnsC.f2.f1.e1 . {
	    after 250 {
		.dnsC.f2.f1.e1 delete [expr [string length [.dnsC.f2.f1.e1 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e2 
	}
	bind .dnsC.f2.f1.e2 <Return> {
	    focus .dnsC.f2.f1.e3
	}
	bind .dnsC.f2.f1.e2 . {
	    after 250 {
		.dnsC.f2.f1.e2 delete [expr [string length [.dnsC.f2.f1.e2 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e3 
	}
	bind .dnsC.f2.f1.e3 <Return> {
	    focus .dnsC.f2.f1.e4
	}
	bind .dnsC.f2.f1.e3 . {
	    after 250 {
		.dnsC.f2.f1.e3 delete [expr [string length [.dnsC.f2.f1.e3 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e4 
	}
	bind .dnsC.f2.f1.e4 <Return> {
	    .dnsC.f2.f1.b1 invoke
	}
	bind .dnsC.f2.f1.e4 . {
	    after 250 {
		.dnsC.f2.f1.e4 delete [expr [string length [.dnsC.f2.f1.e4 get]] -1] end
	    }
	    .dnsC.f2.f1.e1 delete 0 end
	    .dnsC.f2.f1.e2 delete 0 end
	    .dnsC.f2.f1.e3 delete 0 end
	    .dnsC.f2.f1.e4 delete 0 end
	}

	bind .dnsC.f3.f1.e1 <Return> {
	    if { [.dnsC.f3.f1.e1 get] != "" } {
		.dnsC.f3.f2.text insert end [.dnsC.f3.f1.e1 get]
	    }
	    .dnsC.f3.f1.e1 delete 0 end
	}
    } else {
	.dnsC.hidden1 delete 0 end
	.dnsC.hidden1 insert end $currenthost
	wm title .dnsC "DNS Client Editor: $currenthost"
    }

    # Fill boxes with data
    .dnsC.f2.f2.text delete 0 end
    .dnsC.f3.f2.text delete 0 end
    set args ""
    set returndata [queryHost $currenthost "get dnsc" 0 ]
    for { set position 0 } { $position < [llength $returndata] } { incr position 1 } {
	set tmp [lindex $returndata $position ]
	set arg1 [lindex $tmp 0 ]
	set arg2 [lindex $tmp 1 ]
	if { $arg1 == "domain" } {
	    .dnsC.f1.e1 delete 0 end
	    .dnsC.f1.e1 insert end $arg2
	} elseif { $arg1 == "nameserver" } {
	    .dnsC.f2.f2.text insert end $arg2
	} elseif { $arg1 == "search" } {
	    set args [ split $tmp {, } ]
	    for { set position2 1 } { $position2 < [llength $args] } { incr position2 1 } {
		if { [lindex $args $position2 ] != "" } {
		    .dnsC.f3.f2.text insert end [lindex $args $position2 ]
		}
	    }
	}
    }
}




#
# --------------------------------------------------------------------------------------------------------------------------------
# --------------------------                     TASK  PROCEDURES                                ---------------------------------
# --------------------------------------------------------------------------------------------------------------------------------
#

#
# -- Procedure to read tasks from raw data and fill list box window --
#
proc gettask { data window } {  
    $window delete 0 end
    for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	set tmp [split [lindex $data $position ]]
	set task [lindex $tmp 1 ]
	$window insert end $task
    }
}


#
# ----- Procedure for getting the name of a new task -----
#
proc getnewtask { } {
    global stask
    toplevel .newtask -class Dialog
    wm title .newtask "Query?"
    wm iconname .newtask Dialog
    label .newtask.label -text "Please enter the name\nof the new task"
    entry .newtask.entry -width 30
    pack .newtask.label .newtask.entry -side top -fill both -expand 1
    focus .newtask.entry
    bind .newtask.entry <Return> {
	set stask [.newtask.entry get]
	destroy .newtask
    }
}




#
# Task update procedure
#
# USE Tasks   CurrentHost CurrentLine SelectedData
#            CurrentHost = host to update
#            CurrentLine = line number 
#            SelectedData = data
## NOTE: Data passed to this routine will be ignored.  Only the data from the client will be used

proc Tasks { currenthost currentline line args } {
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    set host $currenthost
    if { [winfo exists .task ] == 0 } {
	toplevel .task -class Dialog
	wm title .task "Task Editor: $currenthost"
	wm iconname .task "Task Editor"
	entry .task.hidden1 
	entry .task.hidden2
	# NOT GOOD
	.task.hidden1 insert end $currenthost

	frame .task.time -relief raised -bd 2
	set stask [lindex $line 1 ]
	label .task.time.label -text "Host: $currenthost  Task: $stask"
	pack .task.time.label
	frame .task.time.type -relief flat -bd 0
	checkbutton .task.time.type.interval -text "Interval" -variable ttype -onvalue 1 -offvalue 0 -command {
	    if { $ttype == 1 } {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 configure -state disabled
	    } elseif { $ttype == 0 } {
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
	    }
	}
	checkbutton .task.time.type.timed -text "Timed" -variable ttype -onvalue 0 -offvalue 1 -command {
	    if { $ttype == 1 } {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 configure -state disabled
	    } elseif { $ttype == 0 } {
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
	    }
	}
	pack .task.time.type.interval .task.time.type.timed -side left -expand 1 -fill x
	frame .task.time.timeval -relief flat -bd 0
	label .task.time.timeval.label1 -text "Hour(s)"
	entry .task.time.timeval.entry1 -width 10
	label .task.time.timeval.label2 -text "Minute(s)"
	entry .task.time.timeval.entry2 -width 10
	pack .task.time.timeval.label1 .task.time.timeval.entry1 .task.time.timeval.label2 .task.time.timeval.entry2 -side left -fill x -expand 1
	pack .task.time.type .task.time.timeval -side top -fill x
	pack .task.time -fill x
	
	frame .task.outfile -relief raised -bd 2
	label .task.outfile.label -text "Task Output History"
	pack .task.outfile.label -side top -fill x
	frame .task.outfile.frame -relief flat -bd 0
	label .task.outfile.frame.filelabel -text "File name"
	entry .task.outfile.frame.filename 
	pack .task.outfile.frame.filelabel .task.outfile.frame.filename -side left -fill x -expand 1
	pack .task.outfile.frame -side top -fill x
	scale .task.outfile.histval -from 0 -to 35 -resolution 1 -tickinterval 0 -orient horizontal -label "Retention Period (Days)" -variable retain 
	pack .task.outfile.histval -side top -fill x
	pack .task.outfile -fill x
	
	frame .task.warn -relief raised -bd 2
	label .task.warn.label -text "Warning Threshold"
	entry .task.warn.val 
	pack .task.warn.label .task.warn.val -side left -fill x -expand 1
	pack .task.warn -fill x
	
	frame .task.alarm -relief raised -bd 2
	label .task.alarm.label -text "Alarm Threshold   "
	entry .task.alarm.val 
	pack .task.alarm.label .task.alarm.val -side left -fill x -expand 1
	pack .task.alarm -fill x
	
	frame .task.recover -relief raised -bd 2
	label .task.recover.label -text "Recovery Script  "
	entry .task.recover.val 
	pack .task.recover.label .task.recover.val -side left -fill x -expand 1
	pack .task.recover -fill x
	
	frame .task.bottm -relief raised -bd 2
	button .task.bottm.apply -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { $current != 9999 } {
		set cmdline "cdadmin modifynum $current "
	    } else {
		set cmdline "cdadmin add "
	    }
	    if { $ttype == 0 } {
		append cmdline "T[ .time.timeval.entry1 get ]"
		append cmdline ":[ .time.timeval.entry2 get ]"
	    } else {
		append cmdline [ .time.timeval.entry1 get ]
	    }
	    append cmdline " "
	    append cmdline $stask
	    append cmdline " [expr $retain * 1440 / $interval]"
	    append cmdline " [.outfile.frame.filename get]"
	    append cmdline " [.warn.val get]"
	    append cmdline " [.alarm.val get]"
	    append cmdline " [.recover.val get]"
	    set cmdresult [queryHost $currenthost $cmdline 1 ]
	    puts stdout [lindex [split $cmdresult ":"] 0]
	    if { [lindex [split $cmdresult] 0] == "\{ERROR:" } {
		error 1
	    }
	    puts stdout $cmdresult
	    set returndata [queryHost $currenthost cdadmin 0 ]
	    gettask $returndata .tp.out.text
	}
	
	pack .task.bottm.apply 
	pack .task.bottm -anchor w
    } else {
	.task.outfile.frame.filename delete 0 end
	.task.warn.val delete 0 end
	.task.alarm.val delete 0 end
	.task.recover.val delete 0 end
	.task.time.timeval.entry2 delete 0 end
	.task.time.timeval.entry1 delete 0 end
    }



    #
    #  Task Bindings
    #
    
    bind .tp.out.text <Double-1> { 
	if { [.tp.out.text curselection ] != "" } {
	    set stask [ .tp.out.text get [ .tp.out.text curselection ]]
	    set current [ .tp.out.text curselection ]
	    newstate normal
	    .time.label configure -text "HOST: $currenthost  Task: $stask"
	    set tmp [split [lindex $returndata $current ]]
	    set ttime [lindex $tmp 0 ]
	    set cttype [lindex [split $ttime {} ] 0 ]
	    if { $cttype == "T" | $cttype == "t" } {
		.task.time.label configure -foreground black
		.task.time.label configure -text "Host: $currenthost  Task: $stask"
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
		.task.time.timeval.entry2 delete 0 end
		.task.time.timeval.entry1 delete 0 end
		.task.time.timeval.entry1 insert end [string range [lindex [split $ttime ":"] 0] 1 end]
		.task.time.timeval.entry2 insert end [lindex [split $ttime ":"] 1]
		set ttype 0
	    } else {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 delete 0 end
		.task.time.timeval.entry2 configure -state disabled
		.task.time.timeval.entry1 delete 0 end
		.task.time.timeval.entry1 insert end $ttime
		set ttype 1
	    }
	    set task [lindex $tmp 1 ]
	    set history [lindex $tmp 2 ]
	    set outfile [lindex $tmp 3 ]
	    set warn [lindex $tmp 4 ]
	    set alarm [lindex $tmp 5 ]
	    set recover [lindex $tmp 6 ]
	    if { $history == "-" } {
		set retain 0
	    } else {
		set retain [ expr ($interval * $history) / 1440 ]
	    }
	    .task.outfile.frame.filename delete 0 end
	    .task.warn.val delete 0 end
	    .task.alarm.val delete 0 end
	    .task.recover.val delete 0 end
	    .task.outfile.frame.filename insert end $outfile
	    .task.warn.val insert end $warn
	    .task.alarm.val insert end $alarm
	    .task.recover.val insert end $recover
	}
    } 
    
    bind .task.time.timeval.entry1 <Return> {
	if { $ttype == 0 } {
	    focus .task.time.timeval.entry2
	} else {
	    focus .task.outfile.frame.filename
	}
    }
    
    bind .task.time.timeval.entry2 <Return> {
	focus .task.outfile.frame.filename
    }
    
    bind .task.outfile.frame.filename <Return> {
	focus .task.outfile.histval
    }
    
    bind .task.warn.val <Return> {
	focus .task.alarm.val
    }
    
    bind .task.alarm.val <Return> {
	focus .task.recover.val
    }
    
    bind .task.recover.val <Return> {
	focus .task.bottm
    }


}


#
# Procedure to Enable/Disable Entry
#
proc newstate { state } {
    if { $state == "disabled" } {
	.task.time.label configure -foreground grey
	.task.time.timeval.label1 configure -foreground grey
	.task.time.timeval.label2 configure -foreground grey
	.task.outfile.label configure -foreground grey
	.task.outfile.frame.filelabel configure -foreground grey
	.task.outfile.histval configure -foreground grey
	.task.warn.label configure -foreground grey
	.task.alarm.label configure -foreground grey
	.task.recover.label configure -foreground grey

    } else {
	.task.time.label configure -foreground black
	.task.time.label configure -text "Host:    Task:"
	.task.time.timeval.label1 configure -foreground black
	.task.time.timeval.label2 configure -foreground black
	.task.outfile.label configure -foreground black
	.task.outfile.frame.filelabel configure -foreground black
	.task.outfile.histval configure -foreground black
	.task.warn.label configure -foreground black
	.task.alarm.label configure -foreground black
	.task.recover.label configure -foreground black
    }
    .task.time.type.interval configure -state $state
    .task.time.type.timed configure -state $state
    .task.time.timeval.entry1 configure -state $state
    .task.time.timeval.entry2 configure -state $state
    .task.outfile.frame.filename configure -state $state
    .task.outfile.histval configure -state $state
    .task.warn.val configure -state $state
    .task.alarm.val configure -state $state
    .task.recover.val configure -state $state
    .task.bottm.apply configure -state $state
}


