/* graphics.ibmgl.c */

/* Graphics functions using GL on IBM RS/6000 */

/* VIS-5D version 4.0 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990, 1991, 1992, 1993, 1994  Bill Hibbard, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/






#include <math.h>
#include <stdio.h>
#include <string.h>
#include "graphics.h"
#include "matrix.h"
#include <X11/Xlib.h>
#include <gl/gl.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>


/* API - to globals.c (or private indexed structure);
   except constants */

/*** Private Variables: ***/
static Window wmain;
static float Scale;
static Matrix ModelMat, InvModelMat, ProjMat;
static Matrix IdMat = { 1.0, 0.0, 0.0, 0.0,
			0.0, 1.0, 0.0, 0.0,
                        0.0, 0.0, 1.0, 0.0,
			0.0, 0.0, 0.0, 1.0 };
static int pretty_flag = 0;
static int Perspective = 0;




#define FLIP( A )  ( (A&0xff00ff00) | ((A>>16) & 0xff) | ((A&0xff) <<16) )
#define DOT( a, b )  ( a[0]*b[0] + a[1]*b[1] + a[2]*b[2] )
#define EYEDIST 3.0
#define FOV 400
#define VIEWOBJ 10

static long zfar;
static unsigned int CurColor = 0xffffffff;
static unsigned int CurTrans = 0xff000000;
static float FrontClip;
static float xminplane[4], xmaxplane[4], yminplane[4], ymaxplane[4];

/* setup the default material surface description: */
static float mat1[] = {
   AMBIENT, 0.6, 0.6, 0.6,   /* surface ambient color */
   DIFFUSE, 0.6, 0.6, 0.6,   /* surface diffuse color */
   LMNULL
};

/* setup the light source descriptions: */
static float light1[] = {
   LCOLOR, 0.6, 0.6, 0.6,          /* light color */
   POSITION, 0.0, 0.0, 10.0, 0.0,  /* light location */
   LMNULL
};

static float light2[] = {
   LCOLOR, 0.6, 0.6, 0.6,          /* light color */
   POSITION, 0.0, 0.0, -10.0, 0.0,  /* light location */
   LMNULL
};

/* setup the lighting model description: */
static float lmodel[] = {
   AMBIENT, 0.4, 0.4, 0.4,         /* ambient color */
   LOCALVIEWER, 0.0,               /* infinite viewpoint */
   LMNULL
};

static short screen1[16] = { /* 75% */
   0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd,
   0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd
};
static short screen2[16] = { /* 50% */
   0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555,
   0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555
};
static short screen3[16] = { /* 25% */
   0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222,
   0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222
};
static int screendoor;



void gflush()
{
}



void init_graphics2( alphaflag )
int alphaflag;
{
   HQR_available = 0;
   screendoor = 1;
   Perspec_available = 1;
}



void terminate_graphics()
{
}



Window make_window( title, xpos, ypos, width, height,
                    fontname, fontsize, NPGLwin )
char *title;
int xpos, ypos;
int width, height;
char *fontname;
int fontsize;
Window *NPGLwin;
{
   static long Black = 0xff000000;
   char title2[100];

   strcpy( title2, title );  /* uniquify the window title */
   strcat( title2, "   " );

   if (width==ScrWidth && height==ScrHeight) {
      noborder();
      prefposition( 0, ScrWidth-1, 0, ScrHeight-1 );
   }
   else {
      prefposition( xpos, xpos+width-1,
		    ScrHeight-ypos-1, ScrHeight-ypos-height );
   }
   winopen( title2 );
   winconstraints();

   /* find X window pointer to the GL window */
   wmain = find_window( GfxDpy, GfxScr, RootWindow(GfxDpy,GfxScr), title2 );
   if (!wmain) {
      fprintf(stderr,"Unable to create graphics window\n");
      exit(1);
   }
   wintitle( title );   /* restore window title */

   /* select which X events to receive */
   XSelectInput( GfxDpy, wmain, ExposureMask | PointerMotionMask
		| KeyPressMask | ButtonPressMask | ButtonReleaseMask
		| StructureNotifyMask | VisibilityChangeMask );

   /* setup 24-bit, double buffer, z-buffered window */
   RGBmode();
   doublebuffer();
   gconfig();
   zbuffer( TRUE );
   zfar = getgdesc( GD_ZMAX );

   czclear( Black, zfar );
   swapbuffers();
   czclear( Black, zfar );

   mmode( MVIEWING );     /* multi-matrix mode */

   getsize( &WinWidth, &WinHeight );
   AspectRatio = (float) WinWidth / (float) WinHeight;
   ortho( -1.0, 1.0, -1.0, 1.0, 1.0, -1.0 );  /* this MUST be after mmode()! */
   viewport( 0, (Screencoord) WinWidth-1, 0, (Screencoord) WinHeight-1);

   lmdef( DEFMATERIAL, 10, 0, mat1 );   /* define material, light, model */
   lmdef( DEFLIGHT, 20, 0, light1 );
   lmdef( DEFLIGHT, 21, 0, light2 );
   lmdef( DEFLMODEL, 30, 0, lmodel );
   lmbind( MATERIAL, 10 );
   lmbind( LIGHT1, 20 );
   lmbind( LIGHT2, 21 );
   lmbind( LMODEL, 30 );

   /* Set font */
   FontHeight = getheight();
   FontDescent = getdescender();

   /* init front clipping plane */
   FrontClip = 1.0;

   /* screen door transparency patterns */
   defpattern( 1, 16, screen1 );
   defpattern( 2, 16, screen2 );
   defpattern( 3, 16, screen3 );
   setpattern(0);

   return wmain;
}



/*** set_pointer ******************************************************
   Set mouse pointer image to an arrow or 'busy' clock.
   Input: p - 0 = pointer image
              1 = clock image
**********************************************************************/
void set_pointer( p )
int p;
{
   static unsigned short  hourglass[16] = { 0x1ff0, 0x1ff0, 0x0820, 0x0820,
					    0x0820, 0x0c60, 0x06c0, 0x0100,
					    0x0100, 0x06c0, 0x0c60, 0x0820,
					    0x0820, 0x0820, 0x1ff0, 0x1ff0 };
   if (p) {
      /* make busy cursor */
      curstype( C16X1 );
      defcursor( 1, hourglass );
      setcursor( 1, 0, 0 );
   }
   else {
      /* regulay cursor */
      setcursor( 0, 0, 0 );
   }
}



/*** save_formats *****************************************************
   Return a set of flag bits indicating what image formats we can
   save as.
**********************************************************************/
int save_formats()
{
   return 0;
}



/*** save_window ******************************************************
   Save the current image in the 3-D window to the named file.
   Input:  filename - image filename
           format - SAVE_SGI = SGI .rgb
	            SAVE_GIF = .GIF
		    SAVE_XWD = .xwd   (X window dump)
   Return: 1 = no errors
           0 = error occured.
**********************************************************************/
int save_window( filename, format )
char filename[];
int format;
{
   return 0;
}



/*** print_window *****************************************************
   Print the current image in the 3-D window to the default PostScript
   printer.
   Return: 1 = ok, 0 = error
**********************************************************************/
int print_window( )
{
   return 0;
}





/*** clear_window *****************************************************
   Clear the graphics window.  This is called prior to rendering
   a frame.
**********************************************************************/
void clear_window( bgcolor )
unsigned int bgcolor;
{
   czclear( (unsigned long) bgcolor, zfar );
}



/*
 * Called when window size changes.
 */
void resize_window( width, height )
int width, height;
{
   WinWidth = width;
   WinHeight = height;
   AspectRatio = (float) WinWidth / (float) WinHeight;
   viewport( 0, (Screencoord) WinWidth, 0, (Screencoord) WinHeight);
}



/*** update_window ****************************************************
   Update the 3-D window.  This called when finished rendering a frame.
**********************************************************************/
void update_window()
{
   swapbuffers();
}



/*** set_2d ***********************************************************
   This is to be called prior to any 2-D rendering calls.
**********************************************************************/
void set_2d()
{
   loadmatrix( IdMat );
   ortho2( -0.5, WinWidth-0.5, -0.5, WinHeight-0.5 );
   zfunction( ZF_ALWAYS );
}



/*** set_3d ***********************************************************
   This is to be called prior to any 3-D rendering calls.
   Input:  ctm - the current transformation matrix.
           scale - scaling factor of the ctm.
	   perspec - 0 = parallel, 1 = perspective projection
**********************************************************************/
void set_3d( ctm, scale, perspec )
Matrix ctm;
float scale;
int perspec;
{
   static Matrix flip = { 1.0, 0.0, 0.0, 0.0,
			  0.0, 1.0, 0.0, 0.0,
			  0.0, 0.0,-1.0, 0.0,
			  0.0, 0.0, 0.0, 1.0 };
   float front, rear;

   gflush();   /* to prevent RE2 pipe hangup */

   mat_copy( ModelMat, ctm );
   mat_inv( InvModelMat, ctm );
   InvModelMat[3][0] = InvModelMat[3][1] = InvModelMat[3][2] = 0.0;

   if (perspec) {
      /* Perspective projection */
      front = EYEDIST - scale * 1.75 * FrontClip;
      if (front<0.01)  front = 0.01;
      rear = EYEDIST + scale * 1.75;
      perspective( FOV, AspectRatio, front, rear );
      mat_mul( ModelMat, ModelMat, flip );
      ModelMat[3][2] = -EYEDIST;
      loadmatrix( ModelMat );

      /* make an object, VIEWOBJ, so we can use mapw later */
      makeobj(VIEWOBJ);
      perspective( FOV, AspectRatio, front, rear );
      loadmatrix( ModelMat );
      closeobj();

      /* get projection matrix */
      mmode( MPROJECTION );
      getmatrix( ProjMat );
      mmode( MVIEWING );
   }
   else {
      /* parallel projection */
      /* Note reversed Z! */
      front = scale * 1.75 * FrontClip;
      rear = -scale * 1.75;
      ortho( -AspectRatio, AspectRatio, -1.0, 1.0, front, rear );
      loadmatrix( ModelMat );
   }
   Perspective = perspec;
   Scale = scale;
   zfunction( ZF_LESS );
}



/*** project **********************************************************
   Use current transformation and viewing information to project a
   point p from 3-D graphics coordinates to 2-D window coordinates
   in [0,WinWidth-1]x[0,WinHeight-1].
   Input:  p - 3-D point in graphics space
   Output:  x, y - 2-D point in window pixel coordinates.
**********************************************************************/
void project( p, x, y )
float p[3];
float *x, *y;
{
   float q[4], nx, ny;

   q[0] = p[0];
   q[1] = p[1];
   q[2] = p[2];
   q[3] = 1.0;
   if (Perspective) {
      /* Perspective Projection */
      mat_vecmul4( q, ModelMat );
      mat_vecmul4( q, ProjMat );
      nx = q[0] / q[3];   /* divide by w */
      ny = q[1] / q[3];
      *x = (nx+1.0) / 2.0 * WinWidth;
      *y = (1.0-ny) / 2.0 * WinHeight;
   }
   else {
      /* Parallel Projection */
      mat_vecmul4( q, ModelMat );
      nx = q[0];
      ny = q[1];
      *x = ((nx/AspectRatio+1.0) / 2.0 * WinWidth);
      *y = (1.0-ny) / 2.0 * WinHeight;
   }

}



/*** unproject ********************************************************
   Given a 2-D window coordinate in [0,WinWidth-1]x[0,WinHeight-1],
   return the parametric equation of a line in 3-D such that the
   projection of the line from 3-D to 2-D is a point at the window
   coordinate.
   Input:  x, y - window coordinate.
   Output:  p, d - parametric equation of line:  l = p + t*d
                   NOTE, d will have unit length
**********************************************************************/
void unproject( x, y, p, d )
float x, y;
float p[3], d[3];
{
   float mag;

   if (Perspective) {
      /* Perspective Projection */
      Screencoord sx, sy;
      float a0,a1,a2;
      sx = (Screencoord) x;
      sy = (Screencoord) (WinHeight-y-1);
      mapw( VIEWOBJ, sx, sy, &p[0], &p[1], &p[2], &a0, &a1, &a2 );
      d[0] = a0 - p[0];
      d[1] = a1 - p[1];
      d[2] = a2 - p[2];
   }
   else {
      /* Parallel Projection */
      d[0] = 0.0;
      d[1] = 0.0;
      d[2] = 1.0;
      mat_vecmul3( d, InvModelMat );
      p[0] = ((x/WinWidth) * 2.0 - 1.0) * AspectRatio;
      p[1] = 1.0 - (y/WinHeight) * 2.0;
      p[2] = 0.0;
      p[0] -= ModelMat[3][0];
      p[1] -= ModelMat[3][1];
      p[2] -= ModelMat[3][2];
      mat_vecmul3( p, InvModelMat );
   }
   mag = sqrt( d[0]*d[0] + d[1]*d[1] + d[2]*d[2] );
   d[0] /= mag;
   d[1] /= mag;
   d[2] /= mag;
}




/*** set_color ********************************************************
   Set the current drawing color.  The color is specified in packed
   ARGB format; 1-byte each of Alpha, Red, Green, Blue where Alpha
   is the most-significant byte and blue is the least-signficant byte.
**********************************************************************/
void set_color( c )
unsigned int c;
{
   CurColor = FLIP( c );
   CurTrans = c & 0xff000000;
   if (screendoor) {
      CurTrans = 3 - UNPACK_ALPHA(c) / 64;
   }
   cpack( (unsigned long) CurColor );
}



/*** set_depthcue *****************************************************
   Set the line depth cueing flag on or off.
**********************************************************************/
void set_depthcue( onoff )
int onoff;
{
   short r, g, b;
   if (onoff) {
      r = UNPACK_RED( CurColor );
      g = UNPACK_GREEN( CurColor );
      b = UNPACK_BLUE( CurColor );
      depthcue( TRUE );
      lRGBrange( r/3, g/3, b/3,  r, g, b, 0, zfar );
   }
   else {
      depthcue( FALSE );
   }
}



/*** set_line_width ***************************************************
   Set the width of lines.
   Input:  w - width in pixels.
**********************************************************************/
void set_line_width( w )
float w;
{
/*   linewidthf( w );*/
}



/*** set_frontclip ****************************************************
   Set the position of the front clipping plane.
   Input:  d - front clipping plane z position in [1,-1] where 1 is
               near viewer and -1 is away from viewer.
**********************************************************************/
void set_frontclip( d )
float d;
{
   if (d>1.0)
      FrontClip = 1.0;
   else if (d<-1.0)
      FrontClip = -1.0;
   else
      FrontClip = d;
   /* set_3d() will put new front clipping plane into effect */
}




void set_box_clipping( onoff, xmin, xmax, ymin, ymax )
int onoff;
float xmin, xmax, ymin, ymax;
{
   /* nothing */
}




void set_pretty( onoff )
{
   /* always off */
   pretty_flag = 0;
}




void start_aa_pass( n )
int n;
{
   /* nothing */
}




void end_aa_pass( n )
int n;
{
#ifdef IBM_GL
   /* nothing */
#endif
}



void start_hqtrans_pass()
{
#ifdef IBM_GL
   /* nothing */
#endif
}



int end_hqtrans_pass()
{
   return 0;
}




/*** polytrinorm ******************************************************
   Draw a polytriangle strip with normals.
**********************************************************************/
void polytrinorm( vert, norm, n )
float vert[][3];
float norm[][3];
int n;
{
   register int i;

   lmcolor( LMC_AD );
   cpack( CurColor );
   /* enable transparency */

   if (screendoor)
     setpattern( CurTrans );

   /* Due to strangeness in the IBM hardware (and VOGL?) and their version  */
   /* of GL, we must break the polytriangle strip into individual triangles */
   /* and test to make sure the normal vectors are correct.  Also, we must  */
   /* cull out triangles with zero area.                                    */
   for (i=2; i<n; i++) {
      /* calculate triangle surface normal */
      float v1[3], v2[3], cross[3];
      v1[0] = vert[i][0] - vert[i-2][0];
      v1[1] = vert[i][1] - vert[i-2][1];
      v1[2] = vert[i][2] - vert[i-2][2];
      v2[0] = vert[i-1][0] - vert[i][0];
      v2[1] = vert[i-1][1] - vert[i][1];
      v2[2] = vert[i-1][2] - vert[i][2];
      cross[0] = v1[1]*v2[2] - v1[2]*v2[1];
      cross[1] = v1[2]*v2[0] - v1[0]*v2[2];
      cross[2] = v1[0]*v2[1] - v1[1]*v2[0];
      /* check for zero area */
      if (cross[0]==0.0 && cross[1]==0.0 && cross[2]==0.0)
	continue;

      bgntmesh();

      /* vertex & normal 1 */
      if (DOT(cross,norm[i-2])<0.0) {
	 /* flip normal */
	 float nn[3];
	 nn[0] = -norm[i-2][0];
	 nn[1] = -norm[i-2][1];
	 nn[2] = -norm[i-2][2];
	 n3f( nn );
      }
      else
	n3f( norm[i-2] );
      v3f(vert[i-2]);

      /* vertex & normal 2 */
      if (DOT(cross,norm[i-1])<0.0) {
	 /* flip normal */
	 float nn[3];
	 nn[0] = -norm[i-1][0];
	 nn[1] = -norm[i-1][1];
	 nn[2] = -norm[i-1][2];
	 n3f( nn );
      }
      else
	n3f( norm[i-1] );
      v3f(vert[i-1]);

      /* vertex & normal 3 */
      if (DOT(cross,norm[i])<0.0) {
	 /* flip normal */
	 float nn[3];
	 nn[0] = -norm[i][0];
	 nn[1] = -norm[i][1];
	 nn[2] = -norm[i][2];
	 n3f( nn );
      }
      else
	n3f( norm[i] );
      v3f(vert[i]);

      endtmesh();
   }

   /* Unbind and rebind the material to get around another */
   /* strange IBM GL bug */
   lmbind( MATERIAL, 0 );
   lmbind( MATERIAL, 10 );

   /* turn off transparency */
   setpattern(0);
}



/*** polytri **********************************************************
   Draw a polytriangle strip without shading.
**********************************************************************/
void polytri( vert, n )
float vert[][3];
int n;
{
   int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );
   setpattern( CurTrans );

   bgntmesh();
   for (i=0;i<n;i++) {
      v3f( vert[i] );
   }
   endtmesh();
   setpattern(0);
}



/*** polyline *********************************************************
   Draw a 3-D poly line.
**********************************************************************/
void polyline( vert, n )
float vert[][3];
int n;
{
   register int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );

   bgnline();
   for (i=0;i<n;i++) {
      v3f( vert[i] );
   }
   endline();
}




/*** disjointpolyline *************************************************
   Draw a series of disjoint 3-D lines.
**********************************************************************/
void disjointpolyline( vert, n )
float vert[][3];
int n;
{
   register int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );

   for (i=0;i<n;i+=2) {
      bgnline();
      v3f( vert[i] );
      v3f( vert[i+1] );
      endline();
   }
}





/*** quadmesh *********************************************************
   Draw a quadrilateral mesh with colors at each vertex.
**********************************************************************/
void quadmesh( vert, color, rows, cols )
float vert[][3];
unsigned int color[];
int rows, cols;
{
   int i, j, base1, base2, c;

   lmcolor( LMC_COLOR );

   setpattern( CurTrans ); 

   /* break mesh into strips */
   for (i=0;i<rows-1;i++) {
      base1 = i * cols;
      base2 = (i+1) * cols;
      bgntmesh();
      for (j=0;j<cols;j++) {
         cpack( color[base1+j] );
	 v3f( vert[base1+j] );
 	 cpack( color[base2+j] );
	 v3f( vert[base2+j] );
      }
      endtmesh();
   }

   setpattern(0);
}



/*** quadmeshnorm *****************************************************
   Draw a quadrilateral mesh with normals and colors at each vertex.
**********************************************************************/
void quadmeshnorm( vert, norm, color, rows, cols )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
{
   int i, j, base1, base2, c;

   lmcolor( LMC_AD ); 

   setpattern( CurTrans );

   /* break mesh into strips */
   for (i=0;i<rows-1;i++) {
      base1 = i * cols;
      base2 = (i+1) * cols;
      bgntmesh();
      for (j=0;j<cols;j++) {
	 cpack( color[base1+j] ); 
	 n3f( norm[base1+j] );
	 v3f( vert[base1+j] );
	 cpack( color[base2+j] ); 
	 n3f( norm[base2+j] );
	 v3f( vert[base2+j] );
      }
      endtmesh();
   }

   /* Unbind and rebind the material to get around another */
   /* strange IBM GL (and VOGL?) bug */
   lmbind( MATERIAL, 0 );
   lmbind( MATERIAL, 10 );

   setpattern(0);
}



void quadmeshnorm_translated( vert, norm, color, rows, cols, tx, ty )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
float tx, ty;
{
   pushmatrix();
   translate( tx, ty, 0.0 );

   quadmeshnorm( vert, norm, color, rows, cols );

   popmatrix();
}



/*** polyline2d *******************************************************
   Draw a 2-D poly line.  Coordinates are in pixels with the origin
   in the upper-left corner of the window.  NOTE:  vertices are shorts.
**********************************************************************/
void polyline2d( vert, n )
short vert[][2];
int n;
{
   short v[2];
   int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );
   bgnline();
   for (i=0;i<n;i++) {
      v[0] = vert[i][0];
      v[1] = WinHeight - vert[i][1];
      v2s( v );
   }
   endline();
}


/*** draw_rect ********************************************************
   Draw a rectangle in 2-D.
**********************************************************************/
void draw_rect( x1, y1, x2, y2 )
int x1, y1, x2, y2;
{
   short v[5][2];

   v[0][0] = x1;  v[0][1] = y1;
   v[1][0] = x2;  v[1][1] = y1;
   v[2][0] = x2;  v[2][1] = y2;
   v[3][0] = x1;  v[3][1] = y2;
   v[4][0] = x1;  v[4][1] = y1;
   polyline2d( v, 5 );
}



/*** draw_text ********************************************************
   Draw a text string.  Position coordinates are in pixels with the
   origin being in the upper-left corner.
**********************************************************************/
void draw_text( xpos, ypos, str )
int xpos, ypos;
char *str;
{
   lmcolor( LMC_COLOR );
   cpack( CurColor );
   cmov2i( xpos, WinHeight-ypos );
   charstr( str );
}



/*** text_width *******************************************************
   Return the width of a text string in pixels.
**********************************************************************/
int text_width( str )
char *str;
{
   int dir, ascent, descent;
   XCharStruct overall;

   return strwidth( str );
}



int begin_object()
{
   int n;

   n = genobj();
   makeobj( n );
   return n;
}


int end_object()
{
   closeobj();
}


int draw_object( n )
int n;
{
   callobj( n );
}

