/* $Id: api.c,v 1.114 1996/04/09 17:57:49 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1996 Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#include <unistd.h>
#include <X11/Xlib.h>
#ifdef SGI_GL
#  include "gl/gl.h"
#endif
#ifdef OPENGL
#  include "GL/glx.h"
#endif
#ifdef sgi
#  include <sys/types.h>
#  include <sys/prctl.h>
#  include <sys/sysmp.h>
#endif
#ifdef sunos5
#  include <thread.h>
#endif

#include "analysis.h"
#include "anim.h"
#include "api.h"
#include "box.h"
#include "compute.h"
#include "globals.h"
#include "graphics.h"
#include "grid.h"
#include "image.h"
#include "memory.h"
#include "map.h"
#include "misc.h"
#include "proj.h"
#include "queue.h"
#include "render.h"
#include "save.h"
#include "tclsave.h"
#include "traj.h"
#include "topo.h"
#include "volume.h"
#include "work.h"



#define MEGA 1024*1024
#define MAX(A,B)  ( (A) > (B) ? (A) : (B) )

#define VERY_LARGE_RATIO 0.1


#define CLAMP( X, MIN, MAX )   ( (X)<(MIN) ? (MIN) : ((X>MAX) ? (MAX) : (X)) )



/*
 * Parallelism stuff:
 */
#define MAX_WORKERS 8
#ifdef sgi
  static int WorkerPID[MAX_WORKERS];
#endif
#ifdef suno5
  static thread_t WorkerPID[MAX_WORKERS];
#endif


/*
 * CAVE stuff:
 */
#ifdef CAVE
#  include <malloc.h>
#  ifdef OPENGL
#    include "cave_ogl.h"
#  else
#    include "cave.h"
#  endif
#  include "cave5d.h"
   static void *cave_shmem = NULL;  /* Pool shared by all contexts */
#endif



/* initial xformation matrix */
static MATRIX init_ctm = { 1.0, 0.0, 0.0, 0.0,
                           0.0, 1.0, 0.0, 0.0, 
                           0.0, 0.0, 1.0, 0.0,
                           0.0, 0.0, 0.0, 1.0 };




/**********************************************************************/
/*****                   CONTEXT HANDLING                         *****/
/**********************************************************************/


static Context *ctx_table = NULL;


/*
 * This macro used at the beginning of each function which takes a
 * context index.  It translates the context index to a context pointer
 * and prints an error message if the index is bad.
 */
#define CONTEXT( msg ) \
  Context ctx; \
  if (index<0 || index>=VIS5D_MAX_CONTEXTS || (ctx = ctx_table[index])==NULL) { \
    printf("bad context in %s\n", msg); \
    return VIS5D_BAD_CONTEXT; \
  }


#ifdef LEAVEOUT
/* **** this is a temporary hack for debugging **** */
Context vis5d_get_ctx( int index )
{
   if (index<0 || index>=VIS5D_MAX_CONTEXTS) {
      printf("bad context in vis5d_get_ctx\n");
      return NULL;
   }
   else {
      return ctx_table[index];
   }
}
#endif



/*
 * Initialize the fields of the context to reasonable defaults.
 */
static void init_context( Context ctx )
{
   int i;

   /* initialize everything to zero for starters */
   memset( ctx, 0, sizeof(struct vis5d_context) );

   /* Set non-zero fields */
   ctx->ContnumFlag = 1;
   ctx->DisplayBox = 1;
   ctx->DisplayClock = 1;
   ctx->DisplayCursor = 0;
   ctx->DisplayTraj[0] = 1;
   ctx->CurrentVolume = -1;
   ctx->CursorColor = &ctx->TrajColor[0];
   ctx->BoxColor = PACK_COLOR(255,255,255,255);
   ctx->BgColor = PACK_COLOR(0,0,0,255);
   ctx->LabelColor = PACK_COLOR(255,255,255,255);
   ctx->MapColor = PACK_COLOR(255,255,255,255);


   ctx->Ax = ctx->Ay = ctx->Az = 0.0;

   ctx->UserProjection = -1;
   ctx->UserVerticalSystem = -1;

   ctx->PointerX = ctx->PointerY = -1;

   ctx->FirstArea = -1;

   ctx->FontHeight = 20;
   ctx->FontName[0] = 0;

   ctx->LineWidth = 1.0;
   ctx->DepthCue = 1;

   ctx->Zoom = 1.0;
   memcpy( ctx->CTM, init_ctm, 16*sizeof(float) );

   ctx->MegaBytes = MBS;

   for (i=0;i<MAXVARS;i++) {
      ctx->IsoColorVar[i] = -1;
   }
   for (i=0;i<VIS5D_TRAJ_SETS;i++) {
      ctx->TrajColorVar[i] = -1;
   }

   ALLOC_SEM( ctx->ExtFuncDoneSem, 0 );
}



/* This is a helper function for the next function. */
static void adjust_wind_level_info( Context ctx, int var )
{
   if (var>=0) {
      if (ctx->LowLev[var] + ctx->Nl[var] < ctx->WindNl) {
         ctx->WindNl = ctx->LowLev[var] + ctx->Nl[var];
      }
      if (ctx->LowLev[var] > ctx->WindLow) {
         ctx->WindLow = ctx->LowLev[var];
      }
   }
}


/* Update the WindNl and WindLow values */
static void compute_wind_levels( Context ctx )
{
   int i;
   ctx->WindNl = ctx->MaxNl;
   ctx->WindLow = 0;
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      adjust_wind_level_info( ctx, ctx->Uvar[i] );
      adjust_wind_level_info( ctx, ctx->Vvar[i] );
      adjust_wind_level_info( ctx, ctx->Wvar[i] );
   }
   adjust_wind_level_info( ctx, ctx->TrajU );
   adjust_wind_level_info( ctx, ctx->TrajV );
   adjust_wind_level_info( ctx, ctx->TrajW );
}




/*
 * Allocate a new vis5d context, initialize to default values.
 */
static Context new_context( void )
{
   Context ctx;

#ifdef CAVE
   if (cave_shmem) {
      ctx = amalloc( sizeof(struct vis5d_context), cave_shmem );
   }
   else {
      ctx = (Context) calloc( 1, sizeof(struct vis5d_context) );
   }
#else
   ctx = (Context) calloc( 1, sizeof(struct vis5d_context) );
#endif
   if (ctx) {
      init_context( ctx );
   }
   return ctx;
}



/*
 * Deallocate all the resources attached to the context.  This is used
 * prior to loading a new dataset into a context.
 */
static void clear_context( Context ctx )
{
   int i;

   reinit_memory( ctx );
   FREE_LOCK( ctx->Mutex );
   /* clear graphics tables */
   memset( ctx->SurfTable, 0, sizeof(ctx->SurfTable) );
   memset( ctx->HSliceTable, 0, sizeof(ctx->HSliceTable) );
   memset( ctx->VSliceTable, 0, sizeof(ctx->VSliceTable) );
   memset( ctx->CHSliceTable, 0, sizeof(ctx->CHSliceTable) );
   memset( ctx->CVSliceTable, 0, sizeof(ctx->CVSliceTable) );
   memset( ctx->HWindTable, 0, sizeof(ctx->HWindTable) );
   memset( ctx->VWindTable, 0, sizeof(ctx->VWindTable) );
   memset( ctx->TrajTable, 0, sizeof(ctx->TrajTable) );
   ctx->NumTraj = 0;

   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      ctx->Uvar[i] = ctx->Vvar[i] = ctx->Wvar[i] = -1;
   }
   ctx->TrajU = ctx->TrajV = ctx->TrajW = -1;

   memset( ctx->ExpressionList, 0, sizeof(ctx->ExpressionList) );

   ctx->Zoom = 1.0;
   memcpy( ctx->CTM, init_ctm, 16*sizeof(float) );
   ctx->CursorColor = &ctx->TrajColor[0];
}



/*
 * Deallocate a context and everything it points to.
 */
static void destroy_context( Context ctx, int close_window )
{
   if (close_window) {
      free_graphics( ctx );
   }

#ifdef CAVE
   if (cave_shmem) {
      afree( ctx, cave_shmem);
      afree( ctx->mempool, cave_shmem );
   }
   else {
      free( ctx );
      if (ctx->mempool) {
         free( ctx->mempool );
      }
   }
#else
   free( ctx );
   if (ctx->mempool) {
      free( ctx->mempool );
   }
#endif
}




/**********************************************************************/
/*****                Context-independent functions               *****/
/**********************************************************************/


/*
 * Do one-time initializations which are indepenent of all contexts.
 */
int vis5d_initialize( int cave_mode )
{
   int i;

   init_sync();
   init_queue();
   init_work();

   ALLOC_LOCK( GfxLock );
   ALLOC_LOCK( TrajLock );

#if defined(sgi) || defined(sunos5)
   for (i=0;i<MAX_WORKERS;i++) {
      WorkerPID[i] = 0;
   }
#endif

#ifndef CAVE
   init_graphics();
#endif

   if (cave_mode) {
#ifdef CAVE
      int size = CAVE_MEMORY_SIZE * 1024 * 1024;
      cave_shmem = CAVEUserSharedMemory( size );
      if (!cave_shmem) {
         printf("Error: CAVEUserSharedMemory(%d) failed!\n", CAVE_MEMORY_SIZE);
         exit(1);
      }
      ctx_table = amalloc( sizeof(Context *) * VIS5D_MAX_CONTEXTS,
                               cave_shmem );
#else
      printf("Error: CAVE support not compiled in!\n");
      exit(1);
#endif
   }
   else {
      ctx_table = malloc( sizeof(Context *) * VIS5D_MAX_CONTEXTS );
   }

   for (i=0;i<VIS5D_MAX_CONTEXTS;i++) {
      ctx_table[i] = NULL;
   }


   return 0;
}


/*
 * Call this clean up function before exiting.
 */
int vis5d_terminate( int close_windows )
{
   int i;

   if (close_windows) {
      for (i=0;i<VIS5D_MAX_CONTEXTS;i++) {
         if (ctx_table[i]) {
            free_graphics( ctx_table[i] );
         }
      }
   }

   FREE_LOCK( GfxLock );
   FREE_LOCK( TrajLock );

   terminate_work();
   terminate_queue();
   term_sync();
   terminate_graphics();

   /* Kill threads */
#ifdef sgi
   for (i=0;i<MAX_WORKERS;i++) {
      if (WorkerPID[i]) {
         kill( WorkerPID[i], SIGKILL );
         WorkerPID[i] = 0;
      }
   }
#endif
#ifdef sunos5
   for (i=0;i<MAX_WORKERS;i++) {
      if (WorkerPID[i]) {
         thr_kill( WorkerPID[i], SIGKILL );
         WorkerPID[i] = 0;
      }
   }
#endif

   return 0;
}


/*
 * Initialize 'nworkers' instances of the queue server tasks.
 */
int vis5d_workers( int nworkers )
{
   if (nworkers>=0 && nworkers<MAX_WORKERS) {
      NumThreads = nworkers + 1;
   }
   return 0;
}



/*
 * On single threaded systems, this function will pull one job off the
 * work queue and do it.  If multiple threads, this is a no-op.
 */
int vis5d_do_work( void )
{
   if (NumThreads==1) {
      int size, waiters;
      get_queue_info( &size, &waiters );
      if (size>0) {
         do_one_task( 0 );
      }
   }
   return 0;
}



/*
 * Check if there is work pending in the job queue.
 * Output:  pending_flag - 0 = queue is empty., 1 = queue is not empty
 */
int vis5d_check_work( int *pending_flag )
{
   if (any_work_pending()) {
      *pending_flag = 1;
   }
   else {
      *pending_flag = 0;
   }
   return 0;
}



/*
 * Block until the job queue is empty.
 */
int vis5d_finish_work( void )
{
   int size, waiters;
   if (NumThreads==1) {
      while (1) {
         get_queue_info( &size, &waiters );
         if (size==0) {
            break;
         }
         else {
            do_one_task( 0 );
         }
      }
   }
   else {
      while (1) {
         get_queue_info( &size, &waiters );
         if (size==0 && waiters==NumThreads-1) {
            break;
         }
      }
   }
   return 0;
}




/**********************************************************************/
/*****                Context-dependent functions                 *****/
/**********************************************************************/



/*
 * Allocate a new vis5d context and return its index number or VIS5D_FAIL
 * if the limit on contexts has been hit.
 */
int vis5d_alloc_context( void )
{
   int i;

   for (i=0;i<VIS5D_MAX_CONTEXTS;i++) {
      if (ctx_table[i]==NULL) {
         return i;
      }
   }

   return VIS5D_FAIL;
}


/*
 * Get rid of one context's resources
 */
int vis5d_destroy_context( int index, int close_window )
{
   Context ctx;

   if (ctx_table[index]) {
      destroy_context( ctx_table[index], close_window );
      ctx_table[index] = NULL;
   }

   return 0;
}



/*
 * Begin initialization stage
 */
int vis5d_init_begin( int index )
{
   Context ctx;
   static int first_time = 1;

   /*printf("sizeof(vis5d_context)=%d\n", sizeof(struct vis5d_context) );*/

   if (first_time) {
      int i;
      for (i=0;i<VIS5D_MAX_CONTEXTS;i++) {
         ctx_table[i] = NULL;
      }
      first_time = 0;
   }

   if (ctx_table[index]) {
      destroy_context( ctx_table[index], 0 /*close_window*/ );
      ctx_table[index] = NULL;
   }

   ctx = ctx_table[index] = new_context();
   init_context( ctx );
   ctx->context_index = index;

   ctx->FontName[0] = 0;
   ctx->InsideInit = 1;

   ctx->LogFlag = 0;

   return 0;
}



/*
 * Make a 3-D rendering window and attach it to this vis5d context.
 * Input:  index - the context number
 *         title - character string window title
 *         x, y - position of window relative to upper-left corner of screen
 *         width, height - size of window
 */
int vis5d_init_window( int index, char *title, int x, int y,
		       int width, int height )
{
   CONTEXT("vis5d_make_window");

   if (make_3d_window( ctx, title, x, y, width, height )) {
      return 0;
   }
   else {
      return VIS5D_FAIL;
   }
}



#ifdef SGI_GL
/*
 * Attach a "pure" IRIS GL window to this vis5d context.
 */
int vis5d_init_gl_window( int index, Display *dpy, Window window, long winid )
{
   CONTEXT("vis5d_init_gl_window");
   use_gl_window( ctx, dpy, window, winid );
   return 0;
}
#endif



#ifdef SGI_GL
/*
 * Attach a "mixed-mode" IRIS GL window to this vis5d context.
 */
int vis5d_init_glx_window( int index, Display *dpy, Window window,
                          GLXconfig *glctx )
{
   CONTEXT("vis5d_init_glx_window");
   use_glx_window( ctx, dpy, window, glctx );
   return 0;
}
#endif



#ifdef OPENGL
/*
 * Attach an OpenGL window and rendering context to this vis5d context.
 */
int vis5d_init_opengl_window( int index, Display *dpy, Window window,
			      GLXContext ctx )
{

   return 0;
}
#endif



/*
 * Specify the size of memory pool for the context in megabytes.
 */
int vis5d_init_memory( int index, int mbs )
{
   CONTEXT("vis5d_init_memory");
   ctx->MegaBytes = mbs;
   return 0;
}



/*
 * Specify the aspect ratio of the 3-D box.
 * Input: index - the context number
 *        alon, alat, ahgt - aspect ratios:  latitude:longitude:height
 */
int vis5d_init_box( int index, float alon, float alat, float ahgt )
{
   CONTEXT("vis5d_init_box");
   ctx->Ax = alon;
   ctx->Ay = alat;
   ctx->Az = ahgt;
   return 0;
}



int vis5d_init_log( int index, float scale, float exponent )
{
   CONTEXT("vis5d_init_log");

   ctx->LogFlag = 1;
   ctx->LogScale = scale;
   ctx->LogExp = exponent;
   return 0;
}



int vis5d_init_map( int index, char *mapname )
{
   CONTEXT("vis5d_init_map");
   strcpy( ctx->MapName, mapname );
   return 0;
}



int vis5d_init_topo( int index, char *toponame, int highres_flag )
{
   CONTEXT("vis5d_init_topo");
   strcpy( ctx->TopoName, toponame );
   ctx->HiResTopo = highres_flag;
   return 0;
}


int vis5d_init_path( int index, char *pathname )
{
   int len;
   CONTEXT("vis5d_init_path");

   strcpy( ctx->Path, pathname );
   /* Make sure Path ends with a / */
   len = strlen(ctx->Path);
   if (len>0 && ctx->Path[len-1]!='/') {
      strcat( ctx->Path, "/" );
   }

   return 0;
}


int vis5d_init_texture( int index, char *texturename )
{
   CONTEXT("vis5d_init_texture");
   strcpy( ctx->TextureName, texturename );
   return 0;
}


int vis5d_init_firstarea( int index, int area )
{
   CONTEXT("vis5d_open_areas");
   ctx->FirstArea = area;
   return 0;
}


int vis5d_init_sequence( int index, char *sequencename )
{
   CONTEXT("vis5d_init_sequence");
   strcpy( ctx->SequenceName, sequencename );
   return 0;
}



/*
 * Override the projection from the data file with this one.
 */
int vis5d_init_projection( int index, int projection, float *projargs )
{
   CONTEXT("vis5d_init_projection");
   ctx->UserProjection = projection;
   if (projargs) {
      ctx->UserProjArgs = malloc( MAXPROJARGS*sizeof(float) );
      memcpy( ctx->UserProjArgs, projargs, MAXPROJARGS*sizeof(float) );
   }
   return 0;
}



/*
 * Override the vertical coordinate system  from the data file with this one.
 */
int vis5d_init_vertical( int index, int vertical, float *vertargs )
{
   CONTEXT("vis5d_init_vertical");
   ctx->UserVerticalSystem = vertical;
   if (vertargs) {
      ctx->UserVertArgs = malloc( MAXVERTARGS*sizeof(float) );
      memcpy( ctx->UserVertArgs, vertargs, MAXVERTARGS*sizeof(float) );
   }
   return 0;
}


static void load_topo_and_map( Context ctx )
{
   char name[1000];

   /*** Load topography ***/
   if (ctx->Path[0]) {
      strcpy( name, ctx->Path );
      strcat( name, ctx->TopoName );
   }
   else {
      strcpy( name, ctx->TopoName );
   }
   if (name[0]) {
      ctx->TopoFlag = init_topo( ctx, name, ctx->TextureFlag, ctx->HiResTopo );
   }
   else {
      ctx->TopoFlag = 0;
   }

   /*** Load texture, areas, or image sequence ***/
   init_image(ctx);
   if (ctx->TextureName[0]) {
      /* Load an SGI .rgb image as the texture */
      ctx->TextureFlag = read_texture_image( ctx, ctx->TextureName );
   }
#ifdef MCIDAS
   else if (ctx->FirstArea>0) {
      /* Read a sequence of McIDAS areas as textures */
      ctx->TextureFlag = read_texture_areas( ctx, ctx->FirstArea );
   }
#endif
   else if (ctx->SequenceName[0]) {
      /* Read a raw image sequence as textures */
      ctx->TextureFlag = read_texture_sequence( ctx, ctx->SequenceName );
   }
   else {
      ctx->TextureFlag = 0;
   }

   /*** Load map ***/
   if (ctx->MapName[0] == 0) {
      /* depending on domain size, pick appropriate default */
      float latn, lats, lonw, lone;
      latlon_bounds(ctx, &lats, &latn, &lonw, &lone);
      if (30.0 < latn && latn < 80.0 &&
          0.0 < lats && lats < 45.0 &&
          80.0 < lonw && lonw < 180.0 &&
          30.0 < lone && lone < 115.0) {
         sprintf( name, "%s%s", ctx->Path, USAFILE);
      }
      else {
         sprintf( name, "%s%s", ctx->Path, WORLDFILE );
      }
   }
   else {
      /* concatenate path and user-supplied map file name */
      if (ctx->Path[0]) {
         strcpy( name, ctx->Path );
         strcat( name, ctx->MapName );
      }
      else {
         strcpy( name, ctx->MapName );
      }
   }
   if (name[0]) {
      ctx->MapFlag = init_map( ctx, name );
   }
   else {
      ctx->MapFlag = 0;
   }

   free_topo( ctx );
}




/*
 * Initialize graphics-related and other global vars AFTER the data
 * set has been loaded.
 */
static void initialize_stuff( Context ctx )
{
   static unsigned int nice_color[] = {
      PACK_COLOR( 0xffU, 0xffU, 0x00U, 0xffU ),
      PACK_COLOR( 0xffU, 0xffU, 0xffU, 0xffU ),
      PACK_COLOR( 0xffU, 0x00U, 0xffU, 0xffU ),
      PACK_COLOR( 0x00U, 0xffU, 0xffU, 0xffU ),
      PACK_COLOR( 0xffU, 0x50U, 0x50U, 0xffU ),
      PACK_COLOR( 0x00U, 0xffU, 0x00U, 0xffU ),
      PACK_COLOR( 0x00U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0x80U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0x80U, 0xffU, 0x80U, 0xffU ),
      PACK_COLOR( 0xffU, 0x80U, 0x80U, 0xffU ),
      PACK_COLOR( 0xffU, 0x80U, 0x00U, 0xffU ),
      PACK_COLOR( 0xffU, 0x00U, 0x80U, 0xffU ),
      PACK_COLOR( 0x80U, 0xffU, 0x00U, 0xffU ),
      PACK_COLOR( 0x00U, 0xffU, 0x80U, 0xffU ),
      PACK_COLOR( 0x80U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0x00U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0xc0U, 0xc0U, 0x40U, 0xffU ),
      PACK_COLOR( 0xc0U, 0x40U, 0xc0U, 0xffU ),
      PACK_COLOR( 0x40U, 0xc0U, 0xc0U, 0xffU ),
      PACK_COLOR( 0x40U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0x80U, 0x80U, 0xffU, 0xffU ),
      PACK_COLOR( 0xc0U, 0xffU, 0xc0U, 0xffU ),
      PACK_COLOR( 0x40U, 0xc0U, 0xffU, 0xffU ),
      PACK_COLOR( 0x04U, 0xffU, 0xc0U, 0xffU ),
      PACK_COLOR( 0xffU, 0x40U, 0xc0U, 0xffU )  };
   int var, time, i, j, k, w;

   /* Button matrix colors */
   k = 0;
   for (i=0; i<MAXVARS; i++) {
      for (j=0; j<5; j++) {
         ctx->Color[i][j] = nice_color[k];
         k++;
         if (k>=sizeof(nice_color)/sizeof(unsigned int)) {
            k = 0;
         }
      }
      /* volume button is always white */
      ctx->Color[i][5] = PACK_COLOR( 0xffU, 0xffU, 0xffU, 0xffU);
   }

   ctx->HWindColor[0] = PACK_COLOR( 0x00U, 0xffU, 0xffU, 0xffU );
   ctx->HWindColor[1] = PACK_COLOR( 0x00U, 0xffU, 0x00U, 0xffU );
   ctx->VWindColor[0] = PACK_COLOR( 0x00U, 0x88U, 0xffU, 0xffU );
   ctx->VWindColor[1] = PACK_COLOR( 0xffU, 0x00U, 0xffU, 0xffU );
   ctx->StreamColor[0] = PACK_COLOR( 0xffU, 0x88U, 0x00U, 0xffU );
   ctx->StreamColor[1] = PACK_COLOR( 0xffU, 0xffU, 0x00U, 0xffU );

   ctx->TrajColor[0] = PACK_COLOR( 0xffU, 0xffU, 0xffU, 0xffU );
   ctx->TrajColor[1] = PACK_COLOR( 0xffU, 0xffU, 0x00U, 0xffU );
   ctx->TrajColor[2] = PACK_COLOR( 0xffU, 0x00U, 0xffU, 0xffU );
   ctx->TrajColor[3] = PACK_COLOR( 0x00U, 0xffU, 0x00U, 0xffU );
   ctx->TrajColor[4] = PACK_COLOR( 0x00U, 0x88U, 0xffU, 0xffU );
   ctx->TrajColor[5] = PACK_COLOR( 0xffU, 0x00U, 0x88U, 0xffU );
   ctx->TrajColor[6] = PACK_COLOR( 0xffU, 0x88U, 0x00U, 0xffU );
   ctx->TrajColor[7] = PACK_COLOR( 0x00U, 0x88U, 0x88U, 0xffU );

   /* Graphics structs */
   for (var=0;var<MAXVARS;var++) {
      for (time=0;time<MAXTIMES;time++) {
         ctx->SurfTable[var][time].valid = 0;
         ctx->HSliceTable[var][time].valid = 0;
         ctx->VSliceTable[var][time].valid = 0;
         ctx->CHSliceTable[var][time].valid = 0;
         ctx->CVSliceTable[var][time].valid = 0;
      }
   }

   for (var=0;var<VIS5D_WIND_SLICES;var++) {
      for (time=0;time<MAXTIMES;time++) {
         ctx->HWindTable[var][time].valid = 0;
         ctx->VWindTable[var][time].valid = 0;
         ctx->StreamTable[var][time].valid = 0;
      }
   }

   /* Initialize surfaces, slices */
   for (var=0;var<ctx->NumVars;var++) {
      init_graphics_pos( ctx, var );
   }
   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      /* hwind slices */
      ctx->HWindLevel[w] = (float) (ctx->WindNl-1) / 2.0;
      new_hslice_pos( ctx, ctx->HWindLevel[w], &ctx->HWindZ[w], &ctx->HWindHgt[w] );
      ctx->HWindDensity[w] = 1.0;
      ctx->HWindScale[w] = 1.0;
      /* vwind slices */
      ctx->VWindR1[w] = (float) (ctx->Nr-1) / 2.0;
      ctx->VWindC1[w] = 0.0;
      new_vslice_pos( ctx, ctx->VWindR1[w], ctx->VWindC1[w],
                      &ctx->VWindX1[w], &ctx->VWindY1[w],
                      &ctx->VWindLat1[w], &ctx->VWindLon1[w] );
      ctx->VWindR2[w] = (float) (ctx->Nr-1) / 2.0;
      ctx->VWindC2[w] = (float) (ctx->Nc-1);
      new_vslice_pos( ctx, ctx->VWindR2[w], ctx->VWindC2[w],
                      &ctx->VWindX2[w], &ctx->VWindY2[w],
                      &ctx->VWindLat2[w], &ctx->VWindLon2[w] );
      ctx->VWindDensity[w] = ctx->VWindScale[w] = 1.0;
      /* stream slices */
      ctx->StreamLevel[w] = (float) (ctx->WindNl-1) / 2.0;
      new_hslice_pos( ctx, ctx->StreamLevel[w], &ctx->StreamZ[w], &ctx->StreamHgt[w] );
      ctx->StreamDensity[w] = 1.0;
   }

   /* initialize cursor position */
   {
      float row, col, lev;

      row = (ctx->Nr-1) / 2.0;
      col = (ctx->Nc-1) / 2.0;
      lev = (ctx->MaxNl-1) / 2.0;

      grid_to_xyz( ctx, -1, -1, 1, &row, &col, &lev, &ctx->CursorX,
                   &ctx->CursorY, &ctx->CursorZ );
   }
}




/*
 * End of initialization stage.
 */
int vis5d_init_end( int index )
{
#ifdef CAVE
   void *pool;
#endif
   int memsize;
   float ratio;
   CONTEXT("vis5d_init_end");


   /*** Memory setup ***/
   if (ctx->MegaBytes==0) {
#ifdef CAVE
      abort();
#endif
      /* use malloc/free */
      if (!init_memory( ctx, 0 )) {
         return VIS5D_FAIL;
      }
      memsize = 0;
   }
   else {
      /* Use bounded memory management */
      if (ctx->MegaBytes<10) {
         ctx->MegaBytes = 10;
      }
      /* use 80% of MegaBytes */
      memsize = (int) ( (float) ctx->MegaBytes * 0.80 ) * MEGA;
#ifdef CAVE
      pool = amalloc(memsize,cave_shmem);
      if (!pool) {
         printf("Error: couldn't allocate %d bytes from shared CAVE pool\n",
                 memsize );
         return VIS5D_OUT_OF_MEMORY;
      }
      if (!init_shared_memory( ctx, pool, memsize )) {
         return VIS5D_FAIL;
      }
#else
      if (!init_memory( ctx, memsize )) {
         return VIS5D_FAIL;
      }
#endif
   }
   /* Let 2/5 of the memory pool be used for caching grid data. */
   if (memsize==0) {
      /* Grid cache size = 100MB */
      if (!init_grid_cache( ctx, 100*1024*1024, &ratio )) {
         return VIS5D_OUT_OF_MEMORY;
      }
   }
   else {
      if (!init_grid_cache( ctx, memsize * 2 / 5, &ratio )) {
         return VIS5D_OUT_OF_MEMORY;
      }
   }
   /* Read some or all of the grid data into main memory now.  If we */
   /* have enough memory, the whole file will be loaded.  Otherwise, */
   /* an arbitrary set of grids will be loaded. */
   if (ctx->PreloadCache) {
      preload_cache(ctx);
   }
   if (memsize!=0) {
      /* check if there's enough memory left after loading the data set */
      int min_mem = MAX( memsize/3, 3*MEGA );
      if (mem_available(ctx)<min_mem) {
         printf("Not enough memory left for graphics (only %d bytes free)\n",
                mem_available(ctx));
         return VIS5D_OUT_OF_MEMORY;
      }
   }
   ctx->VeryLarge = (ratio < VERY_LARGE_RATIO);
   if (ctx->VeryLarge) printf("Using VeryLarge option - graphics may be slow\n");

   init_anim(ctx);

   /*** setup map proj and vert coord system, make 3-D box */
   if (!setup_projection( ctx )) {
      return VIS5D_FAIL;
   }
   if (!setup_vertical_system( ctx )) {
      return VIS5D_FAIL;
   }
   make_box( ctx, ctx->Ax, ctx->Ay, ctx->Az );

   load_topo_and_map( ctx );

   /*** Miscellaneous ***/
   if (ctx->FontName[0]) {
      /* use non-default font */
      set_3d_font( ctx, ctx->FontName, ctx->FontHeight );
   }

   compute_wind_levels(ctx);
   initialize_stuff(ctx);
   if (!init_traj(ctx)) {
      return VIS5D_FAIL;
   }

   if (ctx->CurvedBox) {
      /* Can't do volume rendering with curved box */
      ctx->Volume = NULL;
   }
   else {
      ctx->Volume = alloc_volume( ctx, ctx->Nr, ctx->Nc, ctx->MaxNl );
   }

   /*** Create threads ***/
#ifdef sgi
   if (NumThreads>1 && WorkerPID[0]==0) {
      /* Fork off the worker threads if we haven't already */
      if (NumThreads>1)   WorkerPID[0] = sproc( work, PR_SALL, 1 );
      if (NumThreads>2)   WorkerPID[1] = sproc( work, PR_SALL, 2 );
      if (NumThreads>3)   WorkerPID[2] = sproc( work, PR_SALL, 3 );
      if (NumThreads>4)   WorkerPID[3] = sproc( work, PR_SALL, 4 );
      if (NumThreads>5)   WorkerPID[4] = sproc( work, PR_SALL, 5 );
      if (NumThreads>6)   WorkerPID[5] = sproc( work, PR_SALL, 6 );
      if (NumThreads>7)   WorkerPID[6] = sproc( work, PR_SALL, 7 );
      if (NumThreads>8)   WorkerPID[7] = sproc( work, PR_SALL, 8 );
   }
#endif
#ifdef sunos5
   if (Threadsers>1 && WorkerPID[0]==0) {
      if (NumThreads>1)   thr_create( NULL, 0, work, 1, 0, &WorkerPID[0] );
      if (NumThreads>2)   thr_create( NULL, 0, work, 2, 0, &WorkerPID[1] );
      if (NumThreads>3)   thr_create( NULL, 0, work, 3, 0, &WorkerPID[2] );
      if (NumThreads>4)   thr_create( NULL, 0, work, 4, 0, &WorkerPID[3] );
   }
#endif

   ctx->InsideInit = 0;

   return 0;
}



/*
 * Open the named v5d file and read its header.  This should be called
 * before v5d_init_end().  If it's called after vis5d_init_end() the
 * current dataset will be replace by the named one.
 */
int vis5d_open_gridfile( int index, char *name, int read_flag )
{
   float ratio;
   CONTEXT("vis5d_open_gridfile");

   ctx->PreloadCache = read_flag;

   if (ctx->DataFile[0]) {
      /* already have a dataset loaded, replace it with the new one */
      v5dCloseFile( &ctx->G );
      free_all_graphics( ctx );
      clear_context( ctx );
      strcpy( ctx->DataFile, name );
      if (open_gridfile( ctx, name )) {

         if (ctx->memory_limit==0) {
            /* Grid cache size = 100MB */
            if (!init_grid_cache( ctx, 100*1024*1024, &ratio )) {
               return VIS5D_FAIL;
            }
         }
         else {
            if (!init_grid_cache( ctx, ctx->memory_limit * 2 / 5, &ratio )) {
               return VIS5D_FAIL;
            }
         }
         if (ctx->PreloadCache) {
            preload_cache(ctx);
         }
         ctx->VeryLarge = (ratio < VERY_LARGE_RATIO);
         if (ctx->VeryLarge) printf("Using VeryLarge option - graphics may be slow\n");

         init_anim(ctx);

         /*** setup map proj and vert coord system */
         if (!setup_projection( ctx )) {
            return VIS5D_FAIL;
         }
         if (!setup_vertical_system( ctx )) {
            return VIS5D_FAIL;
         }
         make_box( ctx, ctx->Ax, ctx->Ay, ctx->Az );

         load_topo_and_map( ctx );

         compute_wind_levels(ctx);

         initialize_stuff(ctx);
         if (!init_traj(ctx)) {
            return VIS5D_FAIL;
         }

         if (ctx->CurvedBox) {
            /* Can't do volume rendering with curved box */
            ctx->Volume = NULL;
         }
         else {
            ctx->Volume = alloc_volume( ctx, ctx->Nr, ctx->Nc, ctx->MaxNl );
         }
      }
      else {
         return VIS5D_FAIL;
      }
   }
   else {
      strcpy( ctx->DataFile, name );
      if (open_gridfile( ctx, name )) {
         return 0;
      }
      else {
         return VIS5D_FAIL;
      }
   }
   return 0;
}



/*** Time Functions ***/

int vis5d_get_numtimes( int index, int *numtimes )
{
   CONTEXT("vis5d_get_numtimes");
   *numtimes = ctx->NumTimes;
   return 0;
}


int vis5d_get_time_stamp( int index, int timestep, int *day, int *time )
{
  CONTEXT("vis5d_get_time_stamp")
  if (timestep<0 || timestep>=ctx->NumTimes) {
     return VIS5D_BAD_TIME_STEP;
  }
  else {
     *day = ctx->DayStamp[timestep];
     *time = ctx->TimeStamp[timestep];
     return 0;
  }
}


vis5d_set_time_stamp( int index, int timestep, int day, int time )
{
  CONTEXT("vis5d_set_time_stamp")
  if (timestep<0 || timestep>=ctx->NumTimes) {
     return VIS5D_BAD_TIME_STEP;
  }
  else {
     ctx->DayStamp[timestep] = day;
     ctx->TimeStamp[timestep] = time;
     return 0;
  }
}


int vis5d_set_timestep( int index, int time )
{
  CONTEXT("vis5d_set_timestep")
  if (time<0 || time>=ctx->NumTimes) {
     return VIS5D_BAD_TIME_STEP;
  }
  ctx->CurTime = time;
  ctx->Redraw = 1;
  return 0;
}


int vis5d_get_timestep( int index, int *curtime )
{
  CONTEXT("vis5d_get_timestep")
  *curtime = ctx->CurTime;
  return 0;
}



/*** Variable Functions ***/

int vis5d_get_numvars( int index, int *numvars )
{
   CONTEXT("vis5d_get_numvars");
   *numvars = ctx->NumVars;
   return 0;
}



/*
 * Find the number of the named variable.
 */
int vis5d_find_var( int index, char *name )
{
   int i;
   CONTEXT("vis5d_find_var");

   for (i=0;i<ctx->NumVars;i++) {
      if (strcmp(ctx->VarName[i],name)==0) {
         return i;
      }
   }
   return VIS5D_FAIL;
}


int vis5d_get_var_name( int index, int var, char *name )
{
   CONTEXT("vis5d_get_var_name");
   if (var>=0 && var<ctx->NumVars) {
      strcpy( name, ctx->VarName[var] );
      return 0;
   }
   else {
      return VIS5D_BAD_VAR_NUMBER;
   }
}


int vis5d_get_var_units( int index, int var, char *units )
{
   CONTEXT("vis5d_get_var_units");
   if (var>=0 && var<ctx->NumVars) {
      strcpy( units, ctx->Units[var] );
      return 0;
   }
   else {
      return VIS5D_BAD_VAR_NUMBER;
   }
}


/*
 * Return the type of a variable.
 * Input:  index - the context index
 *         var - variable number
 * Output:  type - one of:
 *                     VIS5D_REGULAR - a regular variable
 *                     VIS5D_CLONE - clone of another variable
 *                     VIS5D_EXT_FUNC - computed with external Fortran function
 *                     VIS5D_EXPRESSION - computed with a simple expression
 */
int vis5d_get_var_type( int index, int var, int *type )
{
   CONTEXT("vis5d_get_var_name");
   if (var>=0 && var<ctx->NumVars) {
      *type = ctx->VarType[var];
      return 0;
   }
   else {
      return VIS5D_BAD_VAR_NUMBER;
   }
}



int vis5d_get_var_info( int index, int var, void *info )
{
   CONTEXT("vis5d_get_var_info");
   if (var>=0 && var<ctx->NumVars) {
      if (ctx->VarType[var]==VIS5D_CLONE) {
         int *cloneof = (int *) info;
         *cloneof = ctx->CloneTable[var];
      }
      else if (ctx->VarType[var]==VIS5D_EXT_FUNC) {
         char *funcname = (char *) info;
         strcpy( funcname, ctx->VarName[var] );
      }
      else if (ctx->VarType[var]==VIS5D_EXPRESSION) {
         char *funcname = (char *) info;
         strcpy( funcname, ctx->ExpressionList[var] );
      }
      return 0;
   }
   else {
      return VIS5D_BAD_VAR_NUMBER;
   }
}


int vis5d_get_var_range( int index, int var, float *min, float *max )
{
  CONTEXT("vis5d_get_var_range")
   if (var>=0 && var<ctx->NumVars) {
      *min = ctx->MinVal[var];
      *max = ctx->MaxVal[var];
      return 0;
   }
   else {
      return VIS5D_BAD_VAR_NUMBER;
   }
}



int vis5d_set_wind_vars( int index,
                         int u1, int v1, int w1,
                         int u2, int v2, int w2,
                         int utraj, int vtraj, int wtraj )
{
   int time, ws;
   CONTEXT("vis5d_set_wind_vars")

   ctx->Uvar[0] = u1;
   ctx->Vvar[0] = v1;
   ctx->Wvar[0] = w1;
   ctx->Uvar[1] = u2;
   ctx->Vvar[1] = v2;
   ctx->Wvar[1] = w2;
   ctx->TrajU = utraj;
   ctx->TrajV = vtraj;
   ctx->TrajW = wtraj;

   compute_wind_levels( ctx );

   /* recompute slices since vars may have changed */
   for (ws=0; ws<VIS5D_WIND_SLICES; ws++) {
      for (time=0; time<ctx->NumTimes; time++) {
         if (ctx->HWindTable[ws][time].valid) {
            request_hwindslice( ctx, time, ws, 0 );
         }
         if (ctx->VWindTable[ws][time].valid) {
            request_vwindslice( ctx, time, ws, 0 );
         }
         if (ctx->StreamTable[ws][time].valid) {
            request_streamslice( ctx, time, ws, 0 );
         }
      }
   }

   return 0;
}


int vis5d_get_wind_vars( int index,
                         int *u1, int *v1, int *w1,
                         int *u2, int *v2, int *w2,
                         int *utraj, int *vtraj, int *wtraj )
{
   CONTEXT("vis5d_get_wind_vars")
   *u1 = ctx->Uvar[0];
   *v1 = ctx->Vvar[0];
   *w1 = ctx->Wvar[0];
   *u2 = ctx->Uvar[1];
   *v2 = ctx->Vvar[1];
   *w2 = ctx->Wvar[1];
   *utraj = ctx->TrajU;
   *vtraj = ctx->TrajV;
   *wtraj = ctx->TrajW;
   return 0;
}



/* Reset the position and attributes of all graphics for the given var. */
int vis5d_reset_var_graphics( int index, int newvar )
{
  CONTEXT("vis5d_reset_var_graphics")
  init_graphics_pos(ctx, newvar);
  return 0;
}



/*** Grid Functions ***/


int vis5d_get_size( int index, int *nr, int *nc, int nl[], int lowlev[],
                    int *maxnl, int *maxnlVar, int *windnl, int *windlow )
{
  int i;
  CONTEXT("vis5d_get_size")

  if (nr)        *nr = ctx->Nr;
  if (nc)        *nc = ctx->Nc;
  if (nl)        for (i=0; i<ctx->NumVars; i++)  nl[i] = ctx->Nl[i];
  if (lowlev)    for (i=0; i<ctx->NumVars; i++)  lowlev[i] = ctx->LowLev[i];
  if (maxnl)     *maxnl = ctx->MaxNl;
  if (maxnlVar)  *maxnlVar = ctx->MaxNlVar;
  if (windnl)    *windnl = ctx->WindNl;
  if (windlow)   *windlow = ctx->WindLow;
  return 0;
}


/*
 * Return a 3-D data grid.
 * Input:  index - the context index
 *         time, var - timestep and variable number
 *         data - address to put data, assumed to be large enough.
 */
int vis5d_get_grid( int index, int time, int var, float *data )
{
   float *grid;
   CONTEXT("vis5d_get_grid");

   grid = get_grid( ctx, time, var );
   memcpy( data, grid, ctx->Nr*ctx->Nc*ctx->Nl[var]*sizeof(float) );
   release_grid( ctx, time, var, grid );
   return 0;
}


/*
 * Put new data into a 3-D grid.
 * Input:  index - the context index
 *         time, var - timestep and variable number
 *         data - array of data, assumed to be the right size.
 */
int vis5d_put_grid( int index, int time, int var, float *data )
{
   CONTEXT("vis5d_get_grid");
   if (put_grid( ctx, time, var, data )) {
      return 0;
   }
   else {
      return VIS5D_FAIL;
   }
}


/*
 * Return a variable's grid value at a specific position, using current
 * timestep.
 */
int vis5d_get_grid_value( int index, int var,
                          float row, float column, float level,
                          float *value )
{
   CONTEXT("vis5d_get_grid_value");
   if (var<0 || var>=ctx->NumVars) {
      return VIS5D_BAD_VAR_NUMBER;
   }
   if (row<0.0 || row>(ctx->Nr-1) ||
       column<0.0 || column>(ctx->Nc-1) ||
       level<0.0 || level>(ctx->Nl[var]-1)) {
      /* position is out of bounds */
      return VIS5D_BAD_VALUE;
   }
   *value = interpolate_grid_value( ctx, ctx->CurTime, var,
                                    row, column, level );
   return 0;
}


/*
 * Control the VeryLarge flag.
 * Input:  index - context index
 *         mode - one of VIS5D_ON, VIS5D_OFF, VIS5D_TOGGLE, or VIS5D_GET
 * Return:  VIS5D_ON or VIS5D_OFF, after function is applied
 */
int vis5d_verylarge_mode( int index, int mode )
{
  int *val;
  CONTEXT("vis5d_verylarge_mode");
  val = &ctx->VeryLarge;
  switch (mode) {
    case VIS5D_OFF:
      *val = 0;
      break;
    case VIS5D_ON:
      if (*val == 0) printf("Using VeryLarge option - graphics may be slow\n");
      *val = 1;
      break;
    case VIS5D_TOGGLE:
      *val = *val ? 0 : 1;
      break;
    case VIS5D_GET:
      break;
    default:
      printf("bad mode (%d) in vis5d_verylarge_mode\n", mode);
      return VIS5D_BAD_MODE;
  }
  return *val;
}


/*** Map Projection and VCS Functions ***/

int vis5d_get_projection( int index, int *projection, float *projargs )
{
   CONTEXT("vis5d_get_projection");
   get_projection( ctx, projection, projargs );
   return 0;
}


int vis5d_get_vertical( int index, int *vertical, float *vertargs )
{
   CONTEXT("vis5d_get_vertical");
   get_vertical_system( ctx, vertical, vertargs );
   return 0;
}


int vis5d_get_curved( int index, int *curved )
{
  CONTEXT("vis5d_get_size")
  *curved = ctx->CurvedBox;
  return 0;
}



/*** Topography and Map Functions ***/

int vis5d_check_topo( int index, int *topoflag )
{
  CONTEXT("vis5d_check_topo")
  *topoflag = ctx->TopoFlag;
  return 0;
}


int vis5d_check_map( int index, int *mapflag )
{
  CONTEXT("vis5d_check_map")
  *mapflag = ctx->MapFlag;
  return 0;
}


int vis5d_check_texture( int index, int *textureflag )
{
  CONTEXT("vis5d_check_texture")
  *textureflag = ctx->TextureFlag;
  return 0;
}


vis5d_get_topo_range( int index, float *MinTopoHgt, float *MaxTopoHgt )
{
  CONTEXT("vis5d_get_topo_range")
  *MinTopoHgt = ctx->MinTopoHgt;
  *MaxTopoHgt = ctx->MaxTopoHgt;
  return 0;
}


int vis5d_recolor_topo( int index, int reset )
{
   CONTEXT("vis5d_recolor_topo")
   if (reset) {
      init_topo_color_table( ctx, ctx->TopoColorTable, 256);
      recolor_topo(ctx, ctx->TopoColorTable, 256);
   }
   else {
      recolor_topo(ctx, ctx->TopoColorTable, 256);
   }
   ctx->Redraw = 1;
   return 0;
}



/*
 * Define a texture map.  The texture image is not copied; just the pointer
 * to it is saved.  The format of the texture image is that which is used
 * by the underlying graphics library (usually either arrays of 4-byte or
 * 1-byte color/intensity values.  The texture's width and height should
 * both be powers of two if using OpenGL.
 * Input:  timestep - which timestep this texture is to be shown
 *         width, height - size of texture in pixels
 *         components - how many color components per pixel:
 *                        1 = grayscale
 *                        3 = rgb
 *                        4 = rgba
 *         image - the texture image
 */
int vis5d_texture_image( int index, int timestep, int width, int height,
                         int components, void *image )
{
   CONTEXT("vis5d_texture");
   define_texture( ctx, timestep, width, height, components, image );
   return 0;
}




/*** Cloning, External Functions, and Expression Functions ***/


/*
 * Make a clone of a variable.
 * Input:  index - the context index
 *         var_to_clone - number of variable to clone
 *         newname - name to give the clone
 * Return:  VIS5D_FAIL if newname already used or the limit on number of
 *              variables has been hit.
 */
int vis5d_make_clone_variable( int index, int var_to_clone, char *newname,
                               int *newvar )
{
   int i, n;
   CONTEXT("vis5d_make_clone_variable")

   if (var_to_clone<0 || var_to_clone>=ctx->NumVars) {
      return VIS5D_BAD_VAR_NUMBER;
   }

   n = vis5d_find_var( index, newname );
   if (n>=0) {
      if (ctx->CloneTable[n]==var_to_clone) {
         /* this clone already made */
         *newvar = n;
         return 0;
      }
      else {
         /* clone's name is already used. */
         return VIS5D_FAIL;
      }
   }

   /* make the clone */
   *newvar = allocate_clone_variable( ctx, newname, var_to_clone );
   if (*newvar<0) {
      return VIS5D_FAIL;
   }
   else {
      return 0;
   }
}




int vis5d_compute_ext_func( int index, char *funcname, int *newvar )
{
  int i, var;
  char newname[100];
  char filename[100];
  int recompute_flag;

  CONTEXT("vis5d_compute_ext_func")

  printf("Computing external function %s\n", funcname );
  var = -1;
  *newvar = -1;
  recompute_flag = 0;

  /* look if this extfunc variable already exists */
  for (i=0;i<ctx->NumVars;i++) {
    if (strcmp(funcname,ctx->VarName[i])==0 && ctx->VarType[i]==VIS5D_EXT_FUNC) {
      /* this variable already exists so just recompute it */
      var = i;
      recompute_flag = 1;
      break;
    }
  }

  if (recompute_flag==0) {
    /* We're going to make a new variable, make sure the name is unique */
    strcpy( newname, funcname );
    for (i=0;i<ctx->NumVars;i++) {
      if (strcmp(funcname, ctx->VarName[i])==0) {
        strcat( newname, "'" );
      }
    }

    var = allocate_extfunc_variable( ctx, newname );
    if (var==-1) {
      /* unable to make a new variable */
      printf("Error: allocate_anal_variable failed\n");
      deallocate_variable( ctx, var );
      return VIS5D_FAIL;
    }
  }

  /* call the external function... */
  strcpy( filename, FUNCTION_PATH );
  strcat( filename, "/" );
  strcat( filename, funcname );

  if (compute_analysis_variable( ctx, var, filename )) {
    /* computation was successful */
    if (recompute_flag) {
      /* delete graphics */
      free_param_graphics( ctx, var );
      /* recalculate currently displayed graphics of var */
      if (ctx->DisplaySurf[var]) {
         for (i=0;i<ctx->NumTimes;i++) {
            request_isosurface( ctx, i, var, i==ctx->CurTime );
         }
      }
      if (ctx->DisplayHSlice[var]) {
         for (i=0;i<ctx->NumTimes;i++) {
            request_hslice( ctx, i, var, i==ctx->CurTime );
         }
      }
      if (ctx->DisplayVSlice[var]) {
         for (i=0;i<ctx->NumTimes;i++) {
            request_vslice( ctx, i, var, i==ctx->CurTime );
         }
      }
      if (ctx->DisplayCHSlice[var]) {
         for (i=0;i<ctx->NumTimes;i++) {
            request_chslice( ctx, i, var, i==ctx->CurTime );
         }
      }
      if (ctx->DisplayCVSlice[var]) {
         for (i=0;i<ctx->NumTimes;i++) {
            request_cvslice( ctx, i, var, i==ctx->CurTime );
         }
      }
      ctx->Redraw = 2;
    }
    else {
      *newvar = var;
    }
    compute_wind_levels(ctx);
    return 0;
  }
  else {
    /* there was an error */
    if (!recompute_flag) {
      deallocate_variable( ctx, var );
    }
    return VIS5D_FAIL;
  }
}



int vis5d_make_expr_var( int index, char *expression, char *newname,
                         char *mess, int *newvar, int *recompute )
{
   int result;
   CONTEXT("vis5d_make_expr_var")

   result = compute_var(ctx, expression, newname, mess, recompute );
   if (result<0) {
      /* error */
      return VIS5D_FAIL;
   }
   else {
      /* should this have same recompute logic as vis5d_compute_ext_func ?? */
      *newvar = result;
      strcpy( ctx->ExpressionList[*newvar], expression );
      return 0;
   }
}



/*** Rendering Functions ***/

int vis5d_signal_redraw( int index, int count )
/* Signal that a redraw is requested */
{
  CONTEXT("vis5d_signal_redraw")
  ctx->Redraw = count;
  return 0;
}


int vis5d_check_redraw( int index, int *redraw )
{
   CONTEXT("vis5d_check_redraw");
   *redraw = ctx->Redraw;
   return 0;
}


int vis5d_draw_frame( int index, int animflag )
{
   CONTEXT("vis5d_draw_frame");
   set_current_window( ctx );
   set_line_width( ctx->LineWidth );
   render_everything( ctx, animflag );
   ctx->Redraw = 0;
   return 0;
}


int vis5d_draw_3d_only( int index, int animflag )
{
   CONTEXT("vis5d_draw_3d_only");
   render_3d_only( ctx, animflag );
   ctx->Redraw = 0;
   return 0;
}


int vis5d_draw_2d_only( int index )
{
   CONTEXT("vis5d_draw_2d_only");
   render_2d_only( ctx );
   ctx->Redraw = 0;
   return 0;
}


int vis5d_swap_frame( int index )
{
   CONTEXT("vis5d_swap_frame");
   set_current_window( ctx );
   swap_3d_window();
   return 0;
}


int vis5d_invalidate_frames( int index )
{
  CONTEXT("vis5d_invalidate_frames")
  ctx->Redraw = 1;
  invalidate_frames(ctx);
  return 0;
}




vis5d_set_pointer( int index, int x, int y )
{
  CONTEXT("vis5d_set_fake_pointer")
  ctx->PointerX = x;
  ctx->PointerY = y;
  return 0;
}




/*
 * Control miscellaneous graphics rendering options.
 * Input:  index - context index
 *         what - one of:  VIS5D_BOX, VIS5D_CLOCK, VIS5D_MAP, VIS5D_TOPO,
 *                VIS5D_PERSPECTIVE, VIS5D_CONTOUR_NUMBERS, VIS5D_GRID_COORDS,
 *                VIS5D_PRETTY, VIS5D_INFO, VIS5D_PROBE,
 *                VIS5D_CURSOR, VIS5D_TEXTURE, or VIS5D_ANIMRECORD.
 *         mode - one of VIS5D_ON, VIS5D_OFF, VIS5D_TOGGLE, or VIS5D_GET
 * Return:  VIS5D_ON or VIS5D_OFF, after function is applied
 */
int vis5d_graphics_mode( int index, int what, int mode )
{
  int *val;
  CONTEXT("vis5d_graphics_mode")
  switch(what) {
    case VIS5D_BOX:
      val = &ctx->DisplayBox;
      break;
    case VIS5D_CLOCK:
      val = &ctx->DisplayClock;
      break;
    case VIS5D_MAP:
      val = &ctx->DisplayMap;
      break;
    case VIS5D_TOPO:
      val = &ctx->DisplayTopo;
      break;
    case VIS5D_PERSPECTIVE:
      val = &ctx->GfxProjection;
      break;
    case VIS5D_CONTOUR_NUMBERS:
      val = &ctx->ContnumFlag;
      break;
    case VIS5D_GRID_COORDS:
      val = &ctx->CoordFlag;
      break;
    case VIS5D_PRETTY:
      val = &ctx->PrettyFlag;
      /* it's up to the graphics library to test for the pretty flag */
      /* and implement whatever's needed. */
#ifdef LEAVEOUT
      if (mode==VIS5D_OFF && ctx->PrettyFlag) {
         set_pretty( 0 );
      }
      else if (mode==VIS5D_ON && !ctx->PrettyFlag) {
         set_pretty( 1 );
      }
#endif
      break;
    case VIS5D_INFO:
      val = &ctx->DisplayInfo;
      break;
    case VIS5D_PROBE:
      val = &ctx->DisplayProbe;
      break;
    case VIS5D_CURSOR:
      val = &ctx->DisplayCursor;
      break;
    case VIS5D_ANIMRECORD:
      val = &ctx->AnimRecord;
      break;
    case VIS5D_TEXTURE:
      val = &ctx->DisplayTexture;
      break;
    case VIS5D_DEPTHCUE:
      val = &ctx->DepthCue;
      break;
    default:
      printf("bad value (%d) in vis5d_graphics_mode(what)\n", what);
      return VIS5D_BAD_CONSTANT;
  }
  switch (mode) {
    case VIS5D_OFF:
      if (*val != 0) {
        ctx->Redraw = 1;
        invalidate_frames(ctx);
      }
      *val = 0;
      break;
    case VIS5D_ON:
      if (*val != 1) {
        ctx->Redraw = 1;
        invalidate_frames(ctx);
      }
      *val = 1;
      break;
    case VIS5D_TOGGLE:
      *val = *val ? 0 : 1;
      ctx->Redraw = 1;
      invalidate_frames(ctx);
      break;
    case VIS5D_GET:
      break;
    default:
      printf("bad mode (%d) in vis5d_graphics_mode\n", mode);
      return VIS5D_BAD_MODE;
  }

  return *val;
}



int vis5d_check_volume( int index, int *volume )
{
   CONTEXT("vis5d_check_volume");
   *volume = ctx->Volume ? 1 : 0;
   return 0;
}





/*
 * Control what "data"-graphics to display.
 * Input:  index - context index
 *         what - one of VIS5D_ISOSURF, VIS5D_HSLICE, VIS5D_VSLICE,
 *                VIS5D_CHSLICE, VIS5D_CVSLICE, VIS5D_VOLUME, VIS5D_TRAJ,
 *                VIS5D_HWIND, VIS5D_VWIND, or VIS5D_STREAM
 *         number - variable number, trajectory set or wind slice number
 *                  depending on 'what'.
 *         mode - one of VIS5D_ON, VIS5D_OFF, VIS5D_TOGGLE, or VIS5D_GET
 * Return:  current value after function is applied
 */
int vis5d_enable_graphics( int index, int type, int number, int mode )
{
  int *val;
  CONTEXT("vis5d_enable_graphics")
  switch(type) {
    case VIS5D_ISOSURF:
      val = &ctx->DisplaySurf[number];
      break;
    case VIS5D_HSLICE:
      val = &ctx->DisplayHSlice[number];
      break;
    case VIS5D_VSLICE:
      val = &ctx->DisplayVSlice[number];
      break;
    case VIS5D_CHSLICE:
      val = &ctx->DisplayCHSlice[number];
      break;
    case VIS5D_CVSLICE:
      val = &ctx->DisplayCVSlice[number];
      break;
    case VIS5D_VOLUME:
      switch (mode) {
        case VIS5D_OFF:
          if (number == ctx->CurrentVolume) {
            ctx->CurrentVolume = -1;
            ctx->Redraw = 1;
            invalidate_frames(ctx);
          }
          break;
        case VIS5D_ON:
          if (number != ctx->CurrentVolume) {
            ctx->CurrentVolume = number;
            ctx->Redraw = 1;
            invalidate_frames(ctx);
          }
          break;
        case VIS5D_TOGGLE:
          if (number == ctx->CurrentVolume) {
            ctx->CurrentVolume = -1;
          }
          else if (number != ctx->CurrentVolume) {
            ctx->CurrentVolume = number;
          }
          ctx->Redraw = 1;
          invalidate_frames(ctx);
          break;
        case VIS5D_GET:
          break;
        default:
          printf("bad mode (%d) in vis5d_enable_graphics\n", mode);
          return VIS5D_BAD_MODE;
      }
      return (number == ctx->CurrentVolume);
    case VIS5D_TRAJ:
      val = &ctx->DisplayTraj[number];
      break;
    case VIS5D_HWIND:
      val = &ctx->DisplayHWind[number];
      break;
    case VIS5D_VWIND:
      val = &ctx->DisplayVWind[number];
      break;
    case VIS5D_STREAM:
      val = &ctx->DisplayStream[number];
      break;
    default:
      return VIS5D_BAD_CONSTANT;
  }
  switch (mode) {
    case VIS5D_OFF:
      if (*val != 0) {
        ctx->Redraw = 1;
        invalidate_frames(ctx);
      }
      *val = 0;
      break;
    case VIS5D_ON:
      if (*val != 1) {
        ctx->Redraw = 1;
        invalidate_frames(ctx);
      }
      *val = 1;
      break;
    case VIS5D_TOGGLE:
      *val = *val ? 0 : 1;
      ctx->Redraw = 1;
      invalidate_frames(ctx);
      break;
    case VIS5D_GET:
      break;
    default:
      printf("bad mode (%d) in vis5d_enable_graphics\n", mode);
      return VIS5D_BAD_MODE;
  }
  return *val;
}


vis5d_get_volume( int index, int *CurrentVolume )
{
  CONTEXT("vis5d_get_volume")
  *CurrentVolume = ctx->CurrentVolume;
  return 0;
}


vis5d_set_volume( int index, int CurrentVolume )
{
  CONTEXT("vis5d_set_volume")
  ctx->CurrentVolume = CurrentVolume;
  return 0;
}



/*
 * Return a pointer to the color of the specified graphics element.
 * Input:  index - the context index
 *         type - one of:  VIS5D_ISOSURF, VIS5D_HSLICE, VIS5D_VSLICE,
 *                VIS5D_CHSLICE, VIS5D_CVSLICE, VIS5D_VOLUME, VIS5D_TRAJ,
 *                VIS5D_HWIND, VIS5D_VWIND, VIS5D_STREAM, VIS5D_BOX,
 *                VIS5D_BACKGROUND or VIS5D_LABEL.
 *         number - variable number, trajectory set number or wind slice
 *                  number depending on 'type'.
 * In/Out:  color - pointer to the color
 *
 * NOTE: THIS IS NO LONGER A VISIBLE API FUNCTION.
 */
static int get_graphics_color_address( int index, int type, int number,
                                       unsigned int **color )
{
  CONTEXT("get_graphics_color")
  switch(type) {
    case VIS5D_ISOSURF:
      *color = &ctx->Color[number][0];
      break;
    case VIS5D_HSLICE:
      *color = &ctx->Color[number][1];
      break;
    case VIS5D_VSLICE:
      *color = &ctx->Color[number][2];
      break;
    case VIS5D_CHSLICE:
      *color = &ctx->Color[number][3];
      break;
    case VIS5D_CVSLICE:
      *color = &ctx->Color[number][4];
      break;
    case VIS5D_VOLUME:
      *color = &ctx->Color[number][5];
      break;
    case VIS5D_TRAJ:
      *color = &ctx->TrajColor[number];
      break;
    case VIS5D_HWIND:
      *color = &ctx->HWindColor[number];
      break;
    case VIS5D_VWIND:
      *color = &ctx->VWindColor[number];
      break;
    case VIS5D_STREAM:
      *color = &ctx->StreamColor[number];
      break;
    case VIS5D_BOX:
      *color = &ctx->BoxColor;
      break;
    case VIS5D_MAP:
      *color = &ctx->MapColor;
      break;
    case VIS5D_BACKGROUND:
      *color = &ctx->BgColor;
      break;
    case VIS5D_LABEL:
      *color = &ctx->LabelColor;
      break;
    default:
      return VIS5D_BAD_CONSTANT;
  }
  return 0;
}



/*
 * Set the color of Vis5D graphic.
 * Input:  index - the context index
 *         type - the graphic type:  VIS5D_ISOSURF, VIS5D_TRAJ, etc.
 *         number - which variable or trajectory set
 *         red, green, blue, alpha - the color specified as 4 floating
 *                                   point values in the interval [0,1].
 */
int vis5d_set_color( int index, int type, int number,
                     float red, float green, float blue, float alpha )
{
   unsigned int *ptr;
   int n;
   int r, g, b, a;

   n = get_graphics_color_address( index, type, number, &ptr );
   if (n) {
      return n;
   }
   r = (int) (red * 255.0);
   g = (int) (green * 255.0);
   b = (int) (blue * 255.0);
   a = (int) (alpha * 255.0);
   *ptr = PACK_COLOR(r,g,b,a);
   return 0;
}



/*
 * Get the color of Vis5D graphic.
 * Input:  index - the context index
 *         type - the graphic type:  VIS5D_ISOSURF, VIS5D_TRAJ, etc.
 *         number - which variable or trajectory set
 * Output: red, green, blue, alpha - returns the color as 4 floating
 *                                   point values in the interval [0,1].
 */
int vis5d_get_color( int index, int type, int number,
                     float *red, float *green, float *blue, float *alpha )
{
   unsigned int *ptr;
   int n;
   int r, g, b, a;

   n = get_graphics_color_address( index, type, number, &ptr );
   if (n) {
      return n;
   }
   *red   = UNPACK_RED( *ptr )   / 255.0;
   *green = UNPACK_GREEN( *ptr ) / 255.0;
   *blue  = UNPACK_BLUE( *ptr )  / 255.0;
   *alpha = UNPACK_ALPHA( *ptr ) / 255.0;
   return 0;
}



/*
 * Set the color of the 3-D cursor to be equal to that of one of the
 * trajectory sets.
 */
int vis5d_set_cursor_color( int index, int traj_set )
{
  CONTEXT("vis5d_set_cursor_color")
  ctx->CursorColor = &ctx->TrajColor[traj_set];
  return 0;
}


/*
 * Return the address of a colortable.
 * Input:  index - context index
 *         type - one of:  VIS5D_CHSLICE, VIS5D_CVSLICE, VIS5D_VOLUME,
 *                or VIS5D_TOPO.
 *         number - variable number if type isn't VIS5D_TOPO
 * InOut:  colors - pointer to address of colortable
 */
int vis5d_get_color_table_address( int index, int type, int number,
                                   unsigned int **colors )
{
  CONTEXT("vis5d_get_color_table_address")
  switch (type) {
    case VIS5D_ISOSURF:
      *colors = ctx->IsoColors[number];
      break;
    case VIS5D_CHSLICE:
      *colors = ctx->CHSliceColors[number];
      break;
    case VIS5D_CVSLICE:
      *colors = ctx->CVSliceColors[number];
      break;
    case VIS5D_VOLUME:
      *colors = ctx->VolumeColors[number];
      break;
    case VIS5D_TOPO:
      *colors = ctx->TopoColorTable;
      break;
    case VIS5D_TRAJ:
      *colors = ctx->TrajColors[number];
      break;
    default:
      return VIS5D_BAD_CONSTANT;
  }
  return 0;
}



/*
 * Return the parameters which describe a colorbar curve.
 * Input:  index - context index
 *         graphic - one of:  VIS5D_ISOSURF, VIS5D_CHSLICE, VIS5D_CVSLICE,
 *                   VIS5D_VOLUME, VIS5D_TRAJ or VIS5D_TOPO.
 *         var - variable number if type isn't VIS5D_TOPO
 * InOut:  params - pointer to address of parameter array
 */
int vis5d_get_colorbar_params( int index, int graphic, int var, float **params )
{
   int i;
   unsigned int temptable[256], *table;
   int same;
   CONTEXT("vis5d_get_colorbar_params")

   switch (graphic) {
      case VIS5D_ISOSURF:
         *params = ctx->IsoColorParams[var];
         break;
      case VIS5D_CHSLICE:
         *params = ctx->CHSliceParams[var];
         break;
      case VIS5D_CVSLICE:
         *params = ctx->CVSliceParams[var];
         break;
      case VIS5D_VOLUME:
         *params = ctx->VolumeColorParams[var];
         break;
      case VIS5D_TRAJ:
         *params = ctx->TrajColorParams[var];
         break;
      case VIS5D_TOPO:
         *params = ctx->TopoColorParams;
         break;
      default:
         return VIS5D_FAIL;
   }

   return 0;
}



/*
 * Set the parameters which describe a colorbar curve, and use them
 * to recompute the color tabel of the colorbar.
 * Input:  index - context index
 *         graphic - one of:  VIS5D_ISOSURF, VIS5D_CHSLICE, VIS5D_CVSLICE,
 *                   VIS5D_VOLUME, VIS5D_TRAJ or VIS5D_TOPO.
 *         var - variable number if type isn't VIS5D_TOPO
 *         params - address of parameter array
 */
int vis5d_set_colorbar_params( int index, int graphic, int var, float params[] )
{
   float *p;
   int i;
   unsigned int *table;
   CONTEXT("vis5d_set_colorbar_params")

   switch (graphic) {
      case VIS5D_ISOSURF:
         p = ctx->IsoColorParams[var];
         break;
      case VIS5D_CHSLICE:
         p = ctx->CHSliceParams[var];
         break;
      case VIS5D_CVSLICE:
         p = ctx->CVSliceParams[var];
         break;
      case VIS5D_VOLUME:
         p = ctx->VolumeColorParams[var];
         break;
      case VIS5D_TRAJ:
         p = ctx->TrajColorParams[var];
         break;
      case VIS5D_TOPO:
         p = ctx->TopoColorParams;
         break;
      default:
         return VIS5D_FAIL;
   }

   for (i=0;i<7;i++) {
      p[i] = params[i];
   }

   vis5d_get_color_table_address( index, graphic, var, &table );
   vis5d_colorbar_recompute( table, 256, p, 1, 1 );
   return 0;
}




/*
 * Reset the RGB and/or Alpha curve parameters to their defaults.
 */
int vis5d_colorbar_init_params( float params[], int rgb_flag, int alpha_flag )
{
   if (rgb_flag) {
      params[CURVE] = DEFAULT_CURVE;
      params[BIAS]  = DEFAULT_BIAS;
   }
   if (alpha_flag) {
      params[ALPHAPOW] = DEFAULT_ALPHAPOW;
   }
   return 0;
}



/*
 * Recompute the RGB and/or Alpha table entries from their parameters.
 */
int vis5d_colorbar_recompute( unsigned int table[], int size, float params[],
                              int rgb_flag, int alpha_flag )
{
   float curve, rfact;
   int i;

   rfact = 0.5 * params[BIAS];
   curve = params[CURVE];

   if (alpha_flag) {
      if (params[ALPHAVAL]==-1) {
         params[MINALPHA] = 255;
         params[MAXALPHA] = 0;
      }
      else {
         params[MINALPHA] = params[ALPHAVAL];
         params[MAXALPHA] = params[ALPHAVAL];
      }
   }


   /* NOTE size-1 because last entry is used for missing data */
   for (i=0;i<size-1;i++) {
      int r,g,b,a;
      float s;

      /* compute s in [0,1] */
      s = (float) i / (float) (size-1);

      if (rgb_flag) {
	 float t = curve * (s - rfact);   /* t in [curve*-0.5,curve*0.5) */
	 r = 128.0 + 127.0 * atan( 7.0*t ) / 1.57;
	 g = 128.0 + 127.0 * (2 * exp(-7*t*t) - 1);
	 b = 128.0 + 127.0 * atan( -7.0*t ) / 1.57;
      }
      else {
         /* use current RGB */
         r = UNPACK_RED(   table[i] );
         g = UNPACK_GREEN( table[i] );
         b = UNPACK_BLUE(  table[i] );
      }

      if (alpha_flag) {
         if (params[ALPHAVAL]==-1) {
            /* Init alpha curve */
            a = 255.0 * pow( s, params[ALPHAPOW] );
         }
         else {
            /* Init constant alpha */
            a = params[ALPHAVAL];
         }
         if (a<params[MINALPHA])  params[MINALPHA] = a;
         if (a>params[MAXALPHA])  params[MAXALPHA] = a;
      }
      else {
         /* don't change alpha */
         a = UNPACK_ALPHA( table[i] );
      }

      /* store new packed color */
      table[i] = PACK_COLOR( r, g, b, a );
   }

   table[size-1] = PACK_COLOR( 0, 0, 0, 0 );
   return 0;
}




int vis5d_colorbar_set_alpha( float params[], float alpha )
{
   params[ALPHAVAL] = alpha;
   if (alpha<0) {
      params[MINALPHA] = 0;
      params[MAXALPHA] = 255;
      params[ALPHAPOW] = DEFAULT_ALPHAPOW;
   }
   else {
      params[MINALPHA] = alpha;
      params[MAXALPHA] = alpha;
      params[ALPHAVAL] = alpha;
   }
   return 0;
}





/*
 * Specify how semi-transparent graphics are to be rendered.
 * Input:  index - the context index
 *         mode - 0 = use screendoor method, 1 = use alpha blending
 */
int vis5d_alpha_mode( int index, int mode )
{
   CONTEXT("vis5d_alpha_mode");
   transparency_mode( ctx, mode );
   return 0;
}


/*
 * Specify the font to use in the 3-D window.
 * Input:  index - the context number
 *         fontname - the name of the font
 *         size - height of the font (only for GL FM fonts)
 */
int vis5d_font( int index, char *fontname, int size )
{
   CONTEXT("vis5d_init_font");
   if (ctx->InsideInit) {
      /* just save font name and size for now */
      strcpy( ctx->FontName, fontname );
      ctx->FontHeight = size;
   }
   else {
      /* set the font now! */
      set_3d_font( ctx, fontname, size );
   }
   return 0;
}


/*
 * Set the width of lines in 3-D window.
 */
int vis5d_linewidth( int index, float width )
{
   CONTEXT("vis5d_linewidth");
   ctx->LineWidth = (float) width;
   return 0;
}


/*
 * Do graphics library initializations.
 */
int vis5d_gl_setup( int index, long win_id, int width, int height )
{
   CONTEXT("vis5d_gl_init");
   context_init( ctx, win_id, width, height );
   return 0;
}



/*** 3-D View Functions ***/

int vis5d_set_matrix( int index, float ctm[4][4] )
{
  CONTEXT("vis5d_set_matrix")
  mat_copy(ctx->CTM, ctm);
  ctx->Redraw = 1;
  invalidate_frames( ctx );
  return 0;
}


int vis5d_get_matrix( int index, float ctm[4][4] )
{
  CONTEXT("vis5d_get_matrix")
  mat_copy(ctm, ctx->CTM);
  return 0;
}


/*
 * Set the view to either North, South, East, West, Top, or Bottom.
 * Input:  view - one of VIS5D_NORTH, VIS5D_SOUTH, .. VIS5D_BOTTOM
 */
vis5d_set_ortho_view( int index, int view )
{
  MATRIX ctm;
  CONTEXT("vis5d_ortho_view")

   switch (view) {
      case VIS5D_BOTTOM:
         make_matrix( 0.0, 180.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
         break;
      case VIS5D_NORTH:
         make_matrix( -90.0, 180.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
         break;
      case VIS5D_EAST:
         make_matrix( -90.0, -90.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
         break;
      case VIS5D_TOP:
         make_matrix( 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
         break;
      case VIS5D_SOUTH:
         make_matrix( -90.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
         break;
      case VIS5D_WEST:
         make_matrix( -90.0, 90.0, 0.0, 1.0, 0.0, 0.0, 0.0, ctm );
         vis5d_set_matrix(index, ctm);
      default:
         return VIS5D_BAD_CONSTANT;
   }

   ctx->FrontClip = 0.0;
   ctx->Zoom = 1.0;

   vis5d_invalidate_frames(index);

   return 0;
}



/* Set an arbitrary view rotation, scale and translation */
int vis5d_set_view( int index, float xrot, float yrot, float zrot,
                    float scale, float xtrans, float ytrans, float ztrans )
{
   MATRIX ctm;
   CONTEXT("vis5d_set_view")

   make_matrix( xrot, yrot, zrot, scale, xtrans, ytrans, ztrans, ctm );
   vis5d_set_matrix(index, ctm);
   vis5d_invalidate_frames(index);
   return 0;
}




int vis5d_get_camera( int index, int *perspec, float *front, float *zoom )
{
   CONTEXT("vis5d_get_camera");

   *perspec = ctx->GfxProjection;
   *front = ctx->FrontClip;
   *zoom = ctx->Zoom;
   return 0;
}



int vis5d_set_camera( int index, int perspec, float front, float zoom )
{
   CONTEXT("vis5d_set_camera");

   ctx->GfxProjection = perspec;
   if (front<0.0) {
      ctx->FrontClip = 0.0;
   }
   else if (front>=1.0) {
      ctx->FrontClip = 0.99;
   }
   else {
      ctx->FrontClip = front;
   }
   ctx->Zoom = zoom;
   return 0;
}



int vis5d_get_box_bounds( int index, float *xmin, float *xmax,
                          float *ymin, float *ymax, float *zmin, float *zmax )
{
   CONTEXT("vis5d_get_box_bounds");
   *xmin = ctx->Xmin;
   *xmax = ctx->Xmax;
   *ymin = ctx->Ymin;
   *ymax = ctx->Ymax;
   *zmin = ctx->Zmin;
   *zmax = ctx->Zmax;
   return 0;
}



/*** Isosurface, Slice, and Trajectory Functions ***/


/*
 * Put an isosurface request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         var - which variable
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_isosurface( int index, int time, int var, int urgent )
{
  CONTEXT("vis5d_make_iso_surface")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_isosurface( ctx, time, var, urgent );
  }
  return 0;
}



int vis5d_set_isosurface( int index, int var, float isolevel )
{
  CONTEXT("vis5d_set_iso_surface")
  ctx->IsoLevel[var] = isolevel;
  return 0;
}


int vis5d_get_isosurface( int index, int var, float *isolevel )
{
  CONTEXT("vis5d_get_iso_surface")
  *isolevel = ctx->IsoLevel[var];
  return 0;
}


/*
 * Return the number of the variable used to color a particular isosurface.
 */
int vis5d_get_isosurface_color_var( int index, int iso_var, int *colorvar )
{
   CONTEXT("vis5d_get_isosurface_color_var");
   *colorvar = ctx->IsoColorVar[iso_var];
   return 0;
}


/*
 * Set the variable used to color a particular isosurface.
 */
int vis5d_set_isosurface_color_var( int index, int iso_var, int colorvar )
{
   CONTEXT("vis5d_set_isosurface_color_var");
   ctx->IsoColorVar[iso_var] = colorvar;
   return 0;
}



static int new_slice_pos( int index, int type, int num )
/* if type = HSLICE, VSLICE, CHSLICE or CVSLICE then num = variable number
   if type = HWIND or VWIND then num = slice number */
{
  CONTEXT("vis5d_new_slice_pos")
  switch (type) {
    case HSLICE:
      new_hslice_pos( ctx, ctx->HSliceLevel[num],
                      &ctx->HSliceZ[num], &ctx->HSliceHgt[num] );
      break;
    case CHSLICE:
      new_hslice_pos( ctx, ctx->CHSliceLevel[num],
                      &ctx->CHSliceZ[num], &ctx->CHSliceHgt[num] );
      break;
    case HWIND:
      new_hslice_pos( ctx, ctx->HWindLevel[num],
                      &ctx->HWindZ[num], &ctx->HWindHgt[num] );
      break;
    case STREAM:
      new_hslice_pos( ctx, ctx->StreamLevel[num],
                      &ctx->StreamZ[num], &ctx->StreamHgt[num] );
      break;
    case VSLICE:
      new_vslice_pos( ctx, ctx->VSliceR1[num], ctx->VSliceC1[num],
                      &ctx->VSliceX1[num], &ctx->VSliceY1[num],
                      &ctx->VSliceLat1[num], &ctx->VSliceLon1[num] );
      new_vslice_pos( ctx, ctx->VSliceR2[num], ctx->VSliceC2[num],
                      &ctx->VSliceX2[num], &ctx->VSliceY2[num],
                      &ctx->VSliceLat2[num], &ctx->VSliceLon2[num] );
      break;
    case CVSLICE:
      new_vslice_pos( ctx, ctx->CVSliceR1[num], ctx->CVSliceC1[num],
                      &ctx->CVSliceX1[num], &ctx->CVSliceY1[num],
                      &ctx->CVSliceLat1[num], &ctx->CVSliceLon1[num] );
      new_vslice_pos( ctx, ctx->CVSliceR2[num], ctx->CVSliceC2[num],
                      &ctx->CVSliceX2[num], &ctx->CVSliceY2[num],
                      &ctx->CVSliceLat2[num], &ctx->CVSliceLon2[num] );
      break;
    case VWIND:
      new_vslice_pos( ctx, ctx->VWindR1[num], ctx->VWindC1[num],
                      &ctx->VWindX1[num], &ctx->VWindY1[num],
                      &ctx->VWindLat1[num], &ctx->VWindLon1[num] );
      new_vslice_pos( ctx, ctx->VWindR2[num], ctx->VWindC2[num],
                      &ctx->VWindX2[num], &ctx->VWindY2[num],
                      &ctx->VWindLat2[num], &ctx->VWindLon2[num] );
      break;
    default:
      printf("bad constant (%d) in vis5d_new_slice_pos\n", type);
      break;
  }
  return 0;
}



/*
 * Put a horizontal contour line slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         var - which variable
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_hslice( int index, int time, int var, int urgent )
{
  CONTEXT("vis5d_make_hslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_hslice(ctx, time, var, urgent);
  }
  return 0;
}


int vis5d_set_hslice( int index, int var, float interval,
                      float low, float high, float level )
{
   float maxlev;
   CONTEXT("vis5d_set_hslice")
   if (var<0 || var>=ctx->NumVars) {
      return VIS5D_BAD_VAR_NUMBER;
   }
   if (ctx->Nl[var]==1) {
      maxlev = ctx->MaxNl-1;
   }
   else {
      maxlev = ctx->Nl[var]-1;
   }
   if (level<0.0) {
      level = 0.0;
   }
   else if (level>maxlev) {
      level = maxlev;
   }
   ctx->HSliceInterval[var] = interval;
   ctx->HSliceLowLimit[var] = low;
   ctx->HSliceHighLimit[var] = high;
   ctx->HSliceLevel[var] = level;
   return new_slice_pos(index, HSLICE, var);
}


int vis5d_get_hslice( int index, int var, float *interval,
                      float *low, float *high, float *level )
{
  CONTEXT("vis5d_get_hslice")
  *interval = ctx->HSliceInterval[var];
  *low = ctx->HSliceLowLimit[var];
  *high = ctx->HSliceHighLimit[var];
  *level = ctx->HSliceLevel[var];
  return 0;
}



/*
 * Put a vertical contour line slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         var - which variable
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_vslice( int index, int time, int var, int urgent )
{
  CONTEXT("vis5d_make_vslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_vslice(ctx, time, var, urgent);
  }
  return 0;
}


int vis5d_set_vslice( int index, int var, float interval,
                      float low, float high,
                      float row0, float col0, float row1, float col1)
{
  CONTEXT("vis5d_set_vslice")
  ctx->VSliceInterval[var] = interval;
  ctx->VSliceLowLimit[var] = low;
  ctx->VSliceHighLimit[var] = high;
  ctx->VSliceR1[var] = CLAMP( row0, 0.0, ctx->Nr-1 );
  ctx->VSliceC1[var] = CLAMP( col0, 0.0, ctx->Nc-1 );
  ctx->VSliceR2[var] = CLAMP( row1, 0.0, ctx->Nr-1 );
  ctx->VSliceC2[var] = CLAMP( col1, 0.0, ctx->Nc-1 );
  return new_slice_pos(index, VSLICE, var);
}


int vis5d_get_vslice( int index, int var,
                      float *interval, float *low, float *high,
                      float *row0, float *col0, float *row1, float *col1 )
{
  CONTEXT("vis5d_get_vslice")
  *interval = ctx->VSliceInterval[var];
  *low = ctx->VSliceLowLimit[var];
  *high = ctx->VSliceHighLimit[var];
  *row0 = ctx->VSliceR1[var];
  *col0 = ctx->VSliceC1[var];
  *row1 = ctx->VSliceR2[var];
  *col1 = ctx->VSliceC2[var];
  return 0;
}


/*
 * Put a horizontal color slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         var - which variable
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_chslice( int index, int time, int var, int urgent )
{
  CONTEXT("vis5d_make_chslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_chslice(ctx, time, var, urgent);
  }
  return 0;
}


int vis5d_set_chslice( int index, int var, float level )
{
   float maxlev;
   CONTEXT("vis5d_set_chslice")
   if (var<0 || var>=ctx->NumVars) {
      return VIS5D_BAD_VAR_NUMBER;
   }
   if (ctx->Nl[var]==1) {
      maxlev = ctx->MaxNl-1;
   }
   else {
      maxlev = ctx->Nl[var]-1;
   }
   if (level<0.0) {
      level = 0.0;
   }
   else if (level>maxlev) {
      level = maxlev;
   }
   ctx->CHSliceLevel[var] = level;
   return new_slice_pos(index, CHSLICE, var);
}


int vis5d_get_chslice( int index, int var, float *level )
{
  CONTEXT("vis5d_get_chslice")
  *level = ctx->CHSliceLevel[var];
  return 0;
}


/*
 * Put a vertical color slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         var - which variable
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_cvslice( int index, int time, int var, int urgent )
{
  CONTEXT("vis5d_make_cvslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_cvslice(ctx, time, var, urgent);
  }
  return 0;
}


int vis5d_set_cvslice( int index, int var,
                       float row0, float col0, float row1, float col1 )
{
  CONTEXT("vis5d_set_cvslice")
    
  ctx->CVSliceR1[var] = CLAMP( row0, 0.0, ctx->Nr-1 );
  ctx->CVSliceC1[var] = CLAMP( col0, 0.0, ctx->Nc-1 );
  ctx->CVSliceR2[var] = CLAMP( row1, 0.0, ctx->Nr-1 );
  ctx->CVSliceC2[var] = CLAMP( col1, 0.0, ctx->Nc-1 );
  return new_slice_pos(index, CVSLICE, var);
}


int vis5d_get_cvslice( int index, int var,
                       float *row0, float *col0, float *row1, float *col1 )
{
  CONTEXT("vis5d_get_cvslice")
  *row0 = ctx->CVSliceR1[var];
  *col0 = ctx->CVSliceC1[var];
  *row1 = ctx->CVSliceR2[var];
  *col1 = ctx->CVSliceC2[var];
  return 0;
}


/*
 * Put a horizontal wind vector slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         slice - which wind slice
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_hwindslice( int index, int time, int slice, int urgent )
{
  CONTEXT("vis5d_make_hwindslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_hwindslice( ctx, time, slice, urgent);
  }
  return 0;
}


int vis5d_set_hwindslice( int index, int ws, float density, float scale,
                          float level )
{
   float maxlev;
   CONTEXT("vis5d_set_hwindslice")
   if (ws<0 || ws>=VIS5D_WIND_SLICES) {
      return VIS5D_BAD_VALUE;
   }
   if (ctx->WindNl==1) {
      maxlev = ctx->MaxNl-1;
   }
   else {
      maxlev = ctx->WindNl-1;
   }
   if (level<0.0) {
      level = 0.0;
   }
   else if (level>maxlev) {
      level = maxlev;
   }
   ctx->HWindLevel[ws] = level;
   ctx->HWindDensity[ws] = density;
   ctx->HWindScale[ws] = scale;
   return new_slice_pos(index, HWIND, ws);
}


int vis5d_get_hwindslice( int index, int ws, float *density, float *scale,
                          float *level )
{
  CONTEXT("vis5d_get_hwindslice")
  *level = ctx->HWindLevel[ws];
  *density = ctx->HWindDensity[ws];
  *scale = ctx->HWindScale[ws];
  return 0;
}


/*
 * Put a vertical wind vector slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         slice - which wind slice
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_vwindslice( int index, int time, int slice, int urgent )
{
  CONTEXT("vis5d_make_vwindslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_vwindslice( ctx, time, slice, urgent);
  }
  return 0;
}


int vis5d_set_vwindslice( int index, int ws,
                          float density, float scale,
                          float row0, float col0, float row1, float col1 )
{
  CONTEXT("vis5d_set_vwindslice")
  ctx->VWindDensity[ws] = density;
  ctx->VWindScale[ws] = scale;
  ctx->VWindR1[ws] = row0;
  ctx->VWindC1[ws] = col0;
  ctx->VWindR2[ws] = row1;
  ctx->VWindC2[ws] = col1;
  return new_slice_pos(index, VWIND, ws);
}


int vis5d_get_vwindslice( int index, int ws, float *density, float *scale,
                          float *row0, float *col0, float *row1, float *col1 )
{
  CONTEXT("vis5d_get_vwindslice")
  *density = ctx->VWindDensity[ws];
  *scale = ctx->VWindScale[ws];
  *row0 = ctx->VWindR1[ws];
  *col0 = ctx->VWindC1[ws];
  *row1 = ctx->VWindR2[ws];
  *col1 = ctx->VWindC2[ws];
  return 0;
}


/*
 * Put a horizontal wind vector slice request into the work queue.
 * Input:  index - the context index
 *         time - which timestep
 *         slice - which wind slice
 *         urgent - 1 = put request at head of queue, 0 = put at back of queue
 */
int vis5d_make_streamslice( int index, int time, int slice, int urgent )
{
  CONTEXT("vis5d_make_streamslice")
  if (!ctx->VeryLarge || time == ctx->CurTime) {
    request_streamslice( ctx, time, slice, urgent);
  }
  return 0;
}


int vis5d_set_streamslice( int index, int ws, float density, float level )
{
  CONTEXT("vis5d_set_streamslice")
  ctx->StreamLevel[ws] = level;
  ctx->StreamDensity[ws] = density;
  return new_slice_pos(index, STREAM, ws);
}


int vis5d_get_streamslice( int index, int ws, float *density, float *level )
{
  CONTEXT("vis5d_get_streamslice")
  *level = ctx->StreamLevel[ws];
  *density = ctx->StreamDensity[ws];
  return 0;
}


int vis5d_make_traj( int index, float row, float col, float lev,
                     int time, int set)
{
  CONTEXT("vis5d_make_traj")
  request_traj(ctx, row, col, lev, time, set, ctx->RibbonFlag,
               ctx->UserTrajStep, ctx->UserTrajLength);
  return 0;
}


int vis5d_set_traj( int index, float step, float length, int ribbon_flag )
{
  CONTEXT("vis5d_set_traj")
  ctx->UserTrajStep = step;
  ctx->UserTrajLength = length;
  ctx->RibbonFlag = ribbon_flag;
  return 0;
}


int vis5d_get_traj( int index, float *step, float *length, int *ribbon_flag )
{
  CONTEXT("vis5d_get_traj")
  *step = ctx->UserTrajStep;
  *length = ctx->UserTrajLength;
  *ribbon_flag = ctx->RibbonFlag;
  return 0;
}


/*
 * Set the variable used to color a trajectory set.
 */
int vis5d_set_trajectory_color_var( int index, int traj_set, int colorvar )
{
   CONTEXT("vis5d_set_trajectory_color_var");
   if (ctx->TrajColorVar[traj_set] != colorvar) {
      /* recolor existing trajectories in the set */
      ctx->TrajColorVar[traj_set] = colorvar;
      request_traj_recoloring( ctx, traj_set );
   }
   return 0;
}


/*
 * Return the number of the variable used to color a trajectory set.
 */
int vis5d_get_trajectory_color_var( int index, int traj_set, int *colorvar )
{
   CONTEXT("vis5d_get_trajecotry_color_var");
   *colorvar = ctx->TrajColorVar[traj_set];
   return 0;
}


int vis5d_delete_last_traj( int index )
{
  CONTEXT("vis5d_del_traj");
  del_last_traj(ctx);
  return 0;
}


int vis5d_delete_traj_set( int index, int set )
{
  CONTEXT("vis5d_del_traj_set");
  del_traj_group(ctx, set);
  return 0;
}


/*
 * Return number of trajectories in the context.
 */
int vis5d_get_num_traj( int index )
{
   CONTEXT("vis5d_get_num_traj");
   return ctx->NumTraj;
}


/*
 * Return parameters of an individual trajectory.
 */
int vis5d_get_traj_info( int index, int trajnum,
                         float *row, float *column, float *level,
                         int *timestep, float *step, float *length,
                         int *group, int *ribbon )
{
   struct traj *t;
   CONTEXT("vis5d_get_traj_info");
   if (trajnum>=ctx->NumTraj) {
      return VIS5D_BAD_VALUE;
   }
   t = ctx->TrajTable[trajnum];

   *row      = t->row;
   *column   = t->col;
   *level    = t->lev;
   *timestep = t->timestep;
   *step     = t->stepmult;
   *length   = t->lengthmult;
   *group    = t->group;
   *ribbon   = t->kind;
   return 0;
}



int vis5d_make_timestep_graphics( int index, int time )
{
  int var, ws;
  CONTEXT("vis5d_make_timestep_graphics")

  for (var=0; var<ctx->NumVars; var++) {
    if (ctx->DisplaySurf[var]) request_isosurface(ctx, time, var, 1 );
    if (ctx->DisplayHSlice[var]) request_hslice(ctx, time, var, 1);
    if (ctx->DisplayVSlice[var]) request_vslice(ctx, time, var, 1);
    if (ctx->DisplayCHSlice[var]) request_chslice(ctx, time, var, 1);
    if (ctx->DisplayCVSlice[var]) request_cvslice(ctx, time, var, 1);
  }
  for (ws=0; ws<VIS5D_WIND_SLICES; ws++) {
    if (ctx->DisplayHWind[ws]) request_hwindslice(ctx, time, ws, 1);
    if (ctx->DisplayVWind[ws]) request_vwindslice(ctx, time, ws, 1);
    if (ctx->DisplayStream[ws]) request_streamslice(ctx, time, ws, 1);
  }
  return 0;
}




int vis5d_free_graphics( int index )
{
   CONTEXT("vis5d_free_graphics")
   free_all_graphics( ctx );
   return 0;
}






/*** Text Label Functions ***/


/*
 * Allocate a new label struct, assign it a unique ID, insert it into
 * the head of the linked list for the context, and disable editing
 * of the previous label (if any).
 */
static struct label *alloc_label( Context ctx )
{
   static int LabelID = 1000;
   struct label *l, *next;

   l = (struct label *) malloc( sizeof(struct label) );
   if (l) {
      l->id = LabelID++;
      if (ctx->FirstLabel && ctx->FirstLabel->state) {
         /* disable editing of prev label */
         if (ctx->FirstLabel->len==0) {
            /* delete zero-length label */
            next = ctx->FirstLabel->next;
            free( ctx->FirstLabel );
            ctx->FirstLabel = next;
         }
         else {
            ctx->FirstLabel->state = 0;
         }
      }
      l->next = ctx->FirstLabel;
      ctx->FirstLabel = l;
   }
   return l;
}


static void compute_label_bounds( Context ctx, struct label *lab )
{
   set_current_window( ctx );
   lab->x1 = lab->x;
   lab->y1 = lab->y + ctx->FontDescent - ctx->FontHeight;
   lab->x2 = lab->x + text_width( lab->text );
   lab->y2 = lab->y + ctx->FontDescent;
}



/*
 * Make a complete text label at position (x,y).
 */
int vis5d_make_label( int index, int x, int y, char *text )
{
   struct label *l;
   CONTEXT("vis5d_make_label");

   l = alloc_label( ctx );
   if (l) {
      strcpy( l->text, text );
      l->len = strlen( text );
      l->x = x;
      l->y = y;
      l->state = 0;
      compute_label_bounds( ctx, l );
      return 0;
   }
   return VIS5D_OUT_OF_MEMORY;
}


/*
 * Start a new text label at x, y.  Characters will be appended onto it
 * with vis5d_edit_label().
 */
int vis5d_new_label( int index, int x, int y, int *label_id )
{
   struct label *l;
   CONTEXT("vis5d_new_label");

   l = alloc_label( ctx );
   if (l) {
      l->text[0] = 0;
      l->len = 0;
      l->x = x;
      l->y = y;
      l->state = 1;
      *label_id = l->id;
      compute_label_bounds( ctx, l );
      return 0;
   }
   return VIS5D_OUT_OF_MEMORY;
}



/*
 * Append the character onto the new label.
 * if chr==BACKSPACE, delete last char
 * if chr==RETURN, finish the label
 * See also: vis5d_new_label
 */
int vis5d_edit_label( int index, char chr )
{
   struct label *lab;
   CONTEXT("vis5d_edit_label");

   /* We can only edit the most recently made label.  It will be at the
    * head of the linked list!
    */

   lab = ctx->FirstLabel;
   if (lab && lab->state==1) {
      if (chr=='\r') {
         /* RETURN key, done editing */
         lab->state = 0;
         if (lab->len==0) {
            /* delete zero-length labels */
            struct label *next = lab->next;
            free( lab );
            ctx->FirstLabel = next;
            return 0;
         }
      }
      else if (chr==8 || chr==127) {
         /* BACKSPACE or DELETE key, delete last character */
         if (lab->len>0) {
            lab->len--;
            lab->text[lab->len] = 0;
         }
      }
      else {
         /* Append the character */
         lab->text[lab->len] = chr;
         lab->len++;
         lab->text[lab->len] = 0;
      }
      compute_label_bounds( ctx, lab );
   }
   return 0;
}



/*
 * Return the ID of the label near point (x,y).
 */
int vis5d_find_label( int index, int *x, int *y, int *label_id )
{
   struct label *lab;
   CONTEXT("vis5d_find_label");

   for (lab=ctx->FirstLabel; lab; lab=lab->next) {
      if (lab->x1<=*x && *x<=lab->x2 && lab->y1<=*y && *y<=lab->y2) {
         *x = lab->x;
         *y = lab->y;
         *label_id = lab->id;
         return 0;
      }
   }
   return VIS5D_FAIL;
}



/*
 * Move the specified label to (x,y).
 */
int vis5d_move_label( int index, int label_id, int x, int y )
{
   struct label *lab;
   CONTEXT("vis5d_move_label");

   for (lab=ctx->FirstLabel; lab; lab=lab->next) {
      if (lab->id==label_id) {
         lab->x = x;
         lab->y = y;
         compute_label_bounds( ctx, lab );
         return 0;
      }
   }
   return VIS5D_BAD_VALUE;
}



/*
 * Delete the specified label.
 */
int vis5d_delete_label( int index, int label_id )
{
   struct label *lab, *prev;
   CONTEXT("vis5d_delete_label");

   prev = NULL;
   for (lab=ctx->FirstLabel; lab; lab=lab->next) {
      if (lab->id == label_id) {
         /* found it */
         if (prev) {
            prev->next = lab->next;
         }
         else {
            ctx->FirstLabel = lab->next;
         }
         free( lab );
         return 0;
      }
      prev = lab;
   }
   return VIS5D_BAD_VALUE;
}


/*
 * Return the nth text label.
 * Input:  index - the context index
 *         n - which label, starting at 1
 * Output:  x, y - the label position
 *          label - the label text
 * Return:  VIS5D_OK - if n was valid
 *          VIS5D_FAIL - if n is larger than the number of labels
 */
int vis5d_get_label( int index, int n, int *x, int *y, char *label )
{
   int i;
   struct label *lab;
   CONTEXT("vis5d_delete_label");

   if (n<1) {
      return VIS5D_FAIL;
   }

   lab = ctx->FirstLabel;
   for (i=0;i<n-1;i++) {
      if (!lab) {
         return VIS5D_FAIL;
      }
      else {
         lab = lab->next;
      }
   }

   if (!lab) {
      return VIS5D_FAIL;
   }

   /* Return nth label's info */
   *x = lab->x;
   *y = lab->y;
   strcpy( label, lab->text );
   lab = lab->next;
   return 0;
}



/*** 3-D Cursor Functions ***/

int vis5d_set_cursor( int index, float x, float y, float z )
{
  CONTEXT("vis5d_set_cursor")
  ctx->CursorX = x;
  ctx->CursorY = y;
  ctx->CursorZ = z;
  return 0;
}


int vis5d_get_cursor( int index, float *x, float *y, float *z )
{
  CONTEXT("vis5d_get_cursor")
  *x = ctx->CursorX;
  *y = ctx->CursorY;
  *z = ctx->CursorZ;
  return 0;
}


/*** 3-D Viewing Window Functions */

int vis5d_get_window( int index, Window *window, int *width, int *height )
{
   CONTEXT("vis5d_get_window");
   *window = ctx->GfxWindow;
   *width = ctx->WinWidth;
   *height = ctx->WinHeight;
   return 0;
}


int vis5d_resize_3d_window( int index, int width, int height )
{
  CONTEXT("vis5d_resize_3d_window")
  set_current_window( ctx );
  resize_3d_window( width, height);
  return 0;
}


int vis5d_get_image_formats( void )
{
   return save_formats();
}


/*
 * Save window image to a file.
 * Input:  index - context index
 *         filename - name of image file.
 *         format:  1 = VIS5D_SGI, 2 = VIS5D_GIF, 4 = VIS5D_XWD,
 *                   8 = VIS5D_PS, 16 = VIS5D_COLOR_PS
 */
int vis5d_save_window( int index, char *filename, int format )
{
   CONTEXT("vis5d_save_window");
   if (filename[0]==0) {
      /* no filename! */
      return VIS5D_FAIL;
   }
   set_current_window( ctx );
   if (save_3d_window( filename, format )){
      return 0;
   }
   else {
      return VIS5D_FAIL;
   }
}


vis5d_print_window( int index )
{
  CONTEXT("vis5d_print_window")
  set_current_window( ctx );
  print_3d_window();
  return 0;
}



/*** Coordinate Conversion Functions ***/

int vis5d_project( int index, float p[3], float *x, float *y )
{
  CONTEXT("vis5d_project")
  set_current_window( ctx );
  project( p, x, y );
  return 0;
}


int vis5d_unproject( int index, float cursor_x, float cursor_y,
                     float point[3], float dir[3] )
{
  CONTEXT("vis5d_unproject")
  set_current_window( ctx );
  unproject( cursor_x, cursor_y, point, dir);
  return 0;
}


int vis5d_xyz_to_grid( int index, int time, int var,
                       float x, float y, float z,
                       float *row, float *col, float *lev )
{
  CONTEXT("vis5d_xyz_to_grid")
  xyz_to_grid(ctx, time, var, x, y, z, row, col, lev);
  return 0;
}


int vis5d_grid_to_xyz( int index, int time, int var,
                       float row, float col, float lev,
                       float *x, float *y, float *z )
{
  float r[1], c[1], l[1];
  CONTEXT("vis5d_grid_to_xyz")
  r[0] = row;
  c[0] = col;
  l[0] = lev;
  grid_to_xyz(ctx, time, var, 1, r, c, l, x, y, z);
  return 0;
}


int vis5d_xyz_to_geo( int index, int time, int var,
                      float x, float y, float z,
                      float *lat, float *lon, float *hgt )
{
  CONTEXT("vis5d_xyz_to_grid")
  xyz_to_geo(ctx, time, var, x, y, z, lat, lon, hgt);
  return 0;
}


int vis5d_geo_to_xyz( int index, int time, int var,
                      float lat, float lon, float hgt,
                      float *x, float *y, float *z )
{
  float la[1], lo[1], hg[1];
  CONTEXT("vis5d_geo_to_xyz")
  la[0] = lat;
  lo[0] = lon;
  hg[0] = hgt;
  geo_to_xyz(ctx, time, var, 1, la, lo, hg, x, y, z);
  return 0;
}


/*** Save and Restore Functions (obsolete in favor of Tcl save/restore) ***/



#ifdef LEAVEOUT
/*
 * Save graphics/settings.
 * Return:  0 = OK
 *          VIS5D_BAD_VALUE = couldn't open output file
 *          VIS5D_FAIL = error while writing file, disk full?
 */
int vis5d_save( int index, char *filename, int gfx, int traj)
{
   CONTEXT("vis5d_save")
   return tclsave( index, filename, gfx, traj );
}
#endif


/*
 * Restore saved graphics/settings.  This restores an old (version 4.1
 * and earlier) binary .SAVE file.  Obsolete in favor of Tcl-based
 * save and restore.
 * Return:  0 = OK
 *          VIS5D_BAD_VALUE = filename not found
 *          VIS5D_FAIL = couldn't restore (bad file??)
 */
int vis5d_restore( int index, char *filename )
{
   CONTEXT("vis5d_restore")

   return restore( ctx, filename );
}




