/* grid.c */


/*
 * Functions for manipulating grids and lists of grids.
 */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "grid.h"
#include "projlist.h"



/*
 * Allocate a grid_info struct and initialize it.
 */
struct grid_info *alloc_grid_info( void )
{
   struct grid_info *g;

   g = (struct grid_info *) calloc( 1, sizeof(struct grid_info) );
   if (g) {
      /* all fields will be 0 thanks to calloc but here's some special cases */
      g->TimeStep = g->VarNum = g->Position = -1;
      g->Proj = NULL;
      g->Vcs = NULL;
      g->SelectBits = ALL_BITS;
   }
   return g;
}



/*
 * Deallocate a grid_info struct.
 */
void free_grid_info( struct grid_info *info )
{
   if (info->FileName) {
      free( info->FileName );
   }
   if (info->VarName) {
      free( info->VarName );
   }
   if (info->Units) {
      free( info->Units );
   }
   if (info->Data) {
      free( info->Data );
   }
   /* "erase" data to help with debugging */
   /*memset( info, 0xff, sizeof(struct grid_info) );*/
   free( info );
}



/*
 * Allocate a grid_list struct and initialize it.
 */
struct grid_db *alloc_grid_db()
{
   struct grid_db *gl;

   /* Allocate w/ every field initialized to zero */
   gl = (struct grid_db *) calloc( sizeof(struct grid_db), 1 );

   return gl;
}


/*
 * Deallocate a grid_db struct.
 */
void free_grid_db( struct grid_db *db )
{
   struct grid_info *g, *next;
   int i;

   /* free grid_info list */
   for (g=db->FirstGrid; g; g=next) {
      next = g->Next;
      free_grid_info( g );
   }

   /* free varname and units strings */
   for (i=0;i<db->NumVars;i++) {
      free( db->VarNames[i] );
      if (db->Units[i]) {
         free( db->Units[i] );
      }
   }

   /* free projections */
   for (i=0;i<db->NumProj;i++) {
      free( db->ProjList[i] );
   }

   /* free VCSs */
   for (i=0;i<db->NumVcs;i++) {
      free( db->VcsList[i]->Args );
      free( db->VcsList[i] );
   }

   free( db );
}



/*
 * Append a grid_info struct onto the tail of the list.
 */
void append_grid( struct grid_info *grid, struct grid_db *db )
{
   int i, inlist;

   if (db->LastGrid) {
      db->LastGrid->Next = grid;
      db->LastGrid = grid;
   }
   else {
      db->FirstGrid = db->LastGrid = grid;
   }
   grid->Next = NULL;
   db->NumGrids++;
   db->Sorted = 0;
}



/*
 * Remove a grid_info struct from a list.
 * Return:  1 = success, 0 = error
 */
int remove_grid( struct grid_info *grid, struct grid_db *db )
{
   struct grid_info *pred;
   int found;

   /* Find predecessor to the grid */
   if (db->FirstGrid==grid) {
      /* grid to remove is first in list */
      db->FirstGrid = grid->Next;
      pred = NULL;
      found = 1;
   }
   else {
      for (pred=db->FirstGrid; pred; pred=pred->Next) {
	 if (pred->Next==grid) {
	    /* found predecessor! */
	    pred->Next = grid->Next;
	    found = 1;
	    break;
	 }
      }
   }
   if (db->LastGrid==grid) {
      db->LastGrid = pred;
   }
   
   db->NumGrids--;
   return found;
}



/*
 * Free all the grids and associated data in a grid_db.
 */
void free_all_grids( struct grid_db *db )
{
   struct grid_info *g, *nextg;
   int i, j;

   /* free grids */
   for (g=db->FirstGrid; g; g=nextg) {
      nextg = g->Next;
      free_grid_info( g );
   }
   db->FirstGrid = db->LastGrid = NULL;
   db->NumGrids = 0;

   /* Free map projections */
   for (i=0;i<db->NumProj;i++) {
      free( db->ProjList[i]->Args );
      free( db->ProjList[i] );
      db->ProjList[i] = NULL;
   }
   db->NumProj = 0;

   /* Free VCSs */
   for (i=0;i<db->NumVcs;i++) {
      free( db->VcsList[i]->Args );
      free( db->VcsList[i] );
      db->VcsList[i] = NULL;
   }
   db->NumVcs = 0;

   /* Clear pointers in grid matrix */
   for (i=0;i<db->NumTimes;i++) {
      for (j=0;j<db->NumVars;j++) {
	 db->Matrix[i][j] = NULL;
      }
   }

   db->NumVars = 0;
   db->NumTimes = 0;

   db->Sorted = 0;
}



/*
 * Print a grid list to stdout.  Just used for debugging.
 */
void print_grid_list( struct grid_db *db )
{
   struct grid_info *g;
   int i = 1;

   printf("  Grid  Date  Time    Variable    Nr  Nc  Nl  Proj#  Vcs#  Filename\n");
   for (g=db->FirstGrid; g; g=g->Next) {
      int projnum = lookup_proj( db, g->Proj );
      int vcsnum = lookup_vcs( db, g->Vcs );
      printf("%c %4d  %05d %06d  %-10s %3d %3d %3d   %3d   %3d   %s\n",
	      g->SelectBits==ALL_BITS ? '*': ' ',
	      i, g->DateStamp, g->TimeStamp, g->VarName, g->Nr, g->Nc, g->Nl,
              projnum, vcsnum,
	      g->FileName );
      i++;
   }

   printf("*=include grid in output file\n");
}




/*
 * "Print" the grid list to generate an array of strings.
 * Return null if no grids in list.
 */
char **sprint_grid_list( struct grid_db *db )
{
   struct grid_info *g;
   int i = 0;
   char **vector;

   if (db->NumGrids==0) {
      return NULL;
   }

   vector = (char **) malloc( db->NumGrids * sizeof(char *) );

   /* See gui.h for column headings */
   for (g=db->FirstGrid; g; g=g->Next) {
      int projnum = lookup_proj( db, g->Proj );
      int vcsnum = lookup_vcs( db, g->Vcs );
      vector[i] = (char *) malloc( 1000 );
      sprintf( vector[i],
	      "%4d  %05d %06d  %-10s%3d %3d %3d   %2d    %2d   %s%c",
	      i+1,
	      g->DateStamp, g->TimeStamp,
	      g->VarName,
	      g->Nr, g->Nc, g->Nl,
	      projnum, vcsnum,
	      g->FileName,
	      g->Sibling ? ',' : ' ' );
      i++;
   }

   return vector;
}



/*
 * Find the maximum number of grid levels of all grids in the list.
 * Input:  db - the grid data base
 * Return:  max number of grid levels in list or -1 if empty list
 */
int find_max_levels( struct grid_db *db )
{
   struct grid_info *g;
   int maxnl = -1;

   for (g=db->FirstGrid; g; g=g->Next) {
      if (g->Nl>maxnl) {
	 maxnl = g->Nl;
      }
   }

   return maxnl;
}
