/* rtp.c
 *
 * Main code module for rtp (real-time plotter) program.
 *
 * Copyright (c) 1999, David Watt <wattd@elcsci.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static char rtp_c_id[] = "$Id: rtp.c,v 1.22 1999/12/05 06:27:22 davidw Exp $";

#include <qmainwindow.h>
#include <qwidget.h>
#include <qsocketnotifier.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qrect.h>
#include <qpainter.h>
#include <qmessagebox.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <math.h>
#include <sys/time.h>
#include <string.h>

#include <deque>
using namespace std;

#include "rtp.h"
#include "expand.xpm"
#include "track.xpm"

/* This utility function expands the passed rectangle by the size of
 * _rbPen.
 */
void PlotWindow::expRectForPen(QRect *r)
{
   r->setCoords(QMAX(0, r->left() - _rbPen.width()),
                QMAX(0, r->top() - _rbPen.width()),
                QMIN(this->width(), int(r->right() + _rbPen.width())),
                QMIN(this->height(), int(r->bottom() + _rbPen.width())));
}

PlotWindow::PlotWindow(QWidget * parent = 0, const char * name = 0, WFlags
                       f = 0)
  : QWidget(parent, name, f), _render(&_points), _rbPen(blue, 2, DashLine)
{
  // don't blank the window before repainting
  setBackgroundMode(NoBackground);

  _buffer = new QPixmap;
  _buffer->resize(size());
  _buffer->fill(white);

  _map.xscale = _map.yscale = _map.viewPort.minX = _map.viewPort.maxX
  = _map.viewPort.minY = _map.viewPort.maxY = 0.0;
  _map.xbias = _map.ybias = 0;

  _allPointsBBox.minX = _allPointsBBox.maxX = _allPointsBBox.minY
  = _allPointsBBox.maxY = _trackingPort.minX = _trackingPort.maxX
  = _trackingPort.minY = _trackingPort.maxY = 0.0;

  _rubberBox = NULL;

  _viewMode = AUTO_SCALE;
  
  // Connect _render to slotPrivateRenderDone and slotRepaintNoErase.
  QObject::connect(&_render, 
    SIGNAL(signalPrivateRenderDone(QPixmap *, RtpMapping&)), this,
    SLOT(slotPrivateRenderDone(QPixmap *, RtpMapping&)));
  QObject::connect(&_render, SIGNAL(signalRepaintWindow()),
    this, SLOT(slotRepaintNoErase()));
}

/* The slot called when the stdin socket notifier wakes us up.
 *
 * The buffering is set up so that the read() puts data into the second
 * half of buff.  We then process as many double pairs as we can, starting
 * from buffBegin.  Whatever data is left over is moved to right before
 * index RTP_STDIN_BUFLEN so it will be processed next time.  The assumption
 * is that every buffer of size RTP_STDIN_BUFLEN has meaningful data for
 * us to remove.
 */
void PlotWindow::slotStdinAwake()
{
#define RTP_STDIN_BUFLEN 1024

  static char buff[2 * RTP_STDIN_BUFLEN];
  static int buffBegin = RTP_STDIN_BUFLEN;
  static bool first = true;

  DoubPt pt;

  // Read as many char's as we can, up to RTP_STDIN_BUFLEN of them
  int numRead 
    = read(STDIN_FILENO, &(buff[RTP_STDIN_BUFLEN]), RTP_STDIN_BUFLEN);
  if (numRead <= 0)
    return;  // We don't try to decode errors if we can't get no data.
  int buffEnd = RTP_STDIN_BUFLEN + numRead;

  deque<DoubPt> newPoints;
  
  // For now, just process each group of two white-space separated tokens.
  int ii = buffBegin;
  while ((ii != buffEnd) && (buffBegin != buffEnd)) {
    while (isspace(buff[buffBegin]) && buffBegin != buffEnd)
      buffBegin++;
    int tokCnt = 0;
    ii = buffBegin;
    while ((ii != buffEnd) && (tokCnt < 2)) {
      if (isspace(buff[ii++]))
        tokCnt++;
    }
    if (2 == tokCnt) {
      if (sscanf(&(buff[buffBegin]), "%lf %lf", &pt.x, &pt.y) != 2) {
        fprintf(stderr,"Exiting.  Couldn't parse stdin.\n");
        exit(-1);
      }

      /* Special handling for first point read... Init _allPointsBBox and push
       * right onto _points.
       */
      if (first) {
        _allPointsBBox.minX = _allPointsBBox.maxX = pt.x;
        _allPointsBBox.minY = _allPointsBBox.maxY = pt.y;
        _map.viewPort = _allPointsBBox;
        _points.push_back(pt);
        first=false;
      } 
      /* Expand the _allPointsBBox as necessary here.  If TRACKING, move
       * the viewPort to track new points.
       */
      else {
        if (pt.x > _allPointsBBox.maxX) _allPointsBBox.maxX = pt.x;
        else if (pt.x < _allPointsBBox.minX) _allPointsBBox.minX = pt.x;
        if (pt.y > _allPointsBBox.maxY) _allPointsBBox.maxY = pt.y;
        else if (pt.y < _allPointsBBox.minY) _allPointsBBox.minY = pt.y;
        if (TRACKING == _viewMode) {
          if (pt.x > _trackingPort.maxX) {
            _trackingPort.minX += pt.x - _trackingPort.maxX;
            _trackingPort.maxX += pt.x - _trackingPort.maxX;
          }
          else if (pt.x < _trackingPort.minX) {
            _trackingPort.maxX += pt.x - _trackingPort.minX;
            _trackingPort.minX += pt.x - _trackingPort.minX;
          } 
          if (pt.y > _trackingPort.maxY) {
            _trackingPort.minY += pt.y - _trackingPort.maxY;
            _trackingPort.maxY += pt.y - _trackingPort.maxY;
          }
          else if (pt.y < _trackingPort.minY) {
            _trackingPort.maxY += pt.y - _trackingPort.minY;
            _trackingPort.minY += pt.y - _trackingPort.minY;
          } 
        }
        newPoints.push_back(pt);
      }

      buffBegin = ii;    // Move begin marker past the pt we just read.
    }
  }       

  // Move whatever might be left in the buffer to right before the
  // half point.  memmove is OK for overlapping memory.
  if (buffEnd - buffBegin > RTP_STDIN_BUFLEN) {
    fprintf(stderr,"Exiting.  Couldn't find two tokens in %d sized buffer.\n",
            RTP_STDIN_BUFLEN);
    exit(-1);
  }
  if (buffEnd - buffBegin > 0)
    memmove(&(buff[RTP_STDIN_BUFLEN - buffEnd + buffBegin]), 
            &(buff[buffBegin]), buffEnd - buffBegin);
  buffBegin = RTP_STDIN_BUFLEN - buffEnd + buffBegin; 

  // If no points were added to newPoints, there is nothing to draw, so
  // return.
  if (newPoints.empty())
    return;

  // Now draw the points from newPoints.
  bool repaintNeeded = false;
  QPainter p;
  p.begin(_buffer);

  deque<DoubPt>::iterator pti = newPoints.begin();
  pt = _points.back();

  while (pti != newPoints.end()) {
    DoubPt newPt = *pti;
    // Pass rtpClip pointers to our copies so clipping doesn't affect
    // points going into _points.
    if (rtpClip(_map.viewPort, &pt, &newPt)) {
      p.drawLine(rtpMap(_map, pt), rtpMap(_map, newPt));
      repaintNeeded = true;
    }
    pt = *pti++;
    _points.push_back(pt); 
  }

  p.end();
  if (repaintNeeded)
    repaint(false);  // Get new points onto window.

  // For either the AUTO_SCALE or TRACKING modes, we may need to schedule a
  // new rendering with a new mapping.  In AUTO_SCALE mode, a new rendering
  // is needed if _allPointsBBBox != _map.viewPort.  Or in
  // TRACKING mode, it is needed if _trackingPort != _map.viewPort.

  BBox *testBox;
  testBox = (AUTO_SCALE == _viewMode) ? &_allPointsBBox : &_trackingPort;
  
  if (  (AUTO_SCALE == _viewMode)
      && memcmp(&_allPointsBBox,  &(_map.viewPort), sizeof(BBox))  )
    _render.quePrivateRender(_allPointsBBox, this->size());

  // For TRACKING mode, we must also make sure that the range following
  // has not eaten up all our resolution.
  else if (  (TRACKING == _viewMode)
	   && memcmp(&_trackingPort,  &(_map.viewPort), sizeof(BBox))  )
  {
    BBox tmpPort;
    tmpPort = _trackingPort;   // Need copy for rtpResCheck to modify
    if (!rtpResCheck(&tmpPort))
      _render.quePrivateRender(_trackingPort, this->size());
    else  // Notify user of problem, switch to USER_FIXED mode.
          // Mode change must come before message box to avoid recursion.
    {
      _viewMode = USER_FIXED;
      QMessageBox::warning(this, caption(), 
        "Resolution is insufficient to track new points.\n"
        "Fixing view port in current position.\n",
        QMessageBox::Ok | QMessageBox::Default, 0, 0);
    }
  }
}

/* slotPrivateRenderDone
 *
 * When the private rendering finishes, it calls back to here.  Take the
 * buffer and map it gives us and reassign our _buffer to match.
 * repaint() before exit.
 */
void PlotWindow::slotPrivateRenderDone(QPixmap *buf, RtpMapping& map)
{
  if (_buffer)
    delete _buffer;
  _buffer = buf;
  _map = map;
  repaint(false);
}

/* Paint by copying the rectangular region from _buffer to the window.
 * If there is a rubber box, we first draw that on top.  Use double-buffering
 * for the rubber box to avoid flicker.
 */  
void PlotWindow::paintEvent(QPaintEvent *event)
{
  if (_rubberBox) {
    QPixmap doubleBuf(event->rect().width(), event->rect().height());
    bitBlt(&doubleBuf, 0, 0, _buffer, event->rect().left(),
           event->rect().top(), event->rect().width(), event->rect().height());
    QPainter p;
    p.begin(&doubleBuf);
    p.setPen(_rbPen);
    p.translate(-event->rect().left(), -event->rect().top());
    p.drawRect(*_rubberBox);
    p.end();
    bitBlt(this, event->rect().left(), event->rect().top(), &doubleBuf, 
           0, 0, event->rect().width(), event->rect().height());
  } else
    bitBlt(this, event->rect().left(), event->rect().top(), _buffer,
           event->rect().left(), event->rect().top(), event->rect().width(),
           event->rect().height());
}

/* resizeEvent
 *
 * Blank the screen, and do a preemptive render.
 */ 
void PlotWindow::resizeEvent(QResizeEvent *event)
{
  _buffer->resize(size());
  _buffer->fill(white);
  _map = _render.newOnlineRender(_map.viewPort, _buffer);
}

/* When the mouse is pressed in the drawing area, initialize 
 * _rbAnchor to the point of press.
 */
void PlotWindow::mousePressEvent(QMouseEvent *event)
{
  _rbAnchor = event->pos();

#ifdef DEBUG_RTP
  fprintf(stderr, "press at %d %d\n", event->x(), event->y());
  fflush(stderr);
#endif
}

/* When the mouse is moved (only when already down), update and redraw
 * _rubberBox.  The repaint area will cover both the old and new box so
 * no artifacts are left.
 */
void PlotWindow::mouseMoveEvent(QMouseEvent *event)
{
  QRect newRB;
  newRB.setLeft(QMIN(event->x(), _rbAnchor.x()));
  newRB.setRight(QMAX(event->x(), _rbAnchor.x()));
  newRB.setTop(QMIN(event->y(), _rbAnchor.y()));
  newRB.setBottom(QMAX(event->y(), _rbAnchor.y()));

  // Compute size of redraw rectangle, or create _rubberBox if not
  // already created.
  QRect redrawRect(newRB);
  if (_rubberBox) {
    redrawRect = redrawRect.unite(*_rubberBox);
    *_rubberBox = newRB;
  }
  else 
    _rubberBox = new QRect(newRB); 

  // Expand the redrawRect by the rubber box pen width.  Then do the repaint.
  expRectForPen(&redrawRect);
  repaint(redrawRect, false);

#ifdef DEBUG_RTP
  fprintf(stderr, "mouse moved %d %d\n", event->x(), event->y());
  fflush(stderr);
#endif
} 

/* When the mouse is released, rescale the window according to the final
 * rectangle.  If the rectangle is empty, or we have no points, do nothing.
 * If the range has OK resolution, switch _viewMode to USER_FIXED, so we'll
 * hold steady where we are.
 */
void PlotWindow::mouseReleaseEvent(QMouseEvent *event)
{
#ifdef DEBUG_RTP
  fprintf(stderr, "mouse released %d %d\n", event->x(), event->y());
  fflush(stderr);
#endif

  bool nullEvent = false, lowRes = false;
  BBox testPort;

  if ((event->x() == _rbAnchor.x()) || (event->y() == _rbAnchor.y())
      || (_map.xscale == 0.0) || (_map.yscale == 0.0))
    nullEvent = true;
  else {

    // Get the boundaries for the new viewPort by reverse mapping the 
    // rectangle into the current _map.
    testPort.minX = QMIN((_rbAnchor.x() - _map.xbias) / _map.xscale,
                         (event->x() - _map.xbias) / _map.xscale);
    testPort.maxX = QMAX((_rbAnchor.x() - _map.xbias) / _map.xscale,
                         (event->x() - _map.xbias) / _map.xscale);
    testPort.minY = QMIN((_rbAnchor.y() - _map.ybias) / _map.yscale,
                         (event->y() - _map.ybias) / _map.yscale);
    testPort.maxY = QMAX((_rbAnchor.y() - _map.ybias) / _map.yscale,
                         (event->y() - _map.ybias) / _map.yscale);

    // Check if resolution is OK.  This will modify testPort
    if (rtpResCheck(&testPort)) {
      lowRes = nullEvent = true;
    }
  }

  // Remove the rubber box.  If nullEvent, we need a repaint here to
  // clear it off the screen.
  if (_rubberBox) { 
    QRect redrawRect(*_rubberBox);
    expRectForPen(&redrawRect);
    delete _rubberBox;
    _rubberBox = NULL;
    if (nullEvent)
      repaint(redrawRect);
  }

  if (lowRes)           // Tell user about resolution problem.
    QMessageBox::warning(this, caption(), 
      "Cannot accommodate requested view port.\n"
      "Resolution is too low.\n", QMessageBox::Ok | QMessageBox::Default,
      0, 0);
  
  // If everything was OK, change viewMode, copy the test viewport
  // to the real thing, and schedule a preemptive render.
  if (!nullEvent) {
    _map.viewPort = testPort;
    _viewMode = USER_FIXED;  // Stick with user defined view.
    _buffer->fill(white);
    _map = _render.newOnlineRender(_map.viewPort, _buffer);
  }
}  
    
   
/* slotExpand switches the widget view mode to full, with auto scaling
 * enabled.  The window is cleared and rendering is scheduled.  Switch
 * _viewMode to AUTO_SCALE so if the bounding box changes, we'll 
 * automatically redraw.
 */
void PlotWindow::slotExpand()
{
  if (AUTO_SCALE == _viewMode)
    return;          // Ignore if already auto-scale.

  _viewMode = AUTO_SCALE;
  _map.viewPort = _allPointsBBox;
  _buffer->fill(white);
  _map = _render.newOnlineRender(_map.viewPort, _buffer);
}

/* slotTrack sets _viewMode to TRACKING, and latches _trackingPort to the
 * current _map.viewPort.
 */
void PlotWindow::slotTrack()
{
  if (TRACKING == _viewMode)
    return;

  _viewMode = TRACKING;
  _trackingPort = _map.viewPort;
}
    
int main(int argc, char *argv[])
{
  QApplication myapp(argc,argv);
  QMainWindow * mainwin = new QMainWindow();
  myapp.setMainWidget(mainwin);
  PlotWindow * plotter = new PlotWindow(mainwin);
  mainwin->setCentralWidget(plotter);
  mainwin->setGeometry(50,500,400,400);

  /* Configure stdin for our use.  Set O_NONBLOCK because we don't want
   * to block while reading input points.  Socket Notifier registers a
   * callback when we get data on stdin, but we don't know that a whole
   * pair of numbers will be there when we read. */
  fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK);  
  QSocketNotifier sn(STDIN_FILENO,QSocketNotifier::Read);
  QObject::connect(&sn, SIGNAL(activated(int)),
                   plotter, SLOT(slotStdinAwake()));

  /* Add toolbar to the main window */
  QToolBar * tb = new QToolBar(mainwin);
  QPixmap expandPm(expand);
  QToolButton * exbut = new QToolButton(expandPm, "View full data set", 0,
     plotter, SLOT(slotExpand()), tb, "View full data set");
  QPixmap trackPm(track);
  QToolButton * trbut = new QToolButton(trackPm, "Track new points", 0,
     plotter, SLOT(slotTrack()), tb, "Track new points");
  mainwin->show();
  return myapp.exec();
}
