/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY


#include <qlayout.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qvalidator.h>
#include <qmsgbox.h>
#include <qpalette.h>
#include <qcolor.h>
#include <qfont.h>
#include <qpmcache.h>

//#include <stdio.h>
#include <math.h>

#include "QdbtTabular-0.31/src/qdbttabcell.h"
#include "QdbtTabular-0.31/src/qdbtsection.h"
#include "pfsingtable.hpp"

MyTabular::MyTabular(QWidget *parent,const char *name,WFlags f) :
  QdbtTabular(parent,name,f)
{
  currentCell=QPoint(-1,-1);
}

MyTabular::~MyTabular()
{
currentCell.setY(0);
}


void MyTabular::mousePressEvent(QMouseEvent *e)
{
  int rowClicked=findRow(e->y());
  int colClicked=findCol(e->x());
  if (rowClicked!=-1){
    if(colClicked>0){
      if(cell(rowClicked,colClicked)->isEditable()){
	editCell(rowClicked,colClicked);
      }
    }
    else{
      setRowSelected(rowClicked,true);
      currentCell.setY(rowClicked);
    }
  }
}


void MyTabular::cellChanged(int row, int col){
  int i=0;
  QDoubleValidator *dval;
  QString *s=new QString(cell(row,col)->text());
  if (col<SingTable::MAGCOLUMN){
    dval=new QDoubleValidator(-10.0,10.0,6,0,"dval");
  }
  if (col==SingTable::MAGCOLUMN){
    dval=new QDoubleValidator(-100.0,100.0,6,0,"dval");
  }
  if (col==SingTable::ANGCOLUMN){
     dval=new QDoubleValidator(0.0,360.0,6,0,"dval");
  }
  if(dval->validate(*s,i)==2){
    emit NewValue(s->toDouble(),row,col);
  }
  else{
    emit BadValue(row);
  }
}

// constructor
SingTable::SingTable(SingList *p_SingularityList,
		     QWidget *parent,const char *name, 
		     WFlags f) : QWidget(parent,name,f)
{
  d_SingularityList=p_SingularityList;
  d_NumRows=0 ;

  // Set the caption of the window
  setCaption("PFlow 2.5: Singularities");

  // The Layout managers
  QGridLayout *layout  = new QGridLayout(this,1,1,3);
  
  tabular=new MyTabular(this);
  tabular->setDimensions(0,5);
  tabular->setMinimumSize(290,100);
  tabular->setMaximumWidth(290);
  tabular->selectByRow(false);
  tabular->setHeaderFont(QFont("helvetica",12,QFont::Bold));
  tabular->setCellFont(QFont("times",10,QFont::Bold));
  // Load the singularity pixmaps
  
  lvort= QPixmapCache::find("lvortex");
  rvort= QPixmapCache::find("rvortex");
  sink=  QPixmapCache::find("sink");
  source= QPixmapCache::find("source");
  dipole= QPixmapCache::find("dipole");
  stream= QPixmapCache::find("fstream");
  
  //SetUp
  QFont face("helvetica",12,QFont::Bold);
  QdbtSection *section;

  section=tabular->section(0);
  section->setText(0);
  section->setResizable(false);
  section->setAlignment(AlignCenter);

  section=tabular->section(1);
  section->setText("X");
  section->setResizable(false);
  section->setAlignment(AlignCenter);

  section=tabular->section(2);
  section->setText("Y");
  section->setResizable(false);
  section->setAlignment(AlignCenter);

  section=tabular->section(3);
  section->setText("Mag");
  section->setResizable(false);
  section->setAlignment(AlignCenter);

  section=tabular->section(4);
  section->setText("Theta");
  section->setResizable(false);
  section->setAlignment(AlignCenter);

  tabular->setColumnWidth(0,30);
  tabular->setColumnWidth(1,60);
  tabular->setColumnWidth(2,60);
  tabular->setColumnWidth(3,60);
  tabular->setColumnWidth(4,60);

  connect(d_SingularityList, SIGNAL(NewItemAdded(Singularity *)),
	  this, SLOT(addSingularity(Singularity *)));
  connect(tabular, SIGNAL(cellEdited(int,int)),
	  tabular, SLOT(cellChanged(int,int)));
  connect(tabular, SIGNAL(NewValue(double,int,int)),
	  this, SLOT(changeSingularity(double,int,int)));
  connect(tabular, SIGNAL(BadValue(int)), 
	  this, SLOT(correctBadEntry(int)));
  
  //Now that tabular is set up
  Singularity *temp=new Singularity(0, 0,
				    Singularity::FREESTREAM,
				    d_SingularityList->getFreeSpeed(),
				    d_SingularityList->getFreeDirection());
  
  addSingularity(temp);


  layout->addWidget(tabular,0,0);
  layout->setColStretch(0,1); // make the table strechable
  layout->setRowStretch(0,1);


  layout->activate();

  resize(layout->mainWidget()->sizeHint());
}

// destructor
SingTable::~SingTable()
{
}

  

void SingTable::addSingularity(Singularity *p_Singularity){
  int newrow;

  newrow=tabular->numRows();
  tabular->setDimensions(newrow+1,5);
  d_NumRows++;
  singToTable(p_Singularity,newrow);

 }

void SingTable::clearList(){

  tabular->setDimensions(1,5);
  Singularity *temp=new Singularity(0,
				    0,
				    Singularity::FREESTREAM,
				    d_SingularityList->getFreeSpeed(),
				    d_SingularityList->getFreeDirection());
  singToTable(temp,0);
  d_NumRows=1;
}

void SingTable::singToTable(Singularity *p_Singularity,int row){
  float X,Y,M,A;

  if (row>d_NumRows-1){
    addSingularity(p_Singularity);
  } else{

    X=p_Singularity->x;
    Y=p_Singularity->y;
    M=p_Singularity->mag;
    A=p_Singularity->ang*180/PI;
    QString contents;
  
    QdbtTableCell bmp;
    QdbtTableCell gray;
    QdbtTableCell editable;

    bmp.setEditable(false);
    bmp.setSelectable(FALSE);
    bmp.setPixmapAlignment(AlignRight);
    bmp.setText(0);
    bmp.setBackground(QColor(70,70,100));

    gray.setEditable(FALSE);
    gray.setText(0);
    gray.setSelectable(FALSE);
    gray.setBackground(QColor(70,70,100));

    editable.setEditable(TRUE);
    editable.setText(0);
    editable.setSelectable(FALSE);
    editable.setBackground(white);
  
    switch (p_Singularity->t){
    case Singularity::SOURCE:
      if(p_Singularity->mag>0){
	bmp.setPixmap(source);
      }
      else{
	bmp.setPixmap(sink);
      };
      tabular->changeCell(&gray,row,ANGCOLUMN);

      break;
    case Singularity::VORTEX:
      if(p_Singularity->mag<0){
	bmp.setPixmap(rvort);
      }
      else{
	bmp.setPixmap(lvort);
      };
      tabular->changeCell(&gray,row,ANGCOLUMN);
      break;
    case Singularity::DIPOLE:
      bmp.setPixmap(dipole);
      contents.sprintf("%+2.3f",A);
      editable.setText(contents);
      tabular->changeCell(&editable,row,ANGCOLUMN);
      break;
    
    };//switch
    if (p_Singularity->t!=Singularity::FREESTREAM){
      tabular->changeCell(&bmp,row,TYPECOLUMN);
      contents.sprintf("%+2.3f",X);
      editable.setText(contents);
      tabular->changeCell(&editable,row,XCOLUMN);
      contents.sprintf("%+2.3f",Y);
      editable.setText(contents);
      tabular->changeCell(&editable,row,YCOLUMN);
      contents.sprintf("%+2.3f",M);
      editable.setText(contents);
      tabular->changeCell(&editable,row,MAGCOLUMN);
    }
    else{
      bmp.setPixmap(stream);
      tabular->changeCell(&bmp,row,TYPECOLUMN);
      tabular->changeCell(&gray,row,XCOLUMN);
      tabular->changeCell(&gray,row,YCOLUMN);
      contents.sprintf("%+2.3f",M);
      editable.setText(contents);
      tabular->changeCell(&editable,row,MAGCOLUMN);
      contents.sprintf("%+2.3f",A);
      editable.setText(contents);
      tabular->changeCell(&editable,row,ANGCOLUMN);
    }
  }
}

void SingTable::changeSingularity(double Val, int row, int col){
  Singularity *target;
  if(row>=1){
    target=d_SingularityList->at(row-1);
    switch(col){
    case XCOLUMN:
      d_SingularityList->change(row-1,(float)Val,target->y,target->t,target->mag,target->ang);
      break;
    case YCOLUMN:
      d_SingularityList->change(row-1,target->x,(float)Val,target->t,target->mag,target->ang);
      break;
    case MAGCOLUMN:
      d_SingularityList->change(row-1,target->x,target->y,target->t,Val,target->ang);
      break;
    case ANGCOLUMN:
      d_SingularityList->change(row-1,target->x,target->y,target->t,target->mag,Val*PI/180);
    }//switch
    singToTable(d_SingularityList->at(row-1),row);
  }
  else{//change the free stream
    QdbtTableCell editable;
    editable.setEditable(TRUE);
    editable.setText(0);
    editable.setSelectable(FALSE);
    editable.setBackground(white);
    QString contents;
    contents.sprintf("%+2.3f",Val);
    editable.setText(contents);  

    switch(col){
    case MAGCOLUMN:
      d_SingularityList->setFreeSpeed(Val);break;
    case ANGCOLUMN:
      d_SingularityList->setFreeDirection(Val*PI/180);
    }//switch
    tabular->changeCell(&editable,0,col);
  }//else
}

void SingTable::correctBadEntry(int row){
  Singularity *prev;
  if (row>0){
    prev=d_SingularityList->at(row-1);
  }
  else{
  prev=new Singularity(0,
		       0,
		       Singularity::FREESTREAM,
		       d_SingularityList->getFreeSpeed(),
		       d_SingularityList->getFreeDirection());
  }
  singToTable(prev,row);
  
  QMessageBox *BadEntry=new QMessageBox("PFlow 2.5 Error","\
You have entered an illegal value.\n\
\n\
Legal Values are:\n\
X,Y: -10.0 to 10.0\n\
Mag: -100.0to 100.0\n\
Theta: 0.0 to 360.0",
					QMessageBox::Warning,
					QMessageBox::Ok,0,0,
					this,"badentry",TRUE,0);
  BadEntry->show();
}

void SingTable::updateall(){
 Singularity *prev;
 prev=new Singularity(0,
		      0,
		      Singularity::FREESTREAM,
		      d_SingularityList->getFreeSpeed(),
		      d_SingularityList->getFreeDirection());
  
 singToTable(prev,0); 
 for(uint i=0;i<d_SingularityList->count();i++){
    singToTable(d_SingularityList->at(i),i+1);
  } 

};
  
//ENDPRETTY
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */

