/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY

#include <qlayout.h>
#include <qchkbox.h>
#include <qpen.h>
#include <qcolor.h>
#include <qprinter.h>
#include "pfdatagraph.hpp"
#include <stdio.h>
#include <math.h>

DataGraph::DataGraph(DataPointList *Dp, QWidget *parent=0, const char *name=0):QFrame(parent,name){
  d_DataPoints=Dp;
  
  connect(d_DataPoints,SIGNAL(ListChanged()),this,SLOT(update()));

  d_Plot=new QwtPlot("Data Point Parameters", this);
  d_Plot->show();

  QGridLayout *l_Glayout=new QGridLayout(this,1,1,0);
  
  l_Glayout->addWidget(d_Plot,0,0);//easier than a new resize event handler

  //Initialize and set up the curves, at first only the Speed and
  //Pressure curves will be plotted

  UXKey = d_Plot->insertCurve("Ux");
  d_Plot->setCurvePen(UXKey,QPen(QColor(0,255,255),2));
  d_Plot->setCurveStyle(UXKey,QwtCurve::NoCurve);

  UYKey = d_Plot->insertCurve("Uy");
  d_Plot->setCurvePen(UYKey,QPen(QColor(255,255,0),2));
  d_Plot->setCurveStyle(UYKey,QwtCurve::NoCurve);

  SpeedKey=d_Plot->insertCurve("Speed");
  d_Plot->setCurvePen(SpeedKey,QPen(QColor(255,255,255),2));
  d_Plot->setCurveStyle(SpeedKey,QwtCurve::Lines);

  PressKey=d_Plot->insertCurve("Pressure");
  d_Plot->setCurvePen(PressKey,QPen(QColor(255,0,255),2));
  d_Plot->setCurveYAxis(PressKey, QwtPlot::yRight);
  d_Plot->setCurveStyle(PressKey,QwtCurve::Lines);
  

  //Now set up the plot pretty thingies
  d_Plot->setTitle("Data Point Parameters");
  d_Plot->enableAxis(QwtPlot::yRight);
  d_Plot->setAxisTitle(QwtPlot::xBottom, "Datapoint");
  d_Plot->setAxisTitle(QwtPlot::yLeft, "Speed");
  d_Plot->setAxisTitle(QwtPlot::yRight, "Pressure");

  d_Plot->setPlotBackground(darkBlue);
  d_Plot->setGridMajPen(QPen(white, 0, DotLine));
  d_Plot->setGridMinPen(QPen(gray, 0 , DotLine));

  d_Plot->setAutoReplot(TRUE);

  d_Plot->enableLegend(TRUE);
  d_Plot->setLegendPos(Qwt::Bottom);
  d_Plot->setLegendFrameStyle(QFrame::Box|QFrame::Sunken);
  
  //Set up the menus

  d_options = new QPopupMenu();
  d_options->insertItem( "Plot Ux",UXBUTTON);
  d_options->insertItem( "Plot Uy",UYBUTTON);
  d_options->insertItem( "Plot Speed",SPEEDBUTTON);
  d_options->insertSeparator();  
  d_options->insertItem( "Plot Pressure",PRESSBUTTON);

  d_options->setCheckable( TRUE );

  d_options->setItemChecked(UXBUTTON,FALSE);
  d_options->setItemChecked(UYBUTTON,FALSE);
  d_options->setItemChecked(SPEEDBUTTON,TRUE);
  d_options->setItemChecked(PRESSBUTTON,TRUE);


  d_options->connectItem(UXBUTTON,this,SLOT(toggleUx()));
  d_options->connectItem(UYBUTTON,this,SLOT(toggleUy()));
  d_options->connectItem(SPEEDBUTTON,this,SLOT(toggleSpeed()));
  d_options->connectItem(PRESSBUTTON,this,SLOT(togglePress()));


  QMenuBar *menu = new QMenuBar( this );
  CHECK_PTR( menu );
  menu->insertItem( "Print",this, SLOT(print()) );
  menu->insertItem( "Options", d_options );
  menu->setSeparator( QMenuBar::InWindowsStyle );
}

void DataGraph::print(){
  QPrinter p;

  p.setPageSize(QPrinter::Letter); 
  p.setOutputToFile(FALSE);
  if (p.setup(0))
    {
      d_Plot->print(p, QwtFltrDim(200));
    }
};

void DataGraph::update(){
  int count=d_DataPoints->count();
  double *ux;
  double *uy;
  double *speed;
  double *press;

  double *xcoord;
  
  xcoord=new double[count];
  ux=new double[count];
  uy=new double[count];
  speed=new double[count];
  press=new double[count];

  //deep copy from the datapoint list into out new arrays, it is what
  //Qwtplot want. Easier than rewriting that thing!!

  for(int i=0;i<count;i++){
    xcoord[i]=(double)(i);
    ux[i]=d_DataPoints->at(i)->uX;
    uy[i]=d_DataPoints->at(i)->uY;
    speed[i]=sqrt(ux[i]*ux[i]+uy[i]*uy[i]);
    press[i]=d_DataPoints->at(i)->pressure;
  }

  d_Plot->setCurveData(UXKey,xcoord,ux,count);
  d_Plot->setCurveData(UYKey,xcoord,uy,count);
  d_Plot->setCurveData(PressKey,xcoord,press,count);
  d_Plot->setCurveData(SpeedKey,xcoord,speed,count);

  //autoRefrsh is set, so the plot takes care of itself.
}

void DataGraph::toggleUx(){
  bool l_Flag=d_options->isItemChecked(UXBUTTON);
  
  if(l_Flag){
    d_Plot->setCurveStyle(UXKey,QwtCurve::NoCurve);
  }
  else{
    d_Plot->setCurveStyle(UXKey,QwtCurve::Lines);
  }

  d_options->setItemChecked(UXBUTTON,!l_Flag);
}

void DataGraph::toggleUy(){
  bool l_Flag=d_options->isItemChecked(UYBUTTON);
  
  if(l_Flag){
    d_Plot->setCurveStyle(UYKey,QwtCurve::NoCurve);
  }
  else{
    d_Plot->setCurveStyle(UYKey,QwtCurve::Lines);
  }

  d_options->setItemChecked(UYBUTTON,!l_Flag);
}
void DataGraph::toggleSpeed(){
  bool l_Flag=d_options->isItemChecked(SPEEDBUTTON);
  
  if(l_Flag){
    d_Plot->setCurveStyle(SpeedKey,QwtCurve::NoCurve);
  }
  else{
    d_Plot->setCurveStyle(SpeedKey,QwtCurve::Lines);
  }

  d_options->setItemChecked(SPEEDBUTTON,!l_Flag);
}
void DataGraph::togglePress(){
  bool l_Flag=d_options->isItemChecked(PRESSBUTTON);
  
  if(l_Flag){
    d_Plot->setCurveStyle(PressKey,QwtCurve::NoCurve);
  }
  else{
    d_Plot->setCurveStyle(PressKey,QwtCurve::Lines);
  }

  d_options->setItemChecked(PRESSBUTTON,!l_Flag);
}

  
//ENDPRETTY
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */
