/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY

#include "pfcalcengine.hpp"

#include <qprogdlg.h>
#include <qpainter.h>
#include <math.h>
#include <stdio.h>

CalcEngine::CalcEngine(SingList *SingularityList, float DeltaT){
  d_DeltaT=DeltaT;
  d_SingularityList=SingularityList;
}

void CalcEngine::setFreeStream(){
  d_FreeX=d_SingularityList->getFreeSpeed()
    *cos(d_SingularityList->getFreeDirection());
  d_FreeY=d_SingularityList->getFreeSpeed()
    *sin(d_SingularityList->getFreeDirection());
}


void CalcEngine::calcStreamlines(StreamList *p_StreamlineList){

  Streamline *l_CurrentStreamline;
  for(l_CurrentStreamline=p_StreamlineList->first();//For loop init 
      l_CurrentStreamline!=NULL; //forloop condition
      l_CurrentStreamline=p_StreamlineList->next()){//start the loop
    calcStreamline(l_CurrentStreamline);
  }
}


void CalcEngine::calcStreamline(Streamline *p_Streamline){

  float l_X,l_Y,l_Xnew,l_Ynew;
  CalcEngine::Velocity l_Velocity;
  //setFreeStream gets called too often, but not worth fancy footwork to make it better
  setFreeStream();
  //Initial Streamline position
  l_X=p_Streamline->first()->x();
  l_Y=p_Streamline->first()->y();

  l_Xnew=l_X;l_Ynew=l_Y;

  p_Streamline->clear();//get rid of the previous calculations
  //Restore the first point after clearing the list
  p_Streamline->append(new FPoint(l_X,l_Y));

  //loop through a simple forward difference integration
  for(int i=0;i<30000;i++){
    l_Velocity=calcVelocity(l_Xnew,l_Ynew);
    
    l_Xnew=l_Xnew+l_Velocity.U*d_DeltaT;
    l_Ynew=l_Ynew+l_Velocity.V*d_DeltaT;

    if((-10.0>l_Xnew)||(10.0<l_Xnew)) break;
    if((-10.0>l_Ynew)||(10.0<l_Ynew)) break;
    if((fabs(l_Velocity.U)<0.0001)&&(fabs(l_Velocity.V)<0.0001)) break;
    //This code is an attempt to eliminate ridiculous dS lengths
    //when a streamline passes through a sink
    //Joe Curran, 8-98
    if((fabs(l_Velocity.U)>50)||(fabs(l_Velocity.V)>50)) break;

    //this conditional prevents us from STORING every points we
    //calculate, this speeds up the graphics end significantly, and
    //make the runtime quite a bit smaller (when was the last time you
    //got smaller AND faster!) the 0.1 is experimentally determined.
    if((fabs(l_Xnew-l_X)+fabs(l_Ynew-l_Y))>0.1){
      p_Streamline->append(new FPoint(l_Xnew,l_Ynew));
      l_X=l_Xnew;
      l_Y=l_Ynew;
    }
  }//streamline for loop
}//int calcstream

float CalcEngine::calcPointSpeed(float p_X, float p_Y){
  Velocity l_V= calcVelocity(p_X,p_Y);
  return sqrt(l_V.U * l_V.U+l_V.V * l_V.V);
}

CalcEngine::Flow CalcEngine::calcPoint(float p_X,float p_Y){
  Velocity V=calcVelocity(p_X,p_Y);
  float l_FreeSpeed=d_SingularityList->getFreeSpeed();
  float Pressure=l_FreeSpeed*l_FreeSpeed-(V.U*V.U+V.V*V.V);
  Flow F;
  F.U=V.U;  F.V=V.V;  F.P=Pressure;
  return F;
}

float CalcEngine::calcPointPressure(float p_X,float p_Y){
  return calcPoint(p_X,p_Y).P;
}
//ENDPRETTY
//This commented out because it is no loger accessible from the base
//form. May be I'll put back in sometime and I won't have to reinvent
//it.

// bool CalcEngine::plotPressure(QPainter *p){

//   int X,Y;

//   bool l_Success=TRUE;

//   QProgressDialog l_Progress( "Calculating Pressures...", "Abort",200, 0 ,"Progress",this,false);

//   for(int i=-100;i<100;i++){
//     l_Progress.setProgress(i+1000 );
//     if (l_Progress.wasCancelled() ){
//       l_Success=FALSE;
//       break;
//     }
//     X=100*i;
//     printf("%d\n",X);
//     for(int j=-100;j<100;j++){
//       Y=100*j;

//       float Pressure=50.0*calcPointPressure((float)i/10.0,(float)j/10.0);
       
//       int C=(int)(50*log(fabs(Pressure)+1.0));
//       if (C>255) C=255;
//       if (Pressure>=0){
// 	p->fillRect(X,Y,101,101,QColor(C,0,0));
//       }
//       else{
// 	p->fillRect(X,Y,101,101,QColor(0,0,C));
//       }
//     }
//   }
//   l_Progress.setProgress(200);
//   return l_Success;
// }
//STARTPRETTY
CalcEngine::Velocity CalcEngine::calcVelocity(float X, 
					      float Y){
  float DelX,DelY,Vx,Vy,U=0,V=0,l_Mag,l_Ang;
  float Rd,R,sinA,cosA;
  
  Singularity *l_Singularity;
  setFreeStream();
  Vx=d_FreeX;
  Vy=d_FreeY;

  
  //Loop through the Singularity list calculating velocity
  //contribution from each singularity
  for(l_Singularity=d_SingularityList->first(); l_Singularity!=NULL; l_Singularity=d_SingularityList->next()){
    DelX=X-(l_Singularity->x);
    DelY=Y-(l_Singularity->y);//Calulate the offset to the current singularity.
    l_Mag=l_Singularity->mag;
    l_Ang=l_Singularity->ang;
    if (((DelX <= (0.0 + d_Epsilon)) && (DelX >= (0.0 - d_Epsilon))) &&
	((DelY <= (0.0 + d_Epsilon)) && (DelY >= (0.0 - d_Epsilon))) ) {
      U = 0.0;
      V = 0.0;
    }
    else{
      switch(l_Singularity->t)
	{
	case Singularity::SOURCE:
	  //Source
	  /* The following is for 2D flow
             Corrected 10/13/98 by Joe Curran */
          U = (l_Mag / (4.0 * PI)) * 2*DelX / (DelX*DelX + DelY*DelY);
          V = (l_Mag / (4.0 * PI)) * 2*DelY / (DelX*DelX + DelY*DelY);
	  //  U = (l_Mag / (4.0 * PI)) * pow((DelX*DelX + DelY*DelY), -1.5) * DelX;
	  // 	  V = (l_Mag / (4.0 * PI)) * pow((DelX*DelX + DelY*DelY), -1.5) * DelY;
	  break;
	case Singularity::VORTEX:
	  //Vortex
	  U = -(l_Mag / (2.0 * PI)) * (DelY / (DelX*DelX + DelY*DelY));
	  V =  (l_Mag / (2.0 * PI)) * (DelX / (DelX*DelX + DelY*DelY));
	  break;
	case Singularity::DIPOLE:
	  //Dipole 
	  Rd=DelY*DelY-DelX*DelX;
	  R=DelY*DelY+DelX*DelX;
	  R=R*R*2.0*PI;
	  sinA=sin(l_Ang);
	  cosA=cos(l_Ang);
	  U = -l_Mag*(Rd*cosA-2.0*DelX*DelY*sinA)/R;
	  V = -l_Mag*(-Rd*sinA-2.0*DelX*DelY*cosA)/R;
	  break;
	};//switch	
    }//else
    Vx=Vx+U;
    Vy=Vy+V;
  }//singularity loop
  Velocity l_Velocity;
  l_Velocity.U=Vx;
  l_Velocity.V=Vy;
  return l_Velocity;
}



//ENDPRETTY

/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */
