/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY
#include <qpushbt.h>
#include <qbttngrp.h>
#include <qwidget.h>
#include <qmenubar.h>
#include <qstring.h>
#include <qprinter.h>
#include <qchkbox.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qpmcache.h>
#include <qlayout.h>
#include <qstring.h>

#include "pfbaseform.hpp"
#include "pfcolorpicker.h"

QPushButton *BaseForm::addButton(char *name,QButtonGroup *grp, 
				 const char* pixkey,
				 int x, int y, int w, int h,
				 char * tip){

  QPushButton *l_PushButton = new QPushButton (name, grp, "srcb");

  QPixmap *pm=QPixmapCache::find(pixkey);
  
  if(pm) 
    l_PushButton->setPixmap(*pm);
  else 
    l_PushButton->setText(pixkey);
  
  l_PushButton->setGeometry(x,y,w,h);
  
  QToolTip::add(l_PushButton,tip);

  return l_PushButton;
};


BaseForm::BaseForm(QWidget *parent, const char *name):QWidget (parent,name)
{

  //BaseClass Initialization
  setCaption("PFlow 2.5.1");
  setGeometry(100,100,600,730);

  //local Variable definitions and initialization.
  const int l_ADDBUTTON_W=33;
  const int l_ADDBUTTON_H=33;


  //Setup Class Data members, watch the creation order.
  d_PFStatus=PFStatus::Instance();

  d_SingularityList = d_PFStatus->getSingList();
  d_Streamlines =  d_PFStatus->getStreamList();

  d_Engine =  d_PFStatus->getCalcEngine();
  d_DataPoints =  d_PFStatus->getDataPointList();
  
  d_Fluid = new PFDisplay(this,"pfdisplay");

  d_Table =  d_PFStatus->getSingTable();
  d_DataTable =  d_PFStatus->getDataTable();
  d_DataGraph =  d_PFStatus->getDataGraph();

  d_LeftClickResponse=-1; //Initialize to doing nothing
  d_RightClickResponse=-1;

  (d_AddGrp =new QButtonGroup("Add:",this));//->setFont(face); 
  d_AddGrp->setExclusive(true);
  (d_ClrGrp = new QButtonGroup("Clear:",this));//->setFont(face);

  //***************************************************************
  //set up menus
  //***************************************************************


  QPopupMenu *file = new QPopupMenu();
  CHECK_PTR( file );
  file->insertItem( "Open ...",  
		    this, SLOT(openFile()) );
  file->insertItem( "New", 
		    this, SLOT(clearAll()) );
  file->insertItem("Save", 
		   this, SLOT(saveFile()) );
  file->insertItem("Save As ...", 
		   this, SLOT(saveFileAs()) );
  file->insertSeparator();
  file->insertItem( "Print ...", 
		    this, SLOT(print()));
  file->insertSeparator();
  file->insertItem( "Exit",  
		    qApp, SLOT(quit()) );

  QPopupMenu *colors = new QPopupMenu();
  CHECK_PTR( file );
  colors->insertItem( "Default Colors",  
		      d_Fluid, SLOT(resetColors()) );
  colors->insertItem( "Fluid Color ...",  
		      this, SLOT(setFluidColor()) );
  colors->insertItem( "Grid Color ...", 
		      this, SLOT(setGridColor()) );
  colors->insertItem("Streamline Color ...", 
		     this, SLOT(setStreamlineColor()) );
  colors->insertItem("Singularity Color ...", 
		     this, SLOT(setSingularityColor()) );
  colors->insertItem("Data Point Color ...", 
		     this, SLOT(setDataPointColor()) );

  d_display = new QPopupMenu();
  CHECK_PTR( d_display);
  d_display->insertItem("Grid", GRIDBUTTON);
  d_display->insertItem("Singularities", SINGBUTTON);
  d_display->insertItem("Streamlines", STREAMBUTTON);
  d_display->insertItem("Data Points", DATABUTTON);
  d_display->setCheckable(TRUE);

  d_display->connectItem(GRIDBUTTON, 
			 this, SLOT(toggleGrid()));
  d_display->connectItem(SINGBUTTON, 
			 this, SLOT(toggleSingularities()));
  d_display->connectItem(DATABUTTON, 
			 this, SLOT(toggleDataPoints()));
  d_display->connectItem(STREAMBUTTON, 
			 this, SLOT(toggleStreamlines()));


  d_display->setItemChecked(GRIDBUTTON, TRUE);
  d_display->setItemChecked(SINGBUTTON, TRUE);
  d_display->setItemChecked(STREAMBUTTON, TRUE);
  d_display->setItemChecked(DATABUTTON, TRUE);


  d_options = new QPopupMenu();
  CHECK_PTR( d_options );
  d_options->insertItem( "Display", d_display);
  d_options->insertSeparator();
  d_options->insertItem( "Snap to Grid", 
			 SNAPBUTTON);
  d_options->insertSeparator();
  d_options->insertItem( "Show Singularity Table", 
			 SINGTABLEBUTTON);
  d_options->insertItem( "Show Datapoint Table", 
			 DATATABLEBUTTON);
  d_options->insertItem( "Show Datapoint Graph", 
			 DATAGRAPHBUTTON);
  d_options->insertSeparator();
  d_options->insertItem( "Print Options...");
  d_options->insertSeparator();
  d_options->insertItem( "Change Colors...", 
			 colors);

  d_options->connectItem(SNAPBUTTON, 
			 this, SLOT(toggleSnap()));
  d_options->connectItem(SINGTABLEBUTTON, 
			 this, SLOT(toggleSingTable()));
  d_options->connectItem(DATATABLEBUTTON, 
			 this, SLOT(toggleDataTable()));
  d_options->connectItem(DATAGRAPHBUTTON, 
			 this, SLOT(toggleDataGraph()));

  d_options->setCheckable( TRUE );
  d_options->setItemChecked(SNAPBUTTON, TRUE);
  d_options->setItemChecked(SINGTABLEBUTTON, TRUE);
  d_options->setItemChecked(DATATABLEBUTTON, FALSE);
  d_options->setItemChecked(DATAGRAPHBUTTON, FALSE);

  QMenuBar *menu = new QMenuBar( this );
  CHECK_PTR( menu );
  menu->insertItem( "File", file );
  menu->insertItem( "Options", d_options );
  menu->insertSeparator();
  menu->insertItem( "About", this, SLOT(showAbout()) );
  menu->setSeparator( QMenuBar::InWindowsStyle );

  //******************************************************
  //Setup layout geometry
  //******************************************************

  QGridLayout *l_layout1=new QGridLayout(this,3,1,0);
  QGridLayout *l_layout2=new QGridLayout(1,2,10);

  l_layout1->addLayout(l_layout2,1,0);

  menu->setFixedHeight(30);
  d_AddGrp->setFixedHeight(60);


  l_layout1->addWidget(menu,0,0);

  l_layout2->addWidget(d_AddGrp,0,0);
  l_layout2->addWidget(d_ClrGrp,0,1);
  l_layout2->setColStretch(0,1);
  l_layout2->setColStretch(1,1);


  l_layout1->addWidget(d_Fluid,2,0);

  l_layout1->setRowStretch(0,0);
  l_layout1->setRowStretch(1,0);
  l_layout1->setRowStretch(2,1);

  l_layout1->activate();

  //******************************************************
  //Set up connections
  //******************************************************

  connect(d_Fluid,SIGNAL(LeftClicked(double,double)),
	  this, SLOT(doLeftClick(double,double)));

  connect(d_Fluid,SIGNAL(RightClicked(double,double)),
	  this, SLOT(doRightClick(double,double)));

  connect(d_SingularityList,SIGNAL(ListChanged()),
	  this,SLOT(reCalc()));

  connect(d_SingularityList,SIGNAL(ListChanged()),
	  d_Table,SLOT(update()));

  connect(d_DataPoints,SIGNAL(ListChanged()),
	  this,SLOT(refresh()));

  connect(d_AddGrp, SIGNAL(clicked(int)), 
	  this, SLOT(setState(int)));

  connect(d_SingularityList,SIGNAL(ListChanged()),
	  d_DataPoints,SLOT(updateall()));

  //*******************************************************
  // These are the buttons to control the fluid field
  //*******************************************************

  addButton("Source",d_AddGrp,"source",
	    10,17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a source")->setToggleButton(true);
  addButton("Sink",d_AddGrp,"sink",
	    10+l_ADDBUTTON_W,17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a sink")->setToggleButton(true);
  addButton("Vortex",d_AddGrp,"lvortex",
	    16+3*l_ADDBUTTON_W,17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a CCW Vortex")->setToggleButton(true);
  addButton("Vortex",d_AddGrp,"rvortex",
	    16+4*l_ADDBUTTON_W,17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a CW Vortex")->setToggleButton(true);
  addButton("Dipole",d_AddGrp,"dipole",
	    13+2*l_ADDBUTTON_W,17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a dipole")->setToggleButton(true);
  addButton("Streamline",d_AddGrp,"stream",
	    (int)(22+5*l_ADDBUTTON_W),17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a streamline")->setToggleButton(true);
  addButton("Datapoint",d_AddGrp,"datapoint",
	    (int)(22+6*l_ADDBUTTON_W),17,l_ADDBUTTON_W,l_ADDBUTTON_H,
	    "Add a data point")->setToggleButton(true);

  //*********************************************************
  // Clear Buttons {this is not an attempt at obfuscated C++}
  //*********************************************************

  connect(addButton("All",d_ClrGrp,"xall",
		    12+2*l_ADDBUTTON_W,17,
		    l_ADDBUTTON_W, l_ADDBUTTON_H,
		    "Clear EVERYTHING!!!"),SIGNAL(clicked()),
	  this,SLOT(clearAll()));
  connect(addButton("Streamlines",d_ClrGrp,"xstream",
		    10,17,
		    l_ADDBUTTON_W, l_ADDBUTTON_H,
		    "Clear streamlines"),SIGNAL(clicked()),
	  this,SLOT(clearStreamlines()));
  connect(addButton("Singularities",d_ClrGrp,"xsing",
		    11+l_ADDBUTTON_W,17,
		    l_ADDBUTTON_W, l_ADDBUTTON_H,
		    "Clear singularities"),SIGNAL(clicked()),
	  this,SLOT(clearSingularities()));
  connect(addButton("DataPoints",d_ClrGrp,"xdatapoints",
		    12+3*l_ADDBUTTON_W,17,
		    l_ADDBUTTON_W, l_ADDBUTTON_H,
		    "Clear datapoints"),SIGNAL(clicked()),
	  this,SLOT(clearDataPoints()));

  d_Fluid->show();  
  d_Table->show();
}


//File handling slots
void BaseForm::openFile(){
  d_Fluid->setStatus(d_PFStatus->openFile());
  setCaption(QString("PFlow 2.5: ")+d_PFStatus->getName());
    d_DataTable->updateall();
    d_Table->updateall();
    d_DataGraph->update();
  reCalc();
};

void BaseForm::saveFile(){
  if(d_PFStatus->hasName())
    d_Fluid->setStatus(QString("Saved ")+d_PFStatus->saveFile());
  else
    saveFileAs();
};

void BaseForm::saveFileAs(){
  QString l_Name=d_PFStatus->saveFileAs();
  d_Fluid->setStatus(l_Name);
  setCaption(QString("PFlow 2.5: ")+d_PFStatus->getName());
};



void BaseForm::print(){
  QPrinter prn;
  prn.setPageSize(QPrinter::Letter);
  prn.setOutputFileName("pflow.ps");
  prn.setOutputToFile(FALSE);
  if(prn.setup(0)){
    d_Fluid->setStatus("Printing...");
    QPainter p;
    p.begin(&prn);
    p.setWindow(-10000,-10000,20000,20000);
    p.setViewport(72,72,468,468);
    p.setPen(QPen(QColor("Black"),100));
    d_Fluid->print(&p);
    p.moveTo(-10000,10000);
    p.lineTo(10000,10000);
    p.lineTo(10000,-10000);    
    p.lineTo(-10000,-10000);    
    p.lineTo(-10000,10000);
    p.end();

    d_Fluid->setStatus("Printing...Complete.");
  }
  else{    
    d_Fluid->setStatus("Print canceled.");
  };
};

void BaseForm::refresh(){
  d_Fluid->repaint(false);
  d_Fluid->setStatus("");
}

void BaseForm::reCalc(){
  d_Fluid->setStatus("Calculating Streamlines...Please Wait");
  d_Engine->calcStreamlines(d_Streamlines);
  refresh();
}
  
void BaseForm::setState(int Signal){
  d_LeftClickResponse=Signal;
}

void BaseForm::doLeftClick(double p_X, double p_Y){
  QString s;

  s.sprintf("No Action Taken: X=%2.3f Y=%2.3f",p_X,p_Y);

  if (!((p_X>9.0)||(p_X< -9.0)||(p_Y>9.0)||(p_Y< -9.0))){
    //I broke this case statement up because when you add a datapoint
    //there is no need to calcStreamLines, there is if you are adding another
    //singularity. (did you see the calcStreamLines at the end?)
    if(d_LeftClickResponse==ADDDATAPOINT){
      d_DataPoints->addNewDataPoint(p_X,p_Y);
      s.sprintf("Added a Data Point at X=%2.3f Y=%2.3f",p_X,p_Y);
    }
    else{
      switch(d_LeftClickResponse){
      case ADDSOURCE:
	d_SingularityList->addNewSingularity(p_X,p_Y,
					     Singularity::SOURCE,
					     5.0,0.0);
	s.sprintf("Placed source at X=%2.3f Y=%2.3f",p_X,p_Y);      
	break;
      case ADDSINK:
	d_SingularityList->addNewSingularity(p_X,p_Y,
					     Singularity::SOURCE,
					     -5.0,0.0);
	s.sprintf("Placed sink at X=%2.3f Y=%2.3f",p_X,p_Y);
	break;
      case ADDRVORTEX:
	d_SingularityList->addNewSingularity(p_X,p_Y,
					     Singularity::VORTEX,
					     5.0,0.0);
	s.sprintf("Placed right vortex at X=%2.3f Y=%2.3f",p_X,p_Y);
	break;
      case ADDLVORTEX:
	d_SingularityList->addNewSingularity(p_X,p_Y,
					     Singularity::VORTEX,
					     -5.0,0.0);
	s.sprintf("Placed left vortex at X=%2.3f Y=%2.3f",p_X,p_Y);
	break;
      case ADDDIPOLE:
	d_SingularityList->addNewSingularity(p_X,p_Y,
					     Singularity::DIPOLE,
					     5.0,0.0);
	s.sprintf("Placed Dipole at X=%2.3f Y=%2.3f, adjust direction in spreadsheet",
		  p_X,p_Y);
	break;
      case ADDSTREAMLINE:
	d_Streamlines->Add(p_X,p_Y);
	s.sprintf("Started a streamline at X=%2.3f Y=%2.3f",
		  p_X,p_Y);
	break;
      }//switch
      d_Engine->calcStreamlines(d_Streamlines);
    }//else
    d_Fluid->repaint(false);
  }//if
  d_Fluid->setStatus(s);
}

void BaseForm::doRightClick(double X, double Y){
  QString s;
  CalcEngine::Flow l_Flow=d_Engine->calcPoint(X,Y);
  s.sprintf("At X=%2.3f Y=%2.3f, U=%2.3f V=%2.3f, P_gauge=%2.3f",
	    X,Y,l_Flow.U,l_Flow.V,l_Flow.P);
  d_Fluid->setStatus(s);
};
	 

void BaseForm::showAbout(){
  AboutBox *l_a=new AboutBox(0,0);
  l_a->exec();
}

void BaseForm::clearStreamlines(){
  d_Streamlines->clear();
  d_Fluid->setStatus("Deleted all streamlines.");
  reCalc();
}

void BaseForm::clearDataPoints(){
  d_DataPoints->clear();
  d_DataTable->clearList();
  d_Fluid->setStatus("Deleteded all data points.");
  reCalc();
}

void BaseForm::clearSingularities(){
  d_SingularityList->clear();
  d_Table->clearList();
  d_Fluid->setStatus("Cleared all singularities.");
  reCalc();
}

void BaseForm::clearAll(){
  d_PFStatus->clear();
  reCalc();
  d_Fluid->setStatus("Cleared everything.");
  setCaption("PFlow 2.5");
}

void BaseForm::toggleSnap(){
  bool l_Flag=d_options->isItemChecked(SNAPBUTTON);
  d_Fluid->setDisplaySnapToGrid(!l_Flag);
  d_options->setItemChecked(SNAPBUTTON,!l_Flag);
  if(l_Flag)
    d_Fluid->setStatus("Snap to grid is off.");
  else
    d_Fluid->setStatus("Snap to grid is on.");
}

void BaseForm::toggleGrid(){
  bool l_Flag=d_display->isItemChecked(GRIDBUTTON);
  d_Fluid->setDisplayGrid(!l_Flag);
  d_display->setItemChecked(GRIDBUTTON,!l_Flag);
}

void BaseForm::toggleSingularities(){
  bool l_Flag=d_display->isItemChecked(SINGBUTTON);
  d_Fluid->setDisplaySingularities(!l_Flag);
  d_display->setItemChecked(SINGBUTTON,!l_Flag);
}

void BaseForm::toggleStreamlines(){
  bool l_Flag=d_display->isItemChecked(STREAMBUTTON);
  d_Fluid->setDisplayStreamlines(!l_Flag);
  d_display->setItemChecked(STREAMBUTTON,!l_Flag);
}

void BaseForm::toggleDataPoints(){
  bool l_Flag=d_display->isItemChecked(DATABUTTON);
  d_Fluid->setDisplayDataPoints(!l_Flag);
  d_display->setItemChecked(DATABUTTON,!l_Flag);
}

void BaseForm::toggleDataTable(){
  bool l_Flag=d_options->isItemChecked(DATATABLEBUTTON);
  if(l_Flag) 
    d_DataTable->hide();
  else
    d_DataTable->show();
  d_options->setItemChecked(DATATABLEBUTTON,!l_Flag);
}

void BaseForm::toggleDataGraph(){
  bool l_Flag=d_options->isItemChecked(DATAGRAPHBUTTON);
  if(l_Flag) 
    d_DataGraph->hide();
  else
    d_DataGraph->show();
  d_options->setItemChecked(DATAGRAPHBUTTON,!l_Flag);
}

void BaseForm::toggleSingTable(){
  bool l_Flag=d_options->isItemChecked(SINGTABLEBUTTON);
  if(l_Flag) 
    d_Table->hide();
  else
    d_Table->show();
  d_options->setItemChecked(SINGTABLEBUTTON,!l_Flag);
}

void BaseForm::setFluidColor(){
  QColor *a=new QColor();
  if (getColor(a)) d_Fluid->setFluidColor(a);
}

void BaseForm::setGridColor(){
  QColor *a=new QColor();
  if (getColor(a)) d_Fluid->setGridColor(a);
};

void BaseForm::setStreamlineColor(){
  QColor *a=new QColor();
  if (getColor(a)) d_Fluid->setStreamlineColor(a);
};

void BaseForm::setSingularityColor(){
  QColor *a=new QColor();
  if (getColor(a)) d_Fluid->setSingularityColor(a);
};

void BaseForm::setDataPointColor(){
  QColor *a=new QColor();
  if (getColor(a)) d_Fluid->setDataPointColor(a);
};



bool BaseForm::getColor(QColor *a)
{
	QColorPicker colorPicker(QApplication::desktop(),"colorPicker");
	if(colorPicker.result()) {
		// user pressed OK button
		*a = colorPicker.getSelectedColor();
		return(TRUE);
	}
	// user pressed CANCEL button
	return(FALSE);
}
//ENDPRETTY
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */
