#include "qwt_sclif.h"

//------------------------------------------------------------
//.H QwtScaleIf|3|02/04/98|Qwt Widget Library|Qwt Programmer's manual
//
//.U	NAME
//	QwtScaleIf - An interface class for widgets containing a scale 
//
//.U	SYNOPSIS
//	#include <qwt_sclif.h>
//
//.U	PUBLIC MEMBERS 
//.R
//	QwtScaleIf::setScale (1) --	Set user-defined scale
//	QwtScaleIf::setScale (2)  -- Set user-defined scale
//	QwtScaleIf::setScaleMaxMajor --
//				Set Max. number of major tick intervals
//	QwtScaleIf::setScaleMaxMinor --
//				Set max. number of minor tick intervals
//	QwtScaleIf::autoScale -- adjust scale automatically
//	QwtScaleIf::scaleMaxMinor -- return max. number of minor tick intervals
//      QwtScaleIf::scaleMaxMajor -- return max. number of major tick intervals
//
//.U	PROTECTED MEMBERS
//.R
//	 QwtScaleIf::hasUserScale -- Check for user-defined scale
//	 QwtScaleIf::scaleChange -- Notify change of scale
//
//.U	DESCRIPTION
//	This interface class is used to provide classes
//	with a protected QwtScaleDraw member and a public 
//	interface to access that scale.
//
//.U	INTERNALS
//	The primary purpose of this class is to define
//	a common interface for classes which are supposed to
//	contain a ScaleDraw class. It provides a protected 
//	ScaleDraw member
//      called d_scale and a couple of public member functions 
//      which allow direct but restricted access 
//	to this scale by the user. 
//	Widgets derived from this class have
//	to implement the member function scaleChange(),
//	which is called to notify changes of the
//	scale parameters and usually requires repainting or
//	resizing respectively.
//	In general, a class derived from QwtScaleIf is 
//      expected to manage the division and the position of its scale internally
//	when no user-defined scale is set. It should take the d_maxMinor
//	and d_maxMajor members into account, which can be set by the user.
//	An implementation can check if a user-defined scale is set by calling the
//	@QwtScaleIf::hasUserScale@ member.
//
//------------------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtScaleIf::QwtScaleIf
//		Construct a QwtScaleIf instance
//
//.u	Syntax
//.f	 QwtScaleIf::QwtScaleIf()
//
//------------------------------------------------------------
QwtScaleIf::QwtScaleIf()
{
    d_userScale = FALSE;
    d_maxMajor = 5;
    d_maxMinor = 3;
    d_scale.setScale(0.0,100.0,d_maxMajor, d_maxMinor);
}

//------------------------------------------------------------
//
//.F	QwtScaleIf::setScale (1)
//		Specify a user-defined scale.
//
//.u	Syntax
//.f	void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
//
//.u	Parameters
//.p	double vmin, double vmax	-- boundary values
//	int logarithmic	--	If != 0, Build a logarithmic scale
//
//.u	Description
//		By default, the widget is supposed to control the range of its scale
//		automatically,	but sometimes it is desirable to have a user-defined
//		scale which is not in sync with
//		the widget's range, e.g. if a logarithmic scale is needed
//		(sliders don't support that) or if the scale is required
//		to have a fixed range (say 0...100%), independent of the
//		widget's range. 
//
//.u	See also
//	@QwtScaleIf::autoScale@
//
//------------------------------------------------------------
void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
{
    setScale(vmin,vmax,0.0,logarithmic);
}


//------------------------------------------------------------
//
//.F	QwtScaleIf::setScale (2)
//		Specify a user-defined scale.
//
//.u	Syntax
//.f	void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
//
//.u	Parameters
//.p	double vmin, double vmax	-- interval boundaries
//	int step			-- major step size
//	int logarithmic			-- If != 0, build a logarithmic scale
//
//.u	Description
//		By default, the widget is supposed to control the range of its scale
//		automatically,	but sometimes it is desirable to have a user-defined
//		scale which is not in sync with
//		the widget's range, e.g. if a logarithmic scale is needed
//		(sliders don't support that) or if the scale is required
//		to have a fixed range (say 0...100%), independent of the
//		widget's range. 
//
//.u	See also
//	@QwtScaleIf::autoScale@
//
//------------------------------------------------------------
void QwtScaleIf::setScale(double vmin, double vmax, double step, int logarithmic)
{
    QwtScaleDiv oldscl(d_scale.scaleDiv());
    
    d_scale.setScale(vmin, vmax, d_maxMajor, d_maxMinor, step, logarithmic);
    d_userScale = TRUE;
    if (oldscl != d_scale.scaleDiv())
       scaleChange();
}


//------------------------------------------------------------
//
//.F	QwtScale::setScale
//	Assign a user-defined scale division
//
//.u	Syntax
//.f	void QwtScale::setScale(const QwtScaleDiv &s)
//
//.u	Parameters
//.p	const QwtScaleDiv &s -- scale division
//
//------------------------------------------------------------
inline void QwtScaleIf::setScale(const QwtScaleDiv &s)
{
    d_scale.setScale(s);
    scaleChange();
}




//------------------------------------------------------------
//
//.F	QwtScaleIf::autoScale
//		Advise the widget to control the scale range
//		internally.
//.u	Syntax
//.f	void QwtScaleIf::autoScale
//
//.u	Description
//		Autoscaling is on by default. 
//		
//.u	See also
//		QwtScaleIf::setScale@
//
//------------------------------------------------------------
void QwtScaleIf::autoScale()
{
	if (!d_userScale) 
	{
		d_userScale = FALSE;
		scaleChange();
	}
}

//------------------------------------------------------------
//
//.F	QwtScaleIf::setScaleMaxMajor
//		Set the maximum number of major tick intervals.
//
//.u	Syntax
//.f	void QwtScaleIf::setScaleMaxMajor(int ticks)
//
//.u	Parameters
//.p	int ticks		--		maximal number of major ticks.
//
//.u	Description
//		The scale's major ticks are calculated automatically such that
//		the number of major intervals does not exceed <ticks>.
//		The default value is 5.
//.u	See Also
//	@^QwtScaleDraw@
//
//------------------------------------------------------------
void QwtScaleIf::setScaleMaxMajor(int ticks)
{
   if (ticks != d_maxMajor)
   {
       d_maxMajor = ticks;
       d_scale.setScale(d_scale.scaleDiv().lBound(), d_scale.scaleDiv().hBound(),
			d_maxMajor, d_maxMinor, 0.0,d_scale.scaleDiv().logScale()); 
       scaleChange();
   }
}

//------------------------------------------------------------
//
//.F	QwtScaleIf::setScaleMaxMinor
//	Set the maximum number of minor tick intervals
//
//.u	Syntax
//.f	void QwtScaleIf::setScaleMaxMinor(int ticks)
//
//.u	Parameters
//.p	int ticks
//
//.u	Description
//		The scale's minor ticks are calculated automatically such that
//		the number of minor intervals does not exceed <ticks>.
//		The default value is 3.
//
//.u	See Also
//	@^QwtScaleDraw@
//------------------------------------------------------------
void QwtScaleIf::setScaleMaxMinor(int ticks)
{
    if ( ticks != d_maxMinor)
    {
	d_maxMinor = ticks;
	d_scale.setScale(d_scale.scaleDiv().lBound(), d_scale.scaleDiv().hBound(),
			 d_maxMajor, d_maxMinor, 0.0, d_scale.scaleDiv().logScale()); 
	scaleChange();
    }
}






