#include "qwt_plot.h"
#include "qwt_plot_dict.h"
#include "qwt_math.h"
#include "qwt_legend.h"

//------------------------------------------------------------
//.C Accessing and Manipulating Curves
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::closestCurve (1)
//	Find the curve which is closest to a
//	specified point in the plotting area.
//
//.u	Syntax
//.f	long QwtPlot::closestCurve(int xpos, int ypos, int &dist) const
//
//.u	Input Parameters
//.p	int xpos, int ypos	-- position in the plotting region
//
//.u	Output Parameters
//.p	int &dist	--	distance in points between (xpos, ypos) and the
//				closest data point
//				
//.u	Return Value
//	Key of the closest curve or 0 if no curve was found.
//
//------------------------------------------------------------
long QwtPlot::closestCurve(int xpos, int ypos, int &dist) const
{
    double x,y;
    int index;
    return closestCurve(xpos, ypos, dist, x,y, index);
}

//------------------------------------------------------------
//
//.F	QwtPlot::closestCurve (2)
//	Find the curve which is closest to a point in
//	the plotting area.
//	Determines the position and index of the closest data point.
//
//.u	Syntax
//.f	long QwtPlot::closestCurve(int xpos, int ypos, int &dist, double &xval, double &yval, int &index) const
//
//.u	Input Parameters
//.p	int xpos, int ypos  -- coordinates of a point in the plotting region
//
//.u	Output Parameters
//.p	double &xval, double &yval  -- values of the closest point in the curve's
//					data array
//	int &dist -- distance in points between (xpos, ypos) and the
//				closest data point
//	int &index -- index of the closest point in the curve's data array
//
//.u	Return Value
//	Key of the closest curve or 0 if no curve was found.
//------------------------------------------------------------
long QwtPlot::closestCurve(int xpos, int ypos, int &dist, double &xval, double &yval, int &index) const
{
    long rv = 0;
    QwtPlotCurve *c;
    QIntDictIterator<QwtPlotCurve> itc(*d_curves);

    double dmin = 1.0e10;
    double d, cx, cy;

    int i;
    
    itc.toFirst();
    while((c = itc.current()))
    {
	for (i=0; i<c->dataSize(); i++)
	{
	    cx = d_map[c->xAxis()].xTransform(c->x(i)) - double(xpos);
	    cy = d_map[c->yAxis()].xTransform(c->y(i)) - double(ypos);

	    d = qwtSqr(cx) + qwtSqr(cy);
	    if (d < dmin)
	    {
		dmin = d;
		rv = itc.currentKey();
		xval = c->x(i);
		yval = c->y(i);
		index = i;
	    }
	}
	
	++itc;
    }
    dist = int(sqrt(dmin));
    return rv;

}



//------------------------------------------------------------
//
//.F	QwtPlot::curveStyle
//	Return the style of the curve indexed by key
//
//.u	Syntax
//.f	QwtCurve::CurveStyle QwtPlot::curveStyle(long key) const
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//.u	See also
//	@QwtPlot::setCurveStyle@
//------------------------------------------------------------
QwtCurve::CurveStyle QwtPlot::curveStyle(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->style();
    }
    else
       return (QwtCurve::CurveStyle) 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveSymbol
//	Return the symbol of the curve indexed by key
//
//.u	Syntax
//.f	const QwtSymbol & QwtPlot::curveSymbol(long key) const
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//.u	Return Value
//		The symbol of the specified curve. If the key is invalid,
//		a symbol of type 'NoSymbol'.
//
//------------------------------------------------------------
const QwtSymbol & QwtPlot::curveSymbol(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->symbol();
    }
    else
       return DummySymbol;
}

//------------------------------------------------------------
//
//.F	QwtPlot::curvePen
//	Return the pen of the curve indexed by key
//
//.u	Syntax
//.f	const QPen & QwtPlot::curvePen(long key) const
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//------------------------------------------------------------
const QPen& QwtPlot::curvePen(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->pen();
    }
    else
       return DummyPen;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveOptions
//	Return the drawing options of the curve indexed by key
//
//.u	Syntax
//.f	int QwtPlot::curveOptions(long key)
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//------------------------------------------------------------
int QwtPlot::curveOptions(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->options();
    }
    else
       return 0;

}

//------------------------------------------------------------
//
//.F	QwtPlot::curveSplineSize
//	Return the spline size of the curve indexed by key
//
//.u	Syntax
//.f	int QwtPlot::curveSplineSize(long key)
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//------------------------------------------------------------
int QwtPlot::curveSplineSize(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->splineSize();
    }
    else
       return 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveTitle
//	Return the title of the curve indexed by key
//
//.u	Syntax
//.f	QString QwtPlot::curveTitle(long key) const
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//------------------------------------------------------------
QString QwtPlot::curveTitle(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->title();
    }
    else
       return QString("");
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveKeys
//	Return an array containing the keys of all curves
//
//.u	Syntax
//.f	QArray<long> QwtPlot::curveKeys() const
//
//.u	Return Value
//		Array containing the keys		
//------------------------------------------------------------
QArray<long> QwtPlot::curveKeys() const
{
    int i;
    QIntDictIterator<QwtPlotCurve> ic(*d_curves);
    QArray<long> rv(d_curves->count());

    ic.toFirst();
    i = 0;
    while (ic.current() && (uint(i) < rv.size()))
    {
	rv[i] = ic.currentKey();
	++ic;
	++i;
    }	
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveXAxis
//	Return the index of the x axis to which a curve is mapped
//
//.u	Syntax
//.f	int QwtPlot::curveXAxis(long key)
//
//.u	Parameters
//.p	long key	 -- Key of the curve
//
//.u	Return Value
//		x axis of the curve or -1 if the key is invalid.
//------------------------------------------------------------
int QwtPlot::curveXAxis(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->xAxis();
    }
    else
       return -1;    
}


//------------------------------------------------------------
//
//.F	QwtPlot::curveYAxis
//	Return the index of the y axis to which a curve is mapped
//
//.u	Syntax
//.f	int QwtPlot::curveYAxis(long key) const
//
//.u	Return Value
//		y axis of the curve or -1 if the key is invalid.
//------------------------------------------------------------
int QwtPlot::curveYAxis(long key) const
{
    QwtPlotCurve *c;

    if ((c = d_curves->find(key)))
    {
	return c->yAxis();
    }
    else
       return -1;    
}


//------------------------------------------------------------
//.-
//.F	QwtPlot::newCurveKey
//	Generate a unique key for a new curve
//
//.u	Syntax
//.f	long QwtPlot::newCurveKey()
//
//.u	Return Value
//		new unique key or 0 if no key could be found.
//
//------------------------------------------------------------
long QwtPlot::newCurveKey()
{
    long newkey = d_curves->count() + 1;

    if (newkey > 1)			// size > 0: check if key exists
    {
	if (d_curves->find(newkey))	// key size+1 exists => there must be a
					// free key <= size
	{
	    // find the first available key <= size
	    newkey = 1;
	    while (newkey <= long(d_curves->count()))
	    {
		if (d_curves->find(newkey))
		   newkey++;
		else
		   break;
	    }

	    // This can't happen. Just paranoia.
	    if (newkey > long(d_curves->count()))
	    {
		while (!d_curves->find(newkey))
		{
		    newkey++;
		    if (newkey > 10000) // prevent infinite loop
		    {
			newkey = 0;
			break;
		    }
		}
	    }
	}
    }
    return newkey;
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::insertCurve
//	Insert a new curve
//
//.u	Syntax
//.f	bool QwtPlot::insertCurve(long key, const char *title, int xAxis, int yAxis)
//
//
//.u	Parameters
//.p	long key  -- unique key
//	const char *title -- title of the curve. Default:0.
//	int xAxis	-- x axis to be attached. Default: xBottom.
//	int yAxis	-- y axis to be attached. Default: yLeft.
//
//.u	Return Value
//		TRUE if the curve has been successfully inserted,
//		FALSE if the specified key already exists or if
//		the memory allocation failed.
//------------------------------------------------------------
bool QwtPlot::insertCurve(long key, const char *title, int xAxis, int yAxis)
{
    bool rv = FALSE;
    
    if (d_curves->find(key) == 0)
    {
	QwtPlotCurve *nc = new QwtPlotCurve(this);
	if (nc) 
	{
	    nc->setXAxis(verifyXAxis(xAxis));
	    nc->setYAxis(verifyYAxis(yAxis));
	    nc->setTitle(title);
	    d_curves->insert(key, nc);
	    if (d_legendEnabled) 
	    {
	    	d_legend->appendItem(nc->title(), nc->symbol(),
				    nc->pen(), key);
	    }
	    autoRefresh();
	    rv = TRUE;
	}
    }
    return rv;
}


//------------------------------------------------------------
//
//.F	QwtPlot::insertCurve
//	Insert a new curve and return a unique key
//
//.u	Syntax
//.f	long QwtPlot::insertCurve(const char *title, int xAxis = xBottom, int yAxis = yLeft)
//
//.u	Parameters
//.p	const char *title	--	title of the new curve
//	int xAxis		--	x axis to be attached. Defaults to xBottom.
//	int yAxis		--	y axis to be attached. Defaults to yLeft.
//
//.u	Return Value
//	The key of the new curve or 0 if no new key could be found
//	or if no new curve could be allocated.
//
//------------------------------------------------------------
long QwtPlot::insertCurve(const char *title, int xAxis, int yAxis)
{

    long newkey = newCurveKey();

    if (newkey > 0)
    {
	insertCurve(newkey, title, xAxis, yAxis);
    }
    return newkey;
}

//------------------------------------------------------------
//
//.F	QwtPlot::removeCurve
//	remove the curve indexed by key
//
//.u	Syntax
//.f	void QwtPlot::removeCurve(long key)
//
//.u	Parameters
//.p	long key	-- Key of the curve
//
//------------------------------------------------------------
bool QwtPlot::removeCurve(long key)
{
    if (d_curves->remove(key))
    {
	if (d_legendEnabled)
	{
	    d_legend->removeItem(d_legend->findFirstKey(key));
	}
	
	autoRefresh();
	return TRUE;
    }
    else
       return FALSE;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurvePen
//	Assign a pen to a curve indexed by key
//
//.u	Syntax
//.f	bool QwtPlot::setCurvePen(long key, const QPen &pen)
//
//.u	Parameters
//.p	long key	-- Key of the curve
//	QPen &pen	-- new pen
//
//.u	Return Value
//	True if the curve exists
//
//------------------------------------------------------------
bool QwtPlot::setCurvePen(long key, const QPen &pen)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setPen(pen);
	if (d_legendEnabled)
	{
	    d_legend->setPen(d_legend->findFirstKey(key), pen);
	}
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveSymbol
//	Assign a symbol to a curve indexed by key
//
//.u	Parameters
//.p	long key, const QwtSymbol &s
//
//.u	Syntax
//.f	bool QwtPlot::setCurveSymbol(long key, const QwtSymbol &s)
//
//.u	Return Value
//	True if the curve exists
//
//------------------------------------------------------------
bool QwtPlot::setCurveSymbol(long key, const QwtSymbol &s)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setSymbol(s);
	if (d_legendEnabled)
	{
	    d_legend->setSymbol(d_legend->findFirstKey(key),s);
	}
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveRawData
//	Attach a curve to the specified arrays without copying
//
//.u	Syntax
//.f	bool QwtPlot::setCurveRawData(long key, double *xdat, double *ydat, int size)
//
//.u	Parameters
//.p	long key	-- Key of the curve
//      double *xdat, double *ydat -- Pointers to x and y data
//	int size	-- size of ydat and ydat
//
//.u	Return Value
//	True if the curve exists
//
//.u	Description
//	This function does not copy the values of the specified arrays.
//	Removing the curve will not delete the data.
//
//.u	Note
//	Don't delete dynamically allocated data while a curve
//	is attached to them.
//
//.u	See also
//     @QwtPlot::setCurveData@, @^QwtCurve#QwtCurve::setRawData@
//------------------------------------------------------------
bool QwtPlot::setCurveRawData(long key, double *xdat, double *ydat, int size)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setRawData(xdat, ydat, size);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveTitle
//
//
//.u	Syntax
//.f	bool QwtPlot::setCurveTitle(long key, const QString &s)
//
//.u	Parameters
//.p	long key, const char *s
//
//.u	Return Value
//
//.u	Description
//
//------------------------------------------------------------
bool QwtPlot::setCurveTitle(long key, const char *s)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setTitle(s);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveData
//	Assign x and y values to a curve.
//
//.u	Syntax
//.f	bool QwtPlot::setCurveData(long key, double *xdat, double *ydat, int size)
//
//.u	Parameters
//.p	long key	--	curve key
//	double *xdat	--	pointer to x values
//	double *ydat	--	pointer to y values
//	int size	--	size of xdat and ydat
//
//.u	Description
//	In opposite to @QwtPlot::setCurveRawData@, this function makes
//	a 'deep copy' of the data.
//
//.u	See also
//     @QwtPlot::setCurveRawData@, @^QwtCurve#QwtCurve::setData@
//------------------------------------------------------------
bool QwtPlot::setCurveData(long key, double *xdat, double *ydat, int size)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setData(xdat, ydat, size);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveStyle
//	Change a curve's style
//
//.u Syntax
//.f	bool QwtPlot::setCurveStyle(long key, QwtCurve::CurveStyle s, int options)
//
//.u Parameters
//.p	long key	-- Key of the curve
//	QwtCurve::CurveStyle s -- display style of the curve
//	int options	-- style options
//
//.u Return Value
//	TRUE if the curve exists
//
//.u See also:
//	@QwtCurve#QwtCurve::setStyle@ for a detailed description of
//	valid styles.
//------------------------------------------------------------
bool QwtPlot::setCurveStyle(long key, QwtCurve::CurveStyle s, int options)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setStyle(s, options);
	rv = TRUE;
    }
  
    return rv;
  
}

//------------------------------------------------------------
//
//.F QwtPlot::setCurveOptions
//	Set the style options of a curve indexed by key
//
//.u Syntax
//.f	bool QwtPlot::setCurveOptions(long key, int t)
//.u Parameters
//.p	long key	--	The curve's key
//	int opt	--	The type of the spline
//
//.u Return Value
//	TRUE if the specified curve exists.
//.u See also
//	@^QwtCurve#QwtCurve::setOptions@for a detailed description of
//	valid options.
//------------------------------------------------------------
bool QwtPlot::setCurveOptions(long key, int opt)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setOptions(opt);
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveSplineSize
//	Set the number of interpolated points of a curve indexed by key
//
//.u	Syntax
//.f	bool QwtPlot::setCurveSplineSize(long key, int s)
//
//.u	Parameters
//.p	long key    --	key of the curve
//	int s       --	size of the spline
//
//.u	Return Value
//	True if the curve exists
//
//------------------------------------------------------------
bool QwtPlot::setCurveSplineSize(long key, int s)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setSplineSize(s);
	rv = TRUE;
    }
    return rv;
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::setCurveXAxis
//	Attach a curve to an x axis
//
//.u	Syntax
//.f	bool QwtPlot::setCurveXAxis(long key, int axis)
//
//.u	Parameters
//.p	long key	-- key of the curve
//	int axis	-- x axis to be attached
//
//------------------------------------------------------------
bool QwtPlot::setCurveXAxis(long key, int axis)
{
    bool rv = FALSE;

    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setXAxis(verifyXAxis(axis));
	rv = TRUE;
    }

    return rv;
    
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setCurveYAxis
//	Attach a curve to an y axis
//
//.u	Syntax
//.f	bool QwtPlot::setCurveYAxis(long key, int axis)
//
//.u	Parameters
//.p	long key	-- key of the curve
//	int axis	-- y axis to be attached
//
//------------------------------------------------------------
bool QwtPlot::setCurveYAxis(long key, int axis)
{
    bool rv = FALSE;

    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setYAxis(verifyYAxis(axis));
	rv = TRUE;
    }

    return rv;
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::setCurveBaseline
//	Set the baseline for a specified curve
//
//.u	Syntax
//.f	void QwtPlot::setCurveBaseline(long key, double ref)
//
//.u	Parameters
//.p	long key  -- curve key
//	double ref -- baseline offset from zero
//
//.u	Return Value
//
//.u	Description
//	The baseline is needed for the curve style QwtCurve::Sticks,
//	cf. @^QwtCurve#QwtCurve::setBaseline@
//------------------------------------------------------------
bool QwtPlot::setCurveBaseline(long key, double ref)
{
    bool rv = FALSE;
    
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
    {
	c->setBaseline(ref);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::curveBaseline
//	Return the baseline offset for a specified curve
//
//.u	Syntax
//.f	double QwtPlot::curveBaseline(long key) const
//
//.u	Parameters
//.p	long key -- curve key
//
//.u    Return value
//	Baseline offset of the specified curve, or 0 if the
//	curve doesn't exist
//
//.u	See also
//	@QwtPlot::setCurveBaseline@
//
//------------------------------------------------------------
double QwtPlot::curveBaseline(long key) const
{
    double rv = 0.0;
    QwtPlotCurve *c;
    if ((c = d_curves->find(key)))
	rv = c->baseline();
    return rv;
    
    
}





