#include "qwt_grid.h"
//------------------------------------------------------------
//
//.H QwtGrid | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//
//.U NAME
//	QwtGrid - A class which draws a coordinate grid
//
//.U SYNOPSIS
//	#include <qwt_grid.h>
//
//.U DESCRIPTION
//	The QwtGrid class can be used to draw a coordinate grid.
//	A coordinate grid consists of major and minor vertical
//	and horizontal gridlines. The locations of the gridlines
//	are determined by the X and Y scale divisions which can
//	be assigned with @QwtGrid::setXDiv@ and @QwtGrid::setYDiv@
//	The @QwtGrid::draw@ member draws the grid within a bounding
//	rectangle.
//
//.U PUBLIC MEMBERS 
//.R
//	QwtGrid::QwtGrid (1) --	Constructor
//	QwtGrid::QwtGrid (2) -- Constructor
//	QwtGrid::~QwtGrid -- Destructor
//	QwtGrid::draw -- Draw the grid 
//	QwtGrid::enableX -- Enable vertical gridlines
//	QwtGrid::enableY -- Enable horizontal gridlines
//	QwtGrid::enableXMin -- Enable minor vertical gridlines
//	QwtGrid::enableYMin -- Enable minor horizontal gridlines
//	QwtGrid::setXDiv -- Assign a scale division for vertical gridlines 
//	QwtGrid::setYDiv -- Assign a scale division for horizontal gridlines
//	QwtGrid::setPen -- Set a pen for major and minor gridlines
//	QwtGrid::setMajPen -- Set a pen for major gridlines
//	QwtGrid::setMinPen -- Set a pen for minor gridlines
//	QwtGrid::majPen -- Return pen for major gridlines
//	QwtGrid::minPen -- Return pen for minor gridlines
//	QwtGrid::xEnabled -- Return TRUE if vertical gridlines enabled
//	QwtGrid::xMinEnabled -- Return TRUE if minor vertical gridlines enabled
//	QwtGrid::yEnabled -- Return TRUE if horizontal gridlines enabled
//	QwtGrid::yMinEnabled -- Return TRUE if vertical gridlines enabled
//	QwtGrid::xScaleDiv -- Return x scale division
//	QwtGrid::yScaleDiv -- Return y scale division
//
//.U PROTECTED MEMBERS 
//.R
//	QwtGrid::gridChanged -- Called when an attribute has changed
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtGrid::QwtGrid
//	Construct a QwtGrid instance
//
//.u	Syntax
//.f	 QwtGrid::QwtGrid()
//
//------------------------------------------------------------
QwtGrid::QwtGrid()
{
    d_xMinEnabled = FALSE;
    d_yMinEnabled = FALSE;
    d_xEnabled = TRUE;
    d_yEnabled = TRUE;
}

//------------------------------------------------------------
//
//.F	QwtGrid::~QwtGrid
//	Destroy a QwtGrid instance
//
//.u	Syntax
//.f	 QwtGrid::~QwtGrid()
//
//------------------------------------------------------------
QwtGrid::~QwtGrid()
{
    
}

//------------------------------------------------------------
//
//.F	QwtGrid::enableX
//	Enable or disable vertical gridlines
//
//.u	Syntax
//.f	void QwtGrid::enableX(bool tf)
//
//.u	Parameters
//.p	bool tf -- TRUE (enable) or FALSE (disable)
//
//.u    See also
//	Minor gridlines can be enabled or disabled using
//	@QwtGrid::enableXMin@ 
//------------------------------------------------------------
void QwtGrid::enableX(bool tf)
{
    d_xEnabled = tf;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::enableY
//	Enable or disable horizontal gridlines
//
//.u	Syntax
//.f	void QwtGrid::enableY(bool tf)
//
//.u	Parameters
//.p	bool tf -- Enable (TRUE) or disable (FALSE)
//
//.u    See also
//	Minor gridlines can be enabled or disabled using
//	@QwtGrid::enableYMin@ 
//------------------------------------------------------------
void QwtGrid::enableY(bool tf)
{
    d_yEnabled = tf;  
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::enableXMin
//      Enable or disable  minor vertical gridlines.
//
//.u	Syntax
//.f	void QwtGrid::enableXMin(bool tf)
//
//.u	Parameters
//.p	bool tf -- Enable (TRUE) or disable (FALSE)
//
//.u    See also
//	@QwtGrid::enableX@
//
//------------------------------------------------------------
void QwtGrid::enableXMin(bool tf)
{
    d_xMinEnabled = tf;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::enableYMin
//	Enable or disable minor horizontal gridlines
//
//.u	Syntax
//.f	void QwtGrid::enableYMin(bool tf)
//
//.u	Parameters
//.p	bool tf -- Enable (TRUE) or disable (FALSE)
//
//.u    See also
//	@QwtGrid::enableY@
//------------------------------------------------------------
void QwtGrid::enableYMin(bool tf)
{
    d_yMinEnabled = tf;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::setXDiv
//	Assign an x axis scale division
//
//.u	Syntax
//.f	void QwtGrid::setXDiv(const QwtScaleDiv &sx)
//
//.u	Parameters
//.p	const QwtScaleDiv &sx -- scale division
//
//.u	Note
//	QwtGrid uses implicit sharing (see Qt Manual) for
//	the scale divisions.
//
//------------------------------------------------------------
void QwtGrid::setXDiv(const QwtScaleDiv &sx)
{
    d_sdx = sx;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::setYDiv
//	Assign a y axis division
//
//.u	Syntax
//.f	void QwtGrid::setYDiv(const QwtScaleDiv &sy)
//
//.u	Parameters
//.p	const QwtScaleDiv &sy -- scale division
//
//.u	Note
//	QwtGrid uses implicit sharing (see Qt Manual) for
//	the scale divisions.
//
//------------------------------------------------------------
void QwtGrid::setYDiv(const QwtScaleDiv &sy)
{
    d_sdy = sy;    
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::setPen
//	Assign a pen for both major and minor gridlines
//
//.u	Parameters
//.p	const QPen &p -- pen
//
//.u	Syntax
//.f	void QwtGrid::setPen(const QPen &p)
//
//------------------------------------------------------------
void QwtGrid::setPen(const QPen &p)
{
    d_majPen = p;
    d_minPen = p;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::setMajPen
//	Assign a pen for the major gridlines
//
//.u	Syntax
//.f	void QwtGrid::setMajPen(const QPen &p)
//
//------------------------------------------------------------
void QwtGrid::setMajPen(const QPen &p)
{
    d_majPen = p;
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid:.setMinPen
//	Assign a pen for the minor gridlines
//
//.u	Syntax
//.f	void QwtGrid:.setMinPen(const QPen &p)
//
//------------------------------------------------------------
void QwtGrid::setMinPen(const QPen &p)
{
    d_minPen = p;  
    gridChanged();
}

//------------------------------------------------------------
//
//.F	QwtGrid::draw
//	Draw the grid
//
//.u	Syntax
//.f	void QwtGrid::draw(QPainter *p, const QRect &r, const QwtDiMap &mx, const QwtDiMap &my)
//
//.u	Parameters
//.p	QPainter *p -- Qt painter
//	const QRect &r -- bounding rectangle
//	const QwtDiMap &mx -- x axis map
//	const QwtDiMap &my -- y axis map
//
//.u	Description
//	The grid is drawn into the bounding rectangle such that the
//	gridlines begin and end at the rectangle's borders. The X and Y
//	maps are used to map the scale divisions into the drawing region
//	screen.
//
//------------------------------------------------------------
void QwtGrid::draw(QPainter *p, const QRect &r, const QwtDiMap &mx, const QwtDiMap &my)
{

    int i, x = 0, y = 0;
    int x1, x2, y1, y2;
    x1 = r.left();
    x2 = r.right();
    y1 = r.top();
    y2 = r.bottom();

    //
    //  draw minor gridlines
    //
    p->setPen(d_minPen);
    
    if (d_xEnabled && d_xMinEnabled)
    {
	for (i=0;i<d_sdx.minCnt();i++)
	{
	    x = mx.transform(d_sdx.minMark(i));
	    if ((x >= x1) && (x <= x2))
	       p->drawLine(x, y1, x, y2);
	}
	
    }

    if (d_yEnabled && d_yMinEnabled)
    {
	for (i=0;i<d_sdy.minCnt();i++)
	{
	    y = my.transform(d_sdy.minMark(i));
	    if ((y >= y1) && (y <= y2))
	       p->drawLine(x1, y, x2, y);
	}
	
    }

    //
    //  draw major gridlines
    //
    p->setPen(d_majPen);
    
    if (d_xEnabled)
    {
	for (i=0;i<d_sdx.majCnt();i++)
	{
	    x = mx.transform(d_sdx.majMark(i));
	    if ((x >= x1) && (x <= x2))
	       p->drawLine(x, y1, x, y2);
	}
	
    }
    
    if (d_yEnabled)
    {
	for (i=0;i<d_sdy.majCnt();i++)
	{
	    y = my.transform(d_sdy.majMark(i));
	    if ((y >= y1) && (y <= y2))
	       p->drawLine(x1, y, x2, y);
	}
    }
    
}

//------------------------------------------------------------
//
//.F	QwtGrid::operator=
//	Assignment operator
//
//.u	Syntax
//.f	const QwtGrid & QwtGrid::operator=(const QwtGrid &g)
//
//.u	Parameters
//.p	const QwtGrid &g -- Grid to be assigned
//
//------------------------------------------------------------
const QwtGrid & QwtGrid::operator=(const QwtGrid &g)
{
    d_xEnabled = g.d_xEnabled;
    d_yEnabled = g.d_yEnabled;
    d_xMinEnabled = g.d_xMinEnabled;
    d_yMinEnabled = g.d_yMinEnabled;
    d_sdx = g.d_sdx;
    d_sdy = g.d_sdy;

    d_majPen = g.d_majPen;
    d_minPen = g.d_minPen;
    
    return *this;
}


//------------------------------------------------------------
//
//.F	QwtGrid::majPen
//	Return the pen for the major gridlines
//
//.u	Syntax
//.f	const QPen & QwtGrid::majPen()
//
//.u	See Also
//	@QwtGrid::setMinPen@, @QwtGrid:.setPen@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::minPen
//	Return the pen for the minor gridlines
//
//.u	Syntax
//.f	const QPen & QwtGrid::minPen()
//
//.u	See Also
//	@QwtGrid::setMinPen@, @QwtGrid:.setPen@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::xEnabled
//	Return TRUE if vertical gridlines are enabled
//
//
//.u	Syntax
//.f	bool QwtGrid::xEnabled()
//
//.u    See Also
//	@QwtGrid::enableY@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::xMinEnabled
//	Return TRUE if minor vertical gridlines are enabled
//
//.u	Syntax
//.f	bool QwtGrid::xMinEnabled()
//
//.u	See Also
//	@QwtGrid::enableXMin@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::yEnabled
//	Return TRUE if horizontal gridlines are enabled
//
//.u	Syntax
//.f	bool QwtGrid::yEnabled()
//
//.u    See Also
//	@QwtGrid::enableY@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::yMinEnabled
//	Return TRUE if minor horizontal gridlines are enabled
//
//.u	Syntax
//.f	bool QwtGrid::yMinEnabled()
//
//.u	See Also
//	@QwtGrid::enableYMin@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::xScaleDiv
//	Return the scale division of the x axis
//
//.u	Syntax
//.f	const QwtScaleDiv & QwtGrid::xScaleDiv()
//
//.u	Description
//.u    See Also
//	@^QwtScaleDiv@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::yScaleDiv
//	Return the scale division of the y axis
//
//.u	Syntax
//.f	const QwtScaleDiv & QwtGrid::yScaleDiv()
//
//.u    See Also
//	@^QwtScaleDiv@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtGrid::gridChanged
//	Notify a change of attributes
//
//.u	Syntax
//.f	void QwtGrid::gridChanged()
//
//.u	Access
//	protected virtual
//
//.u	Description
//	This virtual function is called when an attribute of the grid
//	has changed. It can be redefined by derived classes. The
//	default implementation does nothing.
//
//------------------------------------------------------------











