#include <qwt_drange.h>
#include <qwt_math.h>

const double QwtDblRange::MinRelStep = 1.0e-10;
const double QwtDblRange::DefaultRelStep = 1.0e-2;
const double QwtDblRange::MinEps = 1.0e-10;

//------------------------------------------------------------
//.H QwtDblRange | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//------------------------------------------------------------

//-----------------------------------------------------------
//
//.U NAME
//	QwtDblRange - A class which controls a value of type double within a
//		      closed interval
//.U SYNOPSIS 
//	#include <qwt_drange.h>
//
//.U DESCRIPTION
//	This class is useful as a base class or a member for sliders.
//	It represents an interval of type double within which a value can
//	be moved. The value can be either an arbitrary point inside 
//	the interval (see @QwtDblRange::setValue@), or it can be fitted
//	into a step raster (see @QwtDblRange::fitValue@ and
//	@QwtDblRange::incValue@).
//.P
//	As a special case, a QwtDblRange can be periodic, which means that
//	a value outside the interval will be mapped to a value inside the
//	interval when @QwtDblRange::setValue@, @QwtDblRange::fitValue@,
//	@QwtDblRange::incValue@ or @QwtDblRange::incPages@ are called. 
//
//
//.U DERIVED CLASSES
//	@^QwtSliderBase@,
//	@^QwtCounter@
//
//.U PUBLIC MEMBERS
//.R
//	QwtDblRange::QwtDblRange -- Constructor
//	QwtDblRange::fitValue -- Set value with raster fitting
//	QwtDblRange::incValue -- Increments the value
//	QwtDblRange::incPages -- Increments the value pagewise
//	QwtDblRange::setPeriodic	--   Make the range periodic
//	QwtDblRange::setRange -- Specify range and step size
//	QwtDblRange::setStep --	Change step size
//	QwtDblRange::setValue -- Set value, no raster fitting
//	QwtDblRange::maxValue -- Returns the upper boundary
//	QwtDblRange::minValue -- Returns the lower boundary
//	QwtDblRange::periodic -- Returns the TRUE if periodic
//	QwtDblRange::step -- Returns the step size
//	QwtDblRange::pageSize -- Returns the page size
//	QwtDblRange::value -- Returns the value
//
//.U PROTECTED MEMBERS
//.R
//    QwtDblRange::exactValue -- Returns the exact value
//    QwtDblRange::exactPrevValue -- Returns the exact previous value
//    QwtDblRange::prevValue -- Returns the previous value
//    QwtDblRange::valueChange -- Called when the value has changed
//    QwtDblRange::stepChange -- Called when the step size has changed
//    QwtDblRange::rangeChange -- Called when the range has changed
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//.+ 	 
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDblRange::QwtDblRange
//	Construct a QwtDblRange instance  
//
//.u	Syntax
//.f	 QwtDblRange::QwtDblRange()
//
//.u	Description
//		The range is initialized to [0.0, 100.0], the
//		step size to 1.0, and the value to 0.0.
//------------------------------------------------------------
QwtDblRange::QwtDblRange()
{
    //
    // set some reasonable values
    //
    d_minValue = 0;
    d_maxValue = 100.0;
    d_prevValue = d_exactPrevValue = d_exactValue = d_value = 0.0;
    d_step = 1.0;
    d_periodic = FALSE;

}

//------------------------------------------------------------
//.-
//.F	QwtDblRange::setNewValue
//	For internal use only.
//
//.u	Syntax
//.f	void QwtDblRange::setNewValue((double x, int align)
//
//.u	Parameters
//.p	double x, int align
//
//------------------------------------------------------------
void QwtDblRange::setNewValue(double x, int align)
{
    double vmin,vmax;
    
    d_prevValue = d_value;

    vmin = qwtMin(d_minValue, d_maxValue);
    vmax = qwtMax(d_minValue, d_maxValue);

    // 
    // Range check
    //
    if (x < vmin)
    {
	if ((d_periodic) && (vmin != vmax))
	   d_value = x + ceil( (vmin - x) / (vmax - vmin ) ) 
	      * (vmax - vmin);
	else
	   d_value = vmin;
    }
    else if (x > vmax)
    {
	if ((d_periodic) && (vmin != vmax))
	   d_value = x - ceil( ( x - vmax) / (vmax - vmin )) 
	      * (vmax - vmin);
	else
	   d_value = vmax;
    }
    else
       d_value = x;

    d_exactPrevValue = d_exactValue;
    d_exactValue = d_value;
    
    // align to grid
    if (align)
    {
	if (d_step != 0.0)
	   d_value = d_minValue + rint((d_value - d_minValue) / d_step ) * d_step;
	else
	   d_value = d_minValue;
	
	// correct rounding error at the border
	if (fabs(d_value - d_maxValue) < MinEps * qwtAbs(d_step))
	   d_value = d_maxValue;

	// correct rounding error if value = 0
	if (fabs(d_value) < MinEps * qwtAbs(d_step))
	   d_value = 0.0;
    }

    if (d_prevValue != d_value)
       valueChange();

}

//------------------------------------------------------------
//
//.F	QwtDblRange::fitValue
//	    Adjust the value to the closest point in the step
//	    raster.
//
//.u	Syntax
//.f	   void QwtDblRange::fitValue(double x)
//
//.u	Parameters
//.p	double x -- new value
//
//.u	Note
//	   The value is clipped when it lies outside the range.
//	   When the range is @QwtDblRange::periodic@, it will
//	   be mapped to a point in the interval such that
//.P
//		new value := x + n * (max. value - min. value)
//.P
//	  with an integer number n.
//------------------------------------------------------------
void QwtDblRange::fitValue(double x)
{
    setNewValue(x,1);
}


//------------------------------------------------------------
//
//.F	QwtDblRange::setValue
//	Set a new value without adjusting to the step raster
//
//.u	Syntax
//.f	void QwtDblRange::setValue(double x)
//
//.u	Parameters
//.p	double x -- new value
//
//.u	Note
//	   The value is clipped when it lies outside the range.
//	   When the range is @QwtDblRange::periodic@, it will
//	   be mapped to a point in the interval such that
//.P
//		new value := x + n * (max. value - min. value)
//.P
//	  with an integer number n.
//
//------------------------------------------------------------
void QwtDblRange::setValue(double x)
{
    setNewValue(x,0);
}

//------------------------------------------------------------
//
//.F	QwtDblRange::setRange
//	Specify  range and step size
//
//.u	Syntax
//.f	void QwtDblRange::setRange(double vmin, double vmax, double vstep = 0, double pageSize = 0)
//
//.u	Parameters
//.p	double vmin --	lower boundary of the interval
//	double vmax --  higher boundary of the interval
//	double vstep -- step width
//	int pageSize -- page size in steps
//
//.u	Note
//.i
//	-- A change of the range changes the value if it lies outside the
//	   new range. The current value
//	   will *not* be adjusted to the new step raster.
//	-- vmax < vmin is allowed.
//	-- If the step size is left out or set to zero, it will be
//	   set to 1/100 of the interval length.
//	-- If the step size has an absurd value, it will be corrected
//	   to a better one.
//------------------------------------------------------------
void QwtDblRange::setRange(double vmin, double vmax, double vstep, int pageSize)
{
    
    int rchg;
    
    rchg = ((d_maxValue != vmax) || (d_minValue != vmin));
    
    if (rchg) {
	    d_minValue = vmin;
	    d_maxValue = vmax;
	}
    
    //
    // look if the step width has an acceptable 
    // value or otherwise change it.
    //
    setStep(vstep);

    //
    // limit page size
    //
    d_pageSize = qwtLim(pageSize,0, int(qwtAbs((d_maxValue - d_minValue) / d_step))); 
    
    // 
    // If the value lies out of the range, it 
    // will be changed. Note that it will not be adjusted to 
    // the new step width.
    setNewValue(d_value,0);
    
    // call notifier after the step width has been 
    // adjusted.
    if (rchg)
       rangeChange();
    
}

//------------------------------------------------------------
//
//.F	QwtDblRange::setStep
//	Change the step raster 	
//
//.u	Syntax
//.f	void QwtDblRange::setStep(double vstep)
//
//.u	Parameters
//.p	double vstep -- new step width
//
//.u	Note
//	The value will *not* be adjusted to the new step raster.
//
//------------------------------------------------------------
void QwtDblRange::setStep(double vstep)
{
    double newStep,intv;

    intv = d_maxValue - d_minValue;
    
    if (vstep == 0.0)
       newStep = intv * DefaultRelStep;
    else
    {
	if ((intv > 0) && (vstep < 0) || (intv < 0) && (vstep > 0))
	   newStep = -vstep;
	else
	   newStep = vstep;
	
	if ( fabs(newStep) < fabs(MinRelStep * intv) )
	   newStep = MinRelStep * intv;
    }
    
    if (newStep != d_step)
    {
	d_step = newStep;
	stepChange();
    }

}


//------------------------------------------------------------
//
//.F	QwtDblRange::setPeriodic
//	Make the range periodic
//
//.u	Syntax
//.f	void QwtDblRange::setPeriodic(bool tf)
//
//.u	Parameters
//.p	bool tf
//
//.u	Description
//	When the range is periodic, the value will be set to a point
//	inside the interval such that
//.P
//		point = value + n * width
//.P
//	if the user tries to set a new value which is outside the range.
//	If the range is nonperiodic (the default), values outside the
//	range will be clipped.
//
//------------------------------------------------------------
void QwtDblRange::setPeriodic(bool tf)
{
    d_periodic = tf;
}

//------------------------------------------------------------
//
//.F	QwtDblRange::incValue
//	Increment the value by a specified number of steps
//
//.u	Syntax
//.f	void QwtDblRange::incValue(int nSteps)
//
//.u	Parameters
//.p	int nSteps -- Number of steps to increment
//
//.u	Note
//.i	-- As a result of this operation, the new value will always be
//	   adjusted to the step raster.
//
//------------------------------------------------------------
void QwtDblRange::incValue(int nSteps)
{
    setNewValue(d_value + double(nSteps) * d_step,1);
}

//------------------------------------------------------------
//
//.F	QwtDblRange::incPages
//	Increment the value by a specified number of pages
//
//.u	Syntax
//.f	void QwtDblRange::incPages(int nPages)
//
//.u	Parameters
//.p	int nPages	--	Number of pages to increment.
//				A negative number decrements the value.
//.u	Note
//	The Page size is specified in the constructor.
//
//------------------------------------------------------------
void QwtDblRange::incPages(int nPages)
{
    setNewValue(d_value + double(nPages) * double(d_pageSize) * d_step,1);
}

//------------------------------------------------------------
//
//.F	QwtDblRange::valueChange
//	Notify a change of value
//.u	Syntax
//.f	void QwtDblRange::valueChange()
//
//.u	Access
//	protected virtual
//.u	Description
//	This virtual function is called whenever the value changes.
//	The default implementation does nothing.
//
//------------------------------------------------------------
void QwtDblRange::valueChange()
{
}


//------------------------------------------------------------
//
//.F	QwtDblRange::rangeChange
//	Notify a change of the range
//.u	Syntax
//.f	void QwtDblRange::rangeChange()
//
//.u	Access
//	protected virtual
//.u	Description
//	This virtual function is called whenever the range changes.
//	The default implementation does nothing.
//
//------------------------------------------------------------
void QwtDblRange::rangeChange()
{
}


//------------------------------------------------------------
//
//.F	QwtDblRange::stepChange
//	Notify a change of the step size
//.u	Syntax
//.f	void QwtDblRange::stepChange()
//
//.u	Access
//	protected virtual
//.u	Description
//	This virtual function is called whenever the step size changes.
//	The default implementation does nothing.
//
//------------------------------------------------------------
void QwtDblRange::stepChange()
{
}

//------------------------------------------------------------
//
//.F	QwtDblRange::value
//	Returns the current value
//
//.u	Syntax
//.f	double QwtDblRange::value()
//
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtDblRange::pageSize
//	Returns the page size in steps.
//
//.u	Syntax
//.f	int QwtDblRange::pageSize() const
//
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtDblRange::exactValue
//	  Returns the exact value
//.u	Syntax
//.f	double QwtDblRange::exactValue()
//
//.u	Access
//		protected
//.u	Description
//	The exact value is the value which @QwtDblRange::value@ would return
//	if the value were not adjusted to the step raster. It differs from
//	the current value only if @QwtDblRange::fitValue@ or
//	@QwtDblRange::incValue@ have been used before. This function
//	is intended for internal use in derived classes.
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDblRange::prevValue
//	Returns the previous value
//.u	Access
//	protected
//.u	Syntax
//.f	double QwtDblRange::prevValue()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDblRange::exactPrevValue
//	Returns the exact previous value
//.u	Syntax
//.f	double QwtDblRange::exactPrevValue()
//
//.u	Access
//	protected
//.u	Description
//		See @QwtDblRange::exactValue@
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtDblRange::minValue
//	Returns the value at the first border of the range
//
//.u	Syntax
//.f	double QwtDblRange::minValue()
//
//.u	Description
//	  minValue returns the value which has been specified
//	  as the first parameter in  @QwtDblRange::setRange@.
//.u	See also
//	@QwtDblRange::setRange@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDblRange::maxValue
//	Returns the value of the second border of the range
//
//.u	Syntax
//.f	double QwtDblRange::maxValue()
//
//.u	Description
//	  maxValue returns the value which has been specified
//	  as the second parameter in  @QwtDblRange::setRange@.
//
//.u	See also
//	@QwtDblRange::setRange@
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtDblRange::periodic
//	Returns TRUE if the range is periodic
//
//.u	Syntax
//.f	bool QwtDblRange::periodic()
//
//.u	See also
//	@QwtDblRange::setPeriodic@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDblRange::step
//	Returns the step size
//
//.u	Syntax
//.f	double QwtDblRange::step()
//
//.u	See also
//	@QwtDblRange::setStep@, @QwtDblRange::setRange@
//------------------------------------------------------------
double QwtDblRange::step() const
{
    return qwtAbs(d_step);
}










