#include "qwt_counter.h"
#include <qwt_math.h>
#include "qwt_arrbtn.h"

#include <qlined.h>
#include <qlabel.h>

static const int MAXBTNS=3;
static const int MIN_LBL_WIDTH = 50;

//------------------------------------------------------------
//.H QwtCounter | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//.I counter A QwtCounter with two buttons on each side
//.U NAME
//	QwtCounter - The Counter Widget
//
//.U SYNOPSIS 
//	#include <qwt_counter.h>
//
//.U INHERITED CLASSES 
//	QWidget, @^QwtDblRange@
//
//.U DESCRIPTION
//	A Counter consists of a label displaying a number and
//	one ore more (up to three) push buttons on each side
//	of the label which can be used to increment or decrement
//	the counter's value.
//.P
//	A Counter has a range from a minimum value to a maximum value
//	and a step size. The range can be specified using
//	@^QwtDblRange#QwtDblRange::setRange@.
//	The counter's value is an integer multiple of the step size.
//	The number of steps by which a button increments or decrements
//	the value can be specified using @QwtCounter::setIncSteps@.
//	The number of buttons can be changed with
//	@QwtCounter::setNumButtons@.
//
//.U PUBLIC MEMBERS
//.R
//	QwtCounter::QwtCounter -- Constructor
//	QwtCounter::~QwtCounter -- Destructor
//	QwtCounter::setNumButtons -- Specify number of buttons
//	QwtCounter::setIncSteps -- Specify increment steps for a button 
//	QwtCounter::incSteps -- Return increment steps for a button
//	QwtCounter::numButtons -- Return number of buttons
//	QwtCounter::sizeHint -- Size Hint
//.-
//.U PROTECTED MEMBERS 
//.R
//	QwtCounter::fontChange -- Notify change of font
//	QwtCounter::resizeEvent -- resize widget
//	QwtCounter::timerEvent -- timer event
//.+
//.U SIGNALS
//.R
//	QwtCounter::buttonReleased -- button released
//	QwtCounter::valueChanged -- value changed
//
//.U EXAMPLE
//.c
//      #include <qwt_counter.h>
//
//      QwtCounter *cnt;
//
//      cnt = new QwtCounter(parent, name);
//
//      cnt->setRange(0.0, 100.0, 1.0);                 // From 0.0 to 100, step 1.0
//      cnt->setNumButtons(2);                          // Two buttons each side
//      cnt->setIncSteps(QwtCounter::Button1, 1);       // Button 1 increments 1 step
//      cnt->setIncSteps(QwtCounter::Button2, 20);      // Button 2 increments 20 steps
//
//      connect(cnt, SIGNAL(valueChanged(double)), my_class, SLOT(newValue(double)));
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtCounter::QwtCounter
//	Construct a QwtCounter instance
//
//.u	Syntax
//.f	 QwtCounter::QwtCounter(QWidget *parent = 0, char *name = 0)
//
//.u	Parameters
//.p	QWidget *parent --	parent widget
//	char *name      --	name
//
//.u	Description
//	The default number of buttons is set to 2. The default
//	increments are
//.t
//	Button 1: -- one step
//	Button 2: -- 10 steps
//	Button 3: -- 100 steps
//
//------------------------------------------------------------
QwtCounter::QwtCounter(QWidget *parent, char *name )
: QWidget(parent,name)
{
    int i;
    
    d_nButtons = 2;
    d_Inc[0] = 1;
    d_Inc[1] = 10;
    d_Inc[2] = 100;
    
    d_tmrIntv = 200;

    lblValue = new QLabel(this);
    lblValue->setNum(0.0);
    lblValue->setAlignment(AlignCenter);
    lblValue->setFrameStyle(QFrame::Panel|QFrame::Sunken);

    lblValue->setMargin(1);           // for sizeHint()

    
    for(i=0;i<MAXBTNS;i++)
    {
	btnL[i] = new QwtArrowButton(i+1, QwtArrowButton::Down,this);
	btnR[i] = new QwtArrowButton(i+1, QwtArrowButton::Up, this);
    }

    connect(btnL[0], SIGNAL(pressed()), SLOT(btnL1Dn()));
    connect(btnL[1], SIGNAL(pressed()), SLOT(btnL2Dn()));
    connect(btnL[2], SIGNAL(pressed()), SLOT(btnL3Dn()));
    connect(btnR[0], SIGNAL(pressed()), SLOT(btnR1Dn()));
    connect(btnR[1], SIGNAL(pressed()), SLOT(btnR2Dn()));
    connect(btnR[2], SIGNAL(pressed()), SLOT(btnR3Dn()));
    connect(btnL[0], SIGNAL(released()), SLOT(btnUp()));
    connect(btnL[1], SIGNAL(released()), SLOT(btnUp()));
    connect(btnL[2], SIGNAL(released()), SLOT(btnUp()));
    connect(btnR[0], SIGNAL(released()), SLOT(btnUp()));
    connect(btnR[1], SIGNAL(released()), SLOT(btnUp()));
    connect(btnR[2], SIGNAL(released()), SLOT(btnUp()));
    
    setRange(0.0,1.0,0.001);
    setValue(0.0);
    
}

//------------------------------------------------------------
//
//.F	QwtCounter::~QwtCounter
//	Destructor
//
//.u	Syntax
//.f	 QwtCounter::~QwtCounter()
//
//------------------------------------------------------------
QwtCounter::~QwtCounter()
{
    delete btnL[0];
    delete btnL[1];
    
    delete btnR[0];
    delete btnR[1];
    
    delete lblValue;
}

//------------------------------------------------------------
//
//.F	QwtCounter::setIncSteps
//	Specify the number of steps by which the value
//	is incremented or decremented when a specified button
//	is pushed.
//
//.u	Syntax
//.f	void QwtCounter::setIncSteps(QwtCounter::Button btn, int nSteps)
//
//.u	Parameters
//.p	QwtCounter::Button btn  -- One of QwtCounter::Button1,
//				   QwtCounter::Button2, QwtCounter::Button3
//	int nSteps	--	Number of steps
//
//------------------------------------------------------------
void QwtCounter::setIncSteps(QwtCounter::Button btn, int nSteps)
{
    if (( btn >= 0) && (btn < ButtonCnt)) 
       d_Inc[btn] = nSteps;
}

//------------------------------------------------------------
//
//.F	QwtCounter::incSteps
//	Return the number of steps by which a specified button
//	increments the value
//
//.u	Syntax
//.f	int QwtCounter::incSteps(QwtCounter::Button btn) const
//
//.u	Parameters
//.p	QwtCounter::Button btn -- One of QwtCounter::Button1,
//			    QwtCounter::Button2, QwtCounter::Button3
//
//.u	Return Value
//	Number of steps or 0 if the button is invalid.
//
//------------------------------------------------------------
int QwtCounter::incSteps(QwtCounter::Button btn) const
{
    if (( btn >= 0) && (btn < ButtonCnt)) 
       return d_Inc[btn];
    else
       return 0;
}


//------------------------------------------------------------
//.-
//.F	QwtCounter::valueChange
//	Notify a change of value
//
//.u	Syntax
//.f	void QwtCounter::valueChange()
//
//------------------------------------------------------------
void QwtCounter::valueChange()
{
    lblValue->setNum(value());
    emit valueChanged(value());
}


//------------------------------------------------------------
//
//.F	QwtCounter::setNumButtons
//	Specify the number of buttons on each side of the label
//
//.u	Syntax
//.f	void QwtCounter::setNumButtons(int n)
//
//.u	Parameters
//.p	int n	--    Number of buttons
//
//------------------------------------------------------------
void QwtCounter::setNumButtons(int n)
{

    if ((n > 0) && (n <= ButtonCnt))
       d_nButtons = n;
    resize(size());
}

//------------------------------------------------------------
//
//.F	QwtCounter::numButtons
//	Return the number of buttons on each side of the widget.
//
//.u	Syntax
//.f	int QwtCounter::numButtons() const
//
//------------------------------------------------------------


//
// Private slot:  Button 1 Left Down
//
void QwtCounter::btnL1Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = -d_Inc[0];
    tmrID = startTimer(d_tmrIntv);
}

//
// Private slot:  Button 2 Left Down
//
void QwtCounter::btnL2Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = -d_Inc[1];
    tmrID = startTimer(d_tmrIntv);
}

//
// Private slot:  Button 3 Left Down
//
void QwtCounter::btnL3Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = -d_Inc[2];
    tmrID = startTimer(d_tmrIntv);
}


//
// Private slot:  Button 1 Right Down
//
void QwtCounter::btnR1Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = d_Inc[0];
    tmrID = startTimer(d_tmrIntv);
}

//
// Private slot:  Button 2 Right Down
//
void QwtCounter::btnR2Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = d_Inc[1];
    tmrID = startTimer(d_tmrIntv);
}

//
// Private slot:  Button 3 Right Down
//
void QwtCounter::btnR3Dn()
{
    if (tmrID) killTimer(tmrID);
    d_tick = 0;
    d_incValue = d_Inc[2];
    tmrID = startTimer(d_tmrIntv);
}

//
// Private slot:  Button Released
//
void QwtCounter::btnUp()
{
    if (!d_tick)
    {
	incValue(d_incValue);
	d_tick = 0;
    }
    if (tmrID) killTimer(tmrID);
    emit buttonReleased(value());
}


//------------------------------------------------------------
//.-
//.F	QwtCounter::resizeEvent
//	Resize the counter
//
//.u	Syntax
//.f	void QwtCounter::resizeEvent(QResizeEvent *e)
//
//.u	Parameters
//.p	QResizeEvent *e
//
//------------------------------------------------------------
void QwtCounter::resizeEvent(QResizeEvent *e)
{
    int btnWidth,i;
    
    QSize r(e->size());
    btnWidth = r.height();
    
    btnWidth = qwtMin(r.height(), (r.width() - MIN_LBL_WIDTH) / ( 2 * d_nButtons));
    btnWidth = qwtMax(btnWidth,5);
    
    for (i=0;i<d_nButtons;i++)
    {
	btnL[i]->resize(btnWidth,r.height());
	btnL[i]->move((d_nButtons - i - 1) * btnWidth, 0);
	btnR[i]->resize(btnWidth, r.height());
	btnR[i]->move(r.width() - (d_nButtons - i) * btnWidth, 0);
	btnL[i]->show();
	btnR[i]->show();
    }
    
    for (i=d_nButtons; i<MAXBTNS; i++)
    {
	btnL[i]->hide();
	btnR[i]->hide();
    }

    lblValue->resize(r.width() - 2*d_nButtons * btnWidth, r.height());
    lblValue->move(d_nButtons * btnWidth, 0);

}


//------------------------------------------------------------
//.-
//.F	QwtCounter::timerEvent
//	Timer event handler
//
//.u	Syntax
//.f	void QwtCounter::timerEvent(QTimerEvent *e)
//
//.u	Parameters
//.p	QTimerEvent *e
//
//.u	Description
//		A timer event occurs when a mouse is pressed
//		and hold down on one of the buttons.
//		The default implementation increments the
//		counter's value.
//------------------------------------------------------------
void QwtCounter::timerEvent(QTimerEvent *e)
{
    d_tick = 1;
    incValue(d_incValue);
}

//------------------------------------------------------------
//.-
//.F	QwtCounter::fontChange
//	Notify change of font
//
//.u	Syntax
//.f	void QwtCounter::fontChange(const QFont &f)
//
//.u	Access
//	protected virtual
//
//.u	Parameters
//.p	const QFont &f	-- new font
//
//.u	Description
//		This function updates the fonts of all widgets
//		contained in QwtCounter.
//------------------------------------------------------------
void QwtCounter::fontChange(const QFont& f)
{
    int i;
    lblValue->setFont(font());
    for (i = 0;i <ButtonCnt; i++)
    {
	btnL[i]->setFont(font());
	btnR[i]->setFont(font());
    } 
}

//------------------------------------------------------------
//
//.F	QwtCounter::sizeHint
//	Returns a recommended size for the counter.
//
//.u	Syntax
//.f	QSize QwtCounter::sizeHint() const
//
//------------------------------------------------------------
QSize QwtCounter::sizeHint() const
{
    QSize rv;
    int h =  qwtMax(lblValue->sizeHint().height(), btnL[0]->sizeHint().height());
    rv.setHeight(h);
    rv.setWidth( qwtMax(lblValue->sizeHint().width(), MIN_LBL_WIDTH) + d_nButtons * h * 2); 
    return rv;
}

//------------------------------------------------------------
//.-
//.F	QwtCounter::setEnabled
//	Activates/deactivates user input
//
//.u	Syntax
//.f	void QwtCounter::setEnabled(bool tf)
//
//.u	Parameters
//.p	bool tf -- enabled/disabled
//
//------------------------------------------------------------
void QwtCounter::setEnabled(bool tf)
{
    int i;

    QWidget::setEnabled(tf);
    
    for (i=0;i<ButtonCnt;i++)
    {
	btnL[i]->setEnabled(tf);
	btnR[i]->setEnabled(tf);
    }

    lblValue->setEnabled(tf);
    
}

//------------------------------------------------------------
//
//.F	QwtCounter::buttonReleased
//      A signal which is emitted when a button has been released
//
//
//.u	Syntax
//.f	void QwtCounter::buttonReleased(double value)
//
//.u	Parameters
//.p	double value	-- new value
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtCounter::valueChanged
//	A signal which is emitted when the counter's value has changed
//
//.u	Syntax
//.f	void QwtCounter::valueChanged(double value)
//
//.u	Parameters
//.p	double value	--	new value
//
//------------------------------------------------------------










