#include "qwt_clrfltr.h"
#include "qwt_math.h"
//------------------------------------------------------------
//
//.H QwtColorFilter | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//
//.U NAME
//	QwtColorFilter - A base class for color filters
//
//.U SYNOPSIS 
//	#include <qwt_clrfltr.h>
//
//.U DESCRIPTION
//	A color filter is a class which can be used to modify colors
//	in a specific way. A QwtColorFilter has an @QwtColorFilter::operator()@
//	which takes a reference to a QColor as its argument and returns the
//	result of the modification. The QwtColorFilter class itself does not
//	modify colors, but is intended as a base class for user-defined color
//	filters. Building a color filter is quite easy: Derive a class from
//	QwtColorFilter and redefine the operator() member. Qwt has several
//	pre-defined color filters (see below).
//	
//	Color filters are used by @^QwtPlot#QwtPlot::print@.
//	
//.U PUBLIC MEMBERS
//.R
//	QwtColorFilter::~QwtColorFilter --	Destructor
//	QwtColorFilter::operator() -- Filter function
//
//.U DERIVED CLASSES 
//	@QwtFltrDim@, @QwtFltrGray@, @QwtFltrInv@
//
//.U PREDEFINED FILTERS
//
//.u QwtFltrDim
//
//	This color filter can be used to darken colors.
//
//.P Members:
//.R
//	QwtFltrDim -- Constructor
//	QwtFltrDim::operator() -- Filter function
//	QwtFltrDim::dimFactor -- Return dim factor
//
//.u QwtFltrGray
//
//	This color filter maps colors into a grayscale with optional
//	inversion.
//
//.P Members:
//.R
//	QwtFltrGray::QwtFltrGray -- Constructor
//	QwtFltrGray::operator() -- Filter function
//
//.u QwtFltrInv
//
//	This filter inverts the RGB values of colors.
//
//.P Members:
//.R
//	QwtFltrInv::operator() -- Filter function
//
//.U GLOBAL CONSTANTS
//.t
//	NoColorFilter -- A color filter which returns the color
//			 without modification
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.U	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------


QwtColorFilter NoColorFilter;

//------------------------------------------------------------
//
//.F	QwtColorFilter::~QwtColorFilter
//	Destroys a QwtColorFilter instance
//
//.u	Syntax
//.f	 QwtColorFilter::~QwtColorFilter()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtColorFilter::operator()
//	Filter function
//
//.u	Syntax
//.f	QColor QwtColorFilter::operator()(const QColor &c)
//
//.u	Access
//	public virtual
//
//.u	Parameters
//.p	const QColor &c -- modified color
//
//.u	Description
//	The filter function can be redefined by derived classes.
//	The default implementation returns its argument.
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtFltrGray::QwtFltrGray
//	Construct a QwtFltrGray instance
//
//.u	Syntax
//.f	 QwtFltrGray::QwtFltrGray(bool inv = FALSE)
//
//.u	Parameters
//.p	bool inv -- Invert the result if TRUE
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtFltrGray::operator()
//	Filter function which changes a color to grayscale
//
//.u	Syntax
//.f	QColor QwtFltrGray::operator()(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c -- color to be modified
//
//------------------------------------------------------------
QColor QwtFltrGray::operator()(const QColor &c) const
{
    int g = qGray(c.rgb());
    if (d_inv)
       g = 255 - g;
    return QColor(g, g, g);
}


//------------------------------------------------------------
//
//.F	QwtFltrDim::QwtFltrDim
//	Construct a QwtFltrDim instance
//
//.u	Syntax
//.f	 QwtFltrDim::QwtFltrDim(int dim)
//
//.u	Parameters
//.p	int factor -- Darkening factor. A factor greater
//		      than 100 produces a darker color,
//		      a factor smaller than 100 produces
//		      a brighter color.
//
//.u	Description
//	QwtFltrDim uses QColor::dark() (See Qt Manual)
//
//------------------------------------------------------------
QwtFltrDim::QwtFltrDim(int factor)
{
    d_dim = qwtLim(factor, 1, 10000);
}

//------------------------------------------------------------
//
//.F	QwtFltrDim::operator()
//	Filter function. Darkens or brightens a specified
//	color.
//
//.u	Syntax
//.f	QColor QwtFltrDim::operator()(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c
//
//.u	Description
//	The behaviour of the filter is influenced by the
//	constructor argument (See @QwtFltrDim::QwtFltrDim@
//
//------------------------------------------------------------
QColor QwtFltrDim::operator()(const QColor &c) const
{
    return QColor(c.dark(dimFactor()));
}


//------------------------------------------------------------
//
//.F	QwtFltrInv::operator()
//	Filter function. Inverts the RGB values of a
//	specified color.
//
//.u	Syntax
//.f	QColor QwtFltrInv::operator()(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c -- Color to be modified
//
//------------------------------------------------------------
QColor QwtFltrInv::operator()(const QColor &c) const
{
    int r, g, b;
    c.rgb(&r, &g, &b);
    return QColor(255-r, 255-g, 255-b);
}







