#include <qwt_math.h>
#include "qwt_autoscl.h"

//------------------------------------------------------------
//.H QwtAutoScale|3|01/09/97|Qwt Widget Library|Qwt Programmer's manual
//.U NAME
//  QwtAutoScale - The Qwt Auto-Scaler
//
//.U SYNOPSIS
//  #include <qwt_autoscl.h>
//
//.U DESCRIPTION
//	This class can be  used to generate a scale which may span 
//	multiple ranges of values. A scale consists of a lower boundary,
//	an upper boundary, a vector of major scale ticks and a vector of
//	minor scale ticks which divide it into subintervals.
//	A quick look at the example below will give you an idea
//	of how the auto-scaler works.
//.P
//	The auto-scaler produces "reasonable" major and minor step sizes. For
//	linear scales, the major step size will fit into the pattern {1,2,5}*10^n,
//	where n is an integer. In logarithmic mode (see @QwtAutoScale::setOptions@)
//	the step size is measured in *decades*
//	and the major step size will be adjusted to fit the pattern {1,2,3,5}*10^n,
//	where n is a natural number including zero.
//.P
//	The step size can be manipulated indirectly using @QwtAutoScale::setMaxMajor@.
//	The layout of the scale can be varied with  @QwtAutoScale::setOptions@.
//.P
//	The auto-scaling algorithm can be partly or completely disabled
//	(even temporarily) if a user-defined scale is desired. This can be done with
//	the @QwtAutoScale::setScale@ function. It can be switched off
//	again with @QwtAutoScale::setAutoScale@.
//.P
//	The two @QwtAutoScale::adjust (1)@ members are used to extend
//	the scale if necessary in order to include another range or array of
//	values. The resulting scale division can be obtained with
//	@QwtAutoScale::scaleDiv@The @QwtAutoScale::reset@ resets the scale to zero.
//	
//
//.U PUBLIC MEMBERS 
//.R
//	QwtAutoScale::QwtAutoScale	-- Constructor
//	QwtAutoScale::~QwtAutoScale	-- Destructor
//	QwtAutoScale::adjust (1)	-- Adjust the scale in order to include
//					   all values in a specified array
//	QwtAutoScale::adjust (2)	-- Adjust the scale in order to include
//					   a specified range
//	QwtAutoScale::build		-- Rebuild the scale
//	QwtAutoScale::reset		-- Reset the scale to zero
//	QwtAutoScale::changeOptions	-- Change a set of options
//	QwtAutoScale::setAutoRebuild -- Enable automatic update
//	QwtAutoScale::setAutoScale	-- Re-enable auto-scaling
//	QwtAutoScale::setMargins	-- Specify boundary margins
//	QwtAutoScale::setMaxMajor	-- Set maximal number of major subdivisions
//	QwtAutoScale::setMaxMinor	-- Set maximum number of minor subdivisions
//	QwtAutoScale::setOptions	-- Specify options
//	QwtAutoScale::setReference 	-- Specify reference point
//	QwtAutoScale::setScale	-- Set a fixed scale range and (optionally)
//	QwtAutoScale::scaleDiv		-- Return the scale division
//
//	QwtAutoScale::autoScale -- Check for autoscale mode
//	QwtAutoScale::loMargin -- Return lower margin
//	QwtAutoScale::hiMargin -- Return upper margin
//	QwtAutoScale::maxMajor -- Return max. number of major steps
//	QwtAutoScale::maxMinor -- Return max. number of minor steps
//	QwtAutoScale::option -- Check for specified option
//	QwtAutoScale::options -- Return options
//	QwtAutoScale::reference -- Return reference value
//	QwtAutoScale::scaleDiv -- Return scale divisio
//
//.U    EXAMPLE
//.c
//      #include<qwt_autoscl.h>
//      #include<iostream.h>
//
//      double x1[100];
//      double x2[200];
//      double range_min, range_max;
//
//      QwtAutoScale as;
//
//      ... determine x1 and x1, range_min and range_max here ...
//
//      as.reset();                           // clear it
//      as.adjust(range_min, range_max);      // include a range
//      as.adjust(x1,100);                    // include an array
//      as.adjust(x2,200);                    // include another array
//
//      for (i=0;i<as.scaleDiv().majCnt(); i++)
//      {
//              cout << "Scale tick " << i
//                   << " at " << as.scaleDiv().majMark(i) << "\n";
//      }       
//
//.U	NOTE
//	For logarithmic scales, the step size as well as
//	the margins are measured in *decades*.
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.C	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

static const double MinEps=1.0e-10;

//------------------------------------------------------------
//
//.F	QwtAutoScale::QwtAutoScale
//	Construct a QwtAutoScale instance
//
//.u	Syntax
//.f	 QwtAutoScale::QwtAutoScale()
//------------------------------------------------------------
QwtAutoScale::QwtAutoScale ()
{
    d_autoScale = TRUE;
    d_scaleOpt = None;
    
    d_minValue = 0.0;
    d_maxValue = 0.0;
    d_scaleMin = 0.0;
    d_scaleMax = 0.0;
    d_loMargin = 0.0;
    d_hiMargin = 0.0;
    d_step = 0.0;
    d_maxMajor = 8;
    d_maxMinor = 5;
    d_reset = 1;
    d_autoRebuild = TRUE;
    
}

//------------------------------------------------------------
//
//.F	QwtAutoScale::~QwtAutoScale
//	Destroy a QwtAutoScale instance
//
//.u	Syntax
//.f	 QwtAutoScale::~QwtAutoScale()
//
//------------------------------------------------------------
QwtAutoScale::~QwtAutoScale ()
{
}


//------------------------------------------------------------
//
//.F	QwtAutoScale::adjust (1)
//	Adjust the scale to include a given array of input values.
//
//.u	Syntax
//.f	void QwtAutoScale::adjust(double *., int num, int reset)
//
//.u	Parameters
//.p	double *x	-- array of input values
//	int num		-- size of x
//	int reset = 0	-- if != 0 reset the scale's contents.
//
//.u	Return Value
//
//.u	Description
//	This member function extends the boundaries of the scale
//	and re-calculates the step size if necessary in order
//	to include all values in the array. If the reset
//	parameter has nonzero value, the previous state will
//	be cleared.
//
//------------------------------------------------------------
void QwtAutoScale::adjust(double *x, int num, int reset)
{
    int i;
    if (d_reset || reset)
       d_minValue = d_maxValue = x[0];
    for (i = 0; i < num; i++)
    {
	if (x[i] > d_maxValue)
	   d_maxValue = x[i];
	if (x[i] < d_minValue)
	   d_minValue = x[i];
    }
    d_reset = 0;
    if (d_autoRebuild) build();
    
}



//------------------------------------------------------------
//
//.F	QwtAutoScale::adjust (2)
//	Adjust the scale to include a specified interval
//
//.u	Syntax
//.f	void QwtAutoScale::adjust(double vmin, double vmax, int reset)
//
//.u	Parameters
//.p	double vmin	--	lower border of the specified interval
//	double vmax	--	upper border of the specified interval
//	int reset	--	if nonzero, reset the scale. Defaults to 0.
//
//.u	Description
//	This member function extends the boundaries of the scale
//	and re-calculates the step size if necessary in order
//	to include a specified interval. If the reset
//	parameter has nonzero value, the previous state will
//	be cleared.
//
//------------------------------------------------------------
void QwtAutoScale::adjust(double vmin, double vmax, int reset)
{ 
    double mxv,mnv;
    
    mxv = qwtMax(vmin,vmax);
    mnv = qwtMin(vmin,vmax);
    
    if (d_reset || reset)
    {
	d_minValue = mnv;
	d_maxValue = mxv;
    }
    else
    {
	if (d_minValue > mnv) 
	   d_minValue = mnv;
	if (d_maxValue < mxv)
	   d_maxValue = mxv;
    }
    d_reset = 0;
    if (d_autoRebuild) build();
    
}

//------------------------------------------------------------
//.-
//.F	QwtAutoScale::build
//	Re-build the scale
//
//.u	Description
//
//.u	Syntax
//.f	 void QwtAutoScale::build()
//
//------------------------------------------------------------
void QwtAutoScale::build() 
{

    if (d_reset) return;
    
    if (d_autoScale)
    {
	if (d_scaleOpt & Logarithmic) 
	   buildLogScale(); 
	else 
	   buildLinScale();
    }
    else
       d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			bool(d_scaleOpt & Logarithmic), d_step, FALSE); 
       
}


//------------------------------------------------------------
//.-
//.F	QwtAutoScale::buildLinScale
//	Build a linear scale
//
//.u	Syntax
//.f	void QwtAutoScale::buildLinScale()
//
//------------------------------------------------------------
void QwtAutoScale::buildLinScale ()
{

    double delta, dec, ticks, minval, maxval, step;


    ticks = double (d_maxMajor);


    //
    // If in Autoscale Mode, adjust minval and maxval according to
    // the active scale options, and add the margins
    //
    if (!d_autoScale) return;
    
    minval = d_minValue;		// scale boundaries are based on the
    maxval = d_maxValue;		// data.

    //
    // add / subtract margins
    //
    if (d_loMargin > 0.0)
       minval -= d_loMargin;
    if (d_hiMargin > 0.0)
       maxval += d_hiMargin;

    //
    //	Correct minval / maxval according to the scale options
    //
    if (d_scaleOpt & Symmetric)
    {
	delta = qwtMax(qwtAbs(d_ref - maxval), qwtAbs(d_ref - minval));
	maxval = d_ref + delta;
	minval = d_ref - delta;	
    }
    else if (d_scaleOpt & IncludeRef)
    {
	if (maxval < d_ref) 
	   maxval = d_ref;
	else if (minval > d_ref) 
	   minval = d_ref;
    }
    
    //
    // first approximation of d_scaleMin and d_scaleMax
    //
    setRange(minval, maxval);
    delta = d_scaleMax - d_scaleMin;


    // dec := maximal power of ten which fits into the interval
    //   [d_scaleMin,d_scaleMax]
    dec = pow (10.0, floor (log10 (delta)));

    //
    //	The following magic line calculates the step size such that
    //      - The number of subintervals will not exceed the maximum
    //        as specified by the user
    //	    - The step size fits {1,2,5}*10^n with a natural number n  
    // 
    step = qwtCeil125(delta * 0.999999 / dec / ticks) * dec;

    //
    //	determine he final values of scaleMin and scaleMax
    //
    if (! (d_scaleOpt & Floating) )
    {
       // adjust of d_scaleMin and d_scaleMax such that both are integer
       // multiples of the step size.
       d_scaleMin = step * floor ((d_scaleMin + MinEps * step) / step);
       d_scaleMax = step * ceil ((d_scaleMax - MinEps * step) / step);
    }

    if (d_scaleOpt & Inverted)
    {
	step = -step;
	d_scldiv.rebuild(d_scaleMax, d_scaleMin, d_maxMajor, d_maxMinor,
			 FALSE, step, FALSE);
    }
    else
    {
	d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			 FALSE, step, TRUE);
    }
    
}


//------------------------------------------------------------
//.-
//.F	QwtAutoScale::buildLogScale
//	build a logarithmic scale
//
//.u	Syntax
//.f	void QwtAutoScale::buildLogScale()
//
//------------------------------------------------------------
void QwtAutoScale::buildLogScale ()
{

    double ticks, decades;
    double delta, minval, maxval, fpart, ipart, step;

    if (!d_autoScale) return;

    
    minval = d_minValue;	// the calculation of scale divisions is based on the
    maxval = d_maxValue;	// input data.

    if (d_loMargin > 0.0)
       minval /= pow(10.0, d_loMargin);
    if (d_hiMargin > 0.0)
       maxval *= pow(10.0, d_hiMargin);

    if (d_scaleOpt & Symmetric)
    {
	delta = qwtMax(maxval / d_lref,  d_lref / minval); 
	maxval = d_lref * delta;
	minval = d_lref / delta;	
    }
    else if (d_scaleOpt & IncludeRef)
    {
	if (maxval < d_lref) 
	   maxval = d_lref;
	else if (minval > d_lref) 
	   minval = d_lref;
    }

    if (d_maxMajor > 0)
       ticks = double (d_maxMajor);
    else
       ticks = 1;

    setRange(minval, maxval);

    // decades included in the interval
    decades = qwtAbs(log10 (d_scaleMax / d_scaleMin));

    // calculate step size in decades
    if ((decades > 1.0) && (decades > ticks))
    {
	// One interval contains more than one decade.
	// The number of decades in an interval is adjusted
	// to be a multiple of 2,3,5, or 10.
	fpart = modf (log10 (ceil (decades * 0.999999 / ticks)), &ipart);
	if (fpart < MinEps)
	   fpart = 1.0;
	else if ((fpart - LOG10_2) < MinEps)
	   fpart = 2.0;
	else if ((fpart - LOG10_3) < MinEps)
	   fpart = 3.0;
	else if ((fpart - LOG10_5) < MinEps)
	   fpart = 5.0;
	else
	   fpart = 10.0;

	step = pow (10, ipart) * fpart;

    }
    else				// The minimal step size is one decade.
    {
	step = 1.0;
    };
    
    if (!(d_scaleOpt & Floating))
    {
	d_scaleMin = pow (10, step * floor ((log10(d_scaleMin) + MinEps * step) / step));
	d_scaleMax = pow (10, step * ceil ((log10(d_scaleMax) - MinEps * step) / step));
    }

    if (d_scaleOpt & Inverted)
    {
	step = -step;
	d_scldiv.rebuild(d_scaleMax, d_scaleMin, d_maxMajor, d_maxMinor, TRUE,
			 step, FALSE);
    }
    else
    {
	d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			 TRUE, step, TRUE);
    }

}

//------------------------------------------------------------
//
//.F	QwtAutoScale::changeOptions
//	Set or reset specified scale options
//
//.u	Syntax
//.f	void QwtAutoScale::changeOptions(int opt, bool tf)
//
//.u	Parameters
//.p	int opt		--	or-combined  scale options
//	bool tf 	--	If TRUE, set the specified options.
//				If FALSE, reset these options.
//
//.u	See also
//      @QwtAutoScale::setOptions@
//------------------------------------------------------------
void QwtAutoScale::changeOptions(int opt, bool tf)
{
    if (tf)
       d_scaleOpt |= opt;
    else
       d_scaleOpt &= (~opt);
    build();
}


//------------------------------------------------------------
//
//.F	QwtAutoScale::reset
//	Set the interval boundaries to zero and clear the
//	scale division
//
//.u	Syntax
//.f	void QwtAutoScale::reset()
//
//.u	Description
//	This member function resets an AutoScale object 
//	to its initial state. It is needed to clean up
//	the scale before or
//	after subsequent @QwtAutoScale::adjust@ calls.
//	The boundaries of the scale are set to zero
//	and the scale division is cleared.
//
//.u	Note
//	A reset doesn't affect the margins.
//
//------------------------------------------------------------
void QwtAutoScale::reset()
{
    d_reset = TRUE;
    d_scldiv.reset();
    d_minValue = 0;
    d_maxValue = 0;
    d_step = 0;
}


//------------------------------------------------------------
//
//.F	QwtAutoScale::setAutoScale
//	Enable auto-scaling
//
//.u	Syntax
//.f	void QwtAutoScale::setAutoScale()
//
//.u	Description
//	This function is used to switch back to auto-scaling mode
//	if the scale has been frozen temporarily (see @QwtAutoScale::setScale@).
//.P
//	When auto-scaling is reactivated, the scale will be rebuild, which
//	means that
//.i
//	-- if adjust or setMaxIntv have been called in between, the scale
//	   will be adjusted to the new conditions.
//	-- if none of these functions and no reset has been called, the old state will
//		be restored.
//	-- if only reset has been called in between, nothing will happen.
//
//.u    See also
//	@QwtAutoScale::setScale@
//------------------------------------------------------------
void QwtAutoScale::setAutoScale()
{
    d_autoScale = TRUE;
    build();
}

//------------------------------------------------------------
//
//.F	QwtAutoScale::setMargins
//	Specify margins at the scale's endpoints
//
//.u	Syntax
//.f	void QwtAutoScale::setMargins(double mlo, double mhi)
//
//
//.u	Parameters
//.p	double mlo     --	minimum distance between
//				the scale's lower boundary
//				and the smallest enclosed value
//	double mhi	--	minimum distance between 
//				the scale's upper boundary
//				and the greatest enclosed value
//
//.u	Description
//	Margins can be used to leave a minimum amount of space between
//	the enclosed intervals and the boundaries of the scale.
//	
//.u	Note
//.i	-- With logarithmic scales, the margins are measured in
//	   decades.
//	-- The margins will not be changed by any other member function.
//	   You should remember this when you call @QwtAutoScale::reset@
//	   or change from a linear to a logarithmic scale. 
//	   
//------------------------------------------------------------
void QwtAutoScale::setMargins(double mlo, double mhi)
{
    d_loMargin = qwtMax(mlo,0.0);
    d_hiMargin = qwtMax(mhi,0.0);
    build();
}


//------------------------------------------------------------
//
//.F	QwtAutoScale::setMaxMajor
//	Specify the maximum number of major intervals
//
//.u	Syntax
//.f	void QwtAutoScale::setMaxMajor(int mx)
//
//.u	Parameters
//.p	int mx		--	maximum number of subintervals
//
//.u	Description
//	The auto-scaler places the major ticks at reasonable
//	points, such that the number of major tick intervals does not exceed
//	the specified maximum number.
//
//------------------------------------------------------------
void QwtAutoScale::setMaxMajor(int mx)
{
    d_maxMajor = qwtMax(mx,1);
    d_maxMajor = qwtMin(mx, 10000);
    build();
}

//------------------------------------------------------------
//
//.F	QwtAutoScale::setMaxMinor
//	Specify the maximum number of minor subdivisions
//	within major scale intervals
//
//.u	Syntax
//.f	void QwtAutoScale::setMaxMinor(int mx)
//
//.u	Parameters
//.p	int mx	 -- maximum number of minor ticks
//
//------------------------------------------------------------
void QwtAutoScale::setMaxMinor(int mx)
{
    d_maxMinor = qwtMin(qwtMax(mx,0), 100);
    build();
}


//------------------------------------------------------------
//.-
//.F	QwtAutoScale::setRange 
//
//.u	Syntax
//.f	void QwtAutoScale::setRange(double x1, double x2)
//
//
//.u	Parameters
//.p	double x1, double x2
//
//------------------------------------------------------------
void QwtAutoScale::setRange(double x1, double x2)
{
    double delta;
    double minval, maxval;
    
    minval = qwtMin(x1, x2);
    maxval = qwtMax(x1, x2);

    if (d_scaleOpt & Logarithmic)
    {
	minval = qwtMin(qwtMax(minval, LOG_MIN), LOG_MAX);
	maxval = qwtMin(qwtMax(maxval,LOG_MIN), LOG_MAX);
    }
    
    delta = maxval - minval;
	
    if (delta <= 0.0)		// all values are equal
    {				
	if (minval > 0)
	{
	    d_scaleMin = minval * 0.5;
	    d_scaleMax = maxval * 1.5;
	}
	else if (minval < 0)
	{
	    d_scaleMin = minval * 1.5;
	    d_scaleMax = maxval * 0.5;
	}
	else		      // all values are zero
	{			
	    d_scaleMin = -0.5;
	    d_scaleMax = 0.5;
	}

	delta = d_scaleMax - d_scaleMin;

    }
    else			// the normal case
    {				
	d_scaleMin = minval;
	d_scaleMax = maxval;
    }
    
}



//------------------------------------------------------------
//
//.F	QwtAutoScale::setScale
//	Specify a user-defined scale and switch off auto-scaling
//	
//.u	Syntax
//.f	void QwtAutoScale::setScale(double xmin, double xmax, double step)
//
//
//.u	Parameters
//.p	double xmin	--	user-defined lower boundary
//	double xmax	--	user-defined upper boundary
//	double step = 0	--	user-defined fixed major step size
//
//.u	Description
//	A fixed scale may be used for different purposes, e.g.
//	zooming. If the step argument is left out or less or equal
//	to zero, the auto-scaler will calculate the major step size
//	size according to the maxMajor setting (see @QwtAutoScale::setMaxMajor@.
//	The fixed-scale mode can switched off using
//	@QwtAutoScale::setAutoScale@, which restores the
//	previous values.
//
//.u	Note
//.i	-- if xmin > xmax, xmax will be the lower boundary.
//	-- if xmin == xmax, the auto-scaler sets the boundaries to (-0.5, 0.5).
//	-- Options and margins have no effect while auto-scaling is switched off.
//
//------------------------------------------------------------
void QwtAutoScale::setScale(double xmin, double xmax, double step)
{

    // turn auto-scaling off and set the
    // scale limits to the desired values
    setRange(xmin,xmax);
    d_autoScale = FALSE;
    d_step = step;
    // rebuild the scale
    build();

}


//------------------------------------------------------------
//
//.F	QwtAutoScale::setOptions
//	Reset scale options and set specified options
//
//.u	Syntax
//.f	void QwtAutoScale::setOptions(int opt)
//
//.u	Parameters
//.p	int opt		--	Combined set of options
//
//.u	Description
//	The behaviour of the auto-scaling algorithm can be changed
//	with the following options:
//.t
//	QwtAutoScale::None	--	Switch all options off.
//	QwtAutoscale::IncludeRef --	Build a scale which includes
//					the reference value.
//	QwtAutoScale::Symmetric	--	Build a scale which is symmetric
//					to the reference value.
//	QwtAutoScale::Logarithmic --	Build a logarithmic scale.
//	QwtAutoScale::Floating	--	The endpoints of the scale
//					are supposed to be equal the outmost included
//					values plus the specified margins (see
//					@QwtAutoScale::setMargins@).
//					If this option is *not* set, the endpoints
//					of the scale will be integer multiples
//					of the step size.
//	QwtAutoScale::Inverted --	Turn the scale upside down.
//
//.u	Note
//.i	-- If the type of scale division is changed from logarithmic to linear
//	   or vice versa, the margins will not be transformed. Note that
//	    the margins count in decades if the scale is logarithmic.
//      -- If a linear scale contains negative values, switching to a
//	   logarithmic scale will cut them off and set the lower scale
//	   boundary to its lowest possible value of 1.0E-100. This effect
//	   is reversible if you
//	   switch back to a linear scale.
//	-- The options have no effect while auto-scaling is turned off
//		(see @QwtAutoScale::setScale@)
//
//.u	Example
//.c
//      #include <qwt_autoscl.h>
//
//      void main() 
//      {
//          QwtAutoScale as;
//
//          // build linear scale with default settings
//          as.adjust(17.45, 87344.0);
//
//                          ...
//
//          // change to logarithmic scale with floating ends
//          as.setOptions(QwtAutoScale::Floating | QwtAutoscale::Logarithmic);
//
//                          ...
//
//          // change to linear, zero-symmetric scale
//          as.setOptions(QwtAutoScale::ZeroSymmetric);
//
//                          ...
//      }
//
//.u	See also
//	@QwtAutoScale::changeOptions@ for a description of the
//	possible options
//------------------------------------------------------------
void QwtAutoScale::setOptions(int opt)
{
    d_scaleOpt = opt;
    build();
}


//------------------------------------------------------------
//
//.F	QwtAutoScale::setReference
//	Specify a reference point
//
//.u	Syntax
//.f	void QwtAutoScale::setReference(double r)
//
//.u	Parameters
//.p	double r	--	new reference value
//
//.u	Description
//	The reference point is needed if the auto-scaler options IncludeRef or
//	Symmetric are active. Its default value is 0 for linear scales and 1 for
//	logarithmic scales.
//
//.u	Note
//	The reference value for logarithmic scales is limited to
//	( LOG_MIN / 2 <= reference <= LOG_MAX /2 ). If the specified
//	value is less than LOG_MIN (defined in qwt_math.h), it will
//	be set to 1.0 for logarithmic scales.
//
//------------------------------------------------------------
void QwtAutoScale::setReference(double r)
{
    d_ref = r;
    
    if (r > LOG_MIN / 2) 
    {
	d_lref = qwtMin(r, LOG_MAX / 2);
    }
    else
       d_lref = 1.0;
    build();
}

//------------------------------------------------------------
//
//.F	QwtAutoScale::autoScale
//	Return TRUE if auto-scaling is active
//
//.u	Syntax
//.f	bool QwtAutoScale::autoScale()
//
//.u	See Also
//	@QwtAutoScale::setAutoScale@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::loMargin
//	Return the margin at the lower end of the scale
//
//.u	Syntax
//.f	double QwtAutoScale::loMargin()
//
//.u	See also
//	@QwtAutoScale::setMargins@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::hiMargin
//	Return the margin at the upper end of the scale
//
//.u	Syntax
//.f	double QwtAutoScale::hiMargin()
//
//.u	See Also
//	@QwtAutoScale::setMargins@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::maxMajor
//	Return the maximum number of major tickmarks
//
//.u	Syntax
//.f	int QwtAutoScale::maxMajor()
//
//.u	See also
//	@QwtAutoScale::setMaxMajor@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::maxMinor
//	Return the maximum number of minor scale ticks
//
//.u	Syntax
//.f	int QwtAutoScale::maxMinor()
//
//.u	See Also
//	@QwtAutoScale::setMaxMinor@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::option
//	Returns TRUE if the specified option is set.
//.u	Syntax
//.f	bool QwtAutoScale::option(int opt) const
//
//.u Parameters
//	int opt	-- option
//
//.u See also
//   @QwtAutoscale::setOptions@
//------------------------------------------------------------
bool QwtAutoScale::option(int opt) const
{
  return bool(d_scaleOpt & opt);
}
//------------------------------------------------------------
//
//.F	QwtAutoScale::options
//	Return options
//
//.u	Syntax
//.f	int QwtAutoScale::options()
//
//.u    See also
//	@QwtAutoScale::setOptions@
//
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::reference
//	Return the reference value
//
//.u	Syntax
//.f	double QwtAutoScale::reference()
//
//
//.u	See also
//	@QwtAutoScale::setReference@
//------------------------------------------------------------
//------------------------------------------------------------
//
//.F	QwtAutoScale::scaleDiv
//	Returns a const reference to the scale division
//
//.u	Syntax
//.f	const QwtScaleDiv& QwtAutoScale::scaleDiv()
//
//.u	Description
//	The scale division consists of two boundary values,
//	an array of major tickmarks and an array of minor
//	tickmarks.
//
//.u    See also
//	@^QwtScaleDiv@
//------------------------------------------------------------









