#include <qapp.h>
#include <math.h>
#include <qpainter.h>
#include "curvdemo2.h"


//------------------------------------------------------------
//		curvdemo2
//
//	This example shows a simple animation featuring
//	QwtCurve and QwtPixFrame
//
//------------------------------------------------------------

//
//   Array Sizes
//
const int Size = 15;
const int USize = 13;

//
//   Arrays holding the values
//
double xval[Size];
double yval[Size];
double zval[Size];
double uval[USize];
double vval[USize];


//
//  CONSTRUCT MAIN WINDOW
//
MainWin::MainWin()
: QWidget()
{
    int i;
    double base, toggle;
    QwtSymbol sym;

    //
    //  pixframe background
    //
    bg = QColor(30,30,50);

    //
    //  pixframe
    //	
    px = new QwtPixFrame(this,"");
    px->setFrameStyle(QFrame::Box|QFrame::Raised);
    px->setLineWidth(2);
    px->setMidLineWidth(3);
    
    //
    //  curve 1
    // 
    crv1.setRange(-1.5, 1.5 ,0, 0.0, 6.28, 0);
    crv1.setPen(QPen(QColor(150,150,200),2));
    crv1.setSplineSize(150);
    crv1.setStyle(QwtCurve::Spline,QwtCurve::Xfy);

    sym.setStyle(QwtSymbol::XCross);
    sym.setPen(QPen(yellow,2));
    sym.setSize(7);
    
    crv1.setSymbol(sym);

    crv1.setRawData(yval,xval,Size);
    
    //
    // curve 2
    //
    crv2.setRange(0.0, 6.28, 0,-3.0,1.1,0);
    crv2.setPen(QPen(QColor(200,150,50), 1, DashDotDotLine));
    crv2.setStyle(QwtCurve::Sticks);
    crv2.setSymbol(QwtSymbol(QwtSymbol::Ellipse,
			     blue,yellow, 
			     QSize(5,5)));

    crv2.setRawData(xval,zval,Size);

    
    //
    //  curve 3
    // 
    crv3.setRange(-1.1, 3.0,0,-1.1, 3.0,0);
    crv3.setPen(QColor(100,200,150));
    crv3.setStyle(QwtCurve::Spline, QwtCurve::Periodic|QwtCurve::Parametric);
    crv3.setSplineSize(200);

    crv3.setRawData(yval,zval,Size);

    //
    //  curve 4
    //
    crv4.setRange(-5, 1.1,0,-1.1, 5.0,0);
    crv4.setPen(red);
    crv4.setStyle(QwtCurve::Spline);
    crv4.setSplineSize(200);

    crv4.setRawData(uval,vval,USize);

    //
    //  initialize values
    //
    base = 2.0 * M_PI / double(USize - 1);
    toggle = 1.0; 
    for (i=0; i< USize; i++)
    {
	uval[i] =  toggle * cos( double(i) * base);
	vval[i] =  toggle * sin( double(i) * base);
	    
	if (toggle == 1.0)
	   toggle = 0.5;
	else
	   toggle = 1.0;
    }

    newValues();

    //
    //  connections
    //
    connect(px, SIGNAL(pixmapResized()), SLOT(drawCurves()));

    //
    // start timer
    //
    tid = startTimer(250);	

}

//
//  DESTRUCTOR
//
MainWin::~MainWin()
{
    killTimer(tid);
}

//
// RESIZE MAIN WINDOW
//
void MainWin::resizeEvent(QResizeEvent *e) 
{
    px->resize(e->size());
    px->move(0,0);
}

//
//  REDRAW PIXMAP
//
void MainWin::drawCurves()
{

    QPainter p;

    px->fillPixmap(bg);
    
    p.begin(px->pixmap());

    //
    //	Draw the curves into the pixmap
    //
    crv1.draw(&p, px->pixmap()->rect());
    crv2.draw(&p, px->pixmap()->rect());
    crv3.draw(&p, px->pixmap()->rect());
    crv4.draw(&p, px->pixmap()->rect());

    p.end();
    
}

//
//  TIMER EVENT
//
void MainWin::timerEvent(QTimerEvent *t)
{
    newValues();
    drawCurves();
    px->repaint(FALSE);
}

//
//  RE-CALCULATE VALUES
//
void MainWin::newValues()
{
    int i;
    static double phs = 0.0;
    double s,c,u;
    
    for (i=0;i<Size;i++)
    {
	xval[i] = 6.28 * double(i) / double(Size -1);
	yval[i] = sin(xval[i] - phs);
	zval[i] = cos(3.0 * (xval[i] + phs));
    }
    
    s = 0.25 * sin(phs);
    c = sqrt(1.0 - s*s);
    for (i=0; i<USize;i++)
    {
	u = uval[i];
	uval[i] = uval[i] * c - vval[i] * s;
	vval[i] = vval[i] * c + u * s;
    }
    
    phs += 0.0628;
    if (phs > 6.28)
       phs = 0.0;
    
}


//
//  MAIN
//
int main (int argc, char **argv)
{
    QApplication a(argc, argv);
    MainWin w;

    a.setMainWidget(&w);
    w.resize(300,300);
    w.show();

    return a.exec();
}











