#include <qapp.h>
#include <math.h>
#include "bode.h"
#include <qlabel.h>
#include <qwt_counter.h>
#include <qwt_plot.h>
#include <qwt_math.h>
#include <qprinter.h>
#include <qpdevmet.h>
#include "cplx.h"

//-----------------------------------------------------------------
//
//	bode.cpp -- A demo program featuring QwtPlot and QwtCounter
//
//	This example demonstrates the mapping of different curves
//	to different axes in a QwtPlot widget. It also shows how to
//	display the cursor position and how to implement zooming.
//
//-----------------------------------------------------------------

const int ArraySize = 200;
QString zoomInfo("Zoom: Press mouse button and drag");
QString cursorInfo("Cursor Pos: Press mouse button in plot region");


InputFrame::InputFrame(QWidget *p, const char* name)
: QFrame(p, name)
{
    lblDamp = new QLabel("Damping Factor", this);
    cntDamp = new QwtCounter(this);
    lblInfo = new QLabel(cursorInfo, this);
    btnZoom = new QPushButton("Zoom", this);
    btnPrint = new QPushButton("Print", this);
    cntDamp->setRange(0.0, 5.0, 0.01);
    cntDamp->setValue(0.0);
    lblDamp->setGeometry(20, 20, 100, 20);
    cntDamp->setGeometry(120, 20, 120, 20);
    btnZoom->setGeometry(300,15,100,25);
    btnPrint->setGeometry(420,15,100,25);

    lblInfo->setFont(QFont("Helvetica", 8));
    lblInfo->setGeometry(20, 5, 230, 10);

}

InputFrame::~InputFrame()
{
    delete lblDamp;
    delete lblInfo;
    delete cntDamp;
    delete btnZoom;
    delete btnPrint;
}



MainWin::MainWin(QWidget *p , const char *name)
: QWidget(p, name)
{
    
    d_zoomActive = d_zoom = 0;
    
    frequency = new double[ArraySize];
    amplitude = new double[ArraySize];
    phase = new double[ArraySize];

    plt = new QwtPlot(this);
    frmInp = new InputFrame(this);

    frmInp->setFrameStyle(QFrame::Panel|QFrame::Raised);
    frmInp->setLineWidth(2);

    plt->setTitle("Frequency Response of a Second-Order System");
    
    crv1 = plt->insertCurve("Amplitude");
    crv2 = plt->insertCurve( "Phase");

    plt->setCurvePen(crv1, QPen(yellow));
    plt->setCurvePen(crv2, QPen(cyan));

    plt->enableGridXMin();
    plt->enableAxis(QwtPlot::yRight);
    plt->setGridMajPen(QPen(white, 0, DotLine));
    plt->setGridMinPen(QPen(gray, 0 , DotLine));

    plt->setAxisTitle(QwtPlot::xBottom, "Normalized Frequency");
    plt->setAxisTitle(QwtPlot::yLeft, "Amplitude [dB]");
    plt->setAxisTitle(QwtPlot::yRight, "Phase [deg]");

    plt->setAxisOptions(QwtPlot::xBottom, QwtAutoScale::Logarithmic);
    plt->setAxisMaxMajor(QwtPlot::xBottom, 6);
    plt->setAxisMaxMinor(QwtPlot::xBottom, 10);
    
    plt->setCurveYAxis(crv1, QwtPlot::yLeft);
    plt->setCurveYAxis(crv2, QwtPlot::yRight);
    
    mrk1 = plt->insertMarker();
    plt->setMarkerLineStyle(mrk1, QwtMarker::VLine);
    plt->setMarkerPos(mrk1, 0.0,0.0);
    plt->setMarkerLabelAlign(mrk1, AlignRight|AlignBottom);
    plt->setMarkerPen(mrk1, QPen(green, 0, DashDotLine));
    plt->setMarkerFont(mrk1, QFont("Helvetica", 10, QFont::Bold));

    mrk2 = plt->insertLineMarker("", QwtPlot::yLeft);
    plt->setMarkerLabelAlign(mrk2, AlignRight|AlignBottom);
    plt->setMarkerPen(mrk2, QPen(QColor(200,150,0), 0, DashDotLine));
    plt->setMarkerFont(mrk2, QFont("Helvetica", 10, QFont::Bold));
    plt->setMarkerSymbol(mrk2, QwtSymbol(QwtSymbol::Diamond, yellow, green, QSize(7,7)));
    
    plt->setPlotBackground(darkBlue);
    plt->enableLegend(TRUE);
    plt->setLegendPos(Qwt::Bottom);
    plt->setLegendFrameStyle(QFrame::Box|QFrame::Sunken);

    // build frequency vector with logarithmic division
    qwtLogSpace(frequency, ArraySize, 0.01, 100);
    damping = 0.0;

    recalc();

    connect(frmInp->cntDamp, SIGNAL(valueChanged(double)), SLOT(setDamp(double))); 
    connect(frmInp->btnPrint, SIGNAL(clicked()), SLOT(print()));
    connect(frmInp->btnZoom, SIGNAL(clicked()), SLOT(zoom()));
    connect(plt, SIGNAL(plotMouseMoved(const QMouseEvent&)),
	    SLOT(plotMouseMoved( const QMouseEvent&)));
    connect(plt, SIGNAL(plotMousePressed(const QMouseEvent &)),
	    SLOT(plotMousePressed( const QMouseEvent&)));
    connect(plt, SIGNAL(plotMouseReleased(const QMouseEvent &)),
	    SLOT(plotMouseReleased( const QMouseEvent&)));
    
    plt->enableOutline(TRUE);
    plt->setOutlinePen(green);
}

MainWin::~MainWin()
{
    delete plt;
    delete frmInp;
    delete[] frequency;
    delete[] phase;
    delete[] amplitude;
}



void MainWin::print()
{

    QBrush br(red);
    QPen pn(yellow);
    
    QwtSymbol sym1;
    sym1.setBrush(br);
    sym1.setPen(pn);
    sym1.setSize(11);

    QPrinter p;

    if (p.setup(0))
    {
	plt->print(p, QwtFltrDim(200));
    }


}

void MainWin::zoom()
{
    if (d_zoomActive)
    {
	// Disable Zooming.
	plt->setAxisAutoScale(QwtPlot::yLeft);
	plt->setAxisAutoScale(QwtPlot::yRight);
	plt->setAxisAutoScale(QwtPlot::xBottom);
	plt->replot();
	d_zoom = FALSE;
	d_zoomActive = 0;
	
    }
    else
       d_zoom = !d_zoom;
    
    if (d_zoom)
    {
	frmInp->btnZoom->setText("Unzoom");
	frmInp->lblInfo->setText(zoomInfo);
    }
    else
    {
	frmInp->btnZoom->setText("Zoom");
	frmInp->lblInfo->setText(cursorInfo);
    }
    
}

//
// set damping factor
//
void MainWin::setDamp(double d)
{
    damping = d;
    recalc();
    plt->replot();
}

void MainWin::plotMouseMoved(const QMouseEvent &e)
{
    
    QPainter p;
    QString lbl = "Freq=";
    QString lbl2;
    
    lbl2.setNum(plt->invTransform(QwtPlot::xBottom, e.pos().x() ), 'g', 3);
    lbl += lbl2 + ",  Ampl=";
    
    lbl2.setNum(plt->invTransform(QwtPlot::yLeft, e.pos().y() ), 'g', 3);
    lbl += lbl2 + ",  Phase=";
    lbl2.setNum(plt->invTransform(QwtPlot::yRight, e.pos().y() ), 'g', 3);
    lbl += lbl2;
    
    frmInp->lblInfo->setText(lbl);
}

void MainWin::plotMousePressed(const QMouseEvent &e)
{
    // store position
    p1 = e.pos();
    
    // update cursor pos display
    plotMouseMoved(e);
    
    if (d_zoom && (!d_zoomActive))
    {
    	plt->setOutlineStyle(Qwt::Rect); 
    }
    else
    {
	plt->setOutlineStyle(Qwt::Cross);
    } 
    
}

void MainWin::plotMouseReleased(const QMouseEvent &e)
{
    int x1, x2, y1, y2;
    int lim;
    
    // some shortcuts
    int axl= QwtPlot::yLeft, axr = QwtPlot::yRight, axb= QwtPlot::xBottom;
    
    if (d_zoom && (!d_zoomActive))
    {
	d_zoomActive = 1;
	
	// Don't invert any scales which aren't inverted
	x1 = qwtMin(p1.x(), e.pos().x());
	x2 = qwtMax(p1.x(), e.pos().x());
	y1 = qwtMin(p1.y(), e.pos().y());
	y2 = qwtMax(p1.y(), e.pos().y());
	
	// limit selected area to a minimum of 11x11 points
	lim = 5 - (y2 - y1) / 2;
	if (lim > 0)
	{
	    y1 -= lim;
	    y2 += lim;
	}
	lim = 5 - (x2 - x1 + 1) / 2;
	if (lim > 0)
	{
	    x1 -= lim;
	    x2 += lim;
	}
	
	// Set fixed scales
	plt->setAxisScale(axl, plt->invTransform(axl,y1), plt->invTransform(axl,y2));
	plt->setAxisScale(axr, plt->invTransform(axr,y1), plt->invTransform(axr,y2));
	plt->setAxisScale(axb, plt->invTransform(axb,x1), plt->invTransform(axb,x2));
	plt->replot();
	

    }
    frmInp->lblInfo->setText(cursorInfo);
    plt->setOutlineStyle(Qwt::Triangle);
    
}

//
// re-calculate frequency response
//
void MainWin::recalc()
{
    cplx g;
    QString sBuffer;
    double f, fmax = 1;
    int i, i3 = 1;
    double f3;
    double amax = -1000.0;
    double arc_conv = 180.0 / M_PI;
    
    for (i=0; i<ArraySize; i++)
    {
	f = frequency[i];
	g = cplx(1.0) / cplx(1.0 - f * f, 2.0 * damping * f);
	amplitude[i] = 20.0 * log10(sqrt( g.real()*g.real() + g.imag()*g.imag()));
	phase[i] = atan2(g.imag(), g.real()) * arc_conv;

	if ((i3 <= 1) && (amplitude[i] < -3.0)) 
	   i3 = i;
	if (amplitude[i] > amax)
	{
	    amax = amplitude[i];
	    fmax = frequency[i];
	}
	
    }
    
    f3 = frequency[i3] - 
       (frequency[i3] - frequency[i3 - 1]) 
	  / (amplitude[i3] - amplitude[i3 -1]) * (amplitude[i3] + 3);
    
    sBuffer.setNum(f3, 'g', 4);
    plt->setMarkerPos(mrk1, f3, 0.0);
    plt->setMarkerLabel(mrk1,QString("-3 dB at f = ") + sBuffer);
    
    sBuffer.setNum(amax, 'g', 4);
    plt->setMarkerPos(mrk2, fmax, amax);
    plt->setMarkerLabel(mrk2, QString("Peak: ") + sBuffer + QString("dB"));
    
    
    // curve 1 copies data (less efficient, but safe)
    plt->setCurveData(crv1, frequency, amplitude, ArraySize);
    
    // curve 2 attaches data (more efficient, but dangerous)
    plt->setCurveRawData(crv2, frequency, phase, ArraySize);

}

void MainWin::resizeEvent(QResizeEvent *e)
{
    QRect r(0, 0, e->size().width(), e->size().height() - 50);
    
    plt->setGeometry(r);
    frmInp->setGeometry(0, r.bottom() + 1, r.width(), 50);
    
}


int main (int argc, char **argv)
{
    int rv;
    int allocContext;
    QApplication a(argc, argv);
    MainWin w;
    a.setMainWidget(&w);
    w.resize(540,400);
    w.show();
    rv = a.exec();
    return rv;
    
}













