/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <qlayout.h>
#include <qtablevw.h>
#include <qpushbt.h>
#include <qpainter.h>

#include "qdbttabular.h"
#include "qdbtheader.h"
#include "qdbttable.h"
#include "qdbtslidebar.h"

/*! 
  \class QdbtTabular qdbttabular.h
  \brief The QdbtTabular widget provides a multi-column list of cells that
         can be scrolled.
         
  Each cell in a QdbtTabular can either be empty or contain a QdbtTableCell. 
  The default behavior of the table is row oriented. The user can select
  one row at a time. Multiple rows can be selected. A selected row is
  displayed with reversed colors. 

  Using selectByRow() individual cells can be selected instead of entire rows.

  Default the table has a header. The header contains a section for each column.
  Sections are push buttons, with the following special properties:
  \arg The user can resize them by click and dragging the divider between the
       buttons. 
  \arg The text of the button can be aligned to the left, center and right.
      
  The ability to resize a column can be turned off.
  The header consists of a number of buttons. Each button is a QdbtSection 
  widget (or a widget derived from it) can be modified and 
  using changeSection().
  
  The number of rows and columns of the table can be (re-)set directly using
  setDimensions() or by inserting \link insertRow() rows\endlink or 
  inserting \link insertCol() columns\endlink. 

  Newly created cells are empty. You can use changeCell() to
  replace an existing cell with another cell.

  Cells can be made editable. This means that their contents can be edited if
  a user presses the middle mousebutton on a cell. 

  \warning The QdbtTablular assumes ownership of all cells and will
  delete them when they are not needed.
  
  <center>
  <table align=center>
  <tr>
  <td>
  <img src="../snapshots/snapshot1.gif">
  <td>
  <img src="../snapshots/snapshot2.gif">
  </tr>
  <tr>
  <td><center>snapshot 1</center>
  <td><center>snapshot 2</center>
  </tr>
  </table>
  </center>
*/

/*! 
  Constructs an empty tabular (dimension 0 x 0) with an empty header.
  The \a parent, \a name and \a f arguments are passed to the 
  QFrame constructor.
*/    

QdbtTabular::QdbtTabular(QWidget *parent,const char *name,WFlags f)
  : QFrame(parent,name,f)
{
  setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
  headerActive=TRUE;
  header=new QdbtHeader(this);
  table=new QdbtTable(this);
  slideBar=new QdbtSlideBar(this);
  connect(slideBar,SIGNAL(sliderPos(int)),header,SLOT(resizeSection(int)));

  // Create some connections between the table and its header, so they can
  // be synchronized
  connect(header,SIGNAL(signalColumnWidth(int,int)),table,
          SLOT(setColumnWidth(int,int)));
  connect(table,SIGNAL(setHeaderOffset(int)),
          header,SLOT(setHeaderOffset(int)));
  connect(header,SIGNAL(setTableOffset(int)),
          table,SLOT(setTableOffset(int)));
  connect(header,SIGNAL(updateTable()),
          table,SLOT(updateTable()));
  connect(header,SIGNAL(exposeColumn(int)),
          table,SLOT(exposeColumn(int)));

  // Pass on the signals from the children.
  connect(header,SIGNAL(sectionClicked(int)),
          SIGNAL(sectionClicked(int)));
  connect(table,SIGNAL(selected(int,bool)),
          SIGNAL(selected(int,bool)));
  connect(table,SIGNAL(activated(int)),
          SIGNAL(activated(int)));
  connect(table,SIGNAL(cellEdited(int,int)),
          SIGNAL(cellEdited(int,int)));
  connect(table,SIGNAL(cellSelected(int,int,bool)),
          SIGNAL(cellSelected(int,int,bool)));
  connect(table,SIGNAL(cellActivated(int,int)),
          SIGNAL(cellActivated(int,int)));
}

/*!
  Destroys the table.
*/

QdbtTabular::~QdbtTabular()
{
}

/*!
 Sets the font that is used for the text in the table's sections to \a font
 \sa headerFont() and setCellFont().
*/

void QdbtTabular::setHeaderFont(const QFont &font)
{
  header->setFont(font);
  repaint();
}

/*!
  Returns the font that is currently used for the header.<p>
  \sa setHeaderFont().
*/

const QFont &QdbtTabular::headerFont() const
{  
  return header->font();
}

/*!
  Sets the font that is used for the text of each cell to <em>font</em>.<p>
  \sa cellFont() and setHeaderFont().  
*/

void QdbtTabular::setCellFont(const QFont &font)
{
  table->setFont(font);
  repaint();
}

/*!
  Returns the font that is currently used for the header.
  \sa setHeaderFont().
*/

const QFont &QdbtTabular::cellFont() const
{
  return table->font();
}

/*!
  \fn void QdbtTabular::clear()
  Removes all cells from the table. 
  This is equivalent to \link setDimensions() setDimensions\endlink(0,0).
*/

/*!
  Set the dimension of the table to \a rows rows and \a cols columns.
  If the table is made smaller, the cells that fall outside
  the table will be deleted. If the table is made larger, empty
  cells are created. All cells that are inside the area
  covered by both the old size and the new size remain unchanged. 

  Try to use this function whenever possible, because it is much more
  efficient than repeatedly inserting rows and columns.
  \sa insertRow(), removeRow(), insertCol(), removeCol() and clear().
*/

void QdbtTabular::setDimensions(int rows,int cols)
{
  table->setDimensions(rows,cols);
  header->setColumns(cols);
}

/*!
  A virtual function that handles the resize events for the table.

  Does not do anything at the moment.
*/

void QdbtTabular::resizeEvent(QResizeEvent *)
{
  //printf("QdbtTabular::resizeEvent()\n");
}

/*!
  A virtual function that handles the repaint function of the table.
*/

void QdbtTabular::drawContents(QPainter *)
{
  // (re)organize the child widgets so that the frame's contents are filled
  //printf("QdbtTabular::drawContents()\n");
  int hh=header->heightHint();   // preferred height of the header
  int w=contentsRect().width();
  int h=contentsRect().height();
  int fw=frameWidth();
  if (headerActive) // do we have a header?
  {
    header->move(fw,fw);
    header->resize(w,hh);
    table->move(fw,fw+hh);
    table->resize(w,h-hh);
  }
  else 
  {
    table->move(fw,fw);
    table->resize(w,h);
  }
}

/*!
  Shows the table's header whenever the table is visible. This is the default 
  behavior. The header consists of a number of sections, one for each column.
  Each section is a button with a text. Pressing the button will emit
  a signal. Each section is separated by a divider. If the column is
  resizable the user can click the left mouse button and drag the divider to 
  resize the column. Pressing the middle mouse button on a divider 
  will resize the column so all cells fit in the column.
  \sa hideHeader(), headerVisible(), sectionClicked() and changeSection().
*/

void QdbtTabular::showHeader()
{
  if (!headerActive)
  {
    headerActive=TRUE;
    header->show();
    repaint();
  }
}

/*!
  Hides the header normally visible above the table. The columns are no longer
  resizable by the user.
  \sa showHeader(), headerVisible().
*/

void QdbtTabular::hideHeader()
{
  if (headerActive)
  {
    headerActive=FALSE;
    header->hide();
    repaint(); 
  }
}

/*!
  Returns \c TRUE if the table's header is currently visible, and
  \c FALSE if the header is currently not visible.
  \sa showHeader(), hideHeader().
*/

bool QdbtTabular::headerVisible() const 
{
  return headerActive;
}

/*!
  Insert a new column at column-index \a index
  A value of \a index=-1 will append a column to the end of the table
  \sa changeSection(), removeCol() and setDimensions().
*/

void QdbtTabular::insertCol(int col)
{
  table->insertCol(col);
  header->insertCol(col);
}

/*!
  Removes the column at index \a index (and its section) from the table.
  All cells in the column will be deleted.
  \sa insertCol() and setDimensions().
*/

void QdbtTabular::removeCol(int col)
{
  table->removeCol(col);
  header->removeCol(col);
}

/*!
  Insert a new row at row-index \a index in the table. 
  A value of \a index=-1 will append a row to the end of the table.
  Each cell in the new row is empty.

  If \e selectable is set to \c TRUE (the default) the row may be selected 
  by the user.

  If \e selectable is set to \c FALSE the row is not selectable by the user.

  Notice that the value of \e selectable only has effect if the selection 
  of rows is enabled.

  See also: removeRow(), selectByRow(), setDimensions(), changeCell(), 
            numRows().
*/

void QdbtTabular::insertRow(int index,bool selectable)
{
  table->insertRow(index,selectable);
}

/*!
  Removes row at \a index from the table. 
  All cells in the row will be deleted.
  \sa insertRow() and setDimensions().
*/

void QdbtTabular::removeRow(int index)
{
  table->removeRow(index);
}

/*!
  Deletes the contents of the cell at row \a row and column \a col
  and replaces it with a deep copy of the cell \a tc. 
  If \a tc is the null pointer, the cell will become empty.

  \sa cell().
*/

void QdbtTabular::changeCell(const QdbtTableCell *tc,int row,int col)
{
  table->changeCell(tc,row,col);
}

/*!
  Deletes the contents of the cell at row \a row and column \a col
  and replaces it with a cell with text \a text, pixmap \a pixmap.
  This pixmap is placed to the left of the text.
  The text color can be specified by \a color, the alignment of the 
  text-pixmap pair is determined by \a align. 
  The following three alignments are possible: 
  \c AlignLeft, \c AlignCenter, and \c AlignRight.

  If the boolean \a edit is \c TRUE the
  cell is editable by the user. If \a edit is \c FALSE the cell cannot
  be altered by the user of the table 
  (It can still be altered from within the program).
  \sa cell().
*/

void QdbtTabular::changeCell(const char *text,const QPixmap *pixmap,
                  int row,int col,const QColor &color,int align,bool edit)
{
  QdbtTableCell tc(text,pixmap,color,align,edit);
  changeCell(&tc,row,col);
}

/*!
  Deletes the contents of the cell at row \a row and column \a col
  and replaces it with a cell with text \a text and no pixmap.
  The text color can be specified by \a color, the alignment of text
  is determined by \a align. 
  The following three alignments are possible: 
  \c AlignLeft, \c AlignCenter, and \c AlignRight.

  If the boolean \a edit</em> is \c TRUE the
  cell is editable by the user. If \a edit is \c FALSE the cell cannot
  be altered by the user of the table 
  (It can however still be altered from within the program).
  \sa cell().
*/

void QdbtTabular::changeCell(const char *text,int row,int col,
                          const QColor &color,int align,bool edit)
{
  changeCell(text,0,row,col,color,align,edit);
}

/*!
  Deletes the contents of the cell at row \a row and column \a col
  and replaces it with a cell with pixmap \a pixmap and no text.
  The alignment of the pixmap is determined by \a align.
  \sa cell().
*/

void QdbtTabular::changeCell(const QPixmap *pixmap,int row,int col,
                          int align)
{
  changeCell(0,pixmap,row,col,black,align);
}

void QdbtTabular::startSlideBar(int minx,int xpos)
{
  //printf("Starting slide bar at %d minx %d\n",xpos,minx);
  int x,y,w,h,xs=minx+frameWidth();
  contentsRect().rect(&x,&y,&w,&h);
  slideBar->setBounds(xs,y,w-xs+2,h);
  slideBar->setBarWidth(2);
  slideBar->updateBar(mapToGlobal(QPoint(xpos,0)));
  slideBar->show();
}

/*!
  Sets the width of column at \a col to \a width pixels.
  \sa columnWidth() and columnWidthHint().
*/

void QdbtTabular::setColumnWidth(int col,int width)
{
  // resizing a column in the header will update the table as well
  header->setColumnWidth(col,width);  
}

/*!
  Returns the width in pixels of the column at index \e col.
  \sa rowHeight() and columnWidthHint().
*/

int QdbtTabular::columnWidth(int col) const
{
  return table->columnWidth(col);
}

/*!
  Returns the height in pixels of the row at index \e row.
  \sa columnWidth().
*/

int QdbtTabular::rowHeight(int row) const
{
  return table->rowHeight(row);
}

/*!
  Returns the minimal width that is required to fully display all cells in
  the column at index <em>col</em> including the header.
  \sa fitAll().
*/

int QdbtTabular::columnWidthHint(int col) const 
{
  int headerWidth=header->columnWidthHint(col);
  int tableWidth=table->columnWidthHint(col);
  return QMAX(headerWidth,tableWidth);
}

/*!
  Returns a pointer to the cell-object at location (\e row,\e col) 
  in the table.
  \sa changeCell().
*/

const QdbtTableCell *QdbtTabular::cell(int row,int col)
{ 
  return table->cell(row,col);
}

/*!
  Returns \c TRUE if the row at index \e index is currently selected.
  Notice that this function only returns useful values if selection 
  by row is enabled.
  \sa setRowSelected() and selectByRow().
*/

bool QdbtTabular::rowSelected(int row) const
{
  return table->rowSelected(row);
}

/*!
  Sets the select state of the row at index \e index to \e status.
  If \e status is \c TRUE the row will be highlighted (selected).
  If \e status is \c FALSE the row will not be highlighted (unselected).
  \sa rowSelected().
*/

void QdbtTabular::setRowSelected(int row,bool status)
{
  table->setRowSelected(row,status);
}

/*!
  Returns the number of rows in the table (not counting the header).
  \sa numCols().
*/

int QdbtTabular::numRows() const
{
  return table->numRows();
}

/*!
  Returns the number of columns in the table.<p>
  \sa numRows().
*/

int QdbtTabular::numCols() const
{
  return table->numCols(); 
}

/*!
  Sets the auto-update option of the table view to enable.

  If enable is \c TRUE (this is the default) then the table updates it self 
  automatically whenever it has changed in some way (for example when 
  a flag is changed). 

  If enable is \c FALSE, the table does NOT repaint itself, or update 
  its internal state variables itself when it is changed. This can be 
  useful to avoid flicker during large changes, and is singularly 
  useless otherwise: Disable auto-update, do the changes, re-enable 
  auto-update, and call update().

  \warning: Do not leave the view in this state for a long time 
  (i.e. between events ). 
  If, for example, the user interacts with the table when auto-update is off, 
  strange things can happen.

  Setting auto-update to \c TRUE does not repaint the view, you must call 
  update() to do this. 
  
  \sa autoUpdate() and update().
*/

void QdbtTabular::setAutoUpdate(bool state)
{
  // in some rare cases we need to update the table before we can switch
  // autoUpdate on again. Seems like a bug in Qt but I'm not sure.....
  if (state && !autoUpdate()) update(); 
  table->setAutoUpdate(state);
}

/*!
  Returns TRUE if the view updates itself automatically whenever 
  it is changed in some way. 
  \sa setAutoUpdate().
*/

bool QdbtTabular::autoUpdate() const
{
  return table->autoUpdate();
}

/*!
  Updates the widget unless updates are disabled or the widget is hidden.
 
  Updating the widget will redraw each cell that is exposed. Cells 
  erase themselves when they are redrawn, thus minimizing the amount 
  of flickering.

  Notice, that this function repaints the table correctly in all cases, 
  whereas repaint() does not.
*/

void QdbtTabular::update()
{
  repaint();
  table->repaint();
}

/*!
  A virtual function that handles the mouse enter events of the table excluding 
  the header.

  Does not do anything at the moment.
*/

void QdbtTabular::enterTableEvent(QEvent *)
{
}

/*!
  A virtual function that handles the mouse leave events of the table excluding 
  the header.

  Does not do anything at the moment.
*/

void QdbtTabular::leaveTableEvent(QEvent *)
{
}

/*!
  Returns the current size of the cell at location 
  (\a row,\a col) in the table.
  The cells themselves are not aware of their size.
*/

QSize QdbtTabular::cellSize(int row,int col) const
{
  return QSize(table->cellWidth(col),table->cellHeight(row));
}

/*!
  Returns the offset in pixels of the cell at location (\a row,\a col) 
  in the table.

  The offset is based on the scrollable part of the table 
  (excluding the header). Therefore, it includes parts of the table that 
  are not visible because of scrolling.
*/

QPoint QdbtTabular::cellOffset(int row,int col) const
{
  return QPoint(table->cellXOffset(col),table->cellYOffset(row));
}

/*!
  Returns the x coordinate in \e table coordinates of the pixel which 
  is currently on the left edge of the view.
  \sa yOffset().
*/

int QdbtTabular::xOffset() const
{
  return table->xOffset();
}

/*!
  Returns the y coordinate in table coordinates of the pixel which 
  is currently on the top edge of the view.
  \sa xOffset().
*/

int QdbtTabular::yOffset() const
{
  return table->yOffset();
}

/*!
  Returns the index of the row at position \a yPos, 
  where \a yPos is in widget coordinates.
  Returns -1 if \a yPos is outside the valid range.
  \sa findCol() and headerHeight().
*/

int QdbtTabular::findRow(int yPos) const
{
  return table->findRow(yPos-headerHeight());
}

/*!
  Returns the index of the column at position \a xPos, where \a xPos
  is in widget coordinates. Returns -1 if \a xPos is outside the valid range. 
  \sa: findRow().
*/

int QdbtTabular::findCol(int xPos) const
{
  return table->findCol(xPos);
}

/*!
  Returns the preferred size of the table. The size is chosen so
  that all cells fit without the need for scrollbars. 

  Notice, that if there are many or large cells in the table, the returned 
  size may be very large.

  \sa fitAll().
*/
 
QSize QdbtTabular::sizeHint() const
{
  int h=headerHeight();
  int w=0,i;
  for (i=0;i<numCols();i++) w+=columnWidthHint(i);
  for (i=0;i<numRows();i++) h+=rowHeight(i);
  return QSize(w+4,h+4);
}

/*!
  Returns the height of the header in pixels. If the header is currently 
  invisible 0 is returned.
*/
 
int QdbtTabular::headerHeight() const
{
  return headerVisible() ? header->heightHint() : 0;
}

/*!
  Enables dynamic tooltips for cells whose text is truncated. 
  The tooltip shows the complete text. Default this feature is enabled.
  \sa disableTooltips().
*/

void QdbtTabular::enableTooltips()
{
  table->enableTooltips();
}

/*!
  Disables tooltips for cells whose text is truncated.
  \sa enableTooltips().
*/

void QdbtTabular::disableTooltips()
{
  table->disableTooltips();
}

/*!
  This function determines how cells are selected.
  If \e enable is set to \c TRUE the user can select cells
  one row at a time. This is the default behavior.
  Only the rows that are selectable (see setRowSelectable()) can 
  be selected by the user.

  If \e enable is set to \c FALSE the user can select
  individual cells. Only cells that are set to be selectable 
  (see QdbtTableCell::setSelectable()) 
  can be selected by the user.

  \sa setRowSelectable().
*/

void QdbtTabular::selectByRow(bool enable)
{
  table->selectByRow(enable);
}

/*!
  This function returns \c TRUE if the user is allowed to change the 
  highlight of the row at index \a index in the table.

  \sa setRowSelectable().
*/

bool QdbtTabular::rowSelectable(int row) const
{
  return table->rowSelectable(row);
}

/*!
  Sets whether or not the row at index \e index may be selected by the user.

  If \e enable is \c TRUE selection is allowed. This is the default.

  If \e enable is \c FALSE selection is denied.
  This function only has effect when selection by row is enabled.

  \sa selectByRow() and rowSelectable().
*/

void QdbtTabular::setRowSelectable(int row,bool enable)
{
  table->setRowSelectable(row,enable);
}

/*!
  Sets the edit focus on the cell at location (\a row,\a col) 
  in the table. Normally, the user is allowed to edit cells by clicking the 
  middle mouse button those cells that are enabled for editing 
  (see QdbtTableCell::setEditable()).
  This function can force the editing of a certain cell 
  regardless of the edit state of the cell.
*/

void QdbtTabular::editCell(int row,int col)
{
  table->editCell(row,col);
}

/*!
  Sets the color palette to \arg p.
*/

void QdbtTabular::setPalette(const QPalette &p)
{
  //printf("QdbtTabular::setPalette()\n");
  QWidget::setPalette(p);
  header->setPalette(p);
  table->setPalette(p);
}

/*!
  Resizes the table's columns so that all cells fit (are not truncated).<p>
  \sa sizeHint() and columnWidthHint().
*/

void QdbtTabular::fitAll()
{
  int c;
  bool upd=autoUpdate();
  setAutoUpdate(FALSE);
  for (c=0;c<numCols();c++) setColumnWidth(c,columnWidthHint(c));
  setAutoUpdate(upd); 
  update();
}

/*!
  Sets a user specified widget representing the section
  of the header above column \a col.
  
  If you want to change the look and/or feel of the 
  default section, you should inherit your own widget from 
  QdbtSection and supply a dynamicly allocated
  object as the \a section argument:
  
  Example:
  
  Assume you have a widget MySection that inherits from
  QdbtSection, then you can change column \e col of a
  table widget \e tabular like this
  \code
  MySection *section=new MySection(tabular);
  tabular->changeSection(section,col);
  \endcode
  
  \warning The QdbtTabular widget assumes ownership of the section
           widgets, do not delete them manually.
  
  \sa section() and QdbtSection.
*/

void QdbtTabular::changeSection(QdbtSection *section,int col)
{
  header->changeSection(section,col);
}

/*!
  Scrolls the table so that the cell at (\a row,\a col), in such a 
  way that the cell is visible for the largest part possible.
  The amount of scrolling is minimized. It is 0 if the cell is already
  completely visible.
*/

void QdbtTabular::exposeCell(int row,int col)
{
  table->exposeCell(row,col);
}

/*!
  Returns a pointer to the widget representing the section of
  the header above column \a col.
  
  You can use this pointer to get or set the text and/or alignment
  of the section.
  
  \warning The table widget assumes control over the resize and location 
           of the section widget. Do not change these properties manually.
  
  \sa changeSection() and QdbtSection.
*/ 

QdbtSection *QdbtTabular::section(int col)
{
  return header->section(col);
}

/*! 
  \fn void QdbtTabular::sectionClicked(int index)
  This signal is emitted if the user clicks on a section button.
  A section represents the header above a column.
  The index of the column is passed as an argument.
*/

/*!
  \fn void QdbtTabular::selected(int index,bool state)
  This signal is emitted whenever the highlight status of a row is changed.
  The index of the row and the new state are passed as arguments.
  If the value of <em>state</em> is <code>TRUE</code> the row is currently highlighted.<p>
  Notice that this signal is only emitted if selection by row is enabled.<p>
  \sa selectByRow() and cellSelected().
*/

/*!
  \fn void QdbtTabular::activated(int row)
  This signal is emitted whenever the user double clicks on a row.
  The index of the row is passed as an argument.

  Notice that this signal is only emitted if selection by row is enabled.
  \sa selectByRow() and cellActivated().
*/

/*!
  \fn void QdbtTabular::cellEdited(int row,int col)
  This signal is emitted when the user has finished editing a 
  cell and the contents of the cell has changed. The 
  location (\a row,\a col) of the cell in the table is passed as an argument.
*/

/*!
  \fn void QdbtTabular::cellActivated(int row,int col) 
  This signal is emitted whenever the user double clicks on a cell.
  The location of the cell is passed as an argument.

  Notice that this signal is only emitted if selection by row is disabled.

  \sa selectByRow() and activated().
*/

/*!
  \fn void QdbtTabular::cellSelected(int row,int col,bool state)
  This signal is emitted whenever the highlight status of a cell is changed.
  The location of the cell and the new state are passed as arguments.
  If the value of \a state is \c TRUE the cell is currently highlighted.

  Notice that this signal is only emitted if selection by row is disabled.

  \sa selectByRow() and selected().
*/

