/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include "qdbtcellmatrix.h"
#include <stdlib.h>

// We assume the 2D array `m' is the owner of all cells
QdbtCellMatrix::QdbtCellMatrix(int r,int c)
{
  m.setAutoDelete(TRUE);
  row=0;col=0;
  resize(r,c);
}

// The 2D array `m' will automatically delete all cells
// so the destructor is empty
QdbtCellMatrix::~QdbtCellMatrix()
{
}

// returns the cell at location (r,c) in the 2D array.
QdbtTableCell *QdbtCellMatrix::at(int r,int c) const
{
  return m[r]->at(c);
}

// Resizes the table to `r' rows and `c' columns.
// If the table grows, new cells are set to 0.
// If the table shrinks, new cells are deleted.
void QdbtCellMatrix::resize(int r,int c)
{
  int i;
  ASSERT(r>=0 && c>=0);
  if (r==row && c==col) return;
  m.resize(r);
  
  // initialize all new rows
  for (i=row;i<r;i++)
  {
    Row **rowArray=m.data();
    rowArray[i]=new Row; 
    rowArray[i]->setAutoDelete(TRUE);
  }
  
  // resize all rows
  for (i=0;i<r;i++)
  {
    m[i]->resize(c);
  }
  row=r;
  col=c;
}


// Inserts a column at index `c'.
// Implemented by resizing the table,
// and shifting cell pointers. 
void QdbtCellMatrix::insertCol(int c)
{
  ASSERT(c>=0 && c<=col);
  int i,j;
  for (i=0;i<row;i++)
  {
    Row *cellRow=m[i];
    cellRow->resize(col+1);
    QdbtTableCell **cellArray=cellRow->data();
    for (j=col;j>c;j--) cellArray[j]=cellArray[j-1];
    cellArray[c]=0;
  }
  col++;
}

// Inserts a row at index `r'.
// Implemented by resizing the table
// and shifting row pointers.
void QdbtCellMatrix::insertRow(int r)
{
  ASSERT(r>=0 && r<=row);
  int i;
  m.resize(row+1);
  Row **rowArray=m.data();
  for (i=row;i>r;i--) rowArray[i]=rowArray[i-1];
  rowArray[r]=new Row(col);
  rowArray[r]->setAutoDelete(TRUE);
  row++;
}

// Removes the column at index `c'.
// Implemented by shifting cell pointers
// and shrinking each row
void QdbtCellMatrix::removeCol(int c)
{
  ASSERT(c>=0 && c<col);
  int i,j;
  for (i=0;i<row;i++)
  {
    Row *cellRow=m[i];
    QdbtTableCell **cellArray=cellRow->data();
    delete cellArray[c];
    for (j=c;j<col-1;j++) cellArray[j]=cellArray[j+1];
    cellArray[col-1]=0; // set to 0 so it will not be deleted
    cellRow->resize(col-1);
  }
  col--;
}

// Removes a column at index `r'.
// Implemented by shifting row pointers
// and shrinking the array of rows.
void QdbtCellMatrix::removeRow(int r)
{
  ASSERT(r>=0 && r<row);
  int i;
  Row **rowArray=m.data();
  delete rowArray[r];
  for (i=r;i<row-1;i++) rowArray[i]=rowArray[i+1];
  rowArray[row-1]=0; // set to 0 so it will not be deleted
  m.resize(row-1);
  row--;
}

// Easy access operator.
// Allows the following statements:  a(r,c)=c;
//                                   c=a(r,c);
// where a is `a' cellmatrix and `c' is pointer to a cell.
QdbtTableCell *&QdbtCellMatrix::operator()(int r,int c)
{
  ASSERT(r>=0 && r<row && c>=0 && c<col);
  return (m[r]->data())[c];
}
