/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <stdlib.h>
#include <qapp.h>
#include <qlayout.h>
#include <qpushbt.h>
#include <qlabel.h>

#include "example1.h"
#include <qdbt/qdbttabular.h>
#include <qdbt/qdbttabcell.h>
#include <qdbt/qdbtsection.h>

// constructor
Example::Example(QWidget *parent,const char *name, WFlags f) 
      : QWidget(parent,name,f)
{
  // Set the caption of the window
  setCaption("Example");

  // The Layout managers
  QGridLayout *layout  = new QGridLayout(this,3,1,5);
  QGridLayout *input   = new QGridLayout(1,5,5);
  QBoxLayout  *buttons = new QBoxLayout(QBoxLayout::LeftToRight);
  
  // Create a new QdbtTabular object
  tabular=new QdbtTabular(this);

  // Set the font of the cells in the table
  tabular->setCellFont(QFont("helvetica",12,QFont::Bold));

  // Set the minimal widget size
  tabular->setMinimumSize(100,100);

  // Load the folder pixmap (No error checking!)
  folder.load("folder.xpm");

  // Load the file pixmap (No error checking!)
  file.load("file.xpm");

  // set the initial table size (this is faster than
  // inserting the columns and rows one by one
  tabular->setDimensions(10,2);

  // Set the font of the header
  tabular->setHeaderFont(QFont("helvetica",16,QFont::Bold));

  // change the heading of column 0 (default is empty)
  QdbtSection *section0=tabular->section(0);
  section0->setText("Name");

  // change the heading and alignment of column 1
  QdbtSection *section1=tabular->section(1);
  section1->setText("Size");
  section1->setAlignment(AlignRight);

  int i;
  for (i=0;i<10;i++)
  {
    QString name,size;
    name.sprintf("%s%d\n",i<5 ? "folder" : "file",i<5 ? i : i-5);
    size.sprintf("%d",i*1024+1024);
    QdbtTableCell cell;

    // define a cell `cell'
    cell.setText(name);
    cell.setPixmap(i<5 ? &folder : &file);
    cell.setEditable(FALSE);
    cell.setAlignment(AlignLeft);
    cell.setPixmapAlignment(AlignLeft);
    cell.setColor(i<5 ? blue : black);
    // let cell (i,0) be a (deep) copy of `cell' 
    tabular->changeCell(&cell,i,0);

    // modify the cell `cell'
    if (i<5) cell.setText(0); else cell.setText(size);
    cell.setPixmap(0);
    cell.setEditable(TRUE);
    cell.setAlignment(AlignRight);
    // let cell (i,1) be a (deep) copy of `cell'
    tabular->changeCell(&cell,i,1);
  }

  // Set the width of the columns, so all cells fit
  tabular->setColumnWidth(0,tabular->columnWidthHint(0));
  tabular->setColumnWidth(1,tabular->columnWidthHint(1));
  
  // Get the height of the default font (can be changed with the -fn option)
  int fh=fontMetrics().height();

  // Create the input widgets
  QLabel *nameLab  = new QLabel("Name",this);
  QLabel *sizeLab  = new QLabel("Size",this);
  nameEdit = new QLineEdit(this);
  sizeEdit = new QLineEdit(this);
  type     = new QComboBox(this);

  // Set their minimum sizes
  nameLab->setMinimumSize(nameLab->sizeHint());
  sizeLab->setMinimumSize(sizeLab->sizeHint());
  type->insertItem("folder");
  type->insertItem("file");
  type->setMinimumSize(type->sizeHint());
  nameEdit->setMinimumSize(50,fh+6); // height depends on the font that is used
  sizeEdit->setMinimumSize(50,fh+6);

  // Create the control button
  QPushButton *add    = new QPushButton("Add",this);
  QPushButton *addCol = new QPushButton("AddCol",this);
  QPushButton *del    = new QPushButton("Del",this);
  QPushButton *delCol = new QPushButton("DelCol",this);
  QPushButton *clear  = new QPushButton("Clear",this);
  QPushButton *header = new QPushButton("Header",this);
  QPushButton *close  = new QPushButton("Close",this);

  // Set their minimum sizes
  add   ->setMinimumSize(add->sizeHint());
  del   ->setMinimumSize(del->sizeHint());
  addCol->setMinimumSize(addCol->sizeHint());
  delCol->setMinimumSize(delCol->sizeHint());
  close ->setMinimumSize(close->sizeHint());
  clear ->setMinimumSize(clear->sizeHint());
  header->setMinimumSize(header->sizeHint());
  header->setToggleButton(TRUE); // make the header button a toggle button
  header->setOn(TRUE);           // and turn it on

  // Add Widgets and layouts to the layout-manager
  layout->addWidget(tabular,0,0);
  layout->addLayout(input  ,1,0);
  layout->addLayout(buttons,2,0);
  layout->setColStretch(0,1); // make the table strechable
  layout->setRowStretch(0,1);

  // Add Widgets to the button layout
  buttons->addStretch(1);
  buttons->addWidget(add);
  buttons->addWidget(del);
  buttons->addWidget(addCol);
  buttons->addWidget(delCol);
  buttons->addWidget(clear);
  buttons->addWidget(header);
  buttons->addWidget(close);

  // Add Widgets to the input layout
  input->addWidget(type,0,4);
  input->addWidget(nameLab,0,0);
  input->addWidget(nameEdit,0,1);
  input->setColStretch(1,1); // make the edit field horizontal stretchable 
  input->addWidget(sizeLab,0,2);
  input->addWidget(sizeEdit,0,3);
  input->setColStretch(3,1);
 
  // don't forget to activate the top layout manager
  layout->activate();

  // Let the close button quit the application
  connect(close,  SIGNAL(clicked()),qApp,SLOT(quit()));
  
  // Let the toggle button hide/show the header
  connect(header, SIGNAL(toggled(bool)),SLOT(toggleHeader(bool)));
 
  // Connect the add button to the addRow() function
  connect(add,    SIGNAL(clicked()),SLOT(addRow()));

  // Connect the del button to the delRow() function
  connect(del,    SIGNAL(clicked()),SLOT(delRows()));
  
  // Connect the add column button to the addCol() function
  connect(addCol, SIGNAL(clicked()),SLOT(addCol()));

  // Connect the del column button to the delCol() function
  connect(delCol, SIGNAL(clicked()),SLOT(delCol()));
  
  // Connect the clean button to the clear() function
  connect(clear,  SIGNAL(clicked()),SLOT(clear()));
  
  // Resize the widget to its minimal size
  resize(layout->mainWidget()->sizeHint());
}

// destructor
Example::~Example()
{
}

// hide/show table's header
void Example::toggleHeader(bool state)
{
  if (state) tabular->showHeader(); else tabular->hideHeader();
}

// add a row to the table
void Example::addRow()
{
  if (nameEdit->text() && strlen(nameEdit->text())>0) // text field not empty
  {
    int nRows=tabular->numRows();
    int nCols=tabular->numCols();
    tabular->insertRow(nRows);     // add a new row at the end of the table

    if (type->currentItem()==0)    // type==folder
    {
      if (nCols>0) tabular->changeCell(nameEdit->text(),&folder,nRows,0,blue,AlignLeft,TRUE);
      if (nCols>1) tabular->changeCell("",nRows,1,blue);
    } 
    else // type->currentItem()==1 // type==file
    {
      if (nCols>0) tabular->changeCell(nameEdit->text(),&file,nRows,0,black,AlignLeft,TRUE);
      if (nCols>1) tabular->changeCell(sizeEdit->text(),nRows,1,black,AlignRight);
    }
  }
}

// delete all selected rows from the table
void Example::delRows()
{
  tabular->setAutoUpdate(FALSE);  // temporarily disable updates, otherwise
                                  // update is called for each row
  int i=0;
  while (i<tabular->numRows()) // not all rows inspected
  {
    if (!tabular->rowSelected(i)) // if row not selected
      i++;                        // proceed with the next row
    else                          // otherwise
      tabular->removeRow(i);      // remove this row (will decrease numRows())
  }
  tabular->setAutoUpdate(TRUE);   // enable the update again
  tabular->repaint();             // repaint the table
}

// add a new column to the table
void Example::addCol()
{
  tabular->setAutoUpdate(FALSE);  // temporarily disable updates, otherwise
  int colIndex=tabular->numCols();
  tabular->insertCol(colIndex);
  tabular->section(colIndex)->setText("New Column");
  tabular->setColumnWidth(colIndex,tabular->columnWidthHint(colIndex));
  // modify the cells of the new column
  int i,nRows=tabular->numRows();
  for (i=0;i<nRows;i++)
  {
    QdbtTableCell cell;
    QString text;
    text.sprintf("cell (%d,%d)",i,colIndex);
    cell.setText(text);
    cell.setEditable(TRUE);
    tabular->changeCell(&cell,i,colIndex);
  }
  tabular->setAutoUpdate(TRUE);   // enable the update again
  tabular->repaint();             // repaint the table
}

// delete the last column from the table (if any)
void Example::delCol()
{
  int nCols=tabular->numCols();
  if (nCols>0)
  {
    tabular->removeCol(nCols-1);
  }
}

// clear the whole table (removes all rows and columns)
void Example::clear()
{
  tabular->clear();
}

int main(int argc,char **argv)
{
  QApplication app(argc, argv);
  Example example;
  app.setMainWidget(&example);
  example.show();
  return app.exec();  
}
