/*
    OpenGUI - Drawing & Windowing library

    Copyright (C) 1996,2000  Marian Krivos

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    nezmar@internet.alcatel.sk
    
    detect.c - detect some graphics chipset
*/

#ifdef __WATCOMC__
#include <i86.h>
#endif

#ifdef __DJGPP__
#include <pc.h>
#include <dpmi.h>
#endif

#include <stdio.h>

#include "config.h"
#include "base.h"
#include "_fastgl.h"

static const int SEQ = 0x3C4;
static int old, old2, SubVers;
static unsigned long buf[16], pci_found;

static int rdinx(short pt, short inx)	//       {read register PT index INX}
 {
	short x;

	if (pt == 0x3C0)
	{
		x = inp(CRTC + 6);
	}							//    {If Attribute Register then reset Flip-Flop}

	outp(pt, inx);
	return inp(pt + 1);
}

static void wrinx(short pt, short inx, short val)	//    {write VAL to register PT index INX}
 {
	short x;

	if (pt == 0x3C0)
	{
		x = inp(CRTC + 6);
		outp(pt, inx);
		outp(pt, val);
	}
	else
	{
		outp(pt, inx);
		outp(pt + 1, val);
	}
}

//  Returns true if the bits in MSK
//  of register PT index RG are
//  read/writable

static int testinx2(short pt, short rg, short msk)
{
	short old, nw1, nw2;

	old = rdinx(pt, rg);
	wrinx(pt, rg, old & (~msk));
	nw1 = rdinx(pt, rg) & msk;
	wrinx(pt, rg, old | msk);
	nw2 = rdinx(pt, rg) & msk;
	wrinx(pt, rg, old);
	return ((nw1 == 0) && (nw2 == msk));
}

// Returns true if the bits in MSK
// of register PT are read/writable}

static int tstrg(short pt, short msk)
{
	short old, nw1, nw2;

	old = inp(pt);
	outp(pt, old & ~msk);
	nw1 = inp(pt) & msk;
	outp(pt, old | msk);
	nw2 = inp(pt) & msk;
	outp(pt, old);
	return ((nw1 == 0) && (nw2 == msk));
}

// Returns true if all bits of
// register PT index RG are
// read/writable.}
static int testinx(short pt, short rg)
{
	return testinx2(pt, rg, 0xff);
}

static void modinx(unsigned pt, unsigned inx, unsigned mask, unsigned nwv)
{
	unsigned temp;

	temp = ((rdinx(pt, inx) & (~mask)) + (nwv & mask));
	wrinx(pt, inx, temp);
}

static void setinx(unsigned pt, unsigned inx, unsigned val)
{
	unsigned x;

	x = rdinx(pt, inx);
	wrinx(pt, inx, x | val);
}

#define PCI_CONF_ADDR  0xcf8
#define PCI_CONF_DATA  0xcfc


static int pci_read_header (unsigned char bus, unsigned char device,
        unsigned char fn, unsigned long *buf)
{
  int i;
  unsigned long bx = ((fn&7)<<8) | ((device&31)<<11) | (bus<<16) |
                        0x80000000;

  for (i=0; i<16; i++) {
        outpl (PCI_CONF_ADDR, bx|(i<<2));
        buf[i] = inpl (PCI_CONF_DATA);
  }

  return 0;
}


/* find a vga device of the specified vendor, and return
   its configuration (16 dwords) in conf 
   return zero if device found.
   */ 
int pci_find_vendor_vga(unsigned long *conf)
{ unsigned long buf[16];
  int bus,device,cont;
  
  cont=1;
  pci_found = 0;
#ifdef __linux__
  if (getenv("IOPERM") == 0) {
        if (iopl(3) < 0) {
	    printf("svgalib: vgapci: cannot get I/O permissions\n");
	    exit(1);
	}
  }
#endif
  for(bus=0;(bus<16)&&cont;bus++)              
    for(device=0;(device<256)&&cont;device++)
	{
      pci_read_header(bus,device,0,buf);
      if ((((buf[2]>>16)&0xffff)==0x0300)) cont=0;
    }
  if (!cont)
  {
  	memcpy(conf,buf,sizeof(buf));
	pci_found = buf[0];
  }
#ifdef __linux__
  if (getenv("IOPERM") == 0)
  	iopl(0);
#endif
  return cont;
}

//  (* First test for Cirrus 54xx *)
static int detect_cirrus(void)
{
	if ((pci_found&0xFFFF) != 0x1013) // no pci detected
	{
	old = rdinx(0x3C4, 6);
	wrinx(0x3C4, 6, 0);
	if (rdinx(0x3C4, 6) == 15)
	{
		wrinx(0x3c4, 6, 0x12);
		if ((rdinx(0x3C4, 6) == 0x12) && testinx2(0x3C4, 0x1E, 0x3F))
		{
			SubVers = rdinx(0x3d4, 0x27);
			if (testinx(0x3CE, 9))
				return 1;		// cirrus CL-54xx
			else if (testinx(0x3C4, 0x19))
				return 2;		// CL-GD62xx
			else
				return 3;		//name:='Cirrus AVGA2 (5402)';
		}
	}
	else
		wrinx(0x3C4, 6, old);

//  (* Now test for 64xx *)

	old = rdinx(0x3CE, 0xA);
	wrinx(0x3CE, 0xA, 0xCE);
	if (rdinx(0x3CE, 0xA) == 0)
	{
		wrinx(0x3CE, 0xA, 0xEC);
		if (rdinx(0x3CE, 0xA) == 1)
		{
			SubVers = rdinx(0x3CE, 0xAA);
			return 4;			// 'Cirrus CL-GD64xx';

		}
	}
	wrinx(0x3CE, 0xA, old);
	return 0;
	}
	return (buf[0]>>16) & 0xffffU;
}

static int detect_trident(void)
{
	short old, val;

	if ((pci_found&0xFFFF) != 0x1023) // no pci detected
	{
		wrinx(SEQ, 0xB, 0);
		SubVers = inp(SEQ + 1);
		old = rdinx(SEQ, 0xE);
		outp(SEQ + 1, 0);
		val = inp(SEQ + 1);
		outp(SEQ + 1, old);
		if ((val & 15) == 2)
		{
			outp(0x3c5, old ^ 2);	// (* Trident should restore bit 1 reversed *)
			return 1;
		}
		return 0;
	}
	return (buf[0]>>16) & 0xffffU;
}

static int detect_tseng(void)
{
	short old2;

	old = rdinx(0x3BF, 3);
	old2 = rdinx(0x3D8, 0xA0);
	outp(0x3BF, 3);
	outp(0x3D8, 0xA0);			//  {Enable ET4000 extensions}

	if (tstrg(0x3CD, 0x3F))
		if (testinx2(CRTC, 0x33, 0xF))
			return 4000;
		else
			return 3000;
	wrinx(0x3d8, 0xA0, old2);
	wrinx(0x3bf, 3, old);
	return 0;
}

static int detect_s3(void)
{
	if ((pci_found&0xFFFF) != 0x5333) // no pci detected
	{
		old = rdinx(CRTC, 0x38);
		wrinx(CRTC, 0x38, 0);		//   {disable extensions}

		if (!testinx2(CRTC, 0x35, 0xF))
		{
			wrinx(CRTC, 0x38, 0x48);
			if (testinx2(CRTC, 0x35, 0xF))
			{
				wrinx(CRTC, 0x38, old);
				return rdinx(CRTC, 0x30);
			}
		}
		wrinx(CRTC, 0x38, old);
		return 0;
	}
	return (buf[0]>>16) & 0xffffU;
}

static int detect_chips(void)
{
#ifdef __MSDOS__
	__dpmi_regs d;

	d.x.ax = 0x5f00;
	__dpmi_int(0x10, &d);
	if (d.h.al == 0x5f)
		return d.h.bl;
#endif
	return 0;
}

static int detect_WesternDigital(void)
{
	SubVers = 0;
	old = rdinx(0x3CE, 15);
	setinx(0x3CE, 15, 0x17);	//  {Lock registers}

	if (!testinx2(0x3CE, 9, 0x7F))
	{
		modinx(0x3CE, 0xF, 0x17, 5);	//   {Unlock again}

		if (testinx2(0x3CE, 9, 0x7F))
		{
			old2 = rdinx(CRTC, 0x29);
			modinx(CRTC, 0x29, 0x8F, 0x85);		// {Unlock WD90Cxx registers}

			if (testinx(CRTC, 0x2B))
				SubVers = 1;	// 'Paradise PVGA1A'

			else
			{
				wrinx(0x3C4, 6, 0x48);
				if (!testinx2(0x3C4, 7, 0xF0))
					SubVers = 2;	// 'Western Digital WD90C00'

				else if (!testinx(0x3C4, 0x10))
				{
					if (testinx2(CRTC, 0x31, 0x68))
						SubVers = 3;	// 'Western Digital WD90C22'

					else if (testinx2(CRTC, 0x31, 0x90))
						SubVers = 4;	// 'Western Digital WD90C20A'

					else
						SubVers = 5;	// 'Western Digital WD90C20';

					wrinx(0x3d4, 0x34, 0xA6);
					if ((rdinx(0x3d4, 0x32) & 0x20) != 0)
						wrinx(0x3d4, 0x34, 0);
				}
				else if (testinx2(0x3C4, 0x14, 0xF))
				{
					SubVers = (rdinx(CRTC, 0x36) << 8) + rdinx(CRTC, 0x37);
					switch (SubVers)
					{
						case 0x3234:	//'Western Digital WD90C24'

						case 0x3236:	//'Western Digital WD90C26'

						case 0x3330:	//'Western Digital WD90C30'

						case 0x3331:	//'Western Digital WD90C31'

						case 0x3333:	//'Western Digital WD90C33'

							break;
						default:
							SubVers = -1;
							break;
					}
				}
				else if (!testinx2(0x3C4, 0x10, 4))
					SubVers = 5;	// 'Western Digital WD90C10'

				else
					SubVers = 6;	//'Western Digital WD90C11';

			}
			wrinx(0x3d4, 0x29, old2);
		}
	}
	wrinx(0x3CE, 0xF, old);
	return SubVers;
}

static int detect_nvidia(void)
{
	if ((pci_found&0xFFFF) != 0x12d2 && (pci_found&0xFFFF) != 0x10de) return 0;
	if (((buf[0]>>16)&0xffff)<0x18) return 0;
 	if (((buf[0]>>16)&0xffff)>0x2F)
		if (((buf[0]>>16)&0xffff)!=0xA0) return 0;
	return buf[0]>>16;
}

static int detect_matrox(void)
{
	if ((pci_found&0xFFFF) != 0x102b) return 0;
	return buf[0]>>16;
}

static int detect_banshee(void)
{
	if ((pci_found&0xFFFF) != 0x121A) return 0;
	if (((buf[0]>>16)&0xffff)<3) return 0; // banshee and better
	return buf[0]>>16;
}

static int detect_rendition(void)
{
	if ((pci_found&0xFFFF) != 0x1163) return 0;
	if (((buf[0]>>16)&0xffff) != 0x2000) return 0;
	return buf[0]>>16;
}

static int detect_permedia(void)
{
	if ((pci_found&0xFFFF) != 0x3d3d) return 0;
	return buf[0]>>16;
}

static int detect_intel(void)
{
	if ((pci_found&0xFFFF) != 0x8086) return 0;
	if (((buf[0]>>16)&0xffff)==0x7800) return 0x7800;
	if (((buf[0]>>16)&0xffff)==0x7121) return 0x7121;
	if (((buf[0]>>16)&0xffff)==0x7123) return 0x7125;
	if (((buf[0]>>16)&0xffff)==0x7125) return 0x7125;
	return 0;
}

int detect_video(int v)
{
	int val, rc = 0;

	if (inp(0x3CC) & 1)
		CRTC = 0x3D4;
	else
		CRTC = 0x3B4;

	pci_find_vendor_vga(buf);
	
	val = detect_intel();
	if (val==0x7800)
		rc = FG_INTEL740;
	else if (val==0x7121)
		rc = FG_INTEL810;
	if (v)
		printf("Testing for INTEL        : %x\n", val);
	if (rc) return rc;

	val = detect_s3();
	if (val)
	{
		if (val >= 225)
			rc = FG_S3V2;
		else
			rc = FG_S3;
	}
	if (v)
		printf("Testing for S3           : %x\n", val);
	if (rc) return rc;

	val = detect_nvidia();
	if (val)
	{
		rc = FG_NVIDIA;
		wrinx(CRTC, 0x6, 0x57); // unlock card
		if (val>0x18)
		{
			modinx(CRTC, 0x11, 0x80, 0); // TNT and above
			wrinx(CRTC, 0x1f, 0x57); // unlock card
		}
	}
	if (v)
		printf("Testing for nVidia       : %x\n", val);
	if (rc) return rc;

	val = detect_matrox();
	if (val)
		rc = FG_MATROX;
	if (v)
		printf("Testing for MATROX       : %x\n", val);
	if (rc) return rc;

	val = detect_permedia();
	if (val)
		rc = FG_PERMEDIA;
	if (v)
		printf("Testing for PERMEDIA     : %x\n", val);
	if (rc) return rc;

	val = detect_banshee();
	if (val)
		rc = FG_BANSHEE;
	if (v)
		printf("Testing for 3DFX VOODOO  : %x\n", val);
	if (rc) return rc;

	val = detect_rendition();
	if (val)
		rc = FG_RENDITION;
	if (v)
		printf("Testing for RENDITION    : %x\n", val);
	if (rc) return rc;

	val = detect_cirrus();
	if (val)
		rc = FG_CIRRUS;
	if (v)
		printf("Testing for CIRRUS LOGIC : %x\n", val);
	if (rc) return rc;

	val = detect_trident();
	if (val)
		rc = FG_TRIDENT;
	if (v)
		printf("Testing for TRIDENT      : %d\n", val);
	if (rc) return rc;

	val = detect_tseng();
	if (val)
	{
		if ((val = 3000) != 0)
			rc = FG_TSENG3;
		else
			rc = FG_TSENG4;
	}
	if (v)
		printf("Testing for TSENG        : %d\n", val);
	if (rc) return rc;

	val = detect_chips();
	if (val)
		rc = FG_CHIPS;
	if (v)
		printf("Testing for CHIPS&TECH   : %d\n", val);
	if (rc) return rc;

	val = detect_WesternDigital();
	if (val)
		rc = FG_WDIGITAL;
	if (v)
		printf("Testing for WESTERN DIGIT: %d\n", val);
	if (rc) return rc;

	return rc=FG_VESA;
}

