/*
//file contains data structures and correcponding functions for message passing
//between the kernel driver and user level process
*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# ifndef __USERDEV_COMMOH_H__
# define __USERDEV_COMMON_H__

/*operation codes*/
# define OP_READ                0
# define OP_WRITE               1
# define OP_POLL                2
# define OP_IOCTL               3
# define OP_OPEN                4
# define OP_FLUSH               5
# define OP_CLOSE               6
# define OP_FSYNC               7
# define OP_FASYNC              8
# define OP_CHECK_MEDIA_CHANGE  9
# define OP_REVALIDATE          10
# define OP_MEDIACTL		11
# define OP_MESSAGE		12
# define OP_REQUEST		13

/*type of message*/
# define SIGNAL_PENDING		1

# define SI_INTERRUPT	0x88	/*signal code to denote that the signal originated due to an interrupt*/

# ifdef __KERNEL__
	# define ALLOCATE(size) (kmalloc((size),GFP_KERNEL))
# else
	# define ALLOCATE(size) (malloc((size)))
# endif

/*structure to specify ioctls accepted by driver process*/
struct ioctl_elem {
	int num;	/*ioctl number*/
	int max_length;	/*maximum accepted data field (either GET/SET)*/
};

/*structure to specify driver process attributes to attach with kernel driver*/
struct attach_record {
	int readfd;			/*file descriptor of pipe to read from*/
	int writefd;			/*file descriptor of pipe to write to*/
	char *devicename;		/*name of the device for which the driver operates*/
	int namelen;			/*length of name*/
	short func_mask;		/*mask denoting the list of functions supported*/
	int num_ioctls;			/*number of ioctls supported*/
	struct ioctl_elem *ioctls;	/*array of ioctl elements*/
	void *data;			/*optional data, used to initialize block device arrays*/
};

/*optional data passed for block devices, block device array values*/
struct blkdev_data {
	int devsize;		/*device size, in kilobytes*/
	int blksize;		/*block size in bytes*/
	int sectsize;		/*sector size in bytes*/
	int maxreadahead;	/*maximum read ahead, in number of pages*/
	int maxsectors;		/*maximum number of sectors in any request*/
};

/*structure to specify attributes for process detach from kernel driver*/
struct detach_record {
	int id;		/*userdev id of the process*/
};

/*structure to specify attributes for requesting the kernel driver to listen
 *for interrupts on irqno and deliver signal signo*/
struct irq_request_record {
	int id;		/*userdev id of the process*/
	int signo;	/*signal number*/
	int irqno;	/*irq number*/
};

/*structure to specify attributes for freeing irq request*/
struct irq_free_record {
	int id;		/*userdev id of the process*/
};

/*structure to specify attributes for requesting a dma channel from kernel*/
struct dma_request_record {
	int id;
	int channel;
};

/*structure to specify attributes for freeing a dma channel*/
struct dma_free_record {
	int id;
};

/*structure to specify attributes for starting dma operation*/
struct dma_start_record {
	int id;
	int mode;
	int count;
	void *buf;
};

/*structure to specify attributes for checking dam operation status*/
struct dma_check_record {
	int id;
};

/*structure to copy data from dma_buffer to user space*/
struct dma_copy_record {
	int id;
	char *buf;
};

/*message packet header structure, sent prior to data packet*/
struct header {
	int opcode;		/*operation code*/
	int request_id;		/*request identifier used to identify response to awaiting request*/
	long packet_size;	/*size of the data packet to follow*/
};

/*data types used in various data packets*/
typedef struct dat {	/*String data type with length and actual data*/
	int len;
	char *elements;
} DATA;			/*incase elements is NULL, we have len=0*/
typedef unsigned int SIZE;
typedef long long OFFSET;

/*function to copy bytes from source to destination*/
void mcopy(void* dst,void* src,int num)
{
# ifdef __KERNEL__
	int i;
	for(i=0;i<num;i++)
		((char*)dst)[i]=((char*)src)[i];
# else
	memcpy(dst,src,num);
# endif
}

/*function to set bytes in destination to a particular value*/
void mset(void* dst,int val,int num)
{
# ifdef __KERNEL__
   	int i;
   	for(i=0;i<num;i++)
		((char*)dst)[i]=(char)val;
# else
   	memset(dst,val,num);
# endif
}    


/*
//Following are the data packets for each of the file operations of the driver
//the sender fills in data in the request packet, converts it to a string and sends it across
//the receiver retreives the packet from the received string and does the needed
//the vice versa is done with the response packet
//
//The functions that convert a structure to string take in the packet header as
//for ensuring atomic write we need a single string for header+packet and by
//passing the header to these functions a copying step is reduced.
//
//Atomic reading is not required the header and packet can be read separately
//as we have only a single thread doing this on either side, both kernel and
//user sides. While we have many writing threads on both sides forcing to have
//atomic writes
*/
/*--------Data Structures and Conversion functions for READ--------*/
/*request packet*/
struct read_send {
	SIZE size;	/*amount to read*/
	OFFSET offset;	/*offset to read from*/
	int flags;	/*file flags*/
};

/*response packet*/
struct read_recv {
	OFFSET offset;	/*new offset returned*/
	DATA data;	/*data read*/
};

/*function to package request arguments into a string*/
int read_send_to_char(struct header *head, struct read_send* rs, char**pstr)
{
	char *str;
	int h = sizeof(struct header);
	head->packet_size = sizeof(struct read_send);
	str = (char*)ALLOCATE(h+sizeof(struct read_send));
	mcopy(str,head,h);
	mcopy(str+h,rs,sizeof(struct read_send));
	*pstr = str;
	return h+sizeof(struct read_send);
}

/*function to retreive request packet from the given string*/
struct read_send* read_send_from_char(char* str)
{
	struct read_send *rs;
	rs = (struct read_send*)ALLOCATE(sizeof(struct read_send));
	mcopy(rs,str,sizeof(struct read_send));
	return rs;
}

/*function to package response arguments into a string*/
int read_recv_to_char(struct header *head, struct read_recv* rr, char**pstr)
{
	char *str;
	int l,len,ret,h;

	ret = rr->data.len;
	if (ret < 0) ret = 0;

	len = sizeof(OFFSET)+sizeof(int)+(ret*sizeof(char));
	h = sizeof(struct header);
	head->packet_size = len;

	str = (char*)ALLOCATE(h+len);
	mcopy(str,head,h);
	l = sizeof(OFFSET)+sizeof(int);
	mcopy(str+h,rr,l);
	mcopy(str+h+l,rr->data.elements,ret);
	*pstr = str;
	return h+len;
}

/*function to retreive response packet from the given string*/
struct read_recv* read_recv_from_char(char* str)
{
	struct read_recv *rr;
	int l,ret;
	rr = (struct read_recv*)ALLOCATE(sizeof(struct read_recv));
	l = sizeof(OFFSET)+sizeof(int);
	mcopy(rr,str,l);
	ret = rr->data.len;
	if (ret < 0) ret = 0;
	rr->data.elements = (ret == 0)?NULL:str+l;
	return rr;
}


/*--------Data Structures and Conversion functions for WRITE--------*/
/*request packet*/
struct write_send {
	OFFSET offset;	/*offset at which to write*/
	int flags;	/*file flags*/
	DATA data;	/*data to write along with length*/
};

/*response packet*/
struct write_recv {
	OFFSET offset;		/*new offset returned*/
	SIZE num_written;	/*number of bytes written*/
};

/*function to package request arguments into a string*/
int write_send_to_char(struct header *head, struct write_send* ws, char **pstr)
{
	char *str;
	int l;
	int h = sizeof(struct header);
	if(ws->data.len < 0) ws->data.len = 0;
	l = h+sizeof(OFFSET)+sizeof(int)*2+(ws->data.len*sizeof(char));
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	l = sizeof(OFFSET)+sizeof(int)*2;
	mcopy(str+h,ws,l);
	mcopy(str+h+l,ws->data.elements,ws->data.len);
	*pstr = str;
	return h+l+ws->data.len;
}

/*function to retreive request packet from the given string*/
struct write_send* write_send_from_char(char* str)
{
	struct write_send* ws;
	int l,s;
	ws = (struct write_send*)ALLOCATE(sizeof(struct write_send));
	l = sizeof(OFFSET)+sizeof(int)*2;
	mcopy(ws,str,l);
	s = ws->data.len;
	if (s < 0) s = 0;
	ws->data.elements = (s == 0)?NULL:str+l;
	return ws;
}

/*function to package response arguments into a string*/
int write_recv_to_char(struct header *head, struct write_recv* wr, char **pstr)
{
	char *str;
	int l;
	int h = sizeof(struct header);
	l = sizeof(struct write_recv);
	str=(char*)ALLOCATE(l+h);
	head->packet_size = l;
	mcopy(str,head,h);
	mcopy(str+h,wr,l);
	*pstr=str;
	return l+h;
}

/*function to retreive response packet from the given string*/
struct write_recv* write_recv_from_char(char* str)
{
	struct write_recv *wr;
	wr = (struct write_recv*)ALLOCATE(sizeof(struct write_recv));
	mcopy(wr,str,sizeof(struct write_recv));
	return wr;
}


/*--------Data Structures and Conversion functions for POLL--------*/
/*request packet*/
struct poll_send {
};

/*response packet*/
struct poll_recv {
	int poll_result;	/*result of polling*/
};

/*function to package request arguments into a string*/
int poll_send_to_char(struct header *head, struct poll_send* ps, char**pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct poll_send));
	head->packet_size = sizeof(struct poll_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,ps,sizeof(struct poll_send));
	return h+sizeof(struct poll_send);
}

/*function to retreive request packet from the given string*/
struct poll_send* poll_send_from_char(char* str)
{
	struct poll_send *ps;
	ps = (struct poll_send*)ALLOCATE(sizeof(struct poll_send));
	mcopy(ps,str,sizeof(struct poll_send));
	return ps;
}

/*function to package response arguments into a string*/
int poll_recv_to_char(struct header *head, struct poll_recv* pr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct poll_recv));
	head->packet_size = sizeof(struct poll_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,pr,sizeof(struct poll_recv));
	return h+sizeof(struct poll_recv);
}

/*function to retreive response packet from the given string*/
struct poll_recv* poll_recv_from_char(char* str)
{
	struct poll_recv *pr;
	pr = (struct poll_recv*)ALLOCATE(sizeof(struct poll_recv));
	mcopy(pr,str,sizeof(struct poll_recv));
	return pr;
}


/*--------Data Structures and Conversion functions for IOCTL--------*/
/*request packet*/
struct ioctl_send {
	int command;	/*ioctl command */
	DATA data;	/*data sent for SET operation*/
};

/*response packet*/
struct ioctl_recv {
	int result;	/*result of ioctl operation*/
	DATA data;	/*data obtained incase of a GET operation*/
};

/*function to package request arguments into a string*/
int ioctl_send_to_char(struct header *head, struct ioctl_send* is, char** pstr)
{
	char *str;
	int l,h;
	if(is->data.len < 0) is->data.len = 0;
	h = sizeof(struct header);
	l = h+sizeof(int)*2 + is->data.len*sizeof(char);
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	mcopy(str+h,is,sizeof(int)*2);
	mcopy(str+h+sizeof(int)*2,is->data.elements,is->data.len);
	*pstr=str;
	return l;
}

/*function to retreive request packet from the given string*/
struct ioctl_send* ioctl_send_from_char(char* str)
{
	struct ioctl_send *is;
	int s;
	is = (struct ioctl_send*)ALLOCATE(sizeof(struct ioctl_send));
	mcopy(is,str,sizeof(int)*2);
	s = is->data.len;
	if(s < 0) s = 0;
	is->data.elements=(s == 0)?NULL:str+sizeof(int)*2;
	return is;
}

/*function to package response arguments into a string*/
int ioctl_recv_to_char(struct header *head, struct ioctl_recv* ir, char** pstr)
{
	char *str;
	int l,h,s;
	s = ir->data.len;
	if(s < 0) s = 0;
	h = sizeof(struct header);
	l = h+sizeof(int)*2 + s*sizeof(char);
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	mcopy(str+h,ir,sizeof(int)*2);
	mcopy(str+h+sizeof(int)*2,ir->data.elements,s);
	*pstr=str;
	return l;
}

/*function to retreive response packet from the given string*/
struct ioctl_recv* ioctl_recv_from_char(char* str)
{
	struct ioctl_recv *ir;
	int s;
	ir = (struct ioctl_recv*)ALLOCATE(sizeof(struct ioctl_recv));
	mcopy(ir,str,sizeof(int)*2);
	s = ir->data.len;
	if (s < 0) s = 0;
	ir->data.elements=(s == 0)?NULL:str+sizeof(int)*2;
	return ir;
}


/*--------Data Structures and Conversion functions for OPEN--------*/
/*request packet*/
struct open_send {
	unsigned int flags;	/*open flags in file structure*/
	mode_t mode;		/*mode of file open*/
};

/*response packet*/
struct open_recv {
	int result;	/*result of open operation*/
};

/*function to package request arguments into a string*/
int open_send_to_char(struct header *head, struct open_send* os, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct open_send));
	head->packet_size = sizeof(struct open_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,os,sizeof(struct open_send));
	return h+sizeof(struct open_send);
}

/*function to retreive request packet from the given string*/
struct open_send* open_send_from_char(char* str)
{
	struct open_send *os;
	os = (struct open_send*)ALLOCATE(sizeof(struct open_send));
	mcopy(os,str,sizeof(struct open_send));
	return os;
}

/*function to package response arguments into a string*/
int open_recv_to_char(struct header *head, struct open_recv* or, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct open_recv));
	head->packet_size = sizeof(struct open_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,or,sizeof(struct open_recv));
	return h+sizeof(struct open_recv);
}

/*function to retreive response packet from the given string*/
struct open_recv* open_recv_from_char(char* str)
{
	struct open_recv *or;
	or = (struct open_recv*)ALLOCATE(sizeof(struct open_recv));
	mcopy(or,str,sizeof(struct open_recv));
	return or;
}


/*--------Data Structures and Conversion functions for FLUSH--------*/
/*request packet*/
struct flush_send {
};

/*response packet*/
struct flush_recv {
	int result;	/*result of flush operation*/
};

/*function to package request arguments into a string*/
int flush_send_to_char(struct header *head, struct flush_send* fs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct flush_send));
	head->packet_size = sizeof(struct flush_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fs,sizeof(struct flush_send));
	return h+sizeof(struct flush_send);
}

/*function to retreive request packet from the given string*/
struct flush_send* flush_send_from_char(char* str)
{
	struct flush_send *fs;
	fs = (struct flush_send*)ALLOCATE(sizeof(struct flush_send));
	mcopy(fs,str,sizeof(struct flush_send));
	return fs;
}

/*function to package response arguments into a string*/
int flush_recv_to_char(struct header *head, struct flush_recv* fr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct flush_recv));
	head->packet_size = sizeof(struct flush_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fr,sizeof(struct flush_recv));
	return h+sizeof(struct flush_recv);
}

/*function to retreive response packet from the given string*/
struct flush_recv* flush_recv_from_char(char* str)
{
	struct flush_recv *fr;
	fr = (struct flush_recv*)ALLOCATE(sizeof(struct flush_recv));
	mcopy(fr,str,sizeof(struct flush_recv));
	return fr;
}


/*--------Data Structures and Conversion functions for CLOSE--------*/
/*request packet*/
struct close_send {
};

/*response packet*/
struct close_recv {
	int result;	/*result close operation*/
};

/*function to package request arguments into a string*/
int close_send_to_char(struct header *head, struct close_send* cs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct close_send));
	head->packet_size = sizeof(struct close_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,cs,sizeof(struct close_send));
	return h+sizeof(struct close_send);
}

/*function to retreive request packet from the given string*/
struct close_send* close_send_from_char(char* str)
{
	struct close_send *cs;
	cs = (struct close_send*)ALLOCATE(sizeof(struct close_send));
	mcopy(cs,str,sizeof(struct close_send));
	return cs;
}

/*function to package response arguments into a string*/
int close_recv_to_char(struct header *head, struct close_recv* cr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct close_recv));
	head->packet_size = sizeof(struct close_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,cr,sizeof(struct close_recv));
	return h+sizeof(struct close_recv);
}

/*function to retreive response packet from the given string*/
struct close_recv* close_recv_from_char(char* str)
{
	struct close_recv *cr;
	cr = (struct close_recv*)ALLOCATE(sizeof(struct close_recv));
	mcopy(cr,str,sizeof(struct close_recv));
	return cr;
}


/*--------Data Structures and Conversion functions for FSYNC--------*/
/*request packet*/
struct fsync_send {
	int datasync;	/*flag that denotes if meta data along with data needs to be sync*/
};

/*response packet*/
struct fsync_recv {
	int result;	/*result of fsync operation*/
};

/*function to package request arguments into a string*/
int fsync_send_to_char(struct header *head, struct fsync_send* fs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct fsync_send));
	head->packet_size = sizeof(struct fsync_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fs,sizeof(struct fsync_send));
	return h+sizeof(struct fsync_send);
}

/*function to retreive request packet from the given string*/
struct fsync_send* fsync_send_from_char(char* str)
{
	struct fsync_send *fs;
	fs = (struct fsync_send*)ALLOCATE(sizeof(struct fsync_send));
	mcopy(fs,str,sizeof(struct fsync_send));
	return fs;
}

/*function to package response arguments into a string*/
int fsync_recv_to_char(struct header *head, struct fsync_recv* fr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct fsync_recv));
	head->packet_size = sizeof(struct fsync_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fr,sizeof(struct fsync_recv));
	return h+sizeof(struct fsync_recv);
}

/*function to retreive response packet from the given string*/
struct fsync_recv* fsync_recv_from_char(char* str)
{
	struct fsync_recv *fr;
	fr = (struct fsync_recv*)ALLOCATE(sizeof(struct fsync_recv));
	mcopy(fr,str,sizeof(struct fsync_recv));
	return fr;
}


/*--------Data Structures and Conversion functions for FASYNC--------*/
/*request packet*/
struct fasync_send {
};

/*response packet*/
struct fasync_recv {
	int result;	/*result of fasync operation*/
};

/*function to package request arguments into a string*/
int fasync_send_to_char(struct header *head, struct fasync_send* fs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct fasync_send));
	head->packet_size = sizeof(struct fasync_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fs,sizeof(struct fasync_send));
	return h+sizeof(struct fasync_send);
}

/*function to retreive request packet from the given string*/
struct fasync_send* fasync_send_from_char(char* str)
{
	struct fasync_send *fs;
	fs = (struct fasync_send*)ALLOCATE(sizeof(struct fasync_send));
	mcopy(fs,str,sizeof(struct fasync_send));
	return fs;
}

/*function to package response arguments into a string*/
int fasync_recv_to_char(struct header *head, struct fasync_recv* fr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct fasync_recv));
	head->packet_size = sizeof(struct fasync_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,fr,sizeof(struct fasync_recv));
	return h+sizeof(struct fasync_recv);
}

/*function to retreive response packet from the given string*/
struct fasync_recv* fasync_recv_from_char(char* str)
{
	struct fasync_recv *fr;
	fr = (struct fasync_recv*)ALLOCATE(sizeof(struct fasync_recv));
	mcopy(fr,str,sizeof(struct fasync_recv));
	return fr;
}


/*--------Data Structures and Conversion functions for CHECK_MEDIA_CHANGE--------*/
/*request packet*/
struct check_media_change_send {
};

/*response packet*/
struct check_media_change_recv {
	int flag;	/*flag denoting result of media change check*/
};

/*function to package request arguments into a string*/
int check_media_change_send_to_char(struct header *head, struct check_media_change_send* cs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct check_media_change_send));
	head->packet_size = sizeof(struct check_media_change_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,cs,sizeof(struct check_media_change_send));
	return h+sizeof(struct check_media_change_send);
}

/*function to retreive request packet from the given string*/
struct check_media_change_send* check_media_change_send_from_char(char* str)
{
	struct check_media_change_send *cs;
	cs = (struct check_media_change_send*)ALLOCATE(sizeof(struct check_media_change_send));
	mcopy(cs,str,sizeof(struct check_media_change_send));
	return cs;
}

/*function to package response arguments into a string*/
int check_media_change_recv_to_char(struct header *head, struct check_media_change_recv* cr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct check_media_change_recv));
	head->packet_size = sizeof(struct check_media_change_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,cr,sizeof(struct check_media_change_recv));
	return h+sizeof(struct check_media_change_recv);
}

/*function to retreive response packet from the given string*/
struct check_media_change_recv* check_media_change_recv_from_char(char* str)
{
	struct check_media_change_recv *cr;
	cr = (struct check_media_change_recv*)ALLOCATE(sizeof(struct check_media_change_recv));
	mcopy(cr,str,sizeof(struct check_media_change_recv));
	return cr;
}


/*--------Data Structures and Conversion functions for REVALIDATE--------*/
/*request packet*/
struct revalidate_send {
};

/*response packet*/
struct revalidate_recv {
	int result;	/*result of media re-validation*/
};

/*function to package request arguments into a string*/
int revalidate_send_to_char(struct header *head, struct revalidate_send* rs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct revalidate_send));
	head->packet_size = sizeof(struct revalidate_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,rs,sizeof(struct revalidate_send));
	return h+sizeof(struct revalidate_send);
}

/*function to retreive request packet from the given string*/
struct revalidate_send* revalidate_send_from_char(char* str)
{
	struct revalidate_send *rs;
	rs = (struct revalidate_send*)ALLOCATE(sizeof(struct revalidate_send));
	mcopy(rs,str,sizeof(struct revalidate_send));
	return rs;
}

/*function to package response arguments into a string*/
int revalidate_recv_to_char(struct header *head, struct revalidate_recv* rr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct revalidate_recv));
	head->packet_size = sizeof(struct revalidate_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,rr,sizeof(struct revalidate_recv));
	return h+sizeof(struct revalidate_recv);
}

/*function to retreive response packet from the given string*/
struct revalidate_recv* revalidate_recv_from_char(char* str)
{
	struct revalidate_recv *rr;
	rr = (struct revalidate_recv*)ALLOCATE(sizeof(struct revalidate_recv));
	mcopy(rr,str,sizeof(struct revalidate_recv));
	return rr;
}


/*--------Data Structures and Conversion functions for MEDIACTL--------*/
/*request packet*/
struct mediactl_send {
	int op;		/*arguments of mediactl call*/
	int optarg;
};

/*response packet*/
struct mediactl_recv {
	int result;	/*result of media re-validation*/
};

/*function to package request arguments into a string*/
int mediactl_send_to_char(struct header *head, struct mediactl_send* rs, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct mediactl_send));
	head->packet_size = sizeof(struct mediactl_send);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,rs,sizeof(struct mediactl_send));
	return h+sizeof(struct mediactl_send);
}

/*function to retreive request packet from the given string*/
struct mediactl_send* mediactl_send_from_char(char* str)
{
	struct mediactl_send *rs;
	rs = (struct mediactl_send*)ALLOCATE(sizeof(struct mediactl_send));
	mcopy(rs,str,sizeof(struct mediactl_send));
	return rs;
}

/*function to package response arguments into a string*/
int mediactl_recv_to_char(struct header *head, struct mediactl_recv* rr, char** pstr)
{
	int h = sizeof(struct header);
	*pstr = (char*)ALLOCATE(h+sizeof(struct mediactl_recv));
	head->packet_size = sizeof(struct mediactl_recv);
	mcopy(*pstr,head,h);
	mcopy(*pstr+h,rr,sizeof(struct mediactl_recv));
	return h+sizeof(struct mediactl_recv);
}

/*function to retreive response packet from the given string*/
struct mediactl_recv* mediactl_recv_from_char(char* str)
{
	struct mediactl_recv *rr;
	rr = (struct mediactl_recv*)ALLOCATE(sizeof(struct mediactl_recv));
	mcopy(rr,str,sizeof(struct mediactl_recv));
	return rr;
}


/*--------Data Structure and Conversion functions for MESSAGE--------*/
/*request packet*/
struct message_send
{
	int type;	/*type of message*/
	int reqid;	/*request id of the request that was being serviced when the message was sent*/
	int size;	/*size of data in message*/
	void *data;	/*data from message source*/
};

/*function to package request arguments into a string*/
int message_send_to_char(struct header *head, struct message_send* ms, char **pstr)
{
	char *str;
	int l,h;
	h = sizeof(struct header);
	l = h+sizeof(int)+sizeof(int)+sizeof(int)+ms->size;
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	l = sizeof(int)+sizeof(int)+sizeof(int);
	mcopy(str+h,ms,l);
	mcopy(str+h+l,ms->data,ms->size);
	*pstr = str;
	return h+l+ms->size;
}

/*function to retreive request packet from the given string*/
struct message_send* message_send_from_char(char* str)
{
	struct message_send* ms;
	int l;
	ms = (struct message_send*)ALLOCATE(sizeof(struct message_send));
	l = sizeof(int)+sizeof(int)+sizeof(int);
	mcopy(ms,str,l);
	if(ms->size < 0) ms->size = 0;
	ms->data = (ms->size == 0)?NULL:str+l;
	return ms;
}


/*--------Data Structures and Conversion functions for REQUEST--------*/
/*request packet*/
struct request_send {
	unsigned long sector;	/*sector number to operate at*/
	int command;		/*operation to perform */
	int length;		/*incase of read set to the amount expected, incase of write it is equal to data.size*/
	int clustersize;	/*set to number of sectors in the current cluster, not used if clustering is not implemented*/
	DATA data;		/*data sent for write operation*/
};

/*response packet*/
struct request_recv {
	int result;	/*result of operation*/
	DATA data;	/*data obtained incase of a read operation*/
};

/*function to package request arguments into a string*/
int request_send_to_char(struct header *head, struct request_send* rs, char** pstr)
{
	char *str;
	int l,h;
	if(rs->data.len < 0) rs->data.len = 0;
	h = sizeof(struct header);
	l = h+sizeof(unsigned long)+sizeof(int)*4 + rs->data.len*sizeof(char);
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	mcopy(str+h,rs,sizeof(unsigned long)+sizeof(int)*4);
	mcopy(str+h+sizeof(unsigned long)+sizeof(int)*4, rs->data.elements, rs->data.len);
	*pstr=str;
	return l;
}

/*function to retreive request packet from the given string*/
struct request_send* request_send_from_char(char* str)
{
	struct request_send *rs;
	int s;
	rs = (struct request_send*)ALLOCATE(sizeof(struct request_send));
	mcopy(rs,str,sizeof(unsigned long)+sizeof(int)*4);
	s = rs->data.len;
	if(s < 0) s = 0;
	rs->data.elements=(s == 0)?NULL:str+sizeof(unsigned long)+sizeof(int)*4;
	return rs;
}

/*function to package response arguments into a string*/
int request_recv_to_char(struct header *head, struct request_recv* rr, char** pstr)
{
	char *str;
	int l,h,s;
	s = rr->data.len;
	if(s < 0) s = 0;
	h = sizeof(struct header);
	l = h+sizeof(int)*2 + s*sizeof(char);
	str = (char*)ALLOCATE(l);
	head->packet_size = l-h;
	mcopy(str,head,h);
	mcopy(str+h,rr,sizeof(int)*2);
	mcopy(str+h+sizeof(int)*2,rr->data.elements,s);
	*pstr=str;
	return l;
}

/*function to retreive response packet from the given string*/
struct request_recv* request_recv_from_char(char* str)
{
	struct request_recv *rr;
	int s;
	rr = (struct request_recv*)ALLOCATE(sizeof(struct request_recv));
	mcopy(rr,str,sizeof(int)*2);
	s = rr->data.len;
	if (s < 0) s = 0;
	rr->data.elements=(s == 0)?NULL:str+sizeof(int)*2;
	return rr;
}

# endif     /*__USERDEV_COMMON_H__*/
