/*file contains implementation of user level ramdisk device driver*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <signal.h>
# include <sys/types.h>
# include "usrdriv.h"
# include <linux/fs.h>
# include <linux/hdreg.h>

# define NUM_IOCTLS 0
# define DEV_NAME "/dev/uramdisk"

/*disk configuration*/
# define DISKSIZE 2*1024*1024	/*2MB disk*/
# define BLKSIZE 1024
# define SECTSIZE 512

int usrid;	/*userdev id*/
char *disk;	/*in memory disk*/

/*SIGINT signal handler to detach from userdev and end driver*/
void signal_handler(int signal)
{
	printf("Detach from userdev returned %d\n",userdev_detach(usrid));
	free(disk);
	exit(0);
}

/*function to handle ioctl requests for ramdisk*/
/*Only get disk geometry and get disk size are implemented by thr driver, others are taken care of by kernel*/
void rd_ioctl(int minor, int command, void *data, int size, int id)
{
	switch(command)
	{
		case HDIO_GETGEO:
		       {
			       struct hd_geometry geo;
			       int s;

			       s = DISKSIZE / SECTSIZE;
			       geo.cylinders = (s & ~0x3f) >> 6;
			       geo.heads = 4;	/*fake geometry*/
			       geo.sectors = 16;
			       geo.start = 4;
			       memcpy(data,&geo,sizeof(struct hd_geometry));
			       break;
		       }
		case BLKGETSIZE:
		       {
				int size = DISKSIZE / 1024;
				size = size * BLKSIZE / SECTSIZE;
				memcpy(data,&size,sizeof(int));
				break;
		       }
	}
	userdev_send_response(minor,id, IOCTL_OP, 0, size, data);
}

/*function to handle open requests for ramdisk*/
void rd_open(int minor, unsigned int flags, mode_t mode, int id)
{
	userdev_send_response(minor,id,OPEN_OP,0,0,NULL);
}

/*function to handle close requests for ramdisk*/
void rd_close(int minor, int id)
{
	userdev_send_response(minor,id,CLOSE_OP,0,0,NULL);
}

/*function to handle check media change requests for ramdisk*/
void rd_check_media_change(int minor, int id)
{
	userdev_send_response(minor,id,CHECK_MEDIA_CHANGE_OP,1,0,NULL);
}

/*function to handle revalidate requests for ramdisk*/
void rd_revalidate(int minor, int id)
{
	userdev_send_response(minor,id,REVALIDATE_OP,0,0,NULL);
}

/*function to handle messages for ramdisk*/
void rd_message(int minor, int type, int reqid, void *data, int size)
{
}

/*function to handle block device requests for ramdisk*/
void rd_request(int minor, int command, long sector, int length, int clustersize, void *data, int datasize, int id)
{
	unsigned long ptr;
	int ret;
	char *d = NULL;
	int s = 0;

	/*printf("command = %d, sector = %d, length = %u, cluster = %d, datasize = %u\n",command,sector,length,clustersize,datasize);*/
	ptr = sector*SECTSIZE;
	if(ptr + length > DISKSIZE) ret = 0;	/*check if request exceeds the size of disk*/
	else
	{
		switch(command)
		{
			case 0:	/*READ*/
				d = disk+ptr;
				s = length;
				ret = 1;
				break;

			case 1:	/*WRITE*/
				memcpy(disk+ptr,data,datasize);
				ret = 1;
				break;

			default:
				ret = 0;
		}
	}
	userdev_send_response(minor, id, REQUEST_OP, ret, s , d);
}

main(int argc, char* argv[])
{
	char fname[] = DEV_NAME;
	struct userdev_ioctl_data ioelm[NUM_IOCTLS] = {};
	struct userdev_operations uops = {
		USERDEV_BLOCK,
		NULL,	/*read*/
		NULL,	/*write*/
		NULL,	/*poll*/
		rd_ioctl,
		rd_open,
		NULL,	/*flush*/
		rd_close,
		NULL,	/*fsync*/
		NULL,	/*fasync*/
		rd_check_media_change,
		rd_revalidate,
		NULL,	/*media_ctl*/
		rd_message,
		rd_request
	};
        struct userdev_blk_data *data = NULL;

	if(argc>1) strcpy(fname,argv[1]);
	signal(SIGINT,signal_handler);

        data = (struct userdev_blk_data*)malloc(sizeof(struct userdev_blk_data));
        data->devsize = DISKSIZE/1024;
        data->blksize = BLKSIZE;
        data->sectsize = SECTSIZE;
        data->maxreadahead = 31;
        data->maxsectors = 5;

	printf("Attach to userdev returned %d\n",usrid = userdev_attach(fname,ioelm,NUM_IOCTLS,&uops,data));
        free(data);

	disk = (char*)malloc(DISKSIZE);
	printf("Listening for requests....\n");
	if(userdev_start() < 0) printf("Cannot start accepting requests\n");

	printf("Detach from userdev returned %d\n",userdev_detach(usrid));
	free(disk);
}
