/*file containing code for the user level device driver for parallel port printer*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <fcntl.h>
# include <string.h>
# include <signal.h>
# include <errno.h>
# include <pthread.h>
# include <sys/types.h>
# include "usrdriv.h"	/*user level device driver framework library calls*/
# include "parport.h"	/*parport device driver library calls*/
#undef LP_STATS
# include "lpproc.h"

#undef LP_DEBUG
#undef LP_READ_DEBUG

/* if you have more than 3 printers, remember to increase LP_NO */
#define LP_NO 	3

/*array of lp_table structures that keep track of info of each device handled by the driver*/
struct lp_struct lp_table[LP_NO] =
{
	[0 ... LP_NO-1] = { NULL, 0, LP_INIT_CHAR, LP_INIT_TIME, LP_INIT_WAIT, NULL,
#ifdef LP_STATS
			0, 0, {0,0,0,0,0,0},
#endif
			0, 0, 0}
};

/*
//Data Structures shared by more than 1 thread protected by a mutex
//Generally we would have 1 write thread per minor device in addition to the
//main thread, as when writing is in progress the only messages that can come
//are the signal messages. The close message that can come will wait for the
//thread to end(join). Since open enforces mutual exclusion, no other process
//can initiate new messages.
*/
struct sig_list *lp_signals = NULL;	/*list of signal messages arrived for various requests*/
pthread_t lp_write_thread[LP_NO];      	/*pthreads that would be used for writing asynchronously*/
unsigned int lp_write_flag[LP_NO];	/*flag set when a write thread is running, used by close to call join*/
pthread_mutex_t	mutex = PTHREAD_MUTEX_INITIALIZER;	/*mutex for accessing above data shared by more than 1 thread*/

int lp_minor_map[LP_NO];	/*map between minor number and index into lp_table*/


/*function to tranform minor number into index into lp_table*/
static int Index(int id)
{
	int i;
	for(i=0;i<LP_NO;i++)
		if (lp_minor_map[i] == id) return i;
	return -1;
}


/*
//function to add a node into the list of signal messages arrived given the
//requset id for which the signal has arrived
*/
void add_sig_list(int reqid)
{
	struct sig_list *elm;
	
	pthread_mutex_lock(&mutex);	/*lock for mutual exclusion*/
	elm = (struct sig_list*)malloc(sizeof(struct sig_list));
	elm->reqid = reqid;		/*create a node and store the request id*/
	elm->next = lp_signals;		/*prepend the node to the list of signal messages*/
	lp_signals = elm;
	pthread_mutex_unlock(&mutex);	/*unlock*/
}


/*
//function to remove all nodes from the list of signal messages for a
//particular request given its id
*/
void del_sig_list(int reqid)
{
    	struct sig_list *ptr,*pqr;
	 
	pthread_mutex_lock(&mutex);	/*mutual exclusion lock*/
    	ptr = lp_signals;	/*pointer to previous node*/
    	pqr = NULL;		/*pointer to current node*/
    	while(ptr!=NULL)
    	{
        	if(ptr->reqid == reqid)		/*if found match for current*/
        	{
            		if(pqr != NULL)		/*check if previous exists*/
            		{
                		pqr->next = ptr->next;	/*if so update previous' next pointer*/
                		free(ptr);
                		ptr = pqr->next;	/*update current pointer to next element to continue search*/
            		}
            		else	/*previous is NULL*/
            		{
                		lp_signals = ptr->next;	/*update start of list*/
                		free(ptr);
                		ptr = lp_signals;	/*set current to start of list*/
            		}
        	}
        	else	/*not matching*/
        	{			
            	pqr = ptr;	/*update current and previous pointers*/
            	ptr = ptr->next;
        	}
    	}
	pthread_mutex_unlock(&mutex);	/*unlock*/
}


/*function to check if a signal message has arrived for the given request*/
int check_sig_list(int reqid)
{
	struct sig_list *lst = lp_signals;

	pthread_mutex_lock(&mutex);	/*lock to prevent changes while searching*/
	while(lst!=NULL)	/*run though the list and search for a match*/
	{
		if(lst->reqid == reqid)
		{
			pthread_mutex_unlock(&mutex);	/*unlock*/
			return 1;	/*if found return 1*/
		}
		lst = lst->next;
	}
	pthread_mutex_unlock(&mutex);	/*unlock*/
	return 0;	/*not found so return 0*/
}


void lp_set_write_in_progress(int id, int val)
{
	pthread_mutex_lock(&mutex);	/*lock to prevent changes*/
	lp_write_flag[Index(id)] = val;
	pthread_mutex_unlock(&mutex);	/*unlock*/
}

int check_write_in_progress(int id)
{
	int val;
	pthread_mutex_lock(&mutex);	/*lock to prevent changes*/
	val = lp_write_flag[Index(id)];
	pthread_mutex_unlock(&mutex);	/*unlock*/
	return val;
}
	

void lpproc_message(int id, int type, int reqid, void *data, int size)
{
	/*if the type of message is SIGNAL_PENDING then add request id to signal message list*/
	/*lock before access as lpproc_write might be accessing it*/
	/*ofcourse no other function would be accessing the list and only one*/
	/*process will have access to the printer at a time*/

	if(type == SIGNAL_PENDING) add_sig_list(reqid);
}


/* Test if printer is ready */
#define	LP_READY(status)		((status) & LP_PBUSY)
/* Test if the printer is not acking the strobe */
#define	LP_NO_ACKING(status)	((status) & LP_PACK)
/* Test if the printer has error conditions */
#define LP_NO_ERROR(status)		((status) & LP_PERRORP)
/* wait for time specific to the device attached*/
#define	lp_wait(minor)	userdev_udelay(LP_WAIT(minor))


/* --- low-level port access ----------------------------------- */
#define r_dtr(x)	(parport_read_data(lp_table[Index(x)].dev->port))
#define r_str(x)	(parport_read_status(lp_table[Index(x)].dev->port))
#define w_ctr(x,y)	do { parport_write_control(lp_table[Index(x)].dev->port, (y)); } while (0)
#define w_dtr(x,y)	do { parport_write_data(lp_table[Index(x)].dev->port, (y)); } while (0)


/*function to schedule some other process for the specified time*/
static __inline__ void lp_schedule(int minor, long timeout)
{
	usleep(timeout * 1000);	/*schedule_timeout(timeout);*/
}


/*function to reset the printer*/
static int lp_reset(int minor)
{
	int retval;
	
	w_ctr(minor, LP_PSELECP);
	userdev_udelay (LP_DELAY);
	w_ctr(minor, LP_PSELECP | LP_PINITP);
	retval = r_str(minor);
	return retval;
}


/*function to print a given character to a specific printer*/
static inline int lp_char(char lpchar, int minor, int reqid)
{
	unsigned long count = 0;

#ifdef LP_STATS
	struct lp_stats *stats;
#endif
	if(check_sig_list(reqid)) return 0;	/*if a signal message arrived for the current request*/

	for (;;)
	{
		unsigned char status;
		int irq_ok = 0;

		status = r_str(minor);	/*read device status*/
		if (LP_NO_ERROR(status))
		{
			if (LP_READY(status)) break;

			 /* This is a crude hack that should be well known at least by Epson device driver developers. -arca */
			irq_ok = (!LP_POLLED(minor) &&  LP_NO_ACKING(status) &&
				  				lp_table[Index(minor)].irq_detected);
			if ((LP_F(minor) & LP_TRUST_IRQ) && irq_ok) break;
		}

		if (++count == LP_CHAR(minor))	/*if the per character wait time expires*/
		{
			if (irq_ok)
			{
				static int first_time = 1;
				/*
				 * The printer is using a buggy handshake, so
				 * revert to polling to not overload the
				 * machine and warn the user that its printer
				 * could get optimized trusting the irq. -arca
				 */
				lp_table[Index(minor)].irq_missed = 1;
				if (first_time)
				{
					first_time = 0;
					fprintf(stderr,"ulp%d: the "
					       "printing could be optimized "
					       "using the TRUST_IRQ flag, "
					       "see the top of "
					       "linux/drivers/char/lp.c\n",
					       Index(minor));
				}
			}
			return 0;
		}
	}

	w_dtr(minor, lpchar);	/*write character to device*/

#ifdef LP_STATS
	stats = &LP_STAT(minor);
	stats->chars++;
#endif

	/* must wait before taking strobe high, and after taking strobe
	   low, according spec.  Some printers need it, others don't. */
	lp_wait(minor);

	if (LP_POLLED(minor))	/* control port takes strobe high */ 
	{
		w_ctr(minor, LP_PSELECP | LP_PINITP | LP_PSTROBE);
		lp_wait(minor);
		w_ctr(minor, LP_PSELECP | LP_PINITP);
	}
   	else 
	{
		/* Epson Stylus Color generate the IRQ on the rising edge of
		 * strobe so clean the irq's information before playing with
		 * the strobe. -arca */
		lp_table[Index(minor)].irq_detected = 0;
		lp_table[Index(minor)].irq_missed = 0;

		/* Be sure that the CPU doesn' t reorder instructions. use mb(); */
		w_ctr(minor, LP_PSELECP | LP_PINITP | LP_PSTROBE | LP_PINTEN);
		lp_wait(minor);
		w_ctr(minor, LP_PSELECP | LP_PINITP | LP_PINTEN);
	}

	 /* Give to the printer a chance to put BUSY low. Really we could
	  * remove this because we could _guess_ that we are slower to reach
	  * again lp_char() than the printer to put BUSY low, but I' d like
	  * to remove this variable from the function I go solve
	  * when I read bug reports ;-). -arca	*/
	lp_wait(minor);

#ifdef LP_STATS
	/* update waittime statistics */
	{
		unsigned int wait;
		if (count > stats->maxwait)
   		{
#ifdef LP_DEBUG
			printf("ulp%d success after %d counts.\n",Index(minor),count);
#endif
			stats->maxwait = count;
		}
		count *= 256;
		wait = (count > stats->meanwait) ? count - stats->meanwait :
	    							   stats->meanwait - count;
		stats->meanwait = (255 * stats->meanwait + count + 128) / 256;
		stats->mdev = ((127 * stats->mdev) + wait + 64) / 128;
	}
#endif

	return 1;
}


/*
//Signal handler for the driver, which is equivalent to the interrupt handler in user space
//
//We have 3 functions each for a different minor number as the current kernel does not copy the 
//value field of siginfo structure onto the stack when signal is being delivered. We could place 
//the minor number in the error code field, although
*/
static void lp_interrupt0(int sig, siginfo_t *info, void *ignored)
{
	int minor,index;

	/*
	if (!info) { printf(" but no data\n"); return; }
	minor = info->si_value.sival_int;	
	*/

	minor = lp_minor_map[0];
	index = Index(minor);
	if(index == -1) return;

	lp_table[index].irq_detected = 1;	/*make note of occurance of irq*/
	lp_table[index].irq_missed = 0;
}

static void lp_interrupt1(int sig, siginfo_t *info, void *ignored)
{
	int minor,index;

	minor = lp_minor_map[1];
	index = Index(minor);
	if(index == -1) return;

	lp_table[index].irq_detected = 1;	/*make note of occurance of irq*/
	lp_table[index].irq_missed = 0;
}

static void lp_interrupt2(int sig, siginfo_t *info, void *ignored)
{
	int minor,index;

	minor = lp_minor_map[2];
	index = Index(minor);
	if(index == -1) return;

	lp_table[index].irq_detected = 1;	/*make note of occurance of irq*/
	lp_table[index].irq_missed = 0;
}



/*function to recover from error and make note of it*/
static void lp_error(int minor)
{
	if (LP_POLLED(minor) || LP_PREEMPTED(minor))
   	{
		printf("lp_error\n");
		usleep(LP_TIMEOUT_POLLED * 1000);	/*schedule_timeout(LP_TIMEOUT_POLLED);*/
		lp_table[Index(minor)].irq_missed = 1;
	}
}


/*function to check status of the parallel port interface*/
static int lp_check_status(int minor)
{
	unsigned int last = lp_table[Index(minor)].last_error;
	unsigned char status = r_str(minor);	/*read status from device*/

	if (status & LP_PERRORP)
		last = 0;			/*no error*/
	else if ((status & LP_POUTPA))
   	{
		if (last != LP_POUTPA)
	   	{
			last = LP_POUTPA;	/*out of paper*/
			fprintf(stderr,"ulp%d out of paper\n", Index(minor));
		}
	}
   	else if (!(status & LP_PSELECD))
   	{
		if (last != LP_PSELECD)
	   	{
			last = LP_PSELECD;	/*device off line*/
			fprintf(stderr,"ulp%d off-line\n", Index(minor));
		}
	} 
	else
   	{
		if (last != LP_PERRORP)
	   	{
			last = LP_PERRORP;	/*device on fire*/
			fprintf(stderr,"ulp%d on fire\n", Index(minor));
		}
	}

	lp_table[Index(minor)].last_error = last;	/*record error*/

	if (last != 0)
   	{
		if (LP_F(minor) & LP_ABORT)
			return 1;
		lp_error(minor);
	}
	return 0;
}


/*function to write data in the given buffer to the device*/
static int lp_write_buf(unsigned int minor, const char *buf, int count, int reqid)
{
	unsigned long copy_size;
	unsigned long total_bytes_written = 0;
	unsigned long bytes_written;
	struct lp_struct *lp = &lp_table[Index(minor)];

	if ((Index(minor) >= LP_NO) || (Index(minor) < 0))
		return -ENXIO;		/*check if index into p_table is valid*/
	if (lp->dev == NULL)
		return -ENXIO;

	lp_table[Index(minor)].last_error = 0;	/*initialize values*/
	lp_table[Index(minor)].irq_detected = 0;
	lp_table[Index(minor)].irq_missed = 1;

	if (LP_POLLED(minor))		/*if device is polled*/
		w_ctr(minor, LP_PSELECP | LP_PINITP);
	else
		w_ctr(minor, LP_PSELECP | LP_PINITP | LP_PINTEN);

	do
   	{	/* while buffer is not empty, read some into the lp_buffer and call lp_char for each char*/
		bytes_written = 0;
		copy_size = (count <= LP_BUFFER_SIZE ? count : LP_BUFFER_SIZE);

		if (memcpy(lp->lp_buffer, buf, copy_size) == NULL)	/*copy into temporary buffer*/
		{
			w_ctr(minor, LP_PSELECP | LP_PINITP);
			return -EFAULT;
		}

		while (copy_size)	/*for the amount copied*/
	   	{
			if (lp_char(lp->lp_buffer[bytes_written], minor, reqid))	/*write a character to device*/
		   	{
				--copy_size;
				++bytes_written;
#ifdef LP_STATS
				lp->runchars++;
#endif
			}
		   	else		/*if write failed*/
		   	{
				int rc = total_bytes_written + bytes_written;

#ifdef LP_STATS				/*update statisitcs*/
				if (lp->runchars > LP_STAT(minor).maxrun)
					LP_STAT(minor).maxrun = lp->runchars;
				LP_STAT(minor).sleeps++;
#endif

				if(check_sig_list(reqid))	/*if a signal message arrived for the current request*/
				{
					del_sig_list(reqid);	/*remove all signal messages for the request and return*/
                    			w_ctr(minor, LP_PSELECP | LP_PINITP);
				    	if (total_bytes_written + bytes_written)
					    	return total_bytes_written + bytes_written;
					else return -EINTR;
				}

#ifdef LP_STATS				/*update statisitcs*/
				lp->runchars = 0;
#endif

				if (lp_check_status(minor))	/*check status, if set to abort on error then return*/
				{
					w_ctr(minor, LP_PSELECP | LP_PINITP);
					return rc ? rc : -EIO;
				}
					/*if polled or interrupt has been missed*/
				if (LP_POLLED(minor) || lp_table[Index(minor)].irq_missed)
				{
#if defined(LP_DEBUG) && defined(LP_STATS)
					fprintf(stderr,"ulp%d sleeping at %d characters for %d jiffies\n", minor, lp->runchars, LP_TIME(minor));
#endif
					lp_schedule(minor, LP_TIME(minor));	/*call schedule to poll after sometime*/
				} 
				else	/*else wait for interrupt*/
			   	{
					if (!lp_table[Index(minor)].irq_detected)
						usleep(LP_TIMEOUT_INTERRUPT * 1000);
						/*interruptible_sleep_on_timeout(&lp->wait_q, LP_TIMEOUT_INTERRUPT);*/
				}
			}
		}

		total_bytes_written += bytes_written;
		buf += bytes_written;	/*update number of bytes written*/
		count -= bytes_written;

	} while (count > 0);

	w_ctr(minor, LP_PSELECP | LP_PINITP);
	return total_bytes_written;
}


struct lp_write_arg	/*data structure that combines the arguments of write userdev operation*/
{			/*into a single structure, so that a reference to it can be passed to  */
	int id;		/*pthread that is created for asynchronously writing data to the device*/
	char *buf;
	int count;
	long long off;
	int flags;
	int reqid;
};


/*function that is a thread that asynchronously writes data to the device*/
static void* lpproc_write_thread(void *p)
{
	unsigned int minor;
	int retv;
	struct lp_write_arg *param;

	param = (struct lp_write_arg*)p;	/*obtain list of parameters from function argument*/
	minor = param->id;
	lp_set_write_in_progress(minor, 1);

#ifdef LP_STATS
	{
		int jiffies = time(NULL);

		/*LP_TIME is in jiffies so multiply by 100 for comparision*/
		if (jiffies-lp_table[Index(minor)].lastcall > LP_TIME(minor)*100)
			lp_table[Index(minor)].runchars = 0;

		lp_table[Index(minor)].lastcall = jiffies;
	}
#endif

	parport_set_ioperms(lp_table[Index(minor)].dev->port);	/*set io-perms again as thread is different process*/
	retv = lp_write_buf(minor, param->buf, param->count, param->reqid);	/*write data*/
	parport_reset_ioperms(lp_table[Index(minor)].dev->port);/*reset io-perms*/

	userdev_send_response(minor, param->reqid, WRITE_OP, param->off, retv, NULL);	/*send back the response in a asynchronous manner*/

	free(param->buf);	/*free memory and exit thread*/
	free(param);
	pthread_exit(0);
 	return NULL;
}


/*function to write data to device - userdev operation*/
static void lpproc_write(int id, char *buf, int count, long long off, int flags, int reqid)
{
	struct lp_write_arg *param;
	
	param = (struct lp_write_arg*)malloc(sizeof(struct lp_write_arg));
	param->id = id;		/*bundle the parameters into a single argument for the thread*/
	param->buf = (char*)malloc(count);
	memcpy(param->buf,buf,count);	/*need to copy buffer into local as need to operate asynchronously*/
	param->count = count;
	param->off = off;
	param->reqid = reqid;
	
	/*create thread to write asynchronously, the response will be sent when the thread completes*/
	pthread_create(&lp_write_thread[Index(id)], NULL, lpproc_write_thread, (void*)param);
}


#ifdef CONFIG_PRINTER_READBACK		/*----------------------------------------------------------------------------*/


/*function to read a nibble from device data register*/
static int lp_read_nibble(int minor) 
{
	unsigned char i;
	i = r_str(minor)>>3;
	i &= ~8;
	if ((i & 0x10) == 0) i |= 8;
	return (i & 0x0f);
}


/*function to terminate read call*/
static void lp_read_terminate(struct parport *port)
{
	parport_write_control(port, (parport_read_control(port) & ~2) | 8);
		/* SelectIN high, AutoFeed low */
	if (parport_wait_peripheral(port, 0x80, 0)) 
		/* timeout, SelectIN high, Autofeed low */
		return;
	parport_write_control(port, parport_read_control(port) | 2);
		/* AutoFeed high */
	parport_wait_peripheral(port, 0x80, 0x80);
		/* no timeout possible, Autofeed low, SelectIN high */
	parport_write_control(port, (parport_read_control(port) & ~2) | 8);
}


/*function to obtain status readback confirming to ieee1284 -- userdev operation*/
static void lpproc_read(int id, unsigned int length, long long off, int flags, int reqid)
{
	int i;
	unsigned int minor=id;
	char *temp = (char*)malloc(sizeof(char)*length);
	unsigned int count = 0;
	unsigned char z = 0;
	unsigned char Byte = 0;
	struct parport *port = lp_table[Index(minor)].dev->port;

	switch (parport_ieee1284_nibble_mode_ok(port, 0))	/*check if device pertains to ieee1284 standard*/
	{
	case 0:
		lp_read_terminate(port);			/* Handshake failed. */  
		userdev_send_response(id, reqid, READ_OP, -EIO, 0, NULL);
		free(temp);
		return;
	case 1:
		lp_read_terminate(port);			/* No data. */
		userdev_send_response(id, reqid, READ_OP, 0, 0, NULL);
		free(temp);
		return;
	default:						/* Data available */
		/* Hack: Wait 10ms (between events 6 and 7) */
        usleep((HZ+99)/100 * 1000);	/*schedule_timeout((HZ/99)/100);*/
        break;
	}

	for (i=0; ; i++)	/*read as much data as the device provides*/
   	{
		parport_frob_control(port, 2, 2); 		/* AutoFeed high */
		if (parport_wait_peripheral(port, 0x40, 0))
	   	{
#ifdef LP_READ_DEBUG
			/* Some peripherals just time out when they've sent all their data.  */
			fprintf(stderr,"%s: read1 timeout.\n", port->name);
#endif
			parport_frob_control(port, 2, 0); 	/* AutoFeed low */
			break;
		}

		z = lp_read_nibble(minor);
		parport_frob_control(port, 2, 0); 		/* AutoFeed low */
		if (parport_wait_peripheral(port, 0x40, 0x40))
	   	{
			fprintf(stderr,"%s: read2 timeout.\n", port->name);
			break;
		}

		if ((i & 1) != 0)	/*assemble nibbles into bytes*/
	   	{
			Byte |= (z<<4);
			*temp = Byte;
			temp++;
			if (++count == length)
				break;
			/* Does the error line indicate end of data? */
			if ((parport_read_status(port) & LP_PERRORP) == LP_PERRORP)
				break;
		} 
		else Byte=z;
	}

	lp_read_terminate(port);
	userdev_send_response(id, reqid, READ_OP, off, count, temp);	/*send the data read*/
	free(temp);
}

#endif	/*------------------------------------------------------------------------------------------------*/


/*function to open a parallel port printer device -- userdev operation*/
static void lpproc_open(int id, unsigned int flags, mode_t mode, int reqid)
{
	unsigned int minor = id;
	unsigned char busy_bit;

	if ((Index(minor) >= LP_NO) || (Index(minor) < 0))	/*check if lp_table index is valid*/
	{
		userdev_send_response(id, reqid, OPEN_OP, -ENXIO, 0, NULL);
		return;
	}
	if ((LP_F(minor) & LP_EXIST) == 0)
	{
		userdev_send_response(id, reqid, OPEN_OP, -ENXIO, 0, NULL);
		return;
	}

    	busy_bit = LP_F(minor) & LP_BUSY;	/*equivalent to: test_and_set(LP_BUSY_BIT_POS,&LP_F(minor))*/
	if (busy_bit)	/*check if the printer is busy*/
	{
		printf("Sorry! printer is busy!\n");
		userdev_send_response(id, reqid, OPEN_OP, -EBUSY, 0, NULL);
		return;
	}
	else
		LP_F(minor) |= LP_BUSY;		/*set busy flag for exclusion*/

	/* If ABORTOPEN is set and the printer is offline or out of paper,
	   we may still want to open it to perform ioctl()s.  Therefore we
	   have commandeered O_NONBLOCK, even though it is being used in
	   a non-standard manner.  This is strictly a Linux hack, and
	   should most likely only ever be used by the tunelp application. */
	if ((LP_F(minor) & LP_ABORTOPEN) && !(flags & O_NONBLOCK))
   	{
		int status;
		status = r_str(minor);		/*read status*/
		if (status & LP_POUTPA)
	   	{
			fprintf(stderr,"ulp%d out of paper\n", Index(minor));
			LP_F(minor) &= ~LP_BUSY;	/*out of paper*/
			userdev_send_response(id, reqid, OPEN_OP, -ENOSPC, 0, NULL);
			return;
		} else if (!(status & LP_PSELECD))
	   	{
			fprintf(stderr,"ulp%d off-line\n", Index(minor));
			LP_F(minor) &= ~LP_BUSY;	/*device offline*/
			userdev_send_response(id, reqid, OPEN_OP, -EIO, 0, NULL);
			return;
		} else if (!(status & LP_PERRORP))
	   	{
			fprintf(stderr,"ulp%d printer error\n", Index(minor));
			LP_F(minor) &= ~LP_BUSY;	/*printer error*/
			userdev_send_response(id, reqid, OPEN_OP, -EIO, 0, NULL);
			return;
		}
	}

	lp_table[Index(minor)].lp_buffer = (char *)malloc(LP_BUFFER_SIZE);	
	if (!lp_table[Index(minor)].lp_buffer)
   	{				/*in case of error reset busy flag and return error*/
		LP_F(minor) &= ~LP_BUSY;
		userdev_send_response(id, reqid, OPEN_OP, -ENOMEM, 0, NULL);
		return;
	}
	lp_set_write_in_progress(minor, 0);
	userdev_send_response(id, reqid, OPEN_OP, 0, 0, NULL);
}


/*function to close the device -- userdev operation*/
static void lpproc_close(int id, int reqid)
{
	unsigned int minor = id;

	/*wait for the write thread to terminate*/
	if(check_write_in_progress(minor))
	{
		pthread_t thd;
		pthread_mutex_lock(&mutex);	/*lock for mutual exclusion*/
		thd = lp_write_thread[Index(minor)];
		pthread_mutex_unlock(&mutex);	/*unlock for mutual exclusion*/
		pthread_join(thd, NULL);
	}

	free(lp_table[Index(minor)].lp_buffer);	/*free the printer buffer*/
	lp_table[Index(minor)].lp_buffer = NULL;
	lp_set_write_in_progress(minor, 0);
	LP_F(minor) &= ~LP_BUSY;		/*reset busy flag*/
	userdev_send_response(id, reqid, CLOSE_OP, 0, 0, NULL);
}


/*function to perform ioctl operations on the device -- userdev operation*/
static void lpproc_ioctl(int id, int cmd, void *arg, int size, int reqid)
{
	unsigned int minor = id;
	unsigned int *ptr;
	int status;
	int retval = 0;

#ifdef LP_DEBUG
	fprintf(stderr,"ulp%d ioctl, cmd: 0x%x, arg: 0x%x\n", Index(minor), cmd, arg);
#endif
	if ((Index(minor) >= LP_NO) || (Index(minor) < 0))
	{
		userdev_send_response(id, reqid, IOCTL_OP, -ENODEV, 0, NULL);
		return;
	}
	if ((LP_F(minor) & LP_EXIST) == 0)
	{
		userdev_send_response(id, reqid, IOCTL_OP, -ENODEV, 0, NULL);
		return;
	}

	switch ( cmd )		/*based on the command number*/
   	{
		case LPTIME:	/*set LP_TIME value*/
			ptr = (unsigned int*)arg;
			LP_TIME(minor) = (*ptr) * HZ/100;
			break;
		case LPCHAR:	/*set LP_CHAR value*/
			ptr = (unsigned int*)arg;
			LP_CHAR(minor) = (*ptr);
			break;
		case LPABORT:	/*set/unset LP_ABORT flag*/
			if (*(int*)arg)
				LP_F(minor) |= LP_ABORT;
			else
				LP_F(minor) &= ~LP_ABORT;
			break;
		case LPABORTOPEN:	/*set/unset LP_ABORTOPEN flag*/
			if (*(int*)arg)
				LP_F(minor) |= LP_ABORTOPEN;
			else
				LP_F(minor) &= ~LP_ABORTOPEN;
			break;
#ifdef OBSOLETED
		case LPCAREFUL:	/*set/unset LP_CAREFUL flag*/
			if (*(int*)arg)
				LP_F(minor) |= LP_CAREFUL;
			else
				LP_F(minor) &= ~LP_CAREFUL;
			break;
#endif
		case LPTRUSTIRQ:/*set/unset LPTRUSTIRQ flag*/
			if (*(int*)arg)
				LP_F(minor) |= LP_TRUST_IRQ;
			else
				LP_F(minor) &= ~LP_TRUST_IRQ;
			break;
		case LPWAIT:	/*set LP_WAIT value*/
			ptr = (unsigned int*)arg;
			LP_WAIT(minor) = (*ptr);
			break;
		case LPSETIRQ: 	/*set irq number*/
			userdev_send_response(id, reqid, IOCTL_OP, -EINVAL, 0, NULL);
			return;
			break;
		case LPGETIRQ:	/*get irq number*/
			if (!memcpy((int *) arg, &LP_IRQ(minor),sizeof(int)))
			{
				userdev_send_response(id, reqid, IOCTL_OP, -EFAULT, 0, NULL);
				return;
			}
			break;
		case LPGETSTATUS:	/*get device status*/
			status = r_str(minor);
			if (!memcpy((int *) arg, &status, sizeof(int)))
			{
				userdev_send_response(id, reqid, IOCTL_OP, -EFAULT, 0, NULL);
				return;
			}
			break;
		case LPRESET:	/*call printer reset*/
			lp_reset(minor);
			break;
#ifdef LP_STATS
		case LPGETSTATS:/*get device statistics*/
			if (!memcpy((int *)arg, &LP_STAT(minor),sizeof(struct lp_stats)))
			{
				userdev_send_response(id, reqid, IOCTL_OP, -EFAULT, 0, NULL);
				return;
			}
			memset(&LP_STAT(minor), 0,sizeof(struct lp_stats));	/*if possible check if user = superuser*/
			break;
#endif
 		case LPGETFLAGS:/*get device flags*/
 			status = LP_F(minor);
			if (!memcpy((int *) arg, &status, sizeof(int)))
			{
				userdev_send_response(id, reqid, IOCTL_OP, -EFAULT, 0, NULL);
				return;
			}
			break;
		default:
			retval = -EINVAL;
	}
	userdev_send_response(id, reqid, IOCTL_OP, retval, size, arg);
}


/*userdev operations for user level printer driver*/
static struct userdev_operations uops = {
	USERDEV_CHAR,
#ifdef CONFIG_PRINTER_READBACK
	lpproc_read,
#else
	NULL,
#endif
	lpproc_write,
	NULL,		/*poll*/
	lpproc_ioctl,
	lpproc_open,
	NULL,		/*flush*/
	lpproc_close,
	NULL,		/*fsync*/
	NULL,		/*fasync*/
	NULL,		/*check_media_change*/
	NULL,		/*revalidate*/
	NULL,		/*mediactl*/
	lpproc_message,
	NULL		/*request*/
};

# ifdef LP_STATS
# define LP_NUM_IOCTLS 13
# else
# define LP_NUM_IOCTLS 12
# endif


/*list of ioctls supported and the maximum size of data for each*/
struct userdev_ioctl_data ioelm[LP_NUM_IOCTLS] = {
	{LPCHAR,sizeof(int)},
	{LPTIME,sizeof(int)},
	{LPABORT,sizeof(int)},
	{LPSETIRQ,0},
	{LPGETIRQ,sizeof(int)},
	{LPWAIT,sizeof(int)},
	{LPCAREFUL,sizeof(int)},
	{LPABORTOPEN,sizeof(int)},
	{LPGETSTATUS,sizeof(int)},
	{LPRESET,0},
#ifdef LP_STATS 
	{LPGETSTATS,sizeof(struct lp_stats)},
#endif
	{LPGETFLAGS,sizeof(int)},
	{LPTRUSTIRQ,sizeof(int)}
};


/* ------------------------ initialisation code ------------------------------------- */

# define LP_DELIM ","	/*separator of components in user specified argument string*/

/*array of strings that identify the mode of operation for the various printer devices*/
static char *parport[LP_NO] = { NULL,  };

/*mode of operation for each of the printer devices*/
static int parport_nr[LP_NO] = { [0 ... LP_NO-1] = LP_PARPORT_UNSPEC };

static int parport_count = 0;	/*count of number of arguments specified by user*/
static int reset = 0;		/*flag which suggests whether device reset has to be called at device initialization*/

/*names of the device filenames for user level printer driver*/
static char *deviceFile[LP_NO] = { "/dev/ulp0", "/dev/ulp1", "/dev/ulp2" };


/*
//function to parse the user specified arguments
//The arguments are of the type: mode1,mode2,mode3 reset 
//where mode1,mode2,mode3 are the mode of operation for the 3 printer devices
//and reset is a flag which tells whether reset needs to be called at device
//initialisation
*/
void lpproc_parse_args(int argc, char*argv[])
{
	char *str,*tok;
	int i;

	if (argc < 2) return;	/*if no argument specified return*/
	str = argv[1];
	if (argc > 2)		/*if second argument exists then assign value to reset flag*/
		reset = atoi(argv[2]);

	tok = strtok(str,LP_DELIM);
	while(tok != NULL)	/*parse first argument with componets separated by LP_DELIM*/
	{
		parport[parport_count] = (char*)malloc(strlen(tok) + 1);
		strcpy(parport[parport_count],tok);
		parport_count++;	/*copy each component into a separate array element*/
		if(parport_count >= LP_NO) break;
		tok = strtok(NULL,LP_DELIM);
	}
}


/*
//function to register a lp device with parport, attach with userdev device
//driver and to install & enable interrupt handling if required
*/
int lp_register(int nr, struct parport *port)
{
	int id;
	int ret;
	void (*lp_interrupt)(int, siginfo_t *, void *);
	
			/*following switch case used to assign a interrupt handler for each minor*/
			/*need not have this if the value field of siginfo was propagated by kernel*/
			/*Also below the signal number is different for each minor*/
	switch(nr)		
	{		
		case 0:lp_interrupt = lp_interrupt0; break;
		case 1:lp_interrupt = lp_interrupt1; break;
		default:
		case 2:lp_interrupt = lp_interrupt2; break;
	}

	lp_table[nr].dev = parport_register_device(port, deviceFile[nr], 0);	/*register with parport*/
	if (lp_table[nr].dev == NULL)
		return 1;
	lp_table[nr].flags |= LP_EXIST;	/*set flags*/


	id = userdev_attach(deviceFile[nr], ioelm, LP_NUM_IOCTLS, &uops, NULL);	/*attach with userdev*/
	if (id < 0)
	{
		fprintf(stderr,"Attach to userdev failed for %s\n",deviceFile[nr]);
		return -1;
	}
	lp_minor_map[nr] = id;		/*map id returned with lp number nr*/
	printf("Attached ulp%d to userdev\n",nr);
	
	if(!LP_POLLED(id))		/*if not polled request for and enable irq*/
	{
		/*we have different signals for each minor*/
		ret = userdev_request_irq(port->irq, LP_SIGNAL+nr, lp_interrupt,id);	
		if(ret) fprintf(stderr,"request for irq returned %d\n",ret);
		parport_enable_irq(port);
	}

	if (reset)			/*if need to reset device do so*/
		lp_reset(id);

	printf("ulp%d: using %s (%s).\n", nr, port->name, 
	       (port->irq == PARPORT_IRQ_NONE)?"polling":"interrupt-driven");
	return 0;
}


/*function to register and initialize printer port interfaces available*/
int lp_init(void)
{
	unsigned int count = 0;
	unsigned int i;
	struct parport *port;

	switch (parport_nr[0])	/*choose based on first argument specified*/
	{
	case LP_PARPORT_OFF:	/*if off do nothing*/
		return 0;

	case LP_PARPORT_UNSPEC:	/*if auto or no argument specified*/
	case LP_PARPORT_AUTO:	/*run though the list of ports with parport and find available ones*/
	        for (port = parport_enumerate(); port; port = port->next)
		   	{
				if (parport_nr[0] == LP_PARPORT_AUTO &&
			    		port->probe_info.class != PARPORT_CLASS_PRINTER)
					continue;
				
				if (!lp_register(count, port))	/*register with parport and userdev*/
					if (++count == LP_NO)
						break;
			}
			break;

	default:		/*by default check for parport ports with number same as the one specified*/
			for (i = 0; i < LP_NO; i++)
				for (port = parport_enumerate(); port; port = port->next)
					if (port->number == parport_nr[i])	
					/*the specified number matches the numerical index of the port*/
				   	{
						if (!lp_register(i, port))
							count++;
						break;
					}
			break;
	}

	if(!count)
		fprintf(stderr,"ulp: driver loaded but no devices found\n");
	else printf("Loaded %d devices\n",count);
	return 0;
}


/*function to initialize driver module*/
int lpproc_init()
{
	if (parport[0])	/*transform parameters from string to numbers*/
   	{
		/* The user gave some parameters.  Let's see what they were.  */
		if (!strncmp(parport[0], "auto", 4))	/*if first=auto then forget rest*/
			parport_nr[0] = LP_PARPORT_AUTO;
		else
	   	{
			int n;
			for (n = 0; n < LP_NO && parport[n]; n++)
		   	{
				if (!strncmp(parport[n], "none", 4))	/*if none then set LP_PARPORT_NONE*/
					parport_nr[n] = LP_PARPORT_NONE;
				else	/*else convert to numerical value and store*/
			   	{
					char *ep;
					unsigned long r = simple_strtoul(parport[n], &ep, 0);
					if (ep != parport[n]) 
						parport_nr[n] = r;
					else
				   	{
						fprintf(stderr,"ulp: bad port specifier `%s'\n", parport[n]);
						return -ENODEV;
					}
				}
			}
		}
	}
	return lp_init();
}


/*function to cleanup printer module*/
void lpproc_cleanup()
{
	unsigned int offset;

	for (offset = 0; offset < LP_NO; offset++)	/*for each in lp_table*/
   	{
		if (lp_table[offset].dev == NULL) continue;

		if (lp_table[offset].dev->port->irq != PARPORT_IRQ_NONE)	/*if irq enabled disable it*/
			parport_disable_irq(lp_table[offset].dev->port);

		parport_unregister_device(lp_table[offset].dev);	/*unregister from parport*/
		userdev_detach(lp_minor_map[offset]);		/*detach from userdev*/
		free(parport[offset]);
	}
}


/*signal handler for SIGINT to terminate the process*/
void signal_handler(int signal)
{
	lpproc_cleanup();	/*clean up things before quiting*/
	parport_cleanup();	/*for graceful closure*/
	exit(0);
}


main(int argc, char *argv[])
{
	/*
	//arguments for parport are specified as an array so that the interface of
	//both printer driver and parport are same as that of the lp and parport
	// kernel modules
	*/
	char *args[PARPORT_NUM_ARGS] = { "0x3bc,-1,auto", "0x378,-1,7", "0x278,-1,none"};

	signal(SIGINT,signal_handler);
	parport_initialize(PARPORT_NUM_ARGS, args);	/*initialize parport*/
	lpproc_parse_args(argc,argv);			/*parse command line arguments*/
	lpproc_init();					/*initialize lpproc module*/

	printf("Listening for requests.....\n");
	/*start listening for requests from kernel userdev driver*/
	if(userdev_start() < 0) printf("Error occured while accepting requests\n");	

	lpproc_cleanup();	/*once done, clean up lpproc module*/
	parport_cleanup();	/*cleanup parport module*/
}
