/*file containing functions for accessing io-ports from user space*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <unistd.h>
# include <sys/io.h>

/*a link list of port numbers of ports for which the process has been granted
access to, ie ioperm has been called on these ports*/
static struct port_list
{
	int port;		/*port number*/
	struct port_list *next;	/*pointer to next element*/
} *perm_list = NULL;

static int io_prev_level = 0;	/*process' current privilige level for access to ports > 0x3ff */
static int count = 0;		/*number of ports for which privilige level was raised*/


/*
//function to set permission for the current process to access the port
//specified. Incase port number > 0x400 then the process io privilige level is
//raised else it is granted permission for access using the ioperm call
*/
int userdev_set_ioperm(int port)
{
	struct port_list *list;
	int ret;
	
	if (port >= 0x400)		/*check if port number > 0x3ff */
	{
		if (io_prev_level == 0)	/*if process is not already priviliged then*/
		{
			ret = iopl(3);	/*raise the process' io privilige level*/
			if (ret < 0) return ret;
			io_prev_level = 3;	/*make a note of this in the static variable*/
		}
		count++;
		return 0;		/*else do nothing*/
	}

	list = perm_list;		/*if port number < 0x400 */
	while(list != NULL)		/* check in port list if the io permission has already been granted*/
	{
		if(list->port == port) break;
		list = list->next;
	}
	if(list != NULL) return 0;	/*if so return*/

	ret = ioperm(port,1,1);		/*else obtain permission using ioperm system call*/
	if (ret < 0) return ret;
	list = (struct port_list*)malloc(sizeof(struct port_list));
	list->port = port;		/*store an entry in the port list to say so*/
	list->next = perm_list;
	perm_list = list;
	return 0;
}


/*
//function to reset permission for a given port for the current process
//Incase port number is > 0x3ff then the number of ports with port numbers
//greater than 0x3ff are accessible for the process (count) is checked. If
//there are no such ports then the io privilige level of the process is lowered
//otherwise its not. 
//Incase of port numbers less than 0x400 the port list is checked and if found
//then the permission is reset using ioperm and the list is updated.
*/
int userdev_reset_ioperm(int port)
{
	struct port_list *list,*ptr;
	int ret;
	
	if (port >= 0x400)	/*incase port number > 0x3ff */
	{
		count--;	/*decrement number of such ports*/
		if ((io_prev_level == 3) && (count == 0))
		{		/*if no more exist, then lower io privilige level*/
			ret = iopl(0);
			if (ret < 0) return ret;
			io_prev_level = 0;
		}
		return 0;
	}
	
	if(perm_list == NULL) return 0;
	list = perm_list;	/*traverse port list to locate a node for the port*/
	if (list->port == port)	/*incase first node is the required one*/
	{
		ret = ioperm(port,1,0);	/*reset permission for the port*/
		if(ret < 0) return ret;
		perm_list = list->next;	/*update the reference for the port list*/
		free(list);
		return ret;
	}

	ptr = list;		/*else search through the list*/
	list = list->next;
	while(list != NULL)
	{
		if (list->port == port) break;	/*found*/
		ptr = list;
		list = list->next;
	}
	if (list == NULL) return 0;	/*not found exit*/

	ret = ioperm(port,1,0);		/*reset permission and update list*/
	if (ret < 0) return ret;
	ptr->next = list->next;
	free(list);
	return ret;
}


/*function to reset permissions to all ports for the current process*/
int userdev_reset_all_perms()
{
	struct port_list *list,*ptr;
	int ret;
	
	if(io_prev_level == 3)	/*reset io privilege level incase its been raised*/
	{
		ret = iopl(0);
		if (ret < 0) return ret;
		io_prev_level = 0;
		count = 0;
	}
	list = perm_list;
	while(list != NULL)	/*traverse the port list and reset io permission for each port on the list*/
	{
		ret = ioperm(list->port,1,0);
		if (ret < 0) break;
		ptr = list;
		list = list->next;
		free(ptr);	/*also free the list*/
	}
	perm_list = list;
	return 0;
}


/*
//following are functions to read/write a byte/word/double_word to the given io
//port assuming permission to do I/O has already been acquired by the process
*/

/*read a byte from given port*/
unsigned char userdev_inportb(int port)
{
	unsigned char val;
	val = inb(port);
	return val;
}


/*write a byte to the given port*/
void userdev_outportb(int port, unsigned char byte)
{
	outb(byte,port);
}
	

/*read a word from given port*/
unsigned short userdev_inportw(int port)
{
	unsigned short val;
	val = inw(port);
	return val;
}


/*write a word to the given port*/
void userdev_outportw(int port, unsigned short byte)
{
	outw(byte,port);
}


/*read a double word from given port*/
unsigned char userdev_inportd(int port)
{
	unsigned int val;
	val = inl(port);
	return val;
}


/*write a double word to the given port*/
void userdev_outportd(int port, unsigned int byte)
{
	outl(byte,port);
}
