/*
//file contains functions for implementing exclusive access to io-address space 
//by user level device drivers
*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <fcntl.h>
# include <sys/file.h>
# include <string.h>

/*file for storing list of io-addresses currently mapped to some device*/
# define IO_REGION_CHECK        "/root/ioregion.dat"

/*temporary file for update operation on the above file*/
# define IO_REGION_CHECK_TMP    "/root/ioregion.tmp"

/*File Format:	each line denotes a single entry of the form*/
	/* base:number:devicename */


/*
//function to read a line from the given file until EOLN/EOF is reached or
//buffer gets full. Similar to fgets()
 */
static char *freadline(char *s, int size, int fd)
{
    	int i=0,ret;
    	char ch;
		 
    	while(i<size-1)		/*read until size-1*/
    	{
        	ret = read(fd,&ch,1);	/*read a character*/
        	if(ret<=0) break;	/*if EOF or error break*/
      	  	s[i++] = ch;		/*store character in buffer*/
        	if (ch=='\n') break;	/*if EOLN break*/
    	}
    	if (i==0) return NULL;
    	s[i] = '\0';		/*terminate string by '\0'*/
    	return s;
}


/*
//function to check if the ioregion specified by base and number of addresses
//is free and available for occupation
*/
int userdev_check_ioregion(int base, int num)
{
	int fd;
	int b,n;
	char buf[100],tmp[10];
	char *ptr1,*ptr2;
	
	fd = open(IO_REGION_CHECK, O_RDONLY);	/*open io-region data file*/
	if (fd < 0)
	{
		fprintf(stderr,"Cannot check region: base = 0x%x, num = %d\n",base,num);
		return -1;
	}

	if (flock(fd,LOCK_EX))			/*lock file for exclusive access*/
	{
		fprintf(stderr,"Could not lock the ioregion data file for exclusive access\n");
		return -1;
	}

	memset(buf,0,sizeof(buf));
	while((freadline(buf,sizeof(buf),fd)) != NULL)	/*read each entry*/
	{
		ptr1 = strchr(buf,':');
		memset(tmp,0,10);
		strncpy(tmp,buf,ptr1-buf);
		b = atoi(tmp);			/*obtain base address*/
		ptr1++;
		ptr2 = strchr(ptr1,':');
		memset(tmp,0,10);
		strncpy(tmp,ptr1,ptr2-ptr1);
		n = atoi(tmp);			/*obtain number of addresses*/
		if (b > base+num-1) break;	/*check for overlap with current request*/
		else if ((b+n-1) >= base)
		{
			close(fd);		/*if overlap then return -1*/
			return -1;
		}
		memset(buf,0,sizeof(buf));
	}
	close(fd);				/*no overlap found so say yes*/
	return 0;
}


/*
//function to request for a range of io addresses starting with base and
//extending to num number of addresses for device name
*/
int userdev_request_ioregion(int base, int num, const char *name)
{
	int fd,fd1;
	int found,clash,b,n;
	char buf[100],tmp[100];
	char *ptr1,*ptr2;
	
	fd = open(IO_REGION_CHECK, O_RDONLY);	/*open io-region dat file for reading*/
	if (fd < 0)
	{
		fprintf(stderr,"Cannot request region: base = 0x%x, num = %d\n",base,num);
		return -1;
	}

	if (flock(fd,LOCK_EX))			/*lock file for exclusive access*/
	{
		fprintf(stderr,"Could not lock the ioregion data file for exclusive access\n");
		return -1;
	}

	fd1 = open(IO_REGION_CHECK_TMP, O_WRONLY|O_TRUNC);	/*open temporary file for writing*/
	if (fd1 < 0)
	{
		fprintf(stderr,"Cannot request region: base = 0x%x, num = %d\n",base,num);
		close(fd);
		return -1;
	}
	
	clash = found = 0;
	memset(buf,0,sizeof(buf));
	while((freadline(buf,sizeof(buf),fd)) != NULL)	/*read an entry*/
	{
		if(!found)			/*if request not entered into file*/
		{				/*search for correct location to keep the list sorted*/
			ptr1 = strchr(buf,':');
			memset(tmp,0,10);
			strncpy(tmp,buf,ptr1-buf);
			b = atoi(tmp);		/*obtain base and number for each entry*/
			ptr1++;
			ptr2 = strchr(ptr1,':');
			memset(tmp,0,10);
			strncpy(tmp,ptr1,ptr2-ptr1);
			n = atoi(tmp);
			if (b > base+num-1)	/*locate place to put the new entry, following entries are of later address*/
			{
				sprintf(tmp,"%d:%d:%s\n",base,num,name);
				write(fd1,tmp,strlen(tmp));
				found = 1;
			}
			else if ((b+n-1) >= base)	/*if clash with existing entry found then abort*/
			{
				clash = 1;
				break;
			}
		}
		write(fd1,buf,strlen(buf),fd1);		/*update each entry read into temporary file*/
		memset(buf,0,sizeof(buf));
	}
	
	close(fd);
	if(clash)			/*abort if request clashes*/
	{
		close(fd1);
		return -1;
	}
	if (!found)			/*if request not entered in between then add at end*/
	{
		sprintf(tmp,"%d:%d:%s\n",base,num,name);
		write(fd1,tmp,strlen(tmp));
	}
	close(fd1);

	fd = open(IO_REGION_CHECK, O_WRONLY|O_TRUNC);	/*transfer data from temporary file to the actual file*/
	fd1 = open(IO_REGION_CHECK_TMP, O_RDONLY);
	while(freadline(buf,sizeof(buf),fd1) != NULL)
		write(fd,buf,strlen(buf));
	close(fd);
	close(fd1);
	return 0;
}


/*
//function to release the requested io address region given the base address
//and the number of addresses to release
*/
int userdev_release_ioregion(int base, int num)
{
	int fd,fd1;
	char str[25],buf[100],*ptr;
	
	fd = open(IO_REGION_CHECK, O_RDONLY);	/*open io-region dat file for reading*/
	if (fd < 0)
	{
		fprintf(stderr,"Cannot release region: base = 0x%x, num = %d\n",base,num);
		return -1;
	}

	if (flock(fd,LOCK_EX))		/*lock file for exclusive access*/
	{
		fprintf(stderr,"Could not lock the ioregion data file for exclusive access\n");
		return -1;
	}

	fd1 = open(IO_REGION_CHECK_TMP, O_WRONLY|O_TRUNC);	/*open temporary file for writing*/
	if (fd1 < 0)
	{
		fprintf(stderr,"Cannot request region: base = 0x%x, num = %d\n",base,num);
		close(fd);
		return -1;
	}

	sprintf(str,"%d:%d:",base,num);	/*construct a temporary entry with passed parameters*/
	memset(buf,0,sizeof(buf));
	while((freadline(buf,sizeof(buf),fd)) != NULL)	/*read each entry from the file*/
	{
		ptr = strstr(buf,str);			/*write to temporary file if entry does not match*/
		if((ptr == NULL) || (ptr != buf))
			write(fd1,buf,strlen(buf));	/*the constructed entry*/
		memset(buf,0,sizeof(buf));
	}
	
	close(fd1);
	close(fd);
	fd = open(IO_REGION_CHECK, O_WRONLY|O_TRUNC);	/*transfer data back to the original file*/
	fd1 = open(IO_REGION_CHECK_TMP, O_RDONLY);
	while(freadline(buf,sizeof(buf),fd1) != NULL)
		write(fd,buf,strlen(buf));
	close(fd);
	close(fd1);
	return 0;
}
