/*file containing implementation of task queue at user level*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <pthread.h>
# include "taskq.h"

static void (*init)();	/*pointer to initialization function set in run_task_queue function, and is the first thing the thread executes*/

/*function that adds(queues) a given task to the given task queue*/
void userdev_queue_task(struct userdev_task *tsk, struct userdev_task_queue *task_list)
{
	struct userdev_task *ptr;
	int count=0;

	if(tsk->sync) return;	/*queue task only if it has not been queued already*/

	tsk->next = NULL;
	tsk->sync = 1;		/*set sync flag to prevent furthur queueing*/
	pthread_mutex_lock(&task_list->mutex);	/*lock the task queue's mutex*/

	ptr = task_list->first;	
	if (ptr == NULL) 	/*if first task in queue update queue's reference*/
	{
		task_list->first = tsk;
		task_list->run = 1;	/*set run flag and wakeup task queue thread incase it is waiting for jobs to arrive*/
		pthread_cond_signal(&task_list->run_cond);
	}
	else			/*else append the task to the end of the list*/
	{
		while(ptr->next != NULL)
		{
			ptr=ptr->next;
			count++;
		}
		ptr->next = tsk;
	}

	pthread_mutex_unlock(&task_list->mutex);/*unlock the task queue's mutex*/
}


/*function to execute all the accumulated tasks in the task queue by continuously polling it*/
static void* poll_task_queue(void *p)
{
	struct userdev_task_queue *q = (struct userdev_task_queue*)p;
	struct userdev_task *ptr;

	if(init) init();	/*call the thread initialisation function*/
	while(!q->stop)	/*keep doing until stop flag is set*/
	{
		pthread_mutex_lock(&q->mutex);	/*lock the task queue's mutex*/
		if(q->first == NULL)	/*check if the timer queue is empty*/
		{
			q->run = 0;	/*in which case set run flag to 0 and wait till a job arrives*/
			while(q->run == 0)
				pthread_cond_wait(&q->run_cond, &q->mutex);
		}
		ptr = q->first;		/*obtain list of tasks*/
		q->first = NULL;	/*and detach it from the task queue*/
		pthread_mutex_unlock(&q->mutex);/*unlock task queue's mutex*/

		while(ptr!=NULL)	/*for each task in the list*/
		{
			ptr->sync = 0;	/*reset sync flag*/
			ptr->routine(ptr->data);/*call the task's associated function passing its data argument*/
			ptr = ptr->next;/*go to next task*/
		}
	}
	pthread_exit(0);
	return NULL;
}


/*function that creates a pthread that continously executes tasks in the task queue*/
void userdev_run_task_queue(struct userdev_task_queue *q, void (*fn)())
{
	init = fn;	/*assing the thread initialisation function*/
	/*create a detached thread for polling task queue*/
	pthread_create(&q->thread,NULL,poll_task_queue,(void*)q);
	pthread_detach(q->thread);
}


/*function to set the taskq_stop flag to stop the taskq thread*/
void userdev_task_queue_cleanup(struct userdev_task_queue *q)
{
	q->stop = 1;
}
