/*
//file contains implementation of the parallel port access functions for IBM PC
//compatible interface cards. Different cards have different implementations
*/

/* This driver should work with any hardware that is broadly compatible
 * with that in the IBM PC.  This applies to the majority of integrated
 * I/O chipsets that are commonly available.  The expected register
 * layout is:
 *
 *	base+0		data
 *	base+1		status
 *	base+2		control
 *
 * In addition, there are some optional registers:
 *
 *	base+3		EPP address
 *	base+4		EPP data
 *	base+0x400	ECP config A
 *	base+0x401	ECP config B
 *	base+0x402	ECP control
 *
 * All registers are 8 bits wide and read/write.  If your hardware differs
 * only in register addresses (eg because your registers are on 32-bit
 * word boundaries) then you can alter the constants in parport_pc.h to
 * accomodate this.
 */
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <unistd.h>
# include <string.h>
# include <errno.h>
# include <stdlib.h>
# include "usrdriv.h"	/*for port io functions*/
# include "parport.h"
# include "parport_pc.h"

#define PARPORT_PC_MAX_PORTS  8	/*not greater than PARPORT_MAX*/

static int user_specified = 0;	/*flag to note if the configuration has been specified by the user*/


/*Following are the implementation of parport interface functions for parport_pc */

/*write byte to enhanced parallel port*/
void parport_pc_write_epp(struct parport *p, unsigned char d)
{
	userdev_outportb(p->base+EPPDATA,d);
}

/*read byte from enhanced parallel port*/
unsigned char parport_pc_read_epp(struct parport *p)
{
	return userdev_inportb(p->base+EPPDATA);
}

/*write byte to enhanced parallel port address register*/
void parport_pc_write_epp_addr(struct parport *p, unsigned char d)
{
	userdev_outportb(p->base+EPPADDR,d);
}

/*read byte from enhanced parallel port address register*/
unsigned char parport_pc_read_epp_addr(struct parport *p)
{
	return userdev_inportb(p->base+EPPADDR);
}

/*check enhanced parallel port for timeout*/
int parport_pc_check_epp_timeout(struct parport *p)
{
	if(!(userdev_inportb(p->base+STATUS) & 1))
		return 0;
	parport_pc_epp_clear_timeout(p);
	return 1;
}

/*read a byte from configuration register*/
unsigned char parport_pc_read_configb(struct parport *p)
{
	return userdev_inportb(p->base+CONFIGB);
}

/*write byte to data register*/
void parport_pc_write_data(struct parport *p, unsigned char d)
{
	userdev_outportb(p->base+DATA,d);
}

/*read byte from data register*/
unsigned char parport_pc_read_data(struct parport *p)
{
	return userdev_inportb(p->base+DATA);
}

/*write byte to control register and update local copy*/
void parport_pc_write_control(struct parport *p, unsigned char d)
{
	struct parport_pc_private *priv = p->private_data;
	priv->ctr = d;		/* update soft copy */
	userdev_outportb(p->base+CONTROL,d);
}

/*read byte from control register, from local copy*/
unsigned char parport_pc_read_control(struct parport *p)
{
	struct parport_pc_private *priv = p->private_data;
	return priv->ctr;
}

/*function to mask the data in control register and compare with the passed value*/
unsigned char parport_pc_frob_control(struct parport *p, unsigned char mask,  unsigned char val)
{
	struct parport_pc_private *priv = p->private_data;
	unsigned char ctr = priv->ctr;
	ctr = (ctr & ~mask) ^ val;
	userdev_outportb(p->base+CONTROL,ctr);
	return priv->ctr = ctr; 	/* update soft copy */
}

/*write byte to status register*/
void parport_pc_write_status(struct parport *p, unsigned char d)
{
	userdev_outportb(p->base+STATUS,d);
}

/*read byte from status register*/
unsigned char parport_pc_read_status(struct parport *p)
{
	return userdev_inportb(p->base+STATUS);
}

/*write byte to control register of extended capabality port*/
void parport_pc_write_econtrol(struct parport *p, unsigned char d)
{
	userdev_outportb(p->base+ECONTROL,d);
}

/*read byte from control register of extended capabality port*/
unsigned char parport_pc_read_econtrol(struct parport *p)
{
	return userdev_inportb(p->base+ECONTROL);
}

/*function to mask the data in the control register of the extended capabality
 * port and compare the result with the passed value*/
unsigned char parport_pc_frob_econtrol(struct parport *p, unsigned char mask,  unsigned char val)
{
	unsigned char old = userdev_inportb(p->base+ECONTROL);
	userdev_outportb(p->base+ECONTROL,((old & ~mask) ^ val));
	return old;
}

/*write a byte to the CONFIGA register of extended capabality port*/
void parport_pc_write_fifo(struct parport *p, unsigned char v)
{
	userdev_outportb(p->base+CONFIGA,v);
}

/*read byte from CONGIGA register of extended capabality port*/
unsigned char parport_pc_read_fifo(struct parport *p)
{
	return userdev_inportb(p->base+CONFIGA);
}

/*disable interrupts from parallel port*/
void parport_pc_disable_irq(struct parport *p)
{
	parport_pc_frob_control(p, 0x10, 0);
}

/*enable interrupts from parallel port*/
void parport_pc_enable_irq(struct parport *p)
{
	parport_pc_frob_control(p, 0x10, 0x10);
}

/*function to release resources acquired by the parport*/
void parport_pc_release_resources(struct parport *p)
{
	parport_release_ioregion(p->base, p->size);	/*release io address space*/
	if(p->modes & PARPORT_MODE_PCECR)
		parport_release_ioregion(p->base+0x400, 3);
}

/*function to acquire resoources for the parport*/
int parport_pc_claim_resources(struct parport *p)
{
	parport_request_ioregion(p->base, p->size, p->name);	/*acquire io address space*/
	if(p->modes & PARPORT_MODE_PCECR)
		parport_request_ioregion(p->base+0x400, 3, p->name);
	return 0;
}

/*function to intialize the parport*/
void parport_pc_init_state(struct parport_state *s)
{
	s->u.pc.ctr = 0xc;
	s->u.pc.ecr = 0x0;
}

/*function to save control information in the parport's state variable*/
void parport_pc_save_state(struct parport *p, struct parport_state *s)
{
	s->u.pc.ctr = parport_pc_read_control(p);
	if(p->modes & PARPORT_MODE_PCECR)
		s->u.pc.ecr = parport_pc_read_econtrol(p);
}

/*function to restore control information from the parport's state variable*/
void parport_pc_restore_state(struct parport *p, struct parport_state *s)
{
	parport_pc_write_control(p, s->u.pc.ctr);
	if(p->modes & PARPORT_MODE_PCECR)
		parport_pc_write_econtrol(p, s->u.pc.ecr);
}

/*function to read a block of data from enhanced parallel port*/
size_t parport_pc_epp_read_block(struct parport *p, void *buf, size_t length)
{
	size_t got = 0;
	for(; got < length; got++)
   	{
		*((char*)buf)++ = userdev_inportb(p->base+EPPDATA);
		if (userdev_inportb(p->base+STATUS) & 0x01)
			break;
	}
	return got;
}

/*function to write a block of data to the enhanced parallel port*/
size_t parport_pc_epp_write_block(struct parport *p, void *buf, size_t length)
{
	size_t written = 0;
	for(; written < length; written++)
   	{
		userdev_outportb(p->base+EPPDATA,*((char*)buf)++);
		if (userdev_inportb(p->base+STATUS) & 0x01)
			break;
	}
	return written;
}

/*function to read a block of data from extended capabality port*/
int parport_pc_ecp_read_block(struct parport *p, void *buf, size_t length, void (*fn)(struct parport *, void *, size_t), void *handle)
{
	return -ENOSYS; 
}

/*function to write a block of data to the extended capabablity port*/
int parport_pc_ecp_write_block(struct parport *p, void *buf, size_t length, void (*fn)(struct parport *, void *, size_t), void *handle)
{
	return -ENOSYS; 
}


/*
//function to set permissions for all ports that are assoicated with a given
//parallel port interface
*/
int parport_pc_set_ioperms(int base)
{
	if(userdev_set_ioperm(base+DATA)) 	goto reset;
	if(userdev_set_ioperm(base+STATUS)) 	goto reset;
	if(userdev_set_ioperm(base+CONTROL)) 	goto reset;
	if(userdev_set_ioperm(base+EPPADDR)) 	goto reset;
	if(userdev_set_ioperm(base+EPPDATA)) 	goto reset;
	if(userdev_set_ioperm(base+CONFIGA)) 	goto reset;
	if(userdev_set_ioperm(base+CONFIGB)) 	goto reset;
	if(userdev_set_ioperm(base+ECONTROL)) 	goto reset;
	return 0;
reset:
	userdev_reset_ioperm(base+DATA);
	userdev_reset_ioperm(base+STATUS);
	userdev_reset_ioperm(base+CONTROL);
	userdev_reset_ioperm(base+EPPADDR);
	userdev_reset_ioperm(base+EPPDATA);
	userdev_reset_ioperm(base+CONFIGA);
	userdev_reset_ioperm(base+CONFIGB);
	userdev_reset_ioperm(base+ECONTROL);
	return -1;
}


/*
//function to reset permissions for all ports that are assoicated with a given
//parallel port interface
*/
void parport_pc_reset_ioperms(int base)
{
	userdev_reset_ioperm(base+DATA);
	userdev_reset_ioperm(base+STATUS);
	userdev_reset_ioperm(base+CONTROL);
	userdev_reset_ioperm(base+EPPADDR);
	userdev_reset_ioperm(base+EPPDATA);
	userdev_reset_ioperm(base+CONFIGA);
	userdev_reset_ioperm(base+CONFIGB);
	userdev_reset_ioperm(base+ECONTROL);
}


/*parport operations for parport_pc module*/
struct parport_operations parport_pc_ops = 
{
	parport_pc_write_data,
	parport_pc_read_data,

	parport_pc_write_control,
	parport_pc_read_control,
	parport_pc_frob_control,

	parport_pc_write_econtrol,
	parport_pc_read_econtrol,
	parport_pc_frob_econtrol,

	parport_pc_write_status,
	parport_pc_read_status,

	parport_pc_write_fifo,
	parport_pc_read_fifo,
	
	parport_pc_release_resources,
	parport_pc_claim_resources,
	
	parport_pc_write_epp,
	parport_pc_read_epp,
	parport_pc_write_epp_addr,
	parport_pc_read_epp_addr,
	parport_pc_check_epp_timeout,

	parport_pc_epp_write_block,
	parport_pc_epp_read_block,

	parport_pc_ecp_write_block,
	parport_pc_ecp_read_block,
	
	parport_pc_init_state,
	parport_pc_save_state,
	parport_pc_restore_state,

	parport_pc_enable_irq,
	parport_pc_disable_irq,
	
	parport_pc_set_ioperms,
	parport_pc_reset_ioperms
};


/* ============== Mode detection ==================== */

/* Clear TIMEOUT BIT in EPP MODE*/
int parport_pc_epp_clear_timeout(struct parport *pb)
{
	unsigned char r;

	if(!(parport_pc_read_status(pb) & 0x01))
		return 1;

	/* To clear timeout some chips require double read */
	parport_pc_read_status(pb);
	r = parport_pc_read_status(pb);
	parport_pc_write_status(pb, r | 0x01); 	/* Some reset by writing 1 */
	parport_pc_write_status(pb, r & 0xfe); 	/* Others by writing 0 */
	r = parport_pc_read_status(pb);

	return !(r & 0x01);
}


/* Checks for port existence, all ports support SPP MODE*/
static int parport_SPP_supported(struct parport *pb)
{
	unsigned char r, w;

	parport_pc_epp_clear_timeout(pb);

	/* Do a simple read-write test to make sure the port exists. */
	w = 0xc;
	parport_pc_write_control(pb, w);

	/* Can we read from the control register?  Some ports don't
	 * allow reads, so read_control just returns a software
	 * copy. Some ports _do_ allow reads, so bypass the software
	 * copy here.  In addition, some bits aren't writable. */
	r = userdev_inportb(pb->base+CONTROL);
	if((r & 0x3f) == w)
   	{
		w = 0xe;
		parport_pc_write_control (pb, w);
		r = userdev_inportb(pb->base+CONTROL);
		parport_pc_write_control (pb, 0xc);
		if ((r & 0x3f) == w)
			return PARPORT_MODE_PCSPP;
	}

	if(user_specified)
		fprintf(stderr,"0x%lx: CTR: wrote 0x%02x, read 0x%02x\n", pb->base, w, r);

	/* Try the data register.  The data lines aren't tri-stated at
	 * this stage, so we expect back what we wrote. */
	w = 0xaa;
	parport_pc_write_data (pb, w);
	r = parport_pc_read_data (pb);
	if(r == w) 
	{
		w = 0x55;
		parport_pc_write_data (pb, w);
		r = parport_pc_read_data (pb);
		if (r == w)
			return PARPORT_MODE_PCSPP;
	}

	if(user_specified)
		fprintf(stderr,"0x%lx: DATA: wrote 0x%02x, read 0x%02x\n", pb->base, w, r);

	/* It's possible that we can't read the control register or
	   the data register.  In that case just believe the user. */
	if(user_specified)
		return PARPORT_MODE_PCSPP;

	return 0;
}


/* Check for ECP
 *
 * Old style XT ports alias io ports every 0x400, hence accessing ECR
 * on these cards actually accesses the CTR.
 *
 * Modern cards don't do this but reading from ECR will return 0xff
 * regardless of what is written here if the card does NOT support
 * ECP.
 *
 * We will write 0x2c to ECR and 0xcc to CTR since both of these
 * values are "safe" on the CTR since bits 6-7 of CTR are unused.
 */
static int parport_ECR_present(struct parport *pb)
{
	unsigned char r;

	parport_pc_write_control (pb, 0xc);
	r = parport_pc_read_control(pb);	
	if((parport_pc_read_econtrol(pb) & 0x3) == (r & 0x3))
   	{
		parport_pc_write_control(pb, r ^ 0x2 ); /* Toggle bit 1 */

		r = parport_pc_read_control(pb);	
		if ((parport_pc_read_econtrol(pb) & 0x2) == (r & 0x2))
			goto no_reg; 	/* Sure that no ECR register exists */
	}
	
	if((parport_pc_read_econtrol(pb) & 0x3 ) != 0x1)
		goto no_reg;

	parport_pc_write_econtrol(pb, 0x34);
	if(parport_pc_read_econtrol(pb) != 0x35)
		goto no_reg;

	parport_pc_write_control(pb, 0xc);

	/* Go to mode 000; SPP, reset FIFO */
	parport_pc_frob_econtrol (pb, 0xe0, 0x00);
	
	return PARPORT_MODE_PCECR;

 no_reg:
	parport_pc_write_control (pb, 0xc);
	return 0;
}


/*check if Extended Capabality Port is supported*/
static int parport_ECP_supported(struct parport *pb)
{
	int i;
	unsigned char oecr;
	
	/* If there is no ECR, we have no hope of supporting ECP. */
	if(!(pb->modes & PARPORT_MODE_PCECR))
		return 0;

	oecr = parport_pc_read_econtrol(pb);
	/*
	 * Using LGS chipset it uses ECR register, but
	 * it doesn't support ECP or FIFO MODE
	 */
	
	parport_pc_write_econtrol(pb, 0xc0); /* TEST FIFO */
	for(i=0; i < 1024 && (parport_pc_read_econtrol(pb) & 0x01); i++)
		parport_pc_write_fifo(pb, 0xaa);

	parport_pc_write_econtrol(pb, oecr);
	return (i==1024)?0:PARPORT_MODE_PCECP;
}


/* EPP mode detection
 * Theory:
 *	Bit 0 of STR is the EPP timeout bit, this bit is 0
 *	when EPP is possible and is set high when an EPP timeout
 *	occurs (EPP uses the HALT line to stop the CPU while it does
 *	the byte transfer, an EPP timeout occurs if the attached
 *	device fails to respond after 10 micro seconds).
 *
 *	This bit is cleared by either reading it (National Semi)
 *	or writing a 1 to the bit (SMC, UMC, WinBond), others ???
 *	This bit is always high in non EPP modes.
 */
static int parport_EPP_supported(struct parport *pb)
{
	/* If EPP timeout bit clear then EPP available */
	if(!parport_pc_epp_clear_timeout(pb))
		return 0;  	/* No way to clear timeout */

	parport_pc_write_control(pb, parport_pc_read_control(pb) | 0x20);
	parport_pc_write_control(pb, parport_pc_read_control(pb) | 0x10);
	parport_pc_epp_clear_timeout(pb);
	
	parport_pc_read_epp(pb);
	userdev_udelay(30);  		/* Wait for possible EPP timeout */
	
	if(parport_pc_read_status(pb) & 0x01)
   	{
		parport_pc_epp_clear_timeout(pb);
		return PARPORT_MODE_PCEPP;
	}

	return 0;
}


/*check for ECP+EPP mode of operation*/
static int parport_ECPEPP_supported(struct parport *pb)
{
	int mode;
	unsigned char oecr;

	if(!(pb->modes & PARPORT_MODE_PCECR))
		return 0;

	oecr = parport_pc_read_econtrol(pb);
	/* Search for SMC style EPP+ECP mode */
	parport_pc_write_econtrol(pb, 0x80);
	
	mode = parport_EPP_supported(pb);

	parport_pc_write_econtrol(pb, oecr);
	
	return mode?PARPORT_MODE_PCECPEPP:0;
}


/* Detect PS/2 support.
 *
 * Bit 5 (0x20) sets the PS/2 data direction; setting this high
 * allows us to read data from the data lines.  In theory we would get back
 * 0xff but any peripheral attached to the port may drag some or all of the
 * lines down to zero.  So if we get back anything that isn't the contents
 * of the data register we deem PS/2 support to be present. 
 *
 * Some SPP ports have "half PS/2" ability - you can't turn off the line
 * drivers, but an external peripheral with sufficiently beefy drivers of
 * its own can overpower them and assert its own levels onto the bus, from
 * where they can then be read back as normal.  Ports with this property
 * and the right type of device attached are likely to fail the SPP test,
 * (as they will appear to have stuck bits) and so the fact that they might
 * be misdetected here is rather academic. 
 */
static int parport_PS2_supported(struct parport *pb)
{
	int ok = 0;
	unsigned char octr = parport_pc_read_control(pb);
  
	parport_pc_epp_clear_timeout(pb);

	parport_pc_write_control(pb, octr | 0x20);  /* try to tri-state the buffer */
	
	parport_pc_write_data(pb, 0x55);
	if(parport_pc_read_data(pb) != 0x55) ok++;

	parport_pc_write_data(pb, 0xaa);
	if(parport_pc_read_data(pb) != 0xaa) ok++;
	
	parport_pc_write_control(pb, octr);          /* cancel input mode */

	return ok?PARPORT_MODE_PCPS2:0;
}


/*check for ECP+PS2 mode of operation*/
static int parport_ECPPS2_supported(struct parport *pb)
{
	int mode;
	unsigned char oecr;

	if(!(pb->modes & PARPORT_MODE_PCECR))
		return 0;

	oecr = parport_pc_read_econtrol(pb);
	parport_pc_write_econtrol(pb, 0x20);
	
	mode = parport_PS2_supported(pb);

	parport_pc_write_econtrol(pb, oecr);
	return mode?PARPORT_MODE_PCECPPS2:0;
}


/* ================= IRQ detection ==================== */
/*
//The following dummy functions added in place of kernel functions.
//Need to be changed incase actual probing has to take place
//
//The probing mechanism is to
//		probe_on();
//		enable interrupt reporting
//		stimulate device to generate interrupt
//		probe_off() returns the irq number on which interrupt was received
*/

/*function to start irq probing*/
int probe_irq_on()	
{
	return 12345;
}


/*function to stop irq probing and return the irq on which interrupt was received*/
int probe_irq_off(int val)	
{
	return PARPORT_IRQ_NONE;
}


/*function to read irq number from device, Only if supports ECP mode*/
static int programmable_irq_support(struct parport *pb)
{
	int irq, intrLine;
	unsigned char oecr = parport_pc_read_econtrol(pb);
	static const int lookup[8] = {
		PARPORT_IRQ_NONE, 7, 9, 10, 11, 14, 15, 5
	};

	parport_pc_write_econtrol(pb,0xE0); 	/* Configuration MODE */
	
	intrLine = (parport_pc_read_configb(pb) >> 3) & 0x07;
	irq = lookup[intrLine];

	parport_pc_write_econtrol(pb, oecr);
	return irq;
}


/*function to probe Extended Capabality Parallel port for irq number*/
static int irq_probe_ECP(struct parport *pb)
{
	int irqs, i;

	/*Activate all interrupts - sti();*/
	irqs = probe_irq_on();
		
	parport_pc_write_econtrol(pb, 0x00);	/* Reset FIFO */
	parport_pc_write_econtrol(pb, 0xd0);	/* TEST FIFO + nErrIntrEn */

	/* If Full FIFO sure that WriteIntrThresold is generated */
	for (i=0; i < 1024 && !(parport_pc_read_econtrol(pb) & 0x02) ; i++) 
		parport_pc_write_fifo(pb, 0xaa);
		
	pb->irq = probe_irq_off(irqs);
	parport_pc_write_econtrol(pb, 0x00);

	if (pb->irq <= 0)
		pb->irq = PARPORT_IRQ_NONE;

	return pb->irq;
}


/*
//function to probe for irq on a enhanced parallel port
//This detection seems that only works in National Semiconductors
//This doesn't work in SMC, LGS, and Winbond 
*/
static int irq_probe_EPP(struct parport *pb)
{
	int irqs;
	unsigned char octr = parport_pc_read_control(pb);
	unsigned char oecr;

#ifndef ADVANCED_DETECT
	return PARPORT_IRQ_NONE;
#endif

	if (pb->modes & PARPORT_MODE_PCECR)
		oecr = parport_pc_read_econtrol(pb);

	/*Activate all interrupts - sti();*/
	irqs = probe_irq_on();

	if (pb->modes & PARPORT_MODE_PCECR)
		parport_pc_frob_econtrol (pb, 0x10, 0x10);
	
	parport_pc_epp_clear_timeout(pb);
	parport_pc_frob_control (pb, 0x20, 0x20);
	parport_pc_frob_control (pb, 0x10, 0x10);
	parport_pc_epp_clear_timeout(pb);

	/* Device isn't expecting an EPP read and generates an irq*/
	parport_pc_read_epp(pb);
	userdev_udelay(20);

	pb->irq = probe_irq_off (irqs);
	if (pb->modes & PARPORT_MODE_PCECR)
		parport_pc_write_econtrol(pb, oecr);
	parport_pc_write_control(pb, octr);

	if (pb->irq <= 0)
		pb->irq = PARPORT_IRQ_NONE;

	return pb->irq;
}


/*function to probe for irq on a simple parallel port*/
static int irq_probe_SPP(struct parport *pb)
{
	int irqs;
	unsigned char octr = parport_pc_read_control(pb);
	unsigned char oecr;

#ifndef ADVANCED_DETECT
	return PARPORT_IRQ_NONE;
#endif

	if (pb->modes & PARPORT_MODE_PCECR)
		oecr = parport_pc_read_econtrol(pb);
	probe_irq_off(probe_irq_on());	/* Clear any interrupts */
	irqs = probe_irq_on();

	if (pb->modes & PARPORT_MODE_PCECR)
		parport_pc_write_econtrol(pb, 0x10);

	parport_pc_write_data(pb,0x00);
	parport_pc_write_control(pb,0x00);
	parport_pc_write_control(pb,0x0c);
	userdev_udelay(5);
	parport_pc_write_control(pb,0x0d);
	userdev_udelay(5);
	parport_pc_write_control(pb,0x0c);
	userdev_udelay(25);
	parport_pc_write_control(pb,0x08);
	userdev_udelay(25);
	parport_pc_write_control(pb,0x0c);
	userdev_udelay(50);

	pb->irq = probe_irq_off(irqs);
	if (pb->irq <= 0)
		pb->irq = PARPORT_IRQ_NONE;	/* No interrupt detected */
	
	if (pb->modes & PARPORT_MODE_PCECR)
		parport_pc_write_econtrol(pb, oecr);
	parport_pc_write_control(pb, octr);
	return pb->irq;
}


/*
// We will attempt to share interrupt requests since other devices
// such as sound cards and network cards seem to like using the
// printer IRQs.
//
// When ECP is available we can autoprobe for IRQs.
// NOTE: If we can autoprobe it, we can register the IRQ.
*/
static int parport_irq_probe(struct parport *pb)
{
	if (pb->modes & PARPORT_MODE_PCECR) {
		pb->irq = programmable_irq_support(pb);
		if (pb->irq != PARPORT_IRQ_NONE)
			goto out;
	}

	if (pb->modes & PARPORT_MODE_PCECP)
		pb->irq = irq_probe_ECP(pb);

	if (pb->irq == PARPORT_IRQ_NONE && 
	    (pb->modes & PARPORT_MODE_PCECPEPP))
		pb->irq = irq_probe_EPP(pb);

	parport_pc_epp_clear_timeout(pb);

	if (pb->irq == PARPORT_IRQ_NONE && (pb->modes & PARPORT_MODE_PCEPP))
		pb->irq = irq_probe_EPP(pb);

	parport_pc_epp_clear_timeout(pb);

	if (pb->irq == PARPORT_IRQ_NONE)
		pb->irq = irq_probe_SPP(pb);

out:
	return pb->irq;
}


/* ============== Initialisation code ================== */


/*
//function to register a parallel port with port
//check if it can obtain permission to use io address spcae specified
//find out the type of interface and the capabalities present
//probe for the irq number on which its interrupt would be sent
//findout other information from the device and fill the parport structure
*/
static int probe_one_port(unsigned long int base, int irq, int dma)
{
	struct parport *p;
	int probedirq = PARPORT_IRQ_NONE;
	
	if (parport_check_ioregion(base, 3)) return 0;	/*check for io-region availabality*/
	if(!(p = parport_register_port(base, irq, dma, &parport_pc_ops)))
		return 0;		/*register with parport*/

	if(parport_pc_set_ioperms(base))/*set io permissions for all ports*/
	{
		fprintf(stderr,"io permisions(0x%x) could not be set\n",base);
		parport_unregister_port(p);
		return 0;
	}

	p->private_data = (struct parport_pc_private*)malloc(sizeof(struct parport_pc_private));
	if(!p->private_data)
   	{
		fprintf(stderr,"parport (0x%lx): no memory!\n", base);
		parport_unregister_port (p);
		return 0;
	}
	((struct parport_pc_private *)(p->private_data))->ctr = 0xc;	/*initial value*/

	if(p->base != 0x3bc)	/*check io region and presence of various capabablities of the port*/
   	{
		if(!parport_check_ioregion(base+0x400,3))
	   	{
			p->modes |= parport_ECR_present(p);	
			p->modes |= parport_ECP_supported(p);
			p->modes |= parport_ECPPS2_supported(p);
		}
		if(!parport_check_ioregion(base+0x3, 5))
	   	{
			p->modes |= parport_EPP_supported(p);
			p->modes |= parport_ECPEPP_supported(p);
		}
	}

	if(!parport_SPP_supported(p))
   	{
		free (p->private_data);
		parport_unregister_port (p);
		return 0;
	}

	p->modes |= PARPORT_MODE_PCSPP | parport_PS2_supported(p);
	p->size = (p->modes & (PARPORT_MODE_PCEPP | PARPORT_MODE_PCECPEPP))?8:3;

	if(p->irq == PARPORT_IRQ_AUTO)	/*probe for irq and find out irq number*/
   	{
		p->irq = PARPORT_IRQ_NONE;
		parport_irq_probe(p);
	}
   	else if(p->irq == PARPORT_IRQ_PROBEONLY)
   	{
		p->irq = PARPORT_IRQ_NONE;
		parport_irq_probe(p);
		probedirq = p->irq;
		p->irq = PARPORT_IRQ_NONE;
	}

	if (probedirq != PARPORT_IRQ_NONE) 
		printf("%s: detected irq %d; use procfs to enable interrupt-driven operation.\n", p->name, probedirq);

	printf("%s: PC-style at 0x%lx", p->name, p->base);

	if(p->irq != PARPORT_IRQ_NONE)
		printf(", irq %d", p->irq);

	if(p->dma == PARPORT_DMA_AUTO)		
		p->dma = PARPORT_DMA_NONE;
	if (p->dma != PARPORT_DMA_NONE)
		printf(", dma %d", p->dma);

	printf(" [");	/*print out the details*/
#define printmode(x) {if(p->modes&PARPORT_MODE_PC##x) { printf("%s%s",f?",":"",#x); f++; } }
	{
		int f = 0;
		printmode(SPP);
		printmode(PS2);
		printmode(EPP);
		printmode(ECP);
		printmode(ECPEPP);
		printmode(ECPPS2);
	}
#undef printmode
	printf("]\n");

	p->flags |= PARPORT_FLAG_COMA;

			/* Done probing.  Now put the port into a sensible start-up state. */
	if (p->modes & PARPORT_MODE_PCECR)
		parport_pc_write_econtrol(p, 0x0);	/* Put the ECP detected port in the more SPP like mode.*/
	parport_pc_write_control(p, 0xc);
	parport_pc_write_data(p, 0);

	if(parport_probe_hook)
		(*parport_probe_hook)(p);

	return 1;
}


/*function to initialize the user specified or standard ports*/
int parport_pc_init(int *io, int *irq, int *dma)
{
	int count = 0, i = 0;
	if(io && *io)	/*ports given by user*/
   	{
		user_specified = 1;
		do
	   	{
			printf("Initializing io = 0x%x, irq = %d, dma = %d\n",*io,*irq,*dma);
			count += probe_one_port(*(io++), *(irq++), *(dma++));
		} while(*io && (++i < PARPORT_PC_MAX_PORTS));
	} 
	else		/*default values*/
   	{
		count += probe_one_port(0x3bc, irq[0], dma[0]);
		count += probe_one_port(0x378, irq[0], dma[0]);
		count += probe_one_port(0x278, irq[0], dma[0]);
	}

	return count;
}


/*======static data structures of the parport_pc module======*/

/*base address of the io address region*/
static int io[PARPORT_PC_MAX_PORTS+1] = { [0 ... PARPORT_PC_MAX_PORTS] = 0 };

/*dma channnel to be used*/
static int dma[PARPORT_PC_MAX_PORTS] = { [0 ... PARPORT_PC_MAX_PORTS-1] = PARPORT_DMA_NONE };

/*irq line to be used*/
static int irqval[PARPORT_PC_MAX_PORTS] = { [0 ... PARPORT_PC_MAX_PORTS-1] = PARPORT_IRQ_PROBEONLY };

/*irq as a string, either 'auto' or 'none' or a specific irq number*/
static char *irq[PARPORT_PC_MAX_PORTS] = { NULL, };

# define __PARPORT_ARGS_SEP__ 	','	/*separator in the argument string*/


/*
//function to parse the user specified string argument to separate out the io
//address, dma channel number and the irq string and store these values in the
//above static data structures
*/
void parport_pc_parse_args(int argc, char *argv[])
{
	int i,c;
	char *ptr,*oldptr,*tmp;
	
	c = 0;
	for(i=0;i<argc;i++)	/*parse each string in the set of strings*/
	{			/*with each component separater by __PARPORT_ARGS_SEP__*/
		oldptr = argv[i];
		ptr = strchr(oldptr,__PARPORT_ARGS_SEP__);
		if (ptr == NULL) 
			continue;
		tmp = (char*)malloc(ptr-oldptr+1);
		memset(tmp,0,ptr-oldptr);
		strncpy(tmp,oldptr,ptr-oldptr);
		tmp[ptr-oldptr]='\0';
		io[c] = strtol(tmp,NULL,16);	/*obtain io address base*/
		free(tmp);

		oldptr = ptr+1;
		ptr = strchr(oldptr,__PARPORT_ARGS_SEP__);
		if (ptr == NULL)
	   	{
			io[c]=0;
			continue;
		}
		tmp = (char*)malloc(ptr-oldptr+1);
		memset(tmp,0,ptr-oldptr);
		strncpy(tmp,oldptr,ptr-oldptr);
		tmp[ptr-oldptr]='\0';
		dma[c] = strtol(tmp,NULL,10);	/*obtain dma channel number*/
		free(tmp);

		oldptr = ptr+1;
		if (oldptr != NULL)
		{
			irq[c] = (char*)malloc(strlen(oldptr)+1);
			strcpy(irq[c],oldptr);	/*obtain irq string*/
		}
		else
	   	{
			io[c]=dma[c]=0;
		   	continue;
		}
		
		c++;
	}
}


/*function to initialize parport_pc module*/
int parport_pc_initialize(int argc, char *argv[])
{	
	unsigned int i,j;

	parport_pc_parse_args(argc,argv);	/*parse arguments*/
	for (i = 0; i < PARPORT_PC_MAX_PORTS && io[i]; i++);	/*find out number of valid ports specified*/
	parport_parse_irqs(i, irq, irqval);	/*convert irq string to irq number*/
	for(j=0;j<i;j++) free(irq[i]);
	return (parport_pc_init(io, irqval, dma)?0:1);		/*initialize the specified ports*/
}


/*function to cleanup parport_pc module*/
void parport_pc_cleanup(void)
{
	struct parport *p = parport_enumerate(), *tmp;	/*obtain the list of ports*/
	while(p)	/*for each*/
   	{
		tmp = p->next;
		if(p->modes & PARPORT_MODE_PCSPP)
	   	{ 
			if(!(p->flags & PARPORT_FLAG_COMA)) 
				parport_quiesce(p);	/*release resources*/
			free(p->private_data);
			parport_pc_reset_ioperms(p->base);	/*reset all io permisssions*/
			parport_unregister_port(p);	/*unregister port*/
		}
		p = tmp;
	}
}
