/*
//file contains the data structures and function declarations of for parport.c
//which forms the main part of the parport device library
*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

#ifndef __USERDEV_PARPORT_H__
#define __USERDEV_PARPORT_H__

# include <asm/ptrace.h>

/* Maximum of 8 ports per machine */
#define PARPORT_MAX  8 

/* Magic numbers */
#define PARPORT_IRQ_NONE  	-1
#define PARPORT_DMA_NONE  	-1
#define PARPORT_IRQ_AUTO  	-2
#define PARPORT_DMA_AUTO  	-2
#define PARPORT_DISABLE   	-2
#define PARPORT_IRQ_PROBEONLY 	-3

/*bit masks for control register*/
#define PARPORT_CONTROL_STROBE    0x1
#define PARPORT_CONTROL_AUTOFD    0x2
#define PARPORT_CONTROL_INIT      0x4
#define PARPORT_CONTROL_SELECT    0x8
#define PARPORT_CONTROL_INTEN     0x10
#define PARPORT_CONTROL_DIRECTION 0x20

/*bit masks for status register*/
#define PARPORT_STATUS_ERROR      0x8
#define PARPORT_STATUS_SELECT     0x10
#define PARPORT_STATUS_PAPEROUT   0x20
#define PARPORT_STATUS_ACK        0x40
#define PARPORT_STATUS_BUSY       0x80


/* Type classes for Plug-and-Play probe.  */
typedef enum {
	PARPORT_CLASS_LEGACY = 0,/* Non-IEEE1284 device */
	PARPORT_CLASS_PRINTER,
	PARPORT_CLASS_MODEM,
	PARPORT_CLASS_NET,
	PARPORT_CLASS_HDC,      /* Hard disk controller */
	PARPORT_CLASS_PCMCIA,
	PARPORT_CLASS_MEDIA,    /* Multimedia device */
	PARPORT_CLASS_FDC,      /* Floppy disk controller */
	PARPORT_CLASS_PORTS,
	PARPORT_CLASS_SCANNER,
	PARPORT_CLASS_DIGCAM,
	PARPORT_CLASS_OTHER,    /* Anything else */
	PARPORT_CLASS_UNSPEC	/* No CLS field in ID */
} parport_device_class;

/* The "modes" entry in parport is a bit field representing the following
 * modes.
 * Note that PARPORT_MODE_PCECPEPP is for the SMC EPP+ECP mode which is NOT
 * 100% compatible with EPP.
 */
#define PARPORT_MODE_PCSPP	0x0001
#define PARPORT_MODE_PCPS2	0x0002
#define PARPORT_MODE_PCEPP	0x0004
#define PARPORT_MODE_PCECP	0x0008
#define PARPORT_MODE_PCECPEPP	0x0010
#define PARPORT_MODE_PCECR	0x0020  /* ECR Register Exists */
#define PARPORT_MODE_PCECPPS2	0x0040


/* Define this later. */
struct parport;

/*parport_pc state, used in structure below*/
struct pc_parport_state {
	unsigned int ctr;
	unsigned int ecr;
};

/* parport state, union of state of various other modules that register ports
 * with parport such as parport_pc module
 */
struct parport_state {
	union {
		struct pc_parport_state pc;
		void *misc; 
	} u;
};

/*list of operations supported be parport interface*/
struct parport_operations {
	void (*write_data)(struct parport *, unsigned char);
	unsigned char (*read_data)(struct parport *);
	void (*write_control)(struct parport *, unsigned char);
	unsigned char (*read_control)(struct parport *);
	unsigned char (*frob_control)(struct parport *, unsigned char mask, unsigned char val);
	void (*write_econtrol)(struct parport *, unsigned char);
	unsigned char (*read_econtrol)(struct parport *);
	unsigned char (*frob_econtrol)(struct parport *, unsigned char mask, unsigned char val);
	void (*write_status)(struct parport *, unsigned char);
	unsigned char (*read_status)(struct parport *);
	void (*write_fifo)(struct parport *, unsigned char);
	unsigned char (*read_fifo)(struct parport *);

    	void (*release_resources)(struct parport *);
	int (*claim_resources)(struct parport *);

	void (*epp_write_data)(struct parport *, unsigned char);
	unsigned char (*epp_read_data)(struct parport *);
	void (*epp_write_addr)(struct parport *, unsigned char);
	unsigned char (*epp_read_addr)(struct parport *);
	int (*epp_check_timeout)(struct parport *);
	size_t (*epp_write_block)(struct parport *, void *, size_t);
	size_t (*epp_read_block)(struct parport *, void *, size_t);

	int (*ecp_write_block)(struct parport *, void *, size_t, void (*fn)(struct parport *, void *, size_t), void *);
	int (*ecp_read_block)(struct parport *, void *, size_t, void (*fn)(struct parport *, void *, size_t), void *);

	void (*init_state)(struct parport_state *);
	void (*save_state)(struct parport *, struct parport_state *);
	void (*restore_state)(struct parport *, struct parport_state *);

	void (*enable_irq)(struct parport *);
	void (*disable_irq)(struct parport *);
	
	int (*set_ioperms)(int);
	void (*reset_ioperms)(int);
};

/*structure giving details of the device registered with parport*/
struct parport_device_info {
	parport_device_class class;	/*device class*/
	const char *class_name;		/*class name*/
	const char *mfr;		/*manufacturer name*/
	const char *model;		/*model name*/
	const char *cmdset;		/*command set*/
	const char *description;	/*description*/
};

/* A parallel port device */
struct pardevice {
	const char *name;		/*device name*/
	struct parport *port;		/*reference to port on which registered*/
	unsigned int flags;		/*device flags*/
	struct parport_state *state;   	/* saved status over preemption */
};

/* A parallel port */
struct parport {
	unsigned long base;			/*base address */
	unsigned int size;			/*IO extent */
	const char *name;			/*port name*/
	int irq;				/*interrupt (or -1 for none) */
	int dma;				/*dma channel no:*/
	unsigned int modes;			/*mode of operation*/
    	struct pardevice *device;		/*device that has registered with the port*/
	struct parport *next;			/*reference to next port*/
	unsigned int flags;			/*flags*/
	struct parport_device_info probe_info; 	/*information of the device registered*/
	struct parport_operations *ops;		/*pointer to parport operations*/
	void *private_data;     		/* for lowlevel driver */
	int number;				/* port index - the `n' in `parportn' */
};


/* parport_register_port registers a new parallel port at the given address (if
 * one does not already exist) and returns a pointer to it.  This entails
 * claiming the I/O region, IRQ and DMA.
 * NULL is returned if initialisation fails. 
 */
struct parport *parport_register_port(unsigned long base, int irq, int dma, struct parport_operations *ops);

/* Unregister a port. */
extern void parport_unregister_port(struct parport *port);

/* release port's hardware resources. */
extern void parport_quiesce(struct parport *);

/* parport_enumerate returns a pointer to the linked list of all the ports
 * in this machine.
 */
struct parport *parport_enumerate(void);

/* parport_register_device declares that a device is connected to a port*/
struct pardevice *parport_register_device(struct parport *port, const char *name, int flags);

/* parport_unregister unregisters a device from the port it is connected to. */
extern void parport_unregister_device(struct pardevice *dev);


/* parport_in_use returns nonzero if there is a device attached to a port. */
#define parport_in_use(x)  ((x)->device != NULL)

/* Flags used to identify what a device does. */
#define PARPORT_DEV_TRAN	0	/* WARNING !! DEPRECATED !! */
#define PARPORT_DEV_EXCL	(1<<1)	/* Need exclusive access. */

/*Flags to moderate port access*/
#define PARPORT_FLAG_COMA	(1<<0)
#define PARPORT_FLAG_EXCL	(1<<1)	/* EXCL driver registered. */


extern void parport_parse_irqs(int, char *[], int irqval[]);
extern int parport_check_ioregion(int,int);
extern int parport_request_ioregion(int,int, const char*);
extern int parport_release_ioregion(int,int);
extern int parport_ieee1284_nibble_mode_ok(struct parport *, unsigned char);
extern int parport_wait_peripheral(struct parport *, unsigned char, unsigned char);

extern int parport_probe(struct parport *port, char *buffer, int len);
extern void parport_probe_one(struct parport *port);
extern void (*parport_probe_hook)(struct parport *port);


/* Generic operations vector through the dispatch table. */
#define parport_write_data(p,x)            (p)->ops->write_data(p,x)
#define parport_read_data(p)               (p)->ops->read_data(p)
#define parport_write_control(p,x)         (p)->ops->write_control(p,x)
#define parport_read_control(p)            (p)->ops->read_control(p)
#define parport_frob_control(p,m,v)        (p)->ops->frob_control(p,m,v)
#define parport_write_econtrol(p,x)        (p)->ops->write_econtrol(p,x)
#define parport_read_econtrol(p)           (p)->ops->read_econtrol(p)
#define parport_frob_econtrol(p,m,v)       (p)->ops->frob_econtrol(p,m,v)
#define parport_write_status(p,v)          (p)->ops->write_status(p,v)
#define parport_read_status(p)             (p)->ops->read_status(p)
#define parport_write_fifo(p,v)            (p)->ops->write_fifo(p,v)
#define parport_read_fifo(p)               (p)->ops->read_fifo(p)
#define parport_release_resources(p)       (p)->ops->release_resources(p)
#define parport_claim_resources(p)         (p)->ops->claim_resources(p)
#define parport_epp_write_data(p,x)        (p)->ops->epp_write_data(p,x)
#define parport_epp_read_data(p)           (p)->ops->epp_read_data(p)
#define parport_epp_write_addr(p,x)        (p)->ops->epp_write_addr(p,x)
#define parport_epp_read_addr(p)           (p)->ops->epp_read_addr(p)
#define parport_epp_check_timeout(p)       (p)->ops->epp_check_timeout(p)
#define parport_enable_irq(p)		   (p)->ops->enable_irq(p)
#define parport_disable_irq(p)		   (p)->ops->disable_irq(p)
#define parport_set_ioperms(p)		   (p)->ops->set_ioperms(p->base)
#define parport_reset_ioperms(p)	   (p)->ops->reset_ioperms(p->base)

#endif /* __USERDEV_PARPORT_H__ */
