/* file contains implementation of functions of parport library module*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <errno.h>
# include "usrdriv.h"	/*for using microsecond delay function and io region access functions*/
# include "parport.h"

/*list of parallel port structures maintained by the process*/
static struct parport *portlist = NULL, *portlist_tail = NULL;

/*function pointer for requesting probing of parallel ports*/
void (*parport_probe_hook)(struct parport *port) = NULL;


/*function to return list of parallel ports available*/
struct parport *parport_enumerate(void)
{
	return portlist;
}


/* function to register a parallel port with parport module along with the set
 * of operations that implement access to the port
 */
struct parport *parport_register_port(unsigned long base, int irq, int dma, struct parport_operations *ops)
{
	struct parport *tmp;
	int portnum;
	char *name;

	if(ops == NULL)	/*check if the parport_operations is present*/
	{
		fprintf(stderr,"Parport operations missing\n");
		return NULL;
	}

	for (tmp = portlist; tmp; tmp = tmp->next)	/*search through the existing port list if a node exists with given port# */
		if (tmp->base == base) return tmp;	/*if so just return it*/

	tmp = (struct parport*)malloc(sizeof(struct parport));	/*else allocate a new one*/
	if (!tmp)
   	{
		fprintf(stderr,"Lack of memory\n");
		return NULL;
	}

	for (portnum = 0; ; portnum++) 		/*find an unused identification number for the port*/
	{
		struct parport *itr = portlist;
		while (itr) 
			if (itr->number == portnum) break;
			else itr = itr->next;

		if (itr == NULL) break;
	}
	
 	memset(tmp, 0, sizeof(struct parport));	/*initialize the port structure with the values passed*/
	tmp->base = base;
	tmp->irq = irq;
	tmp->dma = dma;
	tmp->modes = 0;
 	tmp->next = NULL;
	tmp->device = NULL;
	tmp->flags = 0;
	tmp->ops = ops;
	tmp->number = portnum;
	memset(&tmp->probe_info, 0, sizeof (struct parport_device_info));

	name = (char*)malloc(15);
	if (!name)
   	{
		fprintf(stderr,"Lack of memory\n");
		free(tmp);
		return NULL;
	}
	sprintf(name, "parport%d", portnum);
	tmp->name = name;

	if (portlist_tail)	/*add the port to the port list*/
		portlist_tail->next = tmp;
	portlist_tail = tmp;
	if (!portlist)
		portlist = tmp;

	tmp->probe_info.class = PARPORT_CLASS_LEGACY;  /* assume the worst */
	return tmp;
}


/*function to unregister a given port*/
void parport_unregister_port(struct parport *port)
{
	struct parport *p;

	if (portlist == port)	/*if the first one in the port list*/
	{
		if ((portlist = port->next) == NULL)	/*update port list reference*/
			portlist_tail = NULL;
	}
	else
   	{
		for (p = portlist; (p != NULL) && (p->next != port); p = p->next);
		if (p)		/*find the port structure in the list and remove it from list*/
		{
			if ((p->next = port->next) == NULL)
				portlist_tail = p;
		}
		else fprintf(stderr,"%s not found in portlist\n",port->name);
	}

	if (port->probe_info.class_name)	/*free the memory occupied by port structure's components*/
		free (port->probe_info.class_name);
	if (port->probe_info.mfr)
		free (port->probe_info.mfr);
	if (port->probe_info.model)
		free (port->probe_info.model);
	if (port->probe_info.cmdset)
		free (port->probe_info.cmdset);
	if (port->probe_info.description)
		free (port->probe_info.description);
	free(port->name);
	free(port);
}


/*function to release resources of the port*/
void parport_quiesce(struct parport *port)
{
	if (port->device)	/*check if a device is registered with the port in which case return*/
	{
		fprintf(stderr,"%s: attempt to quiesce active port.\n",port->name);
		return;
	}
	if (port->flags & PARPORT_FLAG_COMA)	/*check if the port has already released resouces*/
	{
		fprintf(stderr,"%s: attempt to quiesce comatose port\n",port->name);
		return;
	}

	if(port->ops->release_resources)	/*call release resource function of the port*/
		port->ops->release_resources(port);
	port->flags |= PARPORT_FLAG_COMA; 	/*note release of resources*/
}


/*function to register a device with a given port*/
struct pardevice *parport_register_device(struct parport *port, const char *name, int flags)
{
	struct pardevice *tmp;

	if (port->flags & PARPORT_FLAG_EXCL)	/*check if port has been set in exclusive mode by some device*/
	{
		fprintf(stderr,"%s:Exclusive access of devices needed, no more accomadated\n",port->name);
		return NULL;
	}

	tmp = (struct pardevice*)malloc(sizeof(struct pardevice));	/*otherwise allocate a new device structure*/
	if (tmp == NULL)
   	{
		fprintf(stderr,"%s: memory squeeze, couldn't register %s.\n", port->name, name);
		return NULL;
	}

	tmp->state = (struct parport_state*)malloc(sizeof(struct parport_state));
	if (tmp->state == NULL)
   	{
		fprintf(stderr,"%s: memory squeeze, couldn't register %s.\n", port->name, name);
		free(tmp);
		return NULL;
	}

	if (port->flags & PARPORT_FLAG_COMA)	/*if port has no resources, acquire them*/
   	{
		if ((port->ops->claim_resources) && (port->ops->claim_resources(port)))
	   	{
			fprintf(stderr,"%s: unable to get hardware to register %s.\n", port->name, name);
			free (tmp->state);
			free (tmp);
			return NULL;
		}
		port->flags &= ~PARPORT_FLAG_COMA;	/*note that the port has resources*/
	}

	tmp->name = name;	/*fill in the pardevice structure*/
	tmp->port = port;
	tmp->flags = flags;
	if (port->ops->init_state)	/*initialization function is called if port is not initialized*/
		port->ops->init_state(tmp->state);

	if (flags & PARPORT_DEV_EXCL)	/*if device requires exclusive access*/
   	{
		if (port->device)	/*if a device has already registered with the port then quit*/
	   	{
			free (tmp->state);
			free (tmp);
			fprintf(stderr,"%s: cannot grant exclusive access for device %s\n", port->name, name); 
			return NULL;
		}
		port->flags |= PARPORT_FLAG_EXCL;	/*set the exclusive flag for the port*/
	}

	port->device = tmp;
	return tmp;
}


/*function to unregister a device from a port*/
void parport_unregister_device(struct pardevice *dev)
{
	struct parport *port;

	port = dev->port;

	if (dev->flags & PARPORT_DEV_EXCL)	/*reset flags of the port if the device requested exclussive access*/
		port->flags &= ~PARPORT_FLAG_EXCL;

	free(dev->state);	/*free the pardevice data structure*/
	free(dev);

	port->device = NULL;	/*release port resources and set device reference to NULL*/
	parport_quiesce(port);
	return;
}


/* function to convert string to long value irrespective of whether its a
 * decimal, octal or hexadecimal value
 */
unsigned long simple_strtoul(const char *cp, char **endp, unsigned int base)
{
    	unsigned long result = 0,value;
			 
    	if (!base) 		/*if no base specified then detect it*/
	{
        	base = 10;
        	if (*cp == '0')	/*if starts with 0 then octal or hexadecimal*/
	   	{
            		base = 8;
            		cp++;
            		if ((*cp == 'x') && isxdigit(cp[1]))/*if next we have x then its hexadecimal*/
		   	{
                		cp++;
                		base = 16;
            		}
        	}
    	}

	/*read each character of the string till we find a non-numerical character*/
    	while (isxdigit(*cp) && (value = isdigit(*cp) ? *cp-'0' : (islower(*cp)
					  ? toupper(*cp) : *cp)-'A'+10) < base)
   	{
        	result = result*base + value;
        	cp++;
    	}

    	if (endp)	/*return pointer to rest of the string*/
        	*endp = (char *)cp;
    	return result;
}


/*function to parse irq type string specified and fill in the associated  numerical value*/
void parport_parse_irqs(int nports, char *irqstr[], int irqval[])
{
	unsigned int i;
	for (i = 0; i < nports && irqstr[i]; i++)	/*read each string in irqstr*/
   	{
		if (!strncmp(irqstr[i], "auto", 4))	/*if = auto*/
			irqval[i] = PARPORT_IRQ_AUTO;
		else if (!strncmp(irqstr[i], "none", 4))
			irqval[i] = PARPORT_IRQ_NONE;	/*if = none*/
		else 
		{
			char *ep;	/*else a numerical value then parse it and obtain integral value*/
			unsigned long r = simple_strtoul(irqstr[i], &ep, 0);
			if (ep != irqstr[i])
				irqval[i] = r;
			else 
			{
				fprintf(stderr,"parport: bad irq specifier `%s'\n", irqstr[i]);
				return;
			}
		}
	}
}


/*function to check if the specified ioregion is available*/
int parport_check_ioregion(int base, int num)
{
	return userdev_check_ioregion(base,num);
}


/*function to request for the given ioregion*/
int parport_request_ioregion(int base, int num, const char *name)
{
	return userdev_request_ioregion(base,num,name);
}


/*function to release the specified ioregion*/
int parport_release_ioregion(int base, int num)
{
	return userdev_release_ioregion(base,num);
}


/*function to intialize the parport library module*/
/*the arguments to the function are for the parport_pc module*/
void parport_initialize(int argc, char *args[])
{
	int ret;

	userdev_calibrate_delay();		/*calibrate loops_per_usec for delay operations*/
	ret = parport_probe_init();	/*initialize parport_probe module*/
	if (!ret) ret = parport_pc_initialize(argc, args);	/*intialize parport_pc module */
}


/*function to cleanup parport module data structure when the program exits*/
void parport_cleanup()
{
	parport_pc_cleanup();
	parport_probe_cleanup();
}
