/*Architecture specific parts of the Floppy driver*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

#ifndef __USERDEV_FLOPPY_H__
#define __USERDEV_FLOPPY_H__

/*
 * The DMA channel used by the floppy controller cannot access data at
 * addresses >= 16MB
 *
 * Went back to the 1MB limit, as some people had problems with the floppy
 * driver otherwise. It doesn't matter much for performance anyway, as most
 * floppy accesses go through the track buffer.
 */

#define SW fd_routine[use_virtual_dma&1]
#define CSW fd_routine[can_use_virtual_dma & 1]

#define fd_inb(port)		userdev_inportb(port)
#define fd_outb(value,port)	userdev_outportb(port,value)

#define fd_request_dma(id)      CSW._request_dma(FLOPPY_DMA,id)
#define fd_free_dma(id)         CSW._free_dma(id)
#define fd_free_irq(id)		userdev_free_irq(id)
#define fd_get_dma_residue(id)  SW._get_dma_residue(id)
#define fd_dma_mem_alloc(size)	SW._dma_mem_alloc(size)
#define fd_dma_setup(addr, size, mode, io, id) 	SW._dma_setup(addr, size, mode, io, id)

#define FLOPPY_CAN_FALLBACK_ON_NODMA

/*variables for virtual dma*/
static int virtual_dma_count;	/*count*/
static int virtual_dma_residue;	/*residue*/
static char *virtual_dma_addr;	/*address*/
static int virtual_dma_mode;	/*mode*/
static int doing_pdma;		/*flag that denotes whether hard dma or virtual dma is being used*/

static int dma_enabled = 0;	/*flag to keep track if dma is enabled or not, so that multiple calls to kernel are avoided*/
pthread_mutex_t dma_enabled_mutex = PTHREAD_MUTEX_INITIALIZER;	/*mutex for above*/

/*floppy interrupt handler*/
static void floppy_hardint(int sig, siginfo_t *info, void *ptr)
{
	register unsigned char st;

	if(!doing_pdma) 	/*if hard dma in operation then call floppy_interrupt function*/
	{
		floppy_interrupt(sig,info,ptr);
		return;
	}

	/*case of virtual dma mode*/
	{
		register int lcount;
		register char *lptr;

		st = 1;
		/*directly perform in/out from the device to simulate DMA operation*/
		for(lcount=virtual_dma_count, lptr=virtual_dma_addr;lcount; lcount--, lptr++) 
		{
			st=userdev_inportb(virtual_dma_port+4) & 0xa0 ;
			if(st != 0xa0)
				break;
			if(virtual_dma_mode)	/*WRITE*/
				userdev_outportb(virtual_dma_port+5, *lptr);
			else			/*READ*/
				*lptr = userdev_inportb(virtual_dma_port+5);
		}
		virtual_dma_count = lcount;
		virtual_dma_addr = lptr;
		st = userdev_inportb(virtual_dma_port+4);	/*read status*/
	}

	if(st == 0x20) return;
	if(!(st & 0x20)) 	/*if status says successful, update local variables and call floppy_interrupt*/
	{
		virtual_dma_residue += virtual_dma_count;
		virtual_dma_count=0;
		doing_pdma = 0;
		floppy_interrupt(sig, info, ptr);
		return;
	}
}

/*function to disable dma on the specified device*/
static void fd_disable_dma(int device_id)
{

	if(! (can_use_virtual_dma & 1))	/*incase hard dma is being used*/
	{
		pthread_mutex_lock(&dma_enabled_mutex);
		if(dma_enabled)		/*incase dma is enabled*/
		{
			dma_enabled = 0;
			userdev_disable_dma(device_id);	/*disable dma*/
		}
		pthread_mutex_unlock(&dma_enabled_mutex);
	}
	doing_pdma = 0;			/*virtual dma case, just update*/
	virtual_dma_residue += virtual_dma_count;
	virtual_dma_count=0;
}

/*virtual dma request_dma_channel function*/
static int vdma_request_dma(int dmanr, int device_id)
{
	return 0;
}

/*no operation*/
static int vdma_nop(int dummy)
{
	return 0;
}

/*virtual dma get_dma_residue function*/
static int vdma_get_dma_residue(int dummy)
{
	return virtual_dma_count + virtual_dma_residue;
}

/*function to install floppy interrupt handler for the given device*/
static int fd_request_irq(int device_id)
{
	if(can_use_virtual_dma)	
		return userdev_request_irq(FLOPPY_IRQ, FLOPPY_SIG, floppy_hardint, device_id);
	else
		return userdev_request_irq(FLOPPY_IRQ, FLOPPY_SIG, floppy_interrupt, device_id);
}

/*function to allocate memory for hard dma operation*/
static unsigned long dma_mem_alloc(unsigned long size)
{
	return (unsigned long) malloc(size);
}

/*function to allocate memory for virtual dma operation*/
static unsigned long vdma_mem_alloc(unsigned long size)
{
	return (unsigned long) malloc(size);

}

#define nodma_mem_alloc(size) vdma_mem_alloc(size)

/*function to  free dma memory*/
static void _fd_dma_mem_free(unsigned long addr, unsigned long size)
{
	free((void *)addr);
}

#define fd_dma_mem_free(addr, size)  _fd_dma_mem_free(addr, size) 

/*function to chose the type of dma operation*/
static void _fd_chose_dma_mode(char *addr, unsigned long size)
{
	if(can_use_virtual_dma == 2) 
		use_virtual_dma = 1;
	else 
		use_virtual_dma = can_use_virtual_dma & 1;
}

#define fd_chose_dma_mode(addr, size) _fd_chose_dma_mode(addr, size)

/*function to initialise parameters for virtual dma*/
static int vdma_dma_setup(char *addr, unsigned long size, int mode, int io, int device_id)
{
	doing_pdma = 1;
	virtual_dma_port = io;
	virtual_dma_mode = (mode == DMA_WRITE);
	virtual_dma_addr = addr;
	virtual_dma_count = size;
	virtual_dma_residue = 0;
	return 0;
}

/*function to initialise parameters for hard dma*/
static int hard_dma_setup(char *addr, unsigned long size, int mode, int io, int device_id)
{
	/* actual, physical DMA */
	pthread_mutex_lock(&dma_enabled_mutex);
	dma_enabled = 1;	/*set dma enabled flag*/
	pthread_mutex_unlock(&dma_enabled_mutex);

	doing_pdma = 0;
	return userdev_start_dma(mode,size,addr,device_id);	/*start dma operation*/
}

/*generic structure for dma operation*/
struct fd_routine_l {
	int (*_request_dma)(int dmanr, int device_id);	/*request for dma channel*/
	int (*_free_dma)(int device_id);		/*free dma chanel*/
	int (*_get_dma_residue)(int device_id);		/*get dma residue*/
	unsigned long (*_dma_mem_alloc) (unsigned long size);	/*allocate dma memory*/
	int (*_dma_setup)(char *addr, unsigned long size, int mode, int io, int device_id);	/*start dma operation*/
} fd_routine[] = {
	{			/*hard dma*/
		userdev_request_dma,
		userdev_free_dma,
		userdev_check_dma,
		dma_mem_alloc,
		hard_dma_setup
	},
	{			/*virtual dma*/
		vdma_request_dma,
		vdma_nop,
		vdma_get_dma_residue,
		vdma_mem_alloc,
		vdma_dma_setup
	}
};


static int FDC1 = 0x3f0;
static int FDC2 = -1;

/*
 * Floppy types are stored in the rtc's CMOS RAM and so rtc_lock
 * is needed to prevent corrupted CMOS RAM in case "insmod floppy"
 * coincides with another rtc CMOS user.		Paul G.
 * Here locking has been ignored bcoz, its inefficient to go to
 * kernel just for acquiring a lock.                    Hari
 */
/*macro for reading from CMOS memory, replicated from kernel*/
# define CMOS_READ(addr) ({ 			 \
	unsigned char v;			 \
	userdev_set_ioperm(0x70);			 \
	userdev_set_ioperm(0x71);			 \
        userdev_outportb(0x70, addr);  /*RTC(0) = 0x70*/ \
        v = userdev_inportb(0x71);     /*RTC(1) = 0x71*/ \
	userdev_reset_ioperm(0x70);			 \
	userdev_reset_ioperm(0x71);			 \
	v;					 \
        })

/*macros to read the type of the floppy controller*/
#define FLOPPY0_TYPE	({				\
	/*unsigned long flags;*/			\
	unsigned char val;				\
	/*spin_lock_irqsave(&rtc_lock, flags);*/	\
	val = (CMOS_READ(0x10) >> 4) & 15;		\
	/*spin_unlock_irqrestore(&rtc_lock, flags);*/	\
	val;						\
})
#define FLOPPY1_TYPE	({				\
	/*unsigned long flags;*/			\
	unsigned char val;				\
	/*spin_lock_irqsave(&rtc_lock, flags);*/	\
	val = CMOS_READ(0x10) & 15;			\
	/*spin_unlock_irqrestore(&rtc_lock, flags);*/	\
	val;						\
})

#define N_FDC 2		/*number of floppy controllers*/
#define N_DRIVE 8	/*number of floppy drives*/

#define FLOPPY_MOTOR_MASK 0xf0

#define AUTO_DMA

# define ARRAY_SIZE(x)	(sizeof(x)/sizeof(x[0]))	/*size of array*/

#endif /*__USERDEV_FLOPPY_H__*/
