/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_menubar.C
 *
 * AUTHOR : Tom Newton
 *
 * This file 
 *
 * DESCRIPTION:
 *
    This implements the mx_menu_item, mx_menu, and mx_menubar classes
 *
 *
 *
 */

#include <mx.h>
#include <mx_menubar.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/ToggleB.h>
#include <Xm/Separator.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>

///////////////////////////////////// mx_menu_item stuff

mx_menu_item::mx_menu_item(mx_menu *parent) : mx_ui_object()
{
    static int sep_num = 0;

    char char_num[30];

    sprintf(char_num, "separator%d", sep_num++);
    
    item_type = separator_type;

    widget = XtVaCreateManagedWidget(
                    char_num, 
                    xmSeparatorWidgetClass,
                    parent->get_widget(), 
                    NULL);
}

mx_menu_item::mx_menu_item(
                    char           *name, 
                    mx_menu        *parent,
                    XtCallbackProc cb,
                    XtPointer       cd) : mx_ui_object()
{
    item_type = normal_type;

    widget = XtVaCreateManagedWidget(
                    name, 
                    xmPushButtonWidgetClass,
                    parent->get_widget(), 
                    NULL);

    if((cb != NULL) && (cd != NULL))
    {
        XtAddCallback(widget, XmNactivateCallback, cb, cd);
    }
}


mx_menu_item::mx_menu_item(
                    char           *name, 
                    mx_menu        *parent, 
                    bool           checked,
                    XtCallbackProc cb,
                    XtPointer      cd) : mx_ui_object()
{
    item_type = checkable_type;

    widget = XtVaCreateManagedWidget(
                            name, 
                            xmToggleButtonWidgetClass,
                            parent->get_widget(),
                            XmNtoggleMode,    XmTOGGLE_BOOLEAN,
                            XmNindicatorType, XmN_OF_MANY,
                            XmNindicatorOn,   XmINDICATOR_CHECK,
                            NULL);

    XtVaSetValues(widget, XmNset, checked ? XmSET : XmUNSET, NULL);

    if((cb != NULL) && (cd != NULL))
    {
        XtAddCallback(widget, XmNvalueChangedCallback, cb, cd);
    }
}

mx_menu_item::~mx_menu_item()
{
}

bool mx_menu_item::is_sensitized()
{
    return XtIsSensitive(widget);
}

bool mx_menu_item::toggle_sensitivity()
{
    bool value = !is_sensitized();

    XtSetSensitive(widget, value ? True : False);
    return value;
}

void mx_menu_item::set_sensitivity(bool s)
{
    XtSetSensitive(widget, s ? True : False);
}

bool mx_menu_item::is_checked()
{
    Boolean is_set;

    if (item_type == checkable_type)
    {
        XtVaGetValues(widget, XmNset, &is_set, NULL);

        if (is_set == XmSET)
        {
            return TRUE;
        }
    }
    return FALSE;
}

void mx_menu_item::set_checked(bool s)
{
    if (item_type == checkable_type)
    {
        XtVaSetValues(widget, XmNset, s ? XmSET : XmUNSET , NULL);

    }
}


///////////////////////////////////// mx_menu stuff

#define MENU_POSTFIX "Menu"



mx_menu::mx_menu(char *name, mx_menubar *parent, bool is_help = FALSE) : mx_ui_object()
{
    char menu_name[200];

    strcpy(menu_name, name);
    strcat(menu_name, MENU_POSTFIX);

    widget = XmCreatePulldownMenu(parent->get_widget(), name, NULL, 0);

    button = XtVaCreateManagedWidget(
                    name, 
                    xmCascadeButtonWidgetClass,
                    parent->get_widget(),
                    XmNsubMenuId, widget, 
                    NULL);

    // special case for help menu which goes on the right
    if (is_help)
    {    
        XtVaSetValues(
                parent->get_widget(),
                XmNmenuHelpWidget, button,
                NULL); 
    }
}

mx_menu::mx_menu(char *name, mx_menu *parent) : mx_ui_object()
{
    char menu_name[200];

    strcpy(menu_name, name);
    strcat(menu_name, MENU_POSTFIX);

    widget = XmCreatePulldownMenu(parent->get_widget(), name, NULL, 0);

    button = XtVaCreateManagedWidget(
                    name, 
                    xmCascadeButtonWidgetClass,
                    parent->get_widget(),
                    XmNsubMenuId, widget, 
                    NULL);
}

mx_menu::~mx_menu()
{
    int          err;
    mx_menu      *menu;
    mx_menu_item *menu_item;

    sub_menus.iterate_start(err);
    MX_ERROR_CHECK(err);

    while ((menu = (mx_menu *)sub_menus.iterate_next(err)) != NULL)
    {
        MX_ERROR_CHECK(err);
        delete menu;
    }

    items.iterate_start(err);
    MX_ERROR_CHECK(err);

    while((menu_item = (mx_menu_item *)items.iterate_next(err)) != NULL)
    {
        MX_ERROR_CHECK(err);
        delete menu_item;
    }

    return;

abort:
    global_error_trace->print();
}

///////////////////////////////////// mx_menubar stuff

mx_menubar::mx_menubar(char *name, mx_window *window) : mx_bar(window)
{
    // this widget goes into the form of the window, not the window
    // itself
    Arg args[1];

    XtSetArg(args[0], XmNtearOffModel,  XmTEAR_OFF_ENABLED);
    widget = XmCreateMenuBar(window->get_form(), name, args, 1);

    XtManageChild(widget);
}

mx_menubar::~mx_menubar()
{
    int err;
    mx_menu *menu;

    sub_menus.iterate_start(err);
    MX_ERROR_CHECK(err);

    while((menu = (mx_menu *) sub_menus.iterate_next(err)) != NULL)
    {
        MX_ERROR_CHECK(err);
        delete menu;
    }
    return;

abort:
    global_error_trace->print();
}

void mx_menubar::set_item_visible(
                    int  &err, 
                    char *menu_name, 
                    char *item_name, 
                    bool s)
{
    mx_menu *menu;
    mx_menu_item *item;

    // get the menu
    menu = (mx_menu *)sub_menus.get(err, menu_name);
    MX_ERROR_CHECK(err);

    // get the item
    item = menu->get_item(err, item_name);
    MX_ERROR_CHECK(err);

    item->set_sensitivity(s);

abort:;
}

mx_menu_item *mx_menubar::get_item(int &err, char *menu_name, char *item_name)
{
    mx_menu *menu;
    mx_menu_item *item;

    // get the menu
    menu = (mx_menu *)sub_menus.get(err, menu_name);
    MX_ERROR_CHECK(err);

    // get the item
    item = menu->get_item(err, item_name);
    MX_ERROR_CHECK(err);

    return item;

abort:;
    return NULL;
}

bool mx_menubar::get_item_checked(int &err, char *menu_name, char *item_name)
{
    mx_menu_item *item;

    item = get_item(err, menu_name, item_name);
    MX_ERROR_CHECK(err);

    return item->is_checked();

abort:;
    return FALSE;
}

void mx_menubar::set_item_checked(int &err, char *menu_name, char *item_name, bool s)
{
    mx_menu_item *item;

    item = get_item(err, menu_name, item_name);
    MX_ERROR_CHECK(err);

    item->set_checked(s);

abort:;
}

mx_menu_item *mx_menu::add_separator()
{
    int err = MX_ERROR_OK;

    mx_menu_item *item = new mx_menu_item(this);

    items.add(err, (char *)item->get_name(), (void *)item);
    MX_ERROR_CHECK(err);

    return item;

abort:
    return NULL;
}

mx_menu_item *mx_menu::add_item(
                        char           *name, 
                        XtCallbackProc cb, 
                        XtPointer      cd)
{
    int err = MX_ERROR_OK;

    mx_menu_item *item = new mx_menu_item(name, this, cb, cd); 

    items.add(err, (char *)item->get_name(), (void *)item);   
    MX_ERROR_CHECK(err);

    return item;

abort:
    return NULL;
}

mx_menu_item *mx_menu::add_checkable_item(
                            char           *name,    
                            bool           is_checked,
                            XtCallbackProc cb, 
                            XtPointer      cd)
{
    int err = MX_ERROR_OK;

    mx_menu_item *item = new mx_menu_item(name, this, is_checked, cb, cd); 

    items.add(err, (char *)item->get_name(), (void *)item);   
    MX_ERROR_CHECK(err);

    return item;

abort:
    return NULL;
}

mx_menu_item *mx_menu::get_item(int &err, char *name)
{
    return (mx_menu_item *)items.get(err, name);
}

mx_menu *mx_menu::add_sub_menu(char *name)
{
    int err = MX_ERROR_OK;

    mx_menu *menu = new mx_menu(name, this);

    sub_menus.add(err, (char *)menu->get_name(), (void *)menu);
    MX_ERROR_CHECK(err);

    return menu;

abort:
    return NULL;
}

mx_menu *mx_menu::get_sub_menu(int &err, char *name)
{
    return (mx_menu *)sub_menus.get(err, name);
}

mx_menu *mx_menubar::add_sub_menu(char *name, bool is_help = FALSE)
{
    int err = MX_ERROR_OK;

    mx_menu *menu = new mx_menu(name, this, is_help);

    sub_menus.add(err, menu->get_name(), (void *)menu);
    MX_ERROR_CHECK(err);

    return menu;

abort:
    return NULL;
}

mx_menu *mx_menubar::get_sub_menu(int &err, char *name)
{
    return (mx_menu *)sub_menus.get(err, name);
}

bool mx_menubar::is_checked(Widget w)
{
    Boolean is_set;

    XtVaGetValues(w, XmNset, &is_set, NULL);
    return is_set == XmSET;
}
