/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_font_metrics_store & mx_font_metrics
 *
 * AUTHOR : Tom Newton
 *
 * 
 *
 * DESCRIPTION:
 *
 * This class is a font class and includes methods for querying a font about
 * its metrics and such like..
 *
 *
 *
 *
 */

#include "mx_font_metrics.h"
#include "mx_type1.h"
#include "mx_xfontname.h"
#include <mx.h>
#include <mx_list.h>
#include <mx_hash_iter.h>
#include <sys/stat.h>
#include <unistd.h>

// global variable
extern char *global_maxhome;

mx_hash mx_font_metrics_store::font_families(20);

uint32 mx_font_metrics::num_instances = 0;
mx_hash mx_font_metrics::iso_to_type1(256);
mx_hash mx_font_metrics::type1_to_iso(256);

#ifdef sun
static void *mx_font_metrics_memmem(const void *haystack, size_t hl,
                                    const void *needle, size_t nl)
{
    uint8 *hs = (uint8 *)haystack;
    uint8 *n = (uint8 *)needle;
    uint8 *end = (hs + hl) - nl;

    while (hs <= end)
    {
        if (0 == memcmp(hs, needle, nl)) return hs;
        hs++;
        hl--;
    }
    return NULL;
}
#else
inline
static void *mx_font_metrics_memmem(const void *haystack, size_t hl,
                                    const void *needle, size_t nl)
{
    return memmem(haystack, hl, needle, nl);
}
#endif

extern char *global_maxhome;

// Local functions

static void decrypt_hex_encryption(int &err, const char *e_buf, 
                                   uint32 e_buf_length, char *d_buf, 
                                   uint32 &d_buf_length); 

static uint32 width_from_char_string(int &err, const char *char_string_bytes,
                                     int32 num_bytes);

static uint32 get_encryption_offset(int &err, const char *font, uint32
                                    font_length);

static unsigned char decrypt(unsigned char cipher, unsigned short int *r);
static unsigned char decrypt_hex(const char *ptr, unsigned short int *r);

mx_font_family::mx_font_family()
{
    for (int i = 0; i < 4; i++)
    {
        styles[i] = NULL;
    }
}

mx_font_family::~mx_font_family()
{
    for (int i = 0; i < 4; i++)
    {
        if (styles[i]) delete styles[i];
    }
}

mx_font_metrics *mx_font_family::get_font_metrics(mx_font_style_t st) const
{
    if (styles[st] != NULL)
    {
        return styles[st];
    }
    else
    {
        for (int i = 0; i < 4; i++)
        {
            if (styles[i] != NULL) return styles[i];
        }
    }
    return NULL;
}
void mx_font_family::set_font_metrics(mx_font_metrics *fm)
{
    if (styles[fm->get_style()] == fm) return;
    if (styles[fm->get_style()] != NULL) delete styles[fm->get_style()];
    styles[fm->get_style()] = fm;
}

void mx_font_family::family_has_B_or_I(bool &b, bool &it)
{
    b = FALSE;
    it = FALSE;

    if (styles[mx_normal])
    {
        if (styles[mx_bold])
        {
            b = TRUE;
            if (styles[mx_bold_italic] && styles[mx_italic])
            {
                it = TRUE;
            }
        }
        else
        {
            if (styles[mx_italic])
            {
                it = TRUE;
            }
        }
    }
}

const char *mx_font_family::get_name()
{
    for (int i = 0; i < 4; i++)
    {
        if (styles[i]) return styles[i]->get_name();
    }
    return NULL;
}

mx_font_metrics::mx_font_metrics()
{
    encoding_to_type1 = type1_to_encoding = NULL;
    family_name = NULL;
    x_name = NULL;
    file_name = NULL;
    ps_name = NULL;
    full_name = NULL;
    style = mx_normal;
    em_ascender = 0;
    em_descender = 0;

    // using an array for character widths
    for (int i = 0; i < MX_FM_CHARSET_SIZE; i++) widths_arr[i] = 0;

    if (num_instances == 0)
    {
        int err = MX_ERROR_OK;
        get_iso_lookup(err);
        MX_ERROR_CHECK(err);
    }

    num_instances++;

    return;
abort:
    global_error_trace->print();
}

mx_font_metrics::~mx_font_metrics()
{
    if (family_name) delete [] family_name;
    if (x_name) delete [] x_name;
    if (file_name) delete [] file_name;
    if (ps_name) delete [] ps_name;
    if (full_name) delete [] full_name;

    num_instances--;

    if (encoding_to_type1 != NULL)
    {
        mx_hash_iterator iter(*encoding_to_type1);
        while (iter.more()) delete [] (char *) iter.data();
        delete encoding_to_type1;
    }
    if (type1_to_encoding != NULL) delete type1_to_encoding;

    if (num_instances == 0)
    {
        // remove all the iso data
        mx_hash_iterator iter(iso_to_type1);
        while (iter.more()) delete [] (char *) iter.data();
    }
}

char *mx_font_metrics::get_name() const
{
    return family_name;
}

char *mx_font_metrics::get_x_name() const
{
    return x_name;
}

char *mx_font_metrics::get_ps_name() const
{
    return ps_name;
}

char *mx_font_metrics::get_full_name() const
{
    return full_name;
}

mx_font_style_t mx_font_metrics::get_style() const
{
    return style;
}

char *mx_font_metrics::get_file_name() const
{
    return file_name;
}

int32 mx_font_metrics::get_em_ascender() const
{
    return em_ascender;
}

int32 mx_font_metrics::get_em_descender() const
{
    return em_descender;
}

uint32 mx_font_metrics::get_em_width(uint16 c) const
{
    if (c > 255) return 0;
    return widths_arr[c];
}

void mx_font_metrics::set_name(const char *new_name)
{
    if (family_name) delete [] family_name;
    family_name = mx_string_copy(new_name);
}

void mx_font_metrics::set_x_name(const char *new_x_name)
{
    if (x_name) delete [] x_name;
    x_name = mx_string_copy(new_x_name);
}

void mx_font_metrics::set_ps_name(const char *new_ps_name)
{
    if (ps_name) delete [] ps_name;
    ps_name = mx_string_copy(new_ps_name);
}

void mx_font_metrics::set_full_name(const char *new_full_name)
{
    if (full_name) delete [] full_name;
    full_name = mx_string_copy(new_full_name);
}

void mx_font_metrics::set_style(mx_font_style_t st)
{
    style = st;
}

void mx_font_metrics::set_file_name(const char *new_file_name)
{
    if (file_name) delete [] file_name;
    file_name = mx_string_copy(new_file_name);
}

void mx_font_metrics::set_em_ascender(int32 a)
{
    em_ascender = a;
}

void mx_font_metrics::set_em_descender(int32 d)
{
    em_descender = d;
}

void mx_font_metrics::set_em_width(uint16 character, uint32 em_width)
{
    int err = MX_ERROR_OK;

    if (character > (MX_FM_CHARSET_SIZE - 1)) 
    {
        MX_ERROR_THROW(err, MX_FM_OUT_OF_RANGE);
    }
    else
    {
        widths_arr[character] = em_width;
    }

    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return;
}

void mx_font_metrics::write_mfm(int &err, char *output_file_name)
{
    FILE *fd;
    char line[200];
    char string_buffer[200];

    fd = fopen(output_file_name, "w");
    if (fd == NULL)
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }

    if (!(family_name && x_name && file_name && ps_name && full_name))
    {
        MX_ERROR_THROW(err, MX_FM_UNINITIALISED);
    }
    else
    {
        fprintf(fd, "Font Name: (%s)\n", full_name);
        fprintf(fd, "PS Name  : (%s)\n", ps_name);
          
        switch(style)
        {
        case mx_normal:
            fprintf(fd, "Family   : (%s)\n", family_name);
            break;
        case mx_bold:
            fprintf(fd, "Family   : (%s) %s\n", family_name, "BOLD");
            break;
        case mx_italic:
            fprintf(fd, "Family   : (%s) %s\n", family_name, "ITALIC");
            break;
        case mx_bold_italic:
            fprintf(fd, "Family   : (%s) %s\n", family_name, "BOLD ITALIC");
            break;
        }

        fprintf(fd, "XName    : %s\n", x_name);
        fprintf(fd, "File Name: %s\n", file_name);
        fprintf(fd, "Ascender : %ld\n", em_ascender);
        fprintf(fd, "Descender: %ld\n", em_descender);

        for (uint32 i = 0; i < MX_FM_CHARSET_SIZE; i++)
        {
            uint32 width = get_em_width(i);

            if (width != 0)
            {
                // output ISO-CHAR-NUM EM-WIDTH pairs
                fprintf(fd, "%ld %ld\n", i, width);
            }
        }
    }
    fclose(fd);
    return;
abort:
    return;
}


// this function is just used below....
static bool get_font_info_line(int &err, FILE *fd, char *line)
{
    if (fgets(line, 132, fd) == NULL)
    {
        err = MX_FM_BAD_INFO;
        MX_ERROR_CHECK(err);
    }
    return TRUE;
abort:
    return FALSE;
}


// copies a string delimited by brackets -> ()
static void get_bracketed_string(int &err, char *result, const char *original)
{
    while (*original != '(' && *original != '\0') original++;
    if (*original == '\0') MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);
    original++;
    while (*original != ')' && *original != '\0') *result++ = *original++;
    if (*original == '\0') MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);
abort:
    *result = '\0';
}

void mx_font_metrics::read_mfm(int &err, char *file_name)
{
    FILE *fd;
    char line[200];
    char string_buffer[200];

    fd = fopen(file_name, "r");
    if (fd == NULL)
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);
        
    if (strncmp(line, "Font Name: ", 11) != 0) 
        MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);  

    get_bracketed_string(err, string_buffer, line);
    MX_ERROR_CHECK(err);

    set_full_name(string_buffer);

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);

    if (strncmp(line, "PS Name  : ", 11) != 0)
        MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);  

    get_bracketed_string(err, string_buffer, line);
    MX_ERROR_CHECK(err);

    set_ps_name(string_buffer);

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);

    if (strncmp(line, "Family   : ", 11) != 0)
        MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);  

    get_bracketed_string(err, string_buffer, line);
    MX_ERROR_CHECK(err);

    set_name(string_buffer);

    if (strstr(line, "BOLD") != NULL)
    {
        if (strstr(line, "ITALIC") != NULL)
        {
            strcat(string_buffer, "I");
            set_style(mx_bold_italic);
        }
        else
        {
            strcat(string_buffer, "B");
            set_style(mx_bold);
        }
    }
    else if (strstr(line, "ITALIC") != NULL)
    {
        strcat(string_buffer, "I");
        set_style(mx_italic);
    }

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);

    if (strncmp(line, "XName    : ", 11) != 0)
        MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);  
    
    sscanf(line, "XName    : %s", string_buffer);
    set_x_name(string_buffer);
        
    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);
    
    if (strncmp(line, "File Name: ", 11) != 0)
        MX_ERROR_THROW(err, MX_FM_INVALID_MFM_FILE);  
    
    sscanf(line, "File Name: %s", string_buffer);
    set_file_name(string_buffer);

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);
    sscanf(line, "Ascender : %ld", &em_ascender);

    get_font_info_line(err, fd, line);
    MX_ERROR_CHECK(err);
    sscanf(line, "Descender: %ld", &em_descender);
    
    while (fgets(line, 132, fd))
    {
        uint32 width;
        uint32 iso_num;
        
        if (sscanf(line, "%ld %ld", &iso_num, &width) == 2)
        {
            set_em_width(iso_num, width);
        }
    }
    
    fclose(fd);
    return;
abort:
    return;
}

void mx_font_metrics::read_type1(int &err, const char *type1_file_name, const char *x_font_name)
{
    char *tmp = NULL; 
    const char *real_file_name = type1_file_name;
    char *file_buffer = NULL;
    uint32 file_buffer_size = 0, header_length = 0;
    FILE *fd = NULL;

    if (strlen(type1_file_name) < 4) MX_ERROR_THROW(err, MX_FM_DECRYPT_ERROR);

    if (strcmp(type1_file_name + strlen(type1_file_name) - 4, ".pfb") == 0)
    {
        // it's a pfb - turn it into a temporary pfa
        tmp = mx_tmpnam(NULL);
        mx_pfb_2_pfa(err, type1_file_name, tmp);
        MX_ERROR_CHECK(err);
        real_file_name = tmp;
    }

    // read the whole font file into memory and operate on that

    file_buffer_size = mx_file_size(err, real_file_name);
    MX_ERROR_CHECK(err);

    file_buffer = new char[file_buffer_size];

    fd = fopen(real_file_name, "r");
    if (fd == NULL)
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }
    if (!fread(file_buffer, file_buffer_size, 1, fd))
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }
    fclose(fd);
    
    set_file_name(type1_file_name);
    set_x_name(x_font_name);

    header_length = get_encryption_offset(err, file_buffer, file_buffer_size);
    MX_ERROR_CHECK(err);

    get_type1_family_name(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_full_name(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_ps_name(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_style(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_bounding_box(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_encoding(err, file_buffer, header_length);
    MX_ERROR_CHECK(err);

    get_type1_char_widths(err, file_buffer + header_length, file_buffer_size -
                          header_length);
    MX_ERROR_CHECK(err);

    delete file_buffer;
    return;
abort:
    return;
}

static uint32 get_encryption_offset(int &err, const char *font, uint32
                                    font_length)
{
    char *ptr;

    ptr = (char *)mx_font_metrics_memmem(font, font_length, 
                                         "currentfile eexec\n", 18);
    if (ptr == NULL)
    {
        MX_ERROR_THROW(err, MX_FM_NO_EEXEC);
    }
    while (*ptr++ != '\n')  ; // do nothing

    return (uint32)(ptr - font);
abort:
    return 0;
}

void mx_font_metrics::get_type1_family_name(int &err, const char
                                            *buffer, uint32 buffer_length)
{
    char name_buffer[100];
    char *xfontname;
    char xfoundry[50], xfamily[50];

    xfontname = get_x_name();

    mx_xfontname_get_arg(err, xfontname, mx_xfontname_foundry_e, xfoundry);
    MX_ERROR_CHECK(err);
    mx_xfontname_get_arg(err, xfontname, mx_xfontname_family_e, xfamily);
    MX_ERROR_CHECK(err);

    xfoundry[0] = toupper(xfoundry[0]);
    xfamily[0] = toupper(xfamily[0]);

    sprintf(name_buffer, "%s-%s", xfoundry, xfamily);

    set_name(name_buffer);

    return;
abort:
    return;
}

void mx_font_metrics::get_type1_full_name(int &err, const char
                                        *buffer, uint32 buffer_length)
{
    char *ptr;
    char name_buffer[50];
    char *name_ptr = name_buffer;

    ptr = (char *)mx_font_metrics_memmem(buffer, buffer_length,
                                         "/FullName", 9);
    if (ptr == NULL)
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }

    // find the beginning of the name
    while (*ptr != '\n' && (uint32)(ptr - buffer) > 0 && *ptr != '(')
    {
        ptr++;
    }

    if ((uint32)(ptr - buffer) == 0 || *ptr == '\n' || *ptr != '(')
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }
    ptr++;

    // copy the name
    while (*ptr != '\n' && (uint32)(ptr - buffer) > 0 && *ptr != ')')
    {
        *name_ptr++ = *ptr++;
    }
    *name_ptr = '\0';

    if (*name_buffer == '\0')
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }

    set_full_name(name_buffer);

    return;
abort:
    return;
}

void mx_font_metrics::get_type1_ps_name(int &err, const char
                                        *buffer, uint32 buffer_length)
{
    char *ptr;
    char name_buffer[50];
    char *name_ptr = name_buffer;

    ptr = (char *)mx_font_metrics_memmem(buffer, buffer_length,
                                         "/FontName", 9);
    if (ptr == NULL)
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }

    ptr++;

    // find the beginning of the name
    while (*ptr != '\n' && (uint32)(ptr - buffer) > 0 && *ptr != '/')
    {
        ptr++;
    }

    if (*ptr != '/')
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }

    // copy the name
    while (*ptr != '\n' && (uint32)(ptr - buffer) > 0 && *ptr != ' ')
    {
        *name_ptr++ = *ptr++;
    }
    *name_ptr = '\0';

    if (*name_buffer == '\0')
    {
        MX_ERROR_THROW(err, MX_FM_NO_FULL_NAME);
    }

    set_ps_name(name_buffer);

    return;
abort:
    return;
}

void mx_font_metrics::get_type1_style(int &err, const char
                                      *buffer, uint32 buffer_length)
{
    uint32 i;
    char *xfontname;
    char xweight[50], xslant[50];

    xfontname = get_x_name();

    mx_xfontname_get_arg(err, xfontname, mx_xfontname_weight_e, xweight);
    MX_ERROR_CHECK(err);
    mx_xfontname_get_arg(err, xfontname, mx_xfontname_slant_e, xslant);
    MX_ERROR_CHECK(err);

    // need to be case insensitive
    xslant[0] = tolower(xslant[0]);
    for (i = 0; xweight[i] != '\0'; i++) xweight[i] = tolower(xweight[i]);

    if (strstr(xweight, "bold") != NULL)
    {
        if (xslant[0] == 'r') 
        {
            set_style(mx_bold);
        }
        else 
        {
            set_style(mx_bold_italic);
        }
    }
    else
    {
        if (xslant[0] == 'r') 
        {
            set_style(mx_normal);
        }
        else
        {
            set_style(mx_italic);
        }
    }

abort:
    return;
}

void mx_font_metrics::get_type1_bounding_box(int &err, const char
                                             *buffer, uint32 buffer_length)
{
    char *ptr;
    uint32 x0, y0, x1, y1;

    ptr = (char *)mx_font_metrics_memmem(buffer, buffer_length,
                                         "/FontBBox", 9);
    if (ptr == NULL)
    {
        MX_ERROR_THROW(err, MX_FM_NO_BOUNDING_BOX);
    }

    // find the beginning of the numbers
    while (*ptr != '\n' && (uint32)(ptr - buffer) > 0 && *ptr != '{' && 
           *ptr != '[')
    {
        ptr++;
    }

    if ((uint32)(ptr - buffer) == 0 || *ptr == '\n' || (*ptr != '{' && 
        *ptr != '['))
    {
        MX_ERROR_THROW(err, MX_FM_NO_BOUNDING_BOX);
    }
    ptr++;

    // copy the numbers
    if (sscanf((char *)ptr, "%ld %ld %ld %ld", &x0, &y0, &x1, &y1) != 4)
    {
        MX_ERROR_THROW(err, MX_FM_NO_BOUNDING_BOX);
    }

    set_em_descender(y0);
    set_em_ascender(y1);

abort:
    return;
}

static void next_line(char *&ptr, const char *buffer, int32 buffer_len)
{
    while (*ptr != '\n' && *ptr != '\r' && (ptr - buffer) < buffer_len)
    {
        ptr++;
    }

    if (ptr - buffer >= buffer_len)
    {
        ptr = NULL;
        return;
    }
    ptr++;
}

void mx_font_metrics::get_type1_encoding(int &err, const char *buffer,
                                         uint32 buffer_length)
{
    char *ptr;
    char slash_encoding[200], standard_encoding[200], def[200];
    char dup_word[10], ps_char_name[100], put_word[100];
    uint32 enc_char_num = 0;
    mx_hash *temp_encoding_to_type1 = NULL;
    mx_hash *temp_type1_to_encoding = NULL;

    ptr = (char *)mx_font_metrics_memmem(buffer, buffer_length,
                                         "/Encoding", 9);
    if (ptr == NULL)
    {
        // if there's no encoding field just use the standard one
        return;
    }

    if (sscanf(ptr, "%s %s %s", slash_encoding, standard_encoding, def) != 3)
    {
        return;
    }

    if (strcmp(standard_encoding, "StandardEncoding") == 0 &&
        strcmp(def, "def") == 0)
    {
        // just use standard encoding
        return;
    }
    
    // find the first line starting with the "dup" keyword
    while (ptr != NULL && strncmp(ptr, "dup ", 4) != 0) 
    {
        next_line(ptr, buffer, buffer_length);
    }

    // couldn't find a dup line
    if (ptr == NULL) return;

    temp_encoding_to_type1 = new mx_hash(256);
    temp_type1_to_encoding = new mx_hash(20);

    while (ptr != NULL && strncmp(ptr, "dup ", 4) == 0)
    {
        if ((sscanf(ptr, "%s %ld %s %s", dup_word, &enc_char_num, 
                    ps_char_name, put_word) == 4) &&
            (ps_char_name[0] == '/') &&
            strcmp(ps_char_name, "/.notdef") != 0)
        {
            char *ch_name = mx_string_copy(ps_char_name + 1);

            temp_encoding_to_type1->add(err, enc_char_num, (void *)ch_name);
            MX_ERROR_CHECK(err);

            // some fonts have more than one name entry. Use the last one
            temp_type1_to_encoding->add_or_repl(err, ch_name, (void *)enc_char_num);
            MX_ERROR_CHECK(err);
        }
        next_line(ptr, buffer, buffer_length);
    }

    encoding_to_type1 = temp_encoding_to_type1;
    type1_to_encoding = temp_type1_to_encoding;

abort:;
}

void mx_font_metrics::get_type1_char_widths(int &err, 
                                            const char *encrypt_buffer, 
                                            uint32 buffer_length)
{
    char *decrypted_buffer;
    uint32 decrypted_buffer_length;
    const char *end_d_buf;
    const char *line_ptr;
    
    // decrypt all of the ASCII hex encrypted stuff

    decrypted_buffer = new char[buffer_length];
    decrypt_hex_encryption(err, encrypt_buffer, buffer_length,
                           decrypted_buffer, decrypted_buffer_length);
    MX_ERROR_CHECK(err);

    // find the CharStrings line

    line_ptr = (char *)mx_font_metrics_memmem(decrypted_buffer,
                                              decrypted_buffer_length,
                                              "/CharStrings", 12);
    if (line_ptr == NULL)
    {
        MX_ERROR_THROW(err, MX_FM_BAD_FONT_ENCRYPTION);
    }

    end_d_buf = decrypted_buffer + decrypted_buffer_length;
    while (*line_ptr != '\n' && *line_ptr != '\r' && line_ptr < end_d_buf)
    {
        line_ptr++; 
    }

    if (line_ptr == end_d_buf)
    {
        MX_ERROR_THROW(err, MX_FM_BAD_FONT_ENCRYPTION);
    }
    line_ptr++;

    // OK we have got the first CharStrings line (or else we'd have got an
    // error). Now get each line until a line with "end" on its own.

    while (strncmp(line_ptr, "end", 3) != 0)
    {
        line_ptr = get_type1_char_width(err, line_ptr);
        MX_ERROR_CHECK(err);

        // skip any white space
        while (isspace(*line_ptr)) line_ptr++;
    }

    delete decrypted_buffer;
abort:
    return;
}


// returns the address of the next line
const char *
mx_font_metrics::get_type1_char_width(int &err, const char *char_strings_line)
{
    // The string is of the form:
    // /<name> <num_bytes> RD <bytes * num_bytes> ND
    // OR:
    // /<name> <num_bytes> -| <bytes * num_bytes> |-
    // For example:
    // /Alpha 186 RD ~186 binary bytes~ ND
    // The binary bytes are encrypted with the same kind of encoding as the
    // eexec encoding.

    char char_name[50] = "";
    uint32 char_width;
    uint32 char_num;
    const char *next_address;

    int32 num_bytes = 0;
    char opening_chars[4] = "RD ";

    // some fonts seem to have a line like this in them:
    // / 108 RD ...
    // Test for this and skip lines like that
    if (char_strings_line[0] == '/' && char_strings_line[1] == ' ')
    {
        if (sscanf(char_strings_line, "/ %ld %s", &num_bytes, 
                   opening_chars) != 2)
        {
            MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
        }
    }
    else
    {
        if (sscanf(char_strings_line, "/%s %ld %s", char_name, &num_bytes,
                   opening_chars) != 3)
        {
            MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
        }
    }
        
    // skip exactly three spaces to get to the binary bytes
    for (uint32 num_spaces = 0; num_spaces < 3; char_strings_line++)
    {
        if (*char_strings_line == ' ') num_spaces++;
    }

    // calculate the next line's address. ie this address plus the length of
    // the binary bit plus the " RD\n" bit.

    next_address = char_strings_line + num_bytes + 4;
    if (next_address[-1] != '\n' && next_address[-1] != '\r')
    {
        MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
    }

    // if we do not use this character name then don't bother decoding it
    char_num = get_iso(err, char_name);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
    }
    else
    {
        MX_ERROR_CHECK(err);

        // decode Charstring bytes
        char_width = width_from_char_string(err, char_strings_line, num_bytes);
        MX_ERROR_CHECK(err);
        
        set_em_width(char_num, char_width);
    }

    return next_address;
abort:
    return NULL;
}

static uint32 width_from_char_string(int &err, const char *char_string_bytes,
                                     int32 num_bytes)
{
    unsigned short int r = 4330;
    int32 value_stack[200];
    int32 value_stack_size = 0;
    bool got_hsbw_command = FALSE, got_sbw_command = FALSE;

    // skip crud
    for (int i = 0; i < 4; i++) decrypt(*char_string_bytes++, &r);
    num_bytes -= 4;

    if (num_bytes < 0) MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);

    // The first command should always be either hsbw OR sbw, so stack the
    // values up and get the command. This is a bit of a long while loop, but
    // there's no way out.

    while (!got_hsbw_command && !got_sbw_command && num_bytes > 0)
    {
        unsigned char plain = decrypt(*char_string_bytes++, &r);
        num_bytes--;

        if (plain == 13)            // got hsbw command (code 13)
        {
            got_hsbw_command = TRUE;
        }
        else if (plain == 12)
        {
            plain = decrypt(*char_string_bytes++, &r);
            num_bytes--;
            if (plain == 7)
            {
                // got sbw command (code 12 7)
                got_sbw_command = TRUE;
            }
            else
            {
                MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
            }
        }
        else if (plain < 32)              // another command..
        {
            MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
        }
        else if (plain < 247)             // 1 byte value
        {
            value_stack[value_stack_size++] = (int32)plain - 139;
        }
        else if (plain < 255)             // 2 byte value
        {
            // got integer in range 108 -> 1131 OR -108 -> 1131

            int32 second_val = decrypt(*char_string_bytes++, &r);
            num_bytes--;

            if (plain < 251)
            {
                value_stack[value_stack_size++] = 
                    (((int32)plain - 247) * 256) + second_val + 108;
            }
            else
            {
                value_stack[value_stack_size++] = 
                    (((int32)plain - 251) * 256) - second_val - 108;
            }
        }
        else                             // 5 byte vvalue
        {
            // get following 4 bytes as int32
            int32 num = 0;
            for (int i = 0; i < 4; i++)
            {
                num <<= 8;
                num |= (int32)decrypt(*char_string_bytes++, &r);
            }
            num_bytes -= 4;

            value_stack[value_stack_size++] = num;
        }
    }

    if (got_hsbw_command)
    {
        return value_stack[1];
    }
    else if (got_sbw_command)
    {
        return value_stack[2];
    }
    else
    {
        MX_ERROR_THROW(err, MX_FM_BAD_CHARSTRING);
    }

    return 0;
abort:    
    return 0;
}

static void decrypt_hex_encryption(int &err, const char *e_buf, 
                                   uint32 e_buf_length, char *d_buf, 
                                   uint32 &d_buf_length)
{
    unsigned short int r = 55665;

    const char *d_start_ptr = d_buf;
    const char *e_end_ptr = e_buf + e_buf_length;

    // discard first four hex characters since they are rubbish
    for (int i = 0; i < 4; i++)
    {
        if (*e_buf == '\n')
        {
            e_buf++;
        }
        else
        {
            decrypt_hex(e_buf, &r);
            e_buf += 2;
        }
    }
    
    while (TRUE)
    {
        if (*e_buf == '\n')
        {
            e_buf++;
        }
        else if (!isxdigit(e_buf[0]) || !isxdigit(e_buf[1]) || 
                 memcmp(e_buf, "00000000000000000000", 20) == 0 ||
                 e_buf >= e_end_ptr)
        {
            // we've got to the bottom now
            d_buf_length = d_buf - d_start_ptr;
            return;
        }
        else
        {
            *d_buf++ = decrypt_hex(e_buf, &r);
            e_buf += 2;
        }
    }
abort:
    return;
}
void mx_font_metrics::get_iso_lookup(int &err)
{
    char iso_conv_file[MAX_PATH_LEN];
    char line[132];                         // 132 characters should be enough
    int iso;
    FILE *fd;
        
    // Read the iso-8859-1 to postcript standard conversion file
    sprintf(iso_conv_file, "%s/text/iso2ps.txt", global_maxhome);

    fd = fopen(iso_conv_file, "r");

    if (fd == NULL) 
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }

    while (fgets(line, 132, fd) != NULL)
    {
        if (strstr(line, "xxx") == NULL) 
        {
            char *char_name = new char[40];
            sscanf(line, "%d /%s", &iso, char_name);

            iso_to_type1.add(err, iso, (void *) char_name);
            MX_ERROR_CHECK(err);

            type1_to_iso.add(err, char_name, (void *) iso);
            MX_ERROR_CHECK(err);
        }
    }
    fclose(fd);

    return;
abort:
    return;
}

uint16 mx_font_metrics::get_iso(int &err, const char *type1_name)
{
    if (type1_to_encoding != NULL) 
    {
        return (int) type1_to_encoding->get(err, type1_name);
    }
    else
    {
        return (int) type1_to_iso.get(err, type1_name);
    }
}

char *mx_font_metrics::get_type1_name(int &err, uint16 iso_num)
{
    if (encoding_to_type1 != NULL)
    {
        return (char *) encoding_to_type1->get(err, iso_num);
    }
    else
    {
        return (char *) iso_to_type1.get(err, iso_num);
    }
}

static unsigned short int c1 = 52845;
static unsigned short int c2 = 22719;

static unsigned char decrypt(unsigned char cipher, unsigned short int *r)
{
    unsigned char plain;

    plain = (cipher ^ (*r >> 8));
    *r = (cipher + *r) * c1 + c2;

    return plain;
}

static unsigned char decrypt_hex(const char *ptr, unsigned short int *r)
{
    char tmp[3] = "00";

    tmp[0] = ptr[0];
    tmp[1] = ptr[1];

    return decrypt(strtoul(tmp, NULL, 16), r);
}

mx_font_metrics_store::mx_font_metrics_store()
{
    int err = MX_ERROR_OK;

    init_type1_fm(err);
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return;
}

mx_font_metrics_store::~mx_font_metrics_store()
{
    mx_hash_iterator iter(font_families);
    while (iter.more()) delete (mx_font_family *)iter.data();
}

mx_font_family *mx_font_metrics_store
::get_font_family(int &err, const char *font_family_name)
{
    return (mx_font_family *)font_families.get(err, font_family_name);
}

static inline void make_lower_case(char *string)
{
    while (*string != '\0')
    {
        *string = tolower(*string);
        string++;
    }
}

mx_font_family *mx_font_metrics_store
::get_nearest_font_family(int &err, const char *name)
{
    mx_font_family *ff;
    char needle[201];
    char haystack[201];
    mx_hash_iterator iter(font_families);

    strncpy(needle, name, 200);
    needle[200] = '\0';
    make_lower_case(needle);

    while (iter.more())
    {
        ff = (mx_font_family *)iter.data();

        strncpy(haystack, ff->get_name(), 200);
        make_lower_case(haystack);

        if (strstr(haystack, needle) != NULL)
        {
            // found a match
            return ff;
        }        
    }

    ff = this->get_font_family(err, get_default_roman_font());
    MX_ERROR_CHECK(err);

    return ff;
abort:
    return NULL;
}

mx_font_metrics *mx_font_metrics_store
::get_font_metrics(int &err, const char *font_family_name, mx_font_style_t style)
{
    mx_font_family *family = get_font_family(err, font_family_name);
    MX_ERROR_CHECK(err);

    return family->get_font_metrics(style);
abort:
    return NULL;
}

uint32 mx_font_metrics_store::get_num_font_families()
{
    int err = MX_ERROR_OK;
    uint32 result = font_families.get_num_entries(err);
    if (err != MX_ERROR_OK)
    {
        global_error_trace->print();
        MX_ERROR_CLEAR(err);
        return 0;
    }
    return result;
}

void mx_font_metrics_store::get_font_family_names(mx_list &list)
{
    int err = MX_ERROR_OK;
    mx_hash::key_t *k = NULL;

    // wipe the list clean
    while (list.get_num_items() > 0) 
    {
        list.remove(err, 0);
        MX_ERROR_CHECK(err);
    }

    font_families.iterate_start(err);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        k = font_families.iterate_next_key(err);
        MX_ERROR_CHECK(err);

        if (k == NULL) break;

        list.append(k->s);
    }

    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return;
}

void mx_font_metrics_store::add_fm(mx_font_metrics *fm)
{
    int err = MX_ERROR_OK;
    mx_font_family *ff;

    ff = get_font_family(err, fm->get_name());
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);

        // This is the first font of this family

        ff = new mx_font_family;
        ff->set_font_metrics(fm);

        font_families.add(err, fm->get_name(), ff);
        MX_ERROR_CHECK(err);
    }
    else
    {
        MX_ERROR_CHECK(err);

        // we've got a font of this family already.

        ff->set_font_metrics(fm);
    }
    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return;
}

void mx_font_metrics_store::init_type1_fm(int &err)
{
    char fonts_dir_path[MAX_PATH_LEN] = "";
    FILE *fd = NULL;
    bool done_loop = FALSE;
    int32 num_fonts = 0;
    struct stat fonts_dir_stat;

    strcpy(fonts_dir_path, global_maxhome);
    strcat(fonts_dir_path, "/fonts/Type1/fonts.dir");

    if (stat(fonts_dir_path, &fonts_dir_stat) != 0)
    {
        MX_ERROR_THROW(err, MX_FMS_BAD_FONTSDIR_FILE);
    }

    fd = fopen(fonts_dir_path, "r");
    if (fd == NULL)
    {
        err = mx_translate_file_error(errno);
        MX_ERROR_CHECK(err);
    }

    if (fscanf(fd, "%ld\n", &num_fonts) != 1 || num_fonts < 0)
    {
        MX_ERROR_THROW(err, MX_FMS_BAD_FONTSDIR_FILE);
    }

    for (int32 i = 0; i < num_fonts; i++)
    {
        char mfm_font_file_name[MAX_PATH_LEN];
        char font_file_name[MAX_PATH_LEN];
        char font_x_name[200];
        struct stat mfm_font_stat, font_stat;
        bool read_font_write_mfm;
        
        // build full font file path name
        strcpy(font_file_name, global_maxhome);
        strcat(font_file_name, "/fonts/Type1/");
        
        // get font file name and its x name from fonts.dir
        if (fscanf(fd, "%s %s\n", font_file_name + strlen(font_file_name),
                  font_x_name) != 2)
        {
            mx_printf_warning("Problem in font description file: %s", fonts_dir_path);
        }
        else
        {
            strcpy(mfm_font_file_name, font_file_name);
            strcat(mfm_font_file_name, ".mfm");

            if (stat(font_file_name, &font_stat) != 0)
            {
                mx_printf_warning("Cannot load font: %s", font_file_name);
            }
            else if (stat(mfm_font_file_name, &mfm_font_stat) == 0 &&
                     mfm_font_stat.st_mtime > font_stat.st_mtime &&
                     mfm_font_stat.st_mtime > fonts_dir_stat.st_mtime)
            {
                // Just read the mfm file since it is up to date.
                mx_font_metrics *fm = new mx_font_metrics;
                
                fm->read_mfm(err, mfm_font_file_name);
                if (err != MX_ERROR_OK)
                {
                    global_error_trace->print();
                    MX_ERROR_CLEAR(err);
                }
                else
                {
                    add_fm(fm);
                }
            }
            else
            {
                // Write new mfm file if either one does not exist OR its
                // last modification time is older than that of the font
                // file
                
                mx_font_metrics *fm = new mx_font_metrics;
                
                fm->read_type1(err, font_file_name, font_x_name);
                if (err != MX_ERROR_OK)
                {
                    global_error_trace->print();
                    MX_ERROR_CLEAR(err);
                }
                else 
                {
                    add_fm(fm);
                    fm->write_mfm(err, mfm_font_file_name);
                    if (err != MX_ERROR_OK)
                    {
                        static bool printed_error = FALSE;

                        MX_ERROR_CLEAR(err);
                        if (!printed_error)
                        {
                            mx_printf_warning("Could not write maxwell font caching files. Please check");
                            mx_printf_warning("permissions for maxwell font directory %s/fonts/Type1", global_maxhome);
                            printed_error = TRUE;
                        }
                    }
                }
            }
        }
    }
abort:
    fclose(fd);
}


