/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_help_window
 *
 * AUTHOR : David Miller
 *
 * DESCRIPTION:
 *
 * The mx_help_window class provides a window in which to view help
 * documents
 *
 */

#include <stdlib.h>

#include <mx.h>
#include <mx_window.h>
#include <mx_help.h>
#include <mx_editor.h>



/*-------------------------------------------------
 * FUNCTION: mx_help_window::mx_help_window
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_window::mx_help_window(int &err,char *rootFile,mx_editor *o = NULL) : mx_window(NULL, "help")
{
    mx_doc_coord_t tl(0,0) ;
    mx_point       size(100,100) ;
    mx_point       increment(1,1) ;
    bool           backward,forward ;
    
    owner = o;

    err = MX_ERROR_OK ;
    
    menu = new mx_help_menubar(this);                
    tools = new mx_help_toolbar(this);        

    target = new mx_help_target(err) ;
    MX_ERROR_CHECK(err) ;

    
    frame = new mx_scrollable_frame(err, 
                                    tl,
                                    size,
                                    1,
                                    *target,
                                    get_form(),
                                    TRUE,
                                    increment,
                                    FALSE) ;
    MX_ERROR_CHECK(err);



    target->setWindow(this) ;

    setFile(err,rootFile,NULL);
    MX_ERROR_CHECK(err) ;
    
    status = new mx_statusbar(this);        

    add_ui_object(menu) ;
    add_ui_object(tools);
    add_ui_object(frame) ;
    add_ui_object(status) ;


    status->update(MX_MAXWELL_VERSION);

    return ;
    
abort:
    return;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_window::~mx_help_window
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_window::~mx_help_window()
{
    // delete the components
    delete tools ;
    delete frame ;
    delete menu ;
    delete status ;
}


/*-------------------------------------------------
 * FUNCTION: mx_help_window::setFile
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_window::setFile(int      &err,
                             char     *fileName,
                             mx_point *setPos) 
{
    mx_help_file *initialHelpFile = target->getHelpFile() ;
    mx_point newPos ;
    bool forward,backward,newFile ;
    mx_point frameTopLeft = frame->getTopLeft();
    
    err = MX_ERROR_OK ;

    // set the top left on the frame target 

    newFile = (setPos == NULL) ;

    if(initialHelpFile != NULL)
        initialHelpFile->setFrameTopLeft(frameTopLeft) ;

    target->setFile(err,fileName,newFile,frameTopLeft,&backward,&forward);
    MX_ERROR_CHECK(err) ;

    if(newFile) 
    {
        newPos = target->getHelpFile()->getFrameTopLeft() ;
    }
    else
    {
        newPos = *setPos ;
    }
    
    frame->isetTopLeft(newPos) ;
            
    resetSize(err,target->getHelpFile()->getFileSize()) ;
    MX_ERROR_CHECK(err);    

    if(initialHelpFile != NULL) 
    {
        frame->refresh(err) ;
        MX_ERROR_CHECK(err) ;
    }
    
    setSensitive(backward,forward) ;
    
    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_window::resetSize
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_window::resetSize(int &err,
                               mx_point &newSize) 
{
    mx_sheet_size sz(newSize);

    // change the sheet layout
    get_target()->getSheetLayout().changeSheets(err,
                                                0,1,&sz) ;
    MX_ERROR_CHECK(err) ;

    // get the frame to resize
    get_frame()->resetSheetLayout(err) ;
    MX_ERROR_CHECK(err) ;


    return ;
  abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_window::setSensitive
 *
 * DESCRIPTION: 
 * 
 *
 */


void mx_help_window::setSensitive(bool setBackward,
                                  bool setForward) 
{
    tools->set_backward_sensitive(setBackward);
    tools->set_forward_sensitive(setForward);
    menu->set_backward_sensitive(setBackward);
    menu->set_forward_sensitive(setForward);    
}

void mx_help_window::forced_close()
{
    if (owner != NULL)
    {
        owner->help_window_closing();
    }
    delete this;
}
