/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_block
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 *
 * DESCRIPTION:
 * block cache 
 *
 *
 *
 *
 */
#ifndef MX_BLOCK_H
#define MX_BLOCK_H

#include <stdlib.h>
#include <mx.h>
#include <mx_attribute.h>

#if 0
#define HALF_NUM_BLOCK_ENTRIES 2
#else
#define HALF_NUM_BLOCK_ENTRIES 34
#endif

#define NUM_BLOCK_ENTRIES (HALF_NUM_BLOCK_ENTRIES * 2)
#define BLOB_SEGMENT_SIZE  MX_FILE_BLOCK_SIZE

typedef struct 
{
    int32 ptrA;
    int32 ptrB;
}
block_ptr_t;

typedef enum 
{
    value_object_e,
    value_blob_e,
    value_int_e,
    value_float_e,
    value_id_e,
    value_string_e,
    value_char_array_e,
    value_int_array_e,
    value_id_array_e,
    value_float_array_e,
    value_blob_segment_e
} 
value_type_t;

typedef struct
{
    unsigned int offset : 12;
    unsigned int block  : 20;
    int length;
}
data_ptr_t;

typedef union
{
    data_ptr_t ptr;

    float  f;
    int32  i;
    int32  id;
    int    size;
}
value_t;

typedef struct 
{
    char    name[MX_ATTRIBUTE_NAME_LENGTH];
    uint32  oid;
    uint32  segment;

    block_ptr_t  ptr;

    unsigned char type;
    value_t       value;
}
entry_t;

typedef struct
{
    uint32      magic;
    uint16      num_entries;
    char        in_list;
    char        freeable;
    char        pointers_modified;
    int32        next_modified_block;
    block_ptr_t ptr0;
}
block_header_t;

typedef struct
{
    block_header_t header;
    entry_t        entries[NUM_BLOCK_ENTRIES];
}
block_t;

#define TREE_MAGIC_NUM    0x71ab15a6
#define DATA_MAGIC_NUM    0x6a51ba17

#define MX_FILE_BLOCK_SIZE 4096

#define MX_BLOCK_DESCRIPTION_LENGTH 200
#define MX_BLOCK_AUTHOR_LENGTH      100
#define MX_BLOCK_VERSION_LENGTH     100

// number of blocks cached - small number only needed
#define MX_BLOCK_CACHE_SIZE 20

typedef struct
{
    uint32 sig;
    int    first_free;
    int    first_free_data;
    int    first_free_data_index;
    int    root_to_use;
    int    dirty_flag;
    int    root_block[2];
    uint32 next_id;
    int    num_spare_blocks;

    char   description[MX_BLOCK_DESCRIPTION_LENGTH];
    char   author[MX_BLOCK_AUTHOR_LENGTH];
    char   version[MX_BLOCK_VERSION_LENGTH];
}
file_header_t;

typedef struct
{
    unsigned char type;
    unsigned char mark;
    uint16   size;
}
area_size_t;

#define MX_BLOCK_FREE_TYPE 0xAA
#define MX_BLOCK_USED_TYPE 0x55

class mx_block : public mx_rtti 
{
    MX_RTTI(mx_block_class_e)

public:

    // attach a block file to an open file descriptor
    mx_block(int &err, int fd);

    // close file
    virtual ~mx_block();

    // flush the cache
    void flush(int &err);

    // read a block - returns NULL on error
    const unsigned char *read(int &err, int bn, bool is_blob = FALSE);

    // mark a block as dirty - can't get swapped out until
    // it's been written
    void set_dirty(int &err, int bn);

    // write a block from the cache
    void write(int &err, int bn);

    // allocate a free one
    int allocate(int &err, bool reserve = TRUE);

    // free a used one
    void free(int &err, int n);

    // reset the free chain to contain no blocks - used
    // during recovery
    void reset_free();

    // get number of blocks
    int num_blocks();

    // set one of the root block pointers
    void set_root(int &err, int root_to_set, int bn);

    // get one of the root block pointers
    int get_root(int &err, int root_to_get);

    // set or clear dirty flag
    void set_dirty_flag(int &err, bool dirty);

    // get dirty flag
    bool get_dirty_flag(int &err);

    // set root_to_use
    void set_root_to_use(int &err, int root_to_use);

    // get root_to_use
    int get_root_to_use(int &err);

    // allocate from the data chain. size <= BLOCK_SIZE
    void allocate_data(int &err, int size, int &bn, int &offset);

    // write data 
    void write_data(int &err, char *data, int size, int bn, int offset);

    // read data 
    void read_data(int &err, unsigned char *data, int size, int bn, int offset);

    // free data - put pack in free chain
    void free_data(int &err, int bn, int offset);

    void reset_free_data();

    // lock a block in memory
    void lock(int &err, int bn);

    // unlock a block
    void unlock(int &err, int bn);

    // get an object id
    uint32 get_next_id();

    // copy into new file
    void duplicate(int &err, int file);

    int size_in_blocks;

    void set_info(int &err, char *description, char *author, char *version);

    const char *get_description();
    const char *get_author();
    const char *get_version();

    int get_num_spare_blocks();
    void extend(int &err, int nb);

private:
    int  fd;

    file_header_t    header;

    int next_touch;

    unsigned char *cache[MX_BLOCK_CACHE_SIZE];
    int           numbers[MX_BLOCK_CACHE_SIZE];
    bool          dirty[MX_BLOCK_CACHE_SIZE];
    bool          is_blob[MX_BLOCK_CACHE_SIZE];
    bool          locked[MX_BLOCK_CACHE_SIZE];
    int           touch[MX_BLOCK_CACHE_SIZE];

    unsigned char dummy[MX_FILE_BLOCK_SIZE];

    void seek_to_block(int &err, int bn);

    void write_header(int &err);
    void remove_from_free_chain(int &err, int bn, int offset);
    void allocate_from_new_block(int &err, int size, int &bn, int &offset);
    int get_free_cache_position(int &err);

    int translate_error(int e);

    void host_block_to_network(block_t *b);
    void network_block_to_host(block_t *b);
    void network_data_block_to_host(block_t *b);
    void host_data_block_to_network(block_t *b);


};

#endif
