/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_db_client
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 *
 *
 *
 */

#include <stdio.h>


#include <mx.h>
//#include <mx_lm.h>

#include <mx_db_client.h>
#include <mx_db_cc.h>
#include <mx_db_object.h>
#include <mx_wp_doc.h>

static uint32 next_docid = 0;

bool mx_db_client_login(int &err)
/*
 * PARAMS  :
 *   OUT   - err       what went wrong
 *
 * ACTION  : logs in - may need to connect to server too
 * RETURNS : TRUE if everything worked, FALSE otherwise. On
 *            : failure, err indicates what went wrong:
 *         :
 *           :    MX_DB_CLIENT_COULDNT_CONNECT 
 *           :    MX_DB_CLIENT_AUTHENTICATION_FAILURE 
 *
 */
{
    mx_db_cc_init(err);
    MX_ERROR_CHECK(err);
    return TRUE;
abort:
    return FALSE;
}

/*
static void mx_db_client_lock_file(
                        int   &err, 
                        char  *file_name,
                        char  **locked_by, 
                        char  **locked_host, 
                        pid_t &locked_pid)
 *
 * PARAMS  :
 *   OUT   - err        what went wrong
 *   IN    - file_name    name of file document
 *
 * ACTION  : tries to lock the file.
 * RETURNS : none
 *
 * NOTES : there is only one error that can be returned - 
 *       :    MX_LM_ALREADY_LOCKED
 *       : all other errors are trapped and cleared as the lock manager is a nice
 *       : extra rather than totally required bit of the system
 *
 */
/*
{
    char *s = mx_expand_path(file_name);

    mx_lm_client_lock_file(err, s, locked_by, locked_host, locked_pid);
    if (err != MX_ERROR_OK)
    {
        if (err == MX_LM_ALREADY_LOCKED)
        {
            char hn[100];

            // if the file is locked by someone on this machine, make sure 
            // that the process is still going
            if (gethostname(hn, 99) == 0)
            {
                if (strcmp(*locked_host, hn) == 0)
                {
                    if (kill(locked_pid, 0) == -1)
                    {
                        // probably means that maxwell crashed (or was killed)
                        // and then restarted. either way no need to worry about
                        // the file already being locked - overide the lock to get
                        // the right pid attached to it
                        mx_lm_client_lock_file(err, s, locked_by, locked_host, locked_pid, TRUE);
                        MX_ERROR_CLEAR(err);
                    }
                }
            }
        }
        else
        {
            // ignore all other errors -
            MX_ERROR_CLEAR(err);
        }
    }
}*/

mx_wp_document *mx_db_client_open_wp_doc( 
                    int                   &err, 
                    char                  *file_name,
                    bool                  recover,
                    char                  **locked_by, 
                    char                  **locked_host, 
                    pid_t                 &locked_pid,
                    mx_file_create_type_t t = mx_file_simple_e)
/*
 * PARAMS  :
 *   OUT   - err        what went wrong
 *   IN    - cid        database connection id
 *   IN    - file_name  name of file document
 *   IN    - recover    whether to attempt to recover a damaged document
 *
 * ACTION  : opens the document creating it if necessary
 * RETURNS : a pointer to the document
 *
 */
{
    mx_wp_document *res;

    err = MX_ERROR_OK;

    // try to lock it first
    // mx_db_client_lock_file(err, file_name, locked_by, locked_host, locked_pid);
    // MX_ERROR_CHECK(err);

    mx_db_cc_set_ignore_update(TRUE);

    mx_db_cc_open_doc(err, next_docid, file_name, recover, t);
    //if (err != MX_ERROR_OK)
    //{
    //    mx_lm_client_unlock_file(mx_expand_path(file_name));
    //}
    MX_ERROR_CHECK(err);

    res = new mx_wp_document(file_name, next_docid);

    next_docid++;
    return res;

abort:
    return NULL;
}

void mx_db_client_close_wp_doc(int &err, mx_wp_document *doc)
/*
 * PARAMS  :
 *   OUT   - err  what went wrong
 *   IN    - doc  the document to close
 *
 * ACTION  : closes the document 
 *
 */
{
    int docid;

    docid = doc->get_docid();

    mx_db_cc_set_ignore_update(FALSE);

    // unlock it
    // mx_lm_client_unlock_file(doc->get_full_file_name());

    mx_db_cc_close_doc(err, docid);
    MX_ERROR_CHECK(err);

    mx_db_cc_set_ignore_update(TRUE);

abort:;
}

mx_wp_document *mx_db_client_open_temporary_wp_doc(int &err, char *page_size = "A4")
/*
 * PARAMS  :
 *   OUT   - err  what went wrong
 *
 * ACTION  : creates a new document in a temporary file (/tmp e.g.) 
 * RETURNS : a pointer to the document. There should be no erros except in
 *         : a disaster.
 *
 */
{
    char *locked_by, *locked_host;
    pid_t locked_pid;

    mx_wp_document *res;

    char *file_name = mx_tmpnam(NULL);

    err = MX_ERROR_OK;

    // lock it first - something serious is wrong if this fails
    //mx_db_client_lock_file(err, file_name, &locked_by, &locked_host, locked_pid);
    //MX_ERROR_CHECK(err);

    mx_db_cc_set_ignore_update(TRUE);

    mx_db_cc_open_doc(err, next_docid, file_name, TRUE);
    //if (err != MX_ERROR_OK)
    //{
    //    mx_lm_client_unlock_file(mx_expand_path(file_name));
    //}
    MX_ERROR_CHECK(err);

    res = new mx_wp_document(file_name, next_docid, page_size);

    res->commit(err);
    MX_ERROR_CHECK(err);

    res->set_temp_flag(TRUE);

    next_docid++;

    return res;

abort:
    return NULL;
}

mx_document *mx_db_client_open_temporary_doc(int &err)
/*
 * PARAMS  :
 *   OUT   - err  what went wrong
 *
 * ACTION  : creates a new document in a temporary file (/tmp e.g.) 
 * RETURNS : a pointer to the document. There should be no erros except in
 *         : a disaster.
 *
 */
{
    char *locked_by, *locked_host;
    pid_t locked_pid;

    mx_document *res;

    char *file_name = mx_tmpnam(NULL);

    err = MX_ERROR_OK;

    // lock it first - something serious is wrong if this fails
    //mx_db_client_lock_file(err, file_name, &locked_by, &locked_host, locked_pid);
    //MX_ERROR_CHECK(err);

    mx_db_cc_set_ignore_update(TRUE);

    mx_db_cc_open_doc(err, next_docid, file_name, TRUE);
    //if (err != MX_ERROR_OK)
    //{
    //    mx_lm_client_unlock_file(mx_expand_path(file_name));
    //}
    MX_ERROR_CHECK(err);

    res = new mx_document(file_name, next_docid);
    res->set_temp_flag(TRUE);

    next_docid++;

    return res;

abort:
    return NULL;
}
