/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_db_client_cache
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 *
 *
 *
 */

#include <stdlib.h>
#include <string.h>

#include <mx_db_cc.h>
#include <mx.h>
#include <mx_hash.h>
#include <mx_hash_iter.h>
#include <mx_error.h>
#include <mx_error_codes.h>
#include <mx_db_object.h>
#include <mx_doc.h>
#include <mx_file.h>

typedef struct _act
{
    bool            new_attr;
    mx_attribute    *attr;
    uint32          touch;
    uint32          size;
} 
attr_cache_t;

typedef struct _segt
{
    unsigned char   *data;
    uint32          touch;
    bool            updated;
} 
segment_t;

typedef struct _bct
{
    uint32          size;
    int             num_segments;
    segment_t       *segments;
    bool            new_blob;
    bool            modified;
    bool            size_changed;
} 
blob_cache_t;
    
typedef struct _oct
{
    bool          new_obj;
    uint32        id;
    mx_db_object  *obj;
    mx_hash       *attrs;    // hash table of attributes
    mx_hash       *blobs;    // hash table of blobs
} 
obj_cache_t;


// a hash table of hash tables:
//
// [docid1] -> hash table of objects by oid (each is * obj_cache_t)
// [docid2] -> ......
//

static mx_hash      *docs=NULL;    
static mx_hash      *files=NULL;    
static mx_hash      *blob_heap_hash=NULL;

static unsigned char    blob_readonly[MX_DB_OBJECT_MAX_READONLY_BLOB_SIZE];

static bool db_updateable = FALSE;

void mx_db_cc_set_ignore_update(bool f)
{
    db_updateable = f;
}

static void add_attr_to_file(
                int          &err, 
                uint32       docid, 
                uint32       oid, 
                mx_attribute *a)
{
    mx_file *f;

    f = (mx_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    f->write_attribute(err, oid, a);
    MX_ERROR_CHECK(err);

    return;

abort:;
    return;
}

void mx_db_cc_read_oid(int &err, uint32 docid, uint32 oid, mx_db_object *o)
{
    obj_cache_t *item;
    mx_hash     *obj_hash;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK( err );
        
    item = new obj_cache_t;

    item->obj = o;
    item->attrs = new mx_hash(ATTR_HASH_START_SIZE);
    item->blobs = new mx_hash(BLOB_HASH_START_SIZE);

    item->new_obj = FALSE;
    item->id = oid;

    obj_hash->add(err, oid, item);
    MX_ERROR_CHECK(err);

abort:;
}

uint32 mx_db_cc_add_oid(int &err, uint32 docid, mx_db_object *o)
{
    obj_cache_t *item;
    mx_hash     *obj_hash;
    uint32      next_id;
    mx_doc_file *f;

    // is the database active?
    if (!db_updateable)
    {
        return 0;
    }

    f = (mx_doc_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    next_id = f->get_next_id();

    obj_hash = (mx_hash *)docs->get(err, docid);

    if (err == MX_HASH_NOT_FOUND) 
    {
        MX_ERROR_CLEAR(err);

        obj_hash = new mx_hash(OBJ_HASH_START_SIZE);
        docs->add(err, docid, obj_hash);
        MX_ERROR_CHECK(err);
    }
    else 
    {
        MX_ERROR_CHECK(err);
    }

    item = new obj_cache_t;

    item->obj = o;
    item->attrs = new mx_hash(ATTR_HASH_START_SIZE);
    item->blobs = new mx_hash(BLOB_HASH_START_SIZE);

    item->new_obj = TRUE;
    item->id = next_id;

    obj_hash->add(err, item->id, item);
    MX_ERROR_CHECK(err);

    return item->id;
abort:;
    return 0;
}

static void delete_an_attr(int &err, attr_cache_t *c)
{
    c->attr->free_value();
    delete c->attr;
    delete c;
}

static void delete_a_blob(int &err, blob_cache_t *c, int docid)
{
    int          i;

    for (i = 0; i<c->num_segments; i++) 
    {
        if(c->segments[i].data!=NULL) 
        {
            delete [] c->segments[i].data;
        }
    }

    if (c->segments != NULL)
    {
        delete [] c->segments;
    }
    delete c;
abort:;
}

static void mx_db_cc_delete_oid_from_cache(int &err, uint32 docid, uint32 oid)
{
    mx_hash      *obj_hash;
    obj_cache_t  *item;
    blob_cache_t *b;
    attr_cache_t *a;
    bool         temp;

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // free the attributes
    item->attrs->iterate_start(err);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        a = (attr_cache_t *)item->attrs->iterate_next(err);
        MX_ERROR_CHECK(err);
    
        if (a == NULL)
        {
            break;
        }

        delete_an_attr(err, a);
        MX_ERROR_CHECK(err);
    }

    // and the blobs
    item->blobs->iterate_start(err);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        b = (blob_cache_t *)item->blobs->iterate_next(err);
        MX_ERROR_CHECK(err);
    
        if (b == NULL)
        {
            break;
        }

        delete_a_blob(err, b, docid);
        MX_ERROR_CHECK(err);
    }

    // delete the attribute hash table
    delete item->attrs;

    // delete the blobs hash table
    delete item->blobs;

    // free the object struct
    delete item;

    // delete the object entry from the object hash
    obj_hash->del(err, oid);
    MX_ERROR_CHECK(err);
abort:;
}


void mx_db_cc_delete_oid( 
            int &err, uint32 docid, uint32 oid)
{
    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    // drop it from the cache - if it's there
    mx_db_cc_delete_oid_from_cache(err, docid, oid);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);

        return;
    }
    MX_ERROR_CHECK(err);

    return;

abort:;
}

mx_db_object *mx_db_cc_get_obj_from_oid(int &err, uint32 docid, uint32 oid)
{
    mx_hash     *obj_hash;
    obj_cache_t    *item;

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    if (item->obj == NULL)
    {
        err = MX_DB_CLIENT_CACHE_NO_SUCH_OID;
        return NULL;
    }
    else
    {
        return item->obj;
    }

abort:
    MX_ERROR_CLEAR(err);
    err = MX_DB_CLIENT_CACHE_NO_SUCH_OID;
    return NULL;
}

void mx_db_cc_set_attribute(int &err, uint32 docid, uint32 oid, mx_attribute *a)
{
    mx_attribute    *copy;
    mx_hash         *obj_hash, *attr_hash;
    obj_cache_t     *item;
    attr_cache_t    *attr;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    // is it too big?
    if (a->size( err )>MAX_ATTRIBUTE_SIZE) 
    {
        MX_ERROR_THROW(err, MX_DB_CLIENT_CACHE_ATTR_TOO_BIG);
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
        MX_ERROR_THROW(err, MX_DB_CLIENT_CACHE_NO_SUCH_OID);
    }
    MX_ERROR_CHECK(err);

    // got the object record

    attr_hash = item->attrs;

    // does the attribute exist already ?
    attr = (attr_cache_t *)(attr_hash->get(err, a->name));
    if (err == MX_ERROR_OK) 
    {
        // yes, delete the old one
        attr->attr->free_value();
        delete attr->attr;
    }
    else 
    {
        // create it
        attr = new attr_cache_t;

        attr_hash->add(err, a->name, attr);
        MX_ERROR_CHECK(err);
    }

    copy = new mx_attribute();
    *copy = *a;
    attr->new_attr = TRUE;
    attr->attr = copy;
abort:;
}


const mx_attribute *mx_db_cc_get_attribute_ro(
                        int    &err, 
                        uint32 docid, 
                        uint32 oid, 
                        char   *name)
{
    mx_hash            *obj_hash, *attr_hash;
    obj_cache_t        *item;
    attr_cache_t    *attr;
    bool            b;

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
        MX_ERROR_THROW(err, MX_DB_CLIENT_CACHE_NO_SUCH_ATTR);
    }
    MX_ERROR_CHECK(err);

    // got the object record
    attr_hash = item->attrs;

    // get the attribute 
    attr = (attr_cache_t *)attr_hash->get(err, name);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
        MX_ERROR_THROW(err, MX_DB_CLIENT_CACHE_NO_SUCH_ATTR);
    }
    MX_ERROR_CHECK(err);

    return attr->attr;

abort:;
    return NULL;
}

void mx_db_cc_del_attribute(int &err, uint32 docid, uint32 oid, char *name)
{
    mx_hash         *obj_hash, *attr_hash;
    obj_cache_t     *item;
    attr_cache_t *attr;
    bool         b;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record

    attr_hash = item->attrs;

    // get the attribute
    attr = (attr_cache_t *)attr_hash->get(err, name);
    MX_ERROR_CHECK(err);

    attr->attr->free_value();
    delete attr->attr;
    delete attr;
    attr_hash->del(err, name);
    MX_ERROR_CHECK(err);
    
abort:;
}

void mx_db_cc_init(int &err)
{
    if (docs == NULL) 
    {
        docs = new mx_hash(DOC_HASH_START_SIZE );
        files = new mx_hash(DOC_HASH_START_SIZE );
        blob_heap_hash = new mx_hash(DOC_HASH_START_SIZE );
    }
}

void mx_db_cc_set_blob_data(int &err, uint32 docid, uint32 oid, char *name, uint32 index, uint32 n_bytes, unsigned char *data)
{
    mx_hash            *obj_hash, *blob_hash;
    obj_cache_t        *item;
    blob_cache_t    *blob;
    int                block_num, start_block, end_block;
    int                start_index, end_index;
    int                in_index;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err,docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record
    blob_hash = item->blobs;

    // get the blob record
    blob = (blob_cache_t *)(blob_hash->get(err, name));
    MX_ERROR_CHECK(err);

    // check size 
    if ((index + n_bytes) > blob->size) 
    {
        MX_ERROR_THROW(err, MX_DB_OBJECT_BLOB_INDEX_RANGE);
    }

    // find start and end block numbers
    start_block = index / BLOB_SEGMENT_SIZE;
    end_block = (index + n_bytes) / BLOB_SEGMENT_SIZE;

    in_index = 0;

    // copy block at a time
    for (block_num = start_block; block_num <= end_block; block_num++) 
    {
        start_index = (block_num == start_block ) ?
                       index - (start_block * BLOB_SEGMENT_SIZE) :
                       0;
        end_index = (block_num == end_block ) ?
                     (index + n_bytes) - (end_block * BLOB_SEGMENT_SIZE) :
                     BLOB_SEGMENT_SIZE;

        if (blob->segments[block_num].data == NULL)
        {
            blob->segments[block_num].data = new unsigned char[BLOB_SEGMENT_SIZE];
        }

        memcpy(
            blob->segments[block_num].data + start_index, 
            data + in_index, 
            end_index - start_index);

        blob->segments[block_num].updated = TRUE;
        in_index += end_index - start_index;
    }
    blob->modified = TRUE;
abort:;
}

const unsigned char *mx_db_cc_get_blob_readonly(int &err, uint32 docid, uint32 oid, char *name, uint32 index, uint32 n_bytes)
{
    if (n_bytes > MX_DB_OBJECT_MAX_READONLY_BLOB_SIZE) 
    {
        MX_ERROR_THROW(err, MX_DB_OBJECT_BLOB_SLICE_SIZE);
    }
    else 
    {
        mx_db_cc_get_blob_data(err, docid, oid, name, index, n_bytes, blob_readonly);
        MX_ERROR_CHECK(err);
    }
    return blob_readonly;
abort:;
    return NULL;
}

void mx_db_cc_create_blob(int &err, uint32 docid, uint32 oid, char *name)
{
    mx_hash            *obj_hash, *blob_hash;
    obj_cache_t        *item;
    blob_cache_t    *blob;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record
    blob_hash = item->blobs;

    (void)blob_hash->get(err, name);
    if (err == MX_ERROR_OK)
    {
        MX_ERROR_THROW(err, MX_DB_CLIENT_CACHE_DUPLICATE_BLOB);
    }
    else
    {
        MX_ERROR_CLEAR(err);
    }

    blob = new blob_cache_t;

    blob->size = 0;
    blob->segments = NULL;
    blob->new_blob = TRUE;
    blob->modified = TRUE;
    blob->size_changed = TRUE;
    blob->num_segments = 0;
    blob_hash->add(err, name, blob);
    MX_ERROR_CHECK(err);
abort:;
}

void mx_db_cc_delete_blob(int &err, uint32 docid, uint32 oid, char *name)
{
    mx_hash      *obj_hash, *blob_hash;
    obj_cache_t  *item;
    blob_cache_t *blob;
    int          i;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record
    blob_hash = item->blobs;

    blob = (blob_cache_t *)(blob_hash->get(err, name));
    MX_ERROR_CHECK(err);

    for (i = 0; i < blob->num_segments; i++) 
    {
        if (blob->segments[i].data != NULL)
        {
            delete [] blob->segments[i].data;
        }
    }
    if (blob->segments != NULL)
    {
        delete [] blob->segments;
    }

    delete blob;

    blob_hash->del(err, name);
    MX_ERROR_CHECK(err);
abort:;
}

uint32 mx_db_cc_get_blob_size(int &err, uint32 docid, uint32 oid, char *name)
{
    mx_hash            *obj_hash, *blob_hash;
    obj_cache_t        *item;
    blob_cache_t    *blob;

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record

    blob_hash = item->blobs;

    blob = (blob_cache_t *)(blob_hash->get(err, name));
    MX_ERROR_CHECK(err);

    return blob->size;
abort:    
    return 0;
}

void mx_db_cc_set_blob_size(int &err, uint32 docid, uint32 oid, char *name, uint32 size)
{
    mx_hash         *obj_hash, *blob_hash;
    obj_cache_t     *item;
    blob_cache_t *blob;
    int             i, new_num_segs, start;
    segment_t     *temp;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record
    blob_hash = item->blobs;

    blob = (blob_cache_t *)(blob_hash->get(err, name));
    MX_ERROR_CHECK(err);

    if (size == blob->size)
    {
        return;
    }

    new_num_segs = (size / BLOB_SEGMENT_SIZE) + 1;
    if (new_num_segs == blob->num_segments) 
    {
        if (blob->segments[new_num_segs - 1].data != NULL)
        {
            // zero out the last bit of the last block
            if (blob->size > size)
            {
                // it's shrunk
                start = size % BLOB_SEGMENT_SIZE;
            }
            else
            {
                // it's grown
                start = blob->size % BLOB_SEGMENT_SIZE;
            }

            memset(
                blob->segments[new_num_segs - 1].data + start,
                0, 
                BLOB_SEGMENT_SIZE - start);
            blob->segments[new_num_segs - 1].updated = TRUE;
        }

        blob->size = size;
        blob->size_changed = TRUE;
        return;
    }

    temp = new segment_t[new_num_segs];

    if (blob->size < size) 
    {
        // extension
        if (blob->num_segments != 0) 
        {
            memcpy(temp, blob->segments, sizeof(segment_t) * blob->num_segments);
        }

        for (i = blob->num_segments; i < new_num_segs; i++) 
        {
            temp[i].updated = TRUE;
            temp[i].data = NULL;
        }

        if (blob->num_segments != 0)
        {
            // zero out the extended bit of the old last segment
            if (temp[blob->num_segments - 1].data != NULL)
            {
                start = blob->size % BLOB_SEGMENT_SIZE;
                memset(    
                    temp[blob->num_segments - 1].data + start, 
                    0, 
                    BLOB_SEGMENT_SIZE - start);
                temp[blob->num_segments - 1].updated = TRUE;
            }
        }

    }
    else 
    {
        // shrinkage
        memcpy(temp, blob->segments, sizeof(segment_t) * new_num_segs);
        for (i = new_num_segs; i < blob->num_segments; i++) 
        {
            if (blob->segments[i].data != NULL)
            {
                delete [] blob->segments[i].data;
            }
        }

        // zero out the truncated bit of the segment
        if (temp[new_num_segs - 1].data != NULL)
        {
            start = size % BLOB_SEGMENT_SIZE;

            memset(    
                temp[new_num_segs - 1].data + start, 
                0, 
                BLOB_SEGMENT_SIZE - start);
            temp[new_num_segs - 1].updated = TRUE;
        }
    }

    if (blob->segments != NULL)
    {
        delete [] blob->segments;
    }
    blob->segments = temp;

    blob->num_segments = new_num_segs;
    blob->size_changed = TRUE;
    blob->size = size;
abort:;
}

void mx_db_cc_get_blob_data(int &err, uint32 docid, uint32 oid, char *name, uint32 index, uint32 n_bytes, unsigned char *data)
{
    mx_hash            *obj_hash, *blob_hash;
    obj_cache_t        *item;
    blob_cache_t    *blob;
    int                block_num, start_block, end_block;
    int                start_index, end_index;
    int                out_index;

    obj_hash = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    // got the object record
    blob_hash = item->blobs;

    // get the blob record
    blob = (blob_cache_t *)(blob_hash->get(err, name));
    MX_ERROR_CHECK(err);

    // check size 
    if ((index + n_bytes) > blob->size) 
    {
        MX_ERROR_THROW(err, MX_DB_OBJECT_BLOB_INDEX_RANGE);
    }

    // find start and end block numbers
    start_block = index / BLOB_SEGMENT_SIZE;
    end_block = (index + n_bytes) / BLOB_SEGMENT_SIZE;

    out_index = 0;

    // copy block at a time
    for (block_num = start_block; block_num <= end_block; block_num++) 
    {
        start_index = (block_num == start_block) ?
                       index - (start_block * BLOB_SEGMENT_SIZE) :
                       0;

        end_index = (block_num == end_block ) ?
                    (index + n_bytes) - (end_block * BLOB_SEGMENT_SIZE) :
                    BLOB_SEGMENT_SIZE;

        if (blob->segments[block_num].data == NULL) 
        {
            if (!blob->new_blob)
            {
                if (blob->segments[block_num].data == NULL) 
                {
                    memset(data + out_index, 0, end_index - start_index);
                }
                else
                {
                    memcpy(data + out_index, blob->segments[block_num].data + start_index, end_index - start_index);
                }
            }
            else
            {
                memset(data + out_index, 0, end_index - start_index);
            }
        }
        else 
        {
            memcpy(data + out_index, blob->segments[block_num].data + start_index, end_index - start_index);
        }
        out_index += end_index - start_index;
    }
abort:;
}

static void create_object(int &err, int docid, int oid)
{
    obj_cache_t *item;
    mx_hash     *obj_hash;

    obj_hash = (mx_hash *)docs->get(err, docid);
    if (err != MX_ERROR_OK)
    {
        MX_ERROR_CLEAR(err);

        obj_hash = new mx_hash(OBJ_HASH_START_SIZE);

        docs->add(err, docid, obj_hash);
        MX_ERROR_CHECK(err);
    }

    item = (obj_cache_t *)obj_hash->get(err, oid);
    if (err == MX_ERROR_OK)
    {
        return;
    }
    else
    {
        MX_ERROR_CLEAR(err);

        item = new obj_cache_t;

        item->obj = NULL;
        item->attrs = new mx_hash(ATTR_HASH_START_SIZE);
        item->blobs = new mx_hash(BLOB_HASH_START_SIZE);

        item->new_obj = FALSE;
        item->id = oid;

        obj_hash->add(err, item->id, item);
        MX_ERROR_CHECK(err);
    }
abort:;
}

static void read_file(
                int      &err, 
                int      docid, 
                mx_file  *file)
{
    mx_sfile_type_t t;

    int oid, length, segnum, num_segs;
    mx_attribute *a;

    char name[100];
    char data[BLOB_SEGMENT_SIZE];

    while (TRUE)
    {
        t = file->next_type(err);
        MX_ERROR_CHECK(err);

        switch (t)
        {
            case mx_sfile_attr_e :
                    a = file->next_attribute(err, oid);
                    MX_ERROR_CHECK(err);

                    create_object(err, docid, oid);
                    MX_ERROR_CHECK(err);

                    mx_db_cc_set_attribute(err, docid, oid, a); 
                    MX_ERROR_CHECK(err);

                    a->free_value();
                    delete a;

                    break;

            case mx_sfile_blob_e :

                    file->next_blob(err, oid, name, length);
                    MX_ERROR_CHECK(err);

                    create_object(err, docid, oid);
                    MX_ERROR_CHECK(err);

                    mx_db_cc_create_blob(err, docid, oid, name);
                    MX_ERROR_CHECK(err);

                    mx_db_cc_set_blob_size(err, docid, oid, name, length);
                    MX_ERROR_CHECK(err);

                    break;
            case mx_sfile_blob_seg_e :

                    file->next_blob_segment(err, oid, segnum, name, data);
                    MX_ERROR_CHECK(err);

                    length = mx_db_cc_get_blob_size(err, docid, oid, name);
                    MX_ERROR_CHECK(err);

                    create_object(err, docid, oid);
                    MX_ERROR_CHECK(err);

                    num_segs = (length / BLOB_SEGMENT_SIZE) + 1;

                    if (segnum == (num_segs - 1))
                    {
                        length = length - (segnum * BLOB_SEGMENT_SIZE);
                    }
                    else
                    {
                        length = BLOB_SEGMENT_SIZE;
                    }

                    mx_db_cc_set_blob_data(
                                err, 
                                docid, 
                                oid, 
                                name, 
                                segnum * BLOB_SEGMENT_SIZE,
                                length,
                                (unsigned char *)data);
                    MX_ERROR_CHECK(err);

                    break;
            case mx_sfile_none_e :
                    return;
        }
    }
abort:;
}

void mx_db_cc_open_doc(
                int                   &err, 
                uint32                docid, 
                char                  *file_name, 
                bool                  recover,
                mx_file_create_type_t t = mx_file_simple_e)
{
    mx_file *doc_file;

    doc_file = new mx_file;

    files->add(err, docid, doc_file);
    MX_ERROR_CHECK(err);

    doc_file->open(err, file_name);

    if (err == MX_FILE_ENOENT)
    {
    MX_ERROR_CLEAR(err);
    }
    else
    {
    MX_ERROR_CHECK(err);

    // read the attributes etc
    read_file(err, docid, doc_file);
    MX_ERROR_CHECK(err);
    }

abort:;
}

static void mx_db_cc_commit_to_simple_file(int &err, mx_file *file, int docid)
{
    mx_hash       *blobs;
    obj_cache_t   *o;
    attr_cache_t  *a;
    blob_cache_t  *b;
    mx_hash::key_t name;
    mx_hash       *doc_objs;

    file->start_transaction(err);
    MX_ERROR_CHECK(err);

    doc_objs = (mx_hash *)docs->get(err, docid);
    MX_ERROR_CHECK(err);

    doc_objs->iterate_start(err);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        o = (obj_cache_t *)doc_objs->iterate_next(err);
        MX_ERROR_CHECK(err);
    
        if (o == NULL)
        {
            break;
        }

        // make sure the attributes are all up to date
        if (o->obj != NULL)
        {
            o->obj->cc_serialise_attributes(err);
            MX_ERROR_CHECK(err);
        }

        o->attrs->iterate_start(err);
        MX_ERROR_CHECK(err);

        while (TRUE)
        {
            a = (attr_cache_t *)o->attrs->iterate_next(err);
            MX_ERROR_CHECK(err);

            if (a == NULL)
            {    
                break;
            }

            file->write_attribute(err, o->id, a->attr);
            MX_ERROR_CHECK(err);
        }

        blobs = o->blobs;

        blobs->iterate_start(err);
        MX_ERROR_CHECK(err);

        while (TRUE)
        {
            b = (blob_cache_t *)blobs->iterate_next_key_data(err, &name);
            MX_ERROR_CHECK(err);

            if (b == NULL)
            {    
                break;
            }

            file->write_blob(err, o->id, name.s, b->size);
            MX_ERROR_CHECK(err);

            for (int i = 0; i < b->num_segments; i++)
            {
                if (b->segments[i].data != NULL)
                {
                    int length;

                    if (i == (b->num_segments - 1))
                    {
                        length = b->size - (i * BLOB_SEGMENT_SIZE);
                    }
                    else
                    {
                        length = BLOB_SEGMENT_SIZE;
                    }

                    file->write_blob_segment(
                                err, 
                                o->id, 
                                i, 
                                length,
                                name.s,
                                (char *)b->segments[i].data);
                    MX_ERROR_CHECK(err);
                }
            }
        }
    }

    file->commit(err);
    MX_ERROR_CHECK(err);

abort:;
}

static void mx_db_cc_commit_to_doc_file(int &err, mx_doc_file *file, int docid)
{
    mx_db_cc_commit_to_simple_file(err, (mx_file *)file, docid);
    MX_ERROR_CHECK(err);
abort:;
}

void mx_db_cc_commit(int &err, uint32 docid)
{
    mx_doc_file  *file;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    file = (mx_doc_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    mx_db_cc_commit_to_doc_file(err, file, docid);
    MX_ERROR_CHECK(err);

abort:;
}

// rollback consists of dropping all objects and attributes in the
// cache that have been added or modified since the start of the 
// transaction. As the cache holds the entire transaction, nothing
// needs to be done to the actual database file
void mx_db_cc_rollback(int &err, uint32 docid)
{
abort:;
}

static void delete_cache(int &err, int docid)
{
    mx_hash      *cache;
    obj_cache_t  *o;
    attr_cache_t *a;
    blob_cache_t *b;

    int i;

    cache = (mx_hash *)docs->get(err, docid);
    if (err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
        return;
    }
    MX_ERROR_CHECK(err);

    cache->iterate_start(err);

    while (TRUE)
    {
        o = (obj_cache_t *)cache->iterate_next(err);
        MX_ERROR_CHECK(err);

        if (o == NULL)
        {
            break;
        }

        o->attrs->iterate_start(err);
        MX_ERROR_CHECK(err);

        while (TRUE)
        {
            a = (attr_cache_t *)o->attrs->iterate_next(err);
            MX_ERROR_CHECK(err);

            if (a == NULL)
            {
                break;
            }

            a->attr->free_value();
            delete a->attr;
            delete a;
        }

        o->blobs->iterate_start(err);
        MX_ERROR_CHECK(err);

        while (TRUE)
        {
            b = (blob_cache_t *)o->blobs->iterate_next(err);
            MX_ERROR_CHECK(err);

            if (b == NULL)
            {
                break;
            }

            for (i = 0; i < b->num_segments; i++)
            {
                if (b->segments[i].data != NULL)
                {
                    delete [] b->segments[i].data;
                }
            }
            if (b->segments != NULL)
            {
                delete [] b->segments;
                delete b;
            }
        }

        delete o->attrs;
        delete o->blobs;
        delete o->obj;
        delete o;
    }

    delete cache;

    docs->del(err, docid);
    MX_ERROR_CHECK(err);

abort:;
}

void mx_db_cc_close_doc(int &err, uint32 docid)
{
    mx_file *file;
    mx_document *doc;
    bool need_to_delete;
    char filename[MAX_PATH_LEN + MAX_FILE_NAME_LEN + 1];

    file = (mx_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    doc = (mx_document *)mx_db_cc_get_obj_from_oid(err, docid, 
                                                   MX_DB_OBJECT_DOCUMENT_ID);
    MX_ERROR_CHECK(err);

    MX_ERROR_ASSERT(err, (doc->is_a(mx_document_class_e) || 
                          doc->is_a(mx_wp_document_class_e)));

    need_to_delete = doc->get_temp_flag();
    strcpy(filename, doc->get_file_name());

    file->close(err);
    MX_ERROR_CHECK(err);

    delete file;

    files->del(err, docid);
    MX_ERROR_CHECK(err);

    delete_cache(err, docid);
    MX_ERROR_CHECK(err);

    if (need_to_delete)
    {
        if (-1 == unlink(filename))
        {
            MX_ERROR_THROW(err, mx_translate_file_error(errno));
        }
    }

abort:;
}

void mx_db_cc_add_known_oid(int &err, uint32 docid, mx_db_object *o, uint32 oid)
{
    obj_cache_t *item;
    mx_hash     *obj_hash;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    obj_hash = (mx_hash *)docs->get(err, docid);

    if (err == MX_HASH_NOT_FOUND) 
    {
        MX_ERROR_CLEAR(err);

        obj_hash = new mx_hash(OBJ_HASH_START_SIZE);
        docs->add(err, docid, obj_hash);
        MX_ERROR_CHECK(err);
    }
    else 
    {
        MX_ERROR_CHECK(err);
    }

    // check whether the item is already there - may be if this is
    // a simple file
    item = (obj_cache_t *)obj_hash->get(err, oid);
    if (err == MX_ERROR_OK)
    {
        item->obj = o;
    }
    else
    {
        MX_ERROR_CLEAR(err);

        item = new obj_cache_t;

        item->obj = o;
        item->attrs = new mx_hash(ATTR_HASH_START_SIZE);
        item->blobs = new mx_hash(BLOB_HASH_START_SIZE);

        item->new_obj = FALSE;
        item->id = oid;

        obj_hash->add(err, item->id, item);
        MX_ERROR_CHECK(err);
    }

    return;
abort:;
    return;
}

const mx_owner_t *mx_db_client_get_owner(int &err, uint32 docid)
{
    mx_file          *f;
    const mx_owner_t *res;

    f = (mx_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    res = f->get_owner(err);
    MX_ERROR_CHECK(err);

    return res;
abort:;
    return NULL;
}

const mx_access_permissions_t *mx_db_client_get_perm(int &err, uint32 docid)
{
    mx_file                       *f;
    const mx_access_permissions_t *res;

    f = (mx_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    res = f->get_perm(err);
    MX_ERROR_CHECK(err);

    return res;
abort:;
    return NULL;
}

const char *mx_db_cc_get_description(uint32 docid)
{
    mx_file *file;
    int     err = MX_ERROR_OK;

    file = (mx_file *)files->get(err, docid);
    if (err != MX_ERROR_OK)
    {
        MX_ERROR_CLEAR(err);
        return "<none set>";
    }
    else
    {
        return file->get_description();
    }
}

const char *mx_db_cc_get_author(uint32 docid)
{
    mx_file *file;
    int     err = MX_ERROR_OK;

    file = (mx_file *)files->get(err, docid);
    if (err != MX_ERROR_OK)
    {
        MX_ERROR_CLEAR(err);
        return "<none set>";
    }
    else
    {
        return file->get_author();
    }
}

const char *mx_db_cc_get_version(uint32 docid)
{
    mx_file *file;
    int     err = MX_ERROR_OK;

    file = (mx_file *)files->get(err, docid);
    if (err != MX_ERROR_OK)
    {
        MX_ERROR_CLEAR(err);
        return "<none set>";
    }
    else
    {
        return file->get_version();
    }
}

void mx_db_cc_set_info(int &err, uint32 docid, char *description, char *author, char *version)
{
    mx_file *file;

    file = (mx_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    file->set_info(err, description, author, version);
    MX_ERROR_CHECK(err);

abort:;
}

bool mx_db_cc_doc_modified(int &err, uint32 docid)
{
    mx_hash     *obj_hash;
    obj_cache_t *o;

    obj_hash=(mx_hash *)docs->get( err,docid );
    MX_ERROR_CHECK( err );

    obj_hash->iterate_start(err);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        o = (obj_cache_t *)obj_hash->iterate_next(err);
        MX_ERROR_CHECK(err);
    
        if (o == NULL)
        {
            return FALSE;
        }
        else
        {
            if ((o->obj != NULL) && (o->obj->get_mem_state() == mx_in_mem_and_modified_e))
            {
                return TRUE;
            }
        }
    }

abort:
    return FALSE;

}

void mx_db_cc_commit_as(int &err, uint32 docid, char *file_name)
{
    mx_doc_file   *file;

    // is the database active?
    if (!db_updateable)
    {
        return;
    }

    file = (mx_doc_file *)files->get(err, docid);
    MX_ERROR_CHECK(err);

    file->duplicate(err, file_name);
    MX_ERROR_CHECK(err);

    mx_db_cc_commit_to_doc_file(err, file, docid);
    MX_ERROR_CHECK(err);

abort:;
}

static void mx_db_cc_copy_attrs(int &err, mx_hash *attrs, int dest_doc, int dest_oid)
{
    mx_hash_iterator iter(*attrs);
    attr_cache_t     *a;

    while (iter.more())
    {
        a = (attr_cache_t *)iter.data();

        mx_db_cc_set_attribute(err, dest_doc, dest_oid, a->attr);
        MX_ERROR_CHECK(err);
    }
abort:;
}

static void mx_db_cc_copy_blobs(
                int     &err, 
                mx_hash *blobs, 
                int     source_doc, 
                int     source_oid, 
                int     dest_doc, 
                int     dest_oid)
{
    mx_hash_iterator iter(*blobs);
    blob_cache_t     *b;
    char *name;
    uint8 buffer[BLOB_SEGMENT_SIZE];
    uint32 i;

    while (iter.more())
    {
        b = (blob_cache_t *)iter.data();
        name = iter.string_key();

        mx_db_cc_create_blob(err, dest_doc, dest_oid, name);
        MX_ERROR_CHECK(err);

        mx_db_cc_set_blob_size(err, dest_doc, dest_oid, name, b->size);
        MX_ERROR_CHECK(err);

        for (i = 0; i < b->size; i += BLOB_SEGMENT_SIZE)
        {
            mx_db_cc_get_blob_data(    
                        err, 
                        source_doc, 
                        source_oid, 
                        name, 
                        i,
                        (b->size - i) > BLOB_SEGMENT_SIZE ? 
                            BLOB_SEGMENT_SIZE :
                            b->size - i,
                        buffer);
            MX_ERROR_CHECK(err);

            mx_db_cc_set_blob_data(
                        err, 
                        dest_doc, 
                        dest_oid, 
                        name, 
                        i,
                        (b->size - i) > BLOB_SEGMENT_SIZE ? 
                            BLOB_SEGMENT_SIZE :
                            b->size - i,
                        buffer);
            MX_ERROR_CHECK(err);
        }
    }
abort:;
}

void mx_db_cc_move_object(int &err, mx_db_object *o, int source_doc, int dest_doc, int &oid)
{
    mx_hash *src_obj_hash;
    int new_oid;

    obj_cache_t *item;

    src_obj_hash = (mx_hash *)docs->get(err, source_doc);
    MX_ERROR_CHECK(err);

    item = (obj_cache_t *)src_obj_hash->get(err, oid);
    MX_ERROR_CHECK(err);

    new_oid = mx_db_cc_add_oid(err, dest_doc, o);
    MX_ERROR_CHECK(err);

    mx_db_cc_copy_attrs(err, item->attrs, dest_doc, new_oid);
    MX_ERROR_CHECK(err);

    mx_db_cc_copy_blobs(err, item->blobs, source_doc, oid, dest_doc, new_oid);
    MX_ERROR_CHECK(err);

    mx_db_cc_delete_oid_from_cache(err, source_doc, oid);
    MX_ERROR_CHECK(err);

    oid = new_oid;
abort:;
}

bool mx_db_cc_get_ignore_update()
{
    return db_updateable;
}

