/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef MX_LIST2_H
#define MX_LIST2_H
/*
 * MODULE/CLASS : mx_list2
 *
 * AUTHOR : David Miller
 *
 * 
 *
 *
 *
 *
 */

#include <mx_std.h>
#include <mx_error.h>

/*-------------------------------------------------
 * CLASS: mx_list2
 *
 * DESCRIPTION: 
 * 
 *  This list class is designed to be reasonably space efficient and also
 *  reasonably fast. You can basically insert and delete items at indexs
 *  and there are also methods to iterate throught the list.
 *
 *  It is optimized such that repeated operations at a given index are 
 *  really quick and so that if you get iterate up through the list it is
 *  quick too.
 *
 */
class mx_list2 {
public:

    mx_list2();
    mx_list2(int &err,const mx_list2 &list);

    ~mx_list2();

    mx_list2& equate(int &err,const mx_list2 &rvalue);

    inline int32 get_num_items() const {return num_items;};

    void * get(int &err, int32 index);

    void insert(int &err, int32 index, void * item);
    void append(int &err, void * item);

    // these return a pointer to the object that has just been overwritten
    // or removed.
    void *overwrite(int &err, int32 index, void * item);
    void *remove(int &err, int32 index);

    void   *find_smallest(int &err, int (*comp_func)(void * one, void * two));
    void   *find_largest(int &err,  int (*comp_func)(void * one, void * two));
    inline int32 find_smallest_index(int &err, int (*comp_func)(void * one, 
                                void * two));
    inline int32 find_largest_index(int &err,  int (*comp_func)(void * one,
                                void * two));

    // returns an index from a pointer if the pointer is in the list.
    int32 find_index(int &err, void * item);

    void iterate(int &err, void (*iter_func)( int &err, uint32 index, 
                          void * item ));
    
    // These functions can be used to iterate through inserted objects
    // you call iterate_start and then repeatedly call iterate_next.
    // iterate_next returns NULL when you have completed the iteration.
    void iterate_start();
    void iterate_end();
    void *iterate_next();
    void *iterate_prev();

private:

    struct elem_t2 
    {
        void * item;
        elem_t2 * next;
        elem_t2 * prev;
    };

    int32    num_items;
    int32    current_index;

    elem_t2 *current_list_elem;
    elem_t2 *first_list_elem;
    elem_t2 *last_list_elem;

    void goto_index(int &err, int32 index);
    void range_check(int &err, int32 &index) const;
    void delete_list(elem_t2 * list);
    elem_t2 *copy_list(int &err, elem_t2 * list, mx_list2 &other);
};

/*-------------------------------------------------
 * FUNCTION: find_smallest_index
 *
 * DESCRIPTION: 
 * 
 * You give this function a comparision function for the objects
 * and it will return the index of the smallest one (according to 
 * the comparison function. find_largest_index does the opposite.
 *
 */
inline int32 mx_list2::
find_smallest_index(int &err, int (*comp_func)(void * one, void * two))
{
    find_smallest(err, comp_func);
    return current_index;
}

inline int32 mx_list2::
find_largest_index(int &err,  int (*comp_func)(void * one, void * two))
{
    find_largest(err, comp_func);
    return current_index;
}

#endif // MX_LIST2_H
