/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE : iterator.c
 *
 * AUTHOR : David Miller
 *
 * The author should insert the name of whoever should own the
 * copyright
 *
 * 
 *
 * DESCRIPTION: Iterator class to use with mx_attribute_value collections
 * Module iterator.c 
 * 
 *
 * 
 */

#include "mx_iterator.h"
#include "mx_attribute.h"


/*-------------------------------------------------
 * FUNCTION: iterator::iterator
 *
 * DESCRIPTION: iterator constructor - need to tell 
 * the collectin to which the iterator is attached that
 * it is there 
 */

iterator::iterator(int &err,collection *collectt,bool inc) 
{
  err = MX_ERROR_OK ;
  
  collect    = collectt ;
  increment  = inc ;
  n          = NULL ;

  if(collect != NULL) 
  {
    collectt->addIterator(err,this) ; 
    MX_ERROR_CHECK(err);
  }    
  return;
 abort:
  return ;
}


/*-------------------------------------------------
 * FUNCTION: iterator::~iterator
 *
 * DESCRIPTION: iterator destructor - need to tell 
 * the collectoin to which the iterator is attached that
 * it is going
 */

iterator::~iterator() 
{
   if(collect != NULL) collect->removeIterator(this) ; 
}


/*-------------------------------------------------
 * FUNCTION: iterator::reset
 *
 * DESCRIPTION: reset an iterator either on a different
 * or a new collection
 */

void iterator::reset(int &err,
             collection *collectt,
             bool incr) 
{
   err = MX_ERROR_OK ;

   increment  = incr ;
   n          = NULL ;

   /* remove the old iterator if the new is different */ 
   
   if(collect != collectt) 
   {
      if(collect != NULL) collect->removeIterator(this) ; 

      if(collectt != NULL) 
      {
     collectt->addIterator(err,this) ;
     MX_ERROR_CHECK(err) ;
      }
      collect = collectt ;
   }
   goto exit ;
  abort:
  exit:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: iterator::start
 *
 * DESCRIPTION: 
 * 
 *
 */

void iterator::start(int                &err,
             mx_it_start_type   startType,
             mx_attribute_value *startValue) 
{
   err = MX_ERROR_OK ;
  
   if(collect == NULL) 
   {
      err = MX_ITERATOR_NO_COLLECTION ;
      goto abort ;
   }

   switch(startType) 
   {
   case mx_it_head:
      n = collect->getHead() ;
      break ;
   case mx_it_tail:
      n = collect->getTail() ;
      break ;
   case mx_it_pos:

      if(startValue == NULL) 
      {
      err = MX_ITERATOR_NULL_START_VALUE ;
      goto abort ;
      }

      n = collect->getPos(startValue->i) ;
      
      break ;
   case mx_it_value:

      if(startValue == NULL) 
      {
      err = MX_ITERATOR_NULL_START_VALUE ;
      goto abort ;
      }

      n = collect->getValue(err,*startValue) ;
      MX_ERROR_CHECK(err) ;

      break ;
   default:
      break ;
   }

   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: iterator::next
 *
 * DESCRIPTION: Get the values of the currnet element from a
 * collection and move onto the next
 *
 */

bool iterator::next(int &err,
            mx_attribute_value *key,
            mx_attribute_value *data) 
{
   err = MX_ERROR_OK ;

   // Invalid collection 

   if(collect == NULL) 
   {
      err = MX_ITERATOR_NO_COLLECTION ;
      goto abort;
   }

   // If we have reached the end return FALSE ;

   if(n == NULL) 
   {
      return FALSE ;
   }
   else
   {
      // Not yet at the end do the next iteration

      if(key != NULL)  *key  = n->key ;
      if(data != NULL) *data = n->data ;

      if(increment) 
      {
     n = collect->getNext(n) ;
      }
      else
      {
     n = collect->getPrev(n) ;
      }

      
      return TRUE ;
   }

  abort:
   return FALSE ;
}

/*-------------------------------------------------
 * FUNCTION: iterator::go
 *
 * DESCRIPTION: Apply go function 
 * 
 *
 */

void iterator::go(int &err)
{
   err = MX_ERROR_OK ;

   mx_attribute_value key ;
   mx_attribute_value data ;

   while(next(err,&key,&data)) 
   {
      go_fn(err,&key,&data) ;
      MX_ERROR_CHECK(err) ;
   }
   MX_ERROR_CHECK(err) ;

   goto exit ;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: go_fn
 *
 * DESCRIPTION: The function attached to the go
 * 
 *
 */

void iterator::go_fn(int                &err,
             mx_attribute_value *key, 
             mx_attribute_value *data) 
{
   err = MX_ERROR_OK ;

   if(key  != NULL) 
   {
      mx_attribute_print("Key",collect->keytype(),key) ;
   }

   if(data != NULL) 
   {
      mx_attribute_print("Data",collect->datatype(),data) ;
   }

   return ;
}
