/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef MX_ARRAY_H
#define MX_ARRAY_H
/*
 * MODULE/CLASS : mx_array
 *
 * AUTHOR : Tom Newton
 *
 * 
 *
 * DESCRIPTION:
 *
 * A growable array class.
 *
 *
 *
 */

#include <mx.h>
#include <assert.h>

#define MX_ARRAY_MAX_NUM_ELEMS ((uint32)(INT_MAX / 2))

template <class T>
class mx_array
{
public:

    // constructors
    inline mx_array(uint32 init_num_elems = 0);
    inline mx_array(const mx_array &a);
    inline ~mx_array();

    // operators
    inline mx_array &operator=(const mx_array &a);
    inline T &operator[](uint32 index);

    inline void grow(uint32 num_extra_elems);
    inline void shrink();

    // no error checking on these so make sure they've got the right parameters
    inline void make_space(uint32 index, uint32 num_elems);
    inline void overwrite(uint32 index, uint32 num_elems, const T *elems);
    inline void insert(uint32 index, uint32 num_elems, const T *elems);
    inline void insert(uint32 index, const T &elem);
    inline void remove(uint32 index, uint32 num_elems);
    inline void move(uint32 index, uint32 num_e, mx_array &dst, uint32 dst_index);

    /********************************************************************/
    // public variables
    /********************************************************************/

    T *elems;
    uint32 num_elems;
    uint32 max_num_elems;
};

template <class T> inline
mx_array<T>::mx_array(uint32 init_num_elems)
{
    if (init_num_elems > MX_ARRAY_MAX_NUM_ELEMS)
    {
        init_num_elems = MX_ARRAY_MAX_NUM_ELEMS;
    }

    num_elems = 0;
    max_num_elems = init_num_elems;

    if (max_num_elems > 0)
    {
        elems = new T[max_num_elems];
    }
    else
    {
        elems = NULL;
    }
}

template <class T> inline
mx_array<T>::mx_array(const mx_array &a)
{
    num_elems = max_num_elems = a.num_elems;
    if (max_num_elems > 0)
    {
        elems = new T[max_num_elems];
    }
    memcpy(elems, a.elems, num_elems * sizeof(T));
}

template <class T> inline
mx_array<T>::~mx_array()
{
    if (max_num_elems > 0)
    {
        delete [] elems;
    }
}

template <class T> inline
mx_array<T> &mx_array<T>::operator=(const mx_array &a)
{
    if (max_num_elems < a.num_elems)
    {
        if (max_num_elems > 0) delete [] elems;
        max_num_elems = a.num_elems;
        elems = new T[max_num_elems];
    }
    num_elems = a.num_elems;
    memcpy(elems, a.elems, num_elems * sizeof(T));
    return *this;
}

template <class T> inline
T &mx_array<T>::operator[](uint32 index)
{
    return elems[index];
}

template <class T> inline
void mx_array<T>::grow(uint32 num_extra_elems)
{
    uint32 num_elems_required = num_elems + num_extra_elems;

    assert(num_extra_elems < MX_ARRAY_MAX_NUM_ELEMS &&
           num_elems_required < MX_ARRAY_MAX_NUM_ELEMS);

    if (num_elems_required > max_num_elems)
    {
        T *old_elems = elems;

        // increase by a factor of a half
        num_elems_required += num_elems_required >> 1;

        if (num_elems_required > MX_ARRAY_MAX_NUM_ELEMS)
        {
            num_elems_required = MX_ARRAY_MAX_NUM_ELEMS;
        }

        max_num_elems = num_elems_required;

        elems = new T[max_num_elems];
        if (old_elems != NULL)
        {
            memcpy(elems, old_elems, num_elems * sizeof(T));
            delete [] old_elems;
        }
    }
}

template <class T> inline
void mx_array<T>::shrink()
{
    if (num_elems < max_num_elems)
    {
        // if the number of characters in the array is less than the maximum

        T *old_elems = elems;
        max_num_elems = num_elems;

        if (max_num_elems > 0)
        {
            elems = new T[max_num_elems];
            memcpy(elems, old_elems, num_elems * sizeof(T));
        }
        else
        {
            elems = NULL;
        }

        delete [] old_elems;
    }
}

template <class T> inline
void mx_array<T>::make_space(uint32 index, uint32 ne)
{
    if (ne > 0)
    {
        grow(ne);
        memmove(elems + index + ne, elems + index, 
                (num_elems - index) * sizeof(T));
    }
    num_elems += ne;
}

template <class T> inline
void mx_array<T>::overwrite(uint32 index, uint32 ne, const T *some_elems)
{
    memcpy(elems + index, some_elems, ne * sizeof(T));
}

template <class T> inline
void mx_array<T>::insert(uint32 index, uint32 ne, const T *some_elems)
{
    make_space(index, ne);
    overwrite(index, ne, some_elems);
}

template <class T> inline
void mx_array<T>::insert(uint32 index, const T &elem)
{
    make_space(index, 1);
    overwrite(index, 1, &elem);
}

template <class T> inline
void mx_array<T>::remove(uint32 index, uint32 ne)
{
    num_elems -= ne;
    memmove(elems + index + ne, elems + index, num_elems - index);
}

template <class T> inline
void mx_array<T>::move(uint32 index, uint32 ne, mx_array &dst, uint32 dst_i)
{
    dst.make_space(dst_i, ne);
    dst.overwrite(dst_i, ne, elems + index);
    remove(index, ne);
}

#endif
