/*
 *		IP_MASQ_FTP ftp masquerading module
 *
 *
 * Version:	@(#)ip_masq_ftp.c 0.01   02/05/96
 *
 * Author:	Wouter Gadeyne
 *		
 *
 * Fixes:
 *	Fred Viles		:	Added support for masq'ed servers.  In
 *					this case it's the server's response to
 *					the external client's PASV that needs
 *					to be re-written in masq_ftp_out.
 *	Wouter Gadeyne		:	Fixed masquerading support of ftp PORT commands
 * 	Juan Jose Ciarlante	:	Code moved and adapted from ip_fw.c
 * 	Keith Owens		:	Add keep alive for ftp control channel
 *	Nigel Metheringham	:	Added multiple port support
 *	
 *
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 * Multiple Port Support
 *	The helper can be made to handle up to MAX_MASQ_APP_PORTS (normally 12)
 *	with the port numbers being defined at module load time.  The module
 *	uses the symbol "ports" to define a list of monitored ports, which can
 *	be specified on the insmod command line as
 *		ports=x1,x2,x3...
 *	where x[n] are integer port numbers.  This option can be put into
 *	/etc/conf.modules (or /etc/modules.conf depending on your config)
 *	where modload will pick it up should you use modload to load your
 *	modules.
 *	
 */

#include <linux/module.h>
#include <asm/system.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <net/protocol.h>
#include <net/tcp.h>
#include <net/ip_masq.h>

#ifndef DEBUG_CONFIG_IP_MASQ_FTP
#define DEBUG_CONFIG_IP_MASQ_FTP 0
#endif

/* 
 * List of ports (up to MAX_MASQ_APP_PORTS) to be handled by helper
 * First port is set to the default port.
 */
int ports[MAX_MASQ_APP_PORTS] = {21}; /* I rely on the trailing items being set to zero */
struct ip_masq_app *masq_incarnations[MAX_MASQ_APP_PORTS];

#define IP_MASQ_APP_PORT(type)        ( type & 0xffff )

static int
masq_ftp_init_1 (struct ip_masq_app *mapp, struct ip_masq *ms)
{
        MOD_INC_USE_COUNT;
        return 0;
}

static int
masq_ftp_done_1 (struct ip_masq_app *mapp, struct ip_masq *ms)
{
        MOD_DEC_USE_COUNT;
        return 0;
}

/*
 * This function parses the IP address and Port number found in PORT commands
 * and PASV responses.  This used to be done in-line, but with four cases it
 * seemed worth encapsulating.  It returns the IP address, or zero if an
 * error is detected.
 */
static __u32 parse_ip_port( char **datap, __u16 *portp )
{
	char	*data = *datap;
	unsigned char p1,p2,p3,p4,p5,p6;

	p1 = simple_strtoul(data, &data, 10);
	if (*data != ',')
		return 0;
	p2 = simple_strtoul(data+1, &data, 10);
	if (*data != ',')
		return 0;
	p3 = simple_strtoul(data+1, &data, 10);
	if (*data != ',')
		return 0;
	p4 = simple_strtoul(data+1, &data, 10);
	if (*data != ',')
		return 0;
	p5 = simple_strtoul(data+1, &data, 10);
	if (*data != ',')
		return 0;
	p6 = simple_strtoul(data+1, &data, 10);

	*datap = data;
	*portp = (p5<<8) | p6;
	return (p1<<24) | (p2<<16) | (p3<<8) | p4;
}

/*
 * This function checks outgoing client-to-server traffic for PASV and PORT
 * commands and server-to-client traffic for PASV replies.  If PASV is seen,
 * it sets a flag in the ip_masq struct to let masq_ftp_in() know that an
 * incoming PASV reply is expected.  If PORT or a PAV reply is seen, it
 * sets up a masquerade entry to handle the data connection and modifies
 * the IP address and port number in the packet data to point to it.
 * An outgoing packet is client-to-server if its destination port matches
 * the mapped port, and it's server-to-client if the source port matches.
 */
int
masq_ftp_out (struct ip_masq_app *mapp, struct ip_masq *ms, struct sk_buff **skb_p, struct device *dev)
{
        struct sk_buff *skb;
	struct iphdr *iph;
	struct tcphdr *th;
	char *p, *data, *data_limit;
	__u32 from;
	__u16 port;
	struct ip_masq *n_ms;
	char buf[28];		/* xxx,xxx,xxx,xxx,ppp,ppp)\000 */
	char rparen;		/* ')' if PASV reply, '\0' if PORT cmd */
        unsigned buf_len;
	int diff;

        skb = *skb_p;
	iph = skb->h.iph;
        th = (struct tcphdr *)&(((char *)iph)[iph->ihl*4]);
        data = (char *)&th[1];
        data_limit = skb->h.raw + skb->len;

	if (htons( ms->dport ) == IP_MASQ_APP_PORT( mapp->type )) {
		/* client-to-server packet.  Check for PASV and PORT */
		if (skb->len >= 6 && (memcmp(data, "PASV\r\n", 6) == 0
		    || memcmp(data, "pasv\r\n", 6) == 0)) {
			ms->flags |= IP_MASQ_F_FTP_PASV;
			return 0;
		}

		while (data < data_limit && *data == ' ')
			++data;	
		if (memcmp(data,"PORT ",5) && memcmp(data,"port ",5))
			return 0;

		data += 5;
		p = data;
		from = parse_ip_port( &data, &port );
		rparen = '\0';
	}
	else {
		/* server-to-client packet, check for PASV reply */
		if ((ms->flags & IP_MASQ_F_FTP_PASV) == 0)
			return 0;	/* quick exit if no PASV seen */

		/* We've just seen a PASV, so this should be the response */
		do {
			if (data >= data_limit)
				return 0;
		} while (*data++ != '(');

		p = data;
		from = parse_ip_port( &data, &port );
		if (*data++ != ')')
			return 0;
		rparen = ')';
		ms->flags &= ~IP_MASQ_F_FTP_PASV;
	}

	if (from == 0 || (*data != '\r' && *data != '\n'))
		return 0;
#if DEBUG_CONFIG_IP_MASQ_FTP
	printk( "%s %X:%X detected\n", (rparen) ? "PASV response" : "PORT",
			from, port );
#endif
	/*
	 * Now update or create an masquerade entry for it
	 */
#if DEBUG_CONFIG_IP_MASQ_FTP
	printk("protocol %d %X:%X %X:%X\n", iph->protocol, from,
	    port, iph->daddr, 0);
#endif	
	n_ms = ip_masq_out_get_2(iph->protocol,
			htonl(from), htons(port),
			iph->daddr, 0);
	if (n_ms) {
		/* existing masquerade, clear timer */
		ip_masq_set_expire(n_ms,0);
	}
	else {
		n_ms = ip_masq_new(dev, IPPROTO_TCP,
				htonl(from), htons(port),
				iph->daddr, 0,
				IP_MASQ_F_NO_DPORT);

		if (n_ms==NULL)
			return 0;
		n_ms->control = ms;		/* keepalive from data to the control channel */
		ms->flags |= IP_MASQ_F_CONTROL;	/* this is a control channel */
	}

	/*
	 * keep for a bit longer than tcp_fin, caller may not reissue
	 * PORT before tcp_fin_timeout.
	 */
	ip_masq_set_expire(n_ms, ip_masq_expire->tcp_fin_timeout*3);

	/*
	 * Replace the old PORT cmd/PASV response with the new one
	 */
	from = ntohl(n_ms->maddr);
	port = ntohs(n_ms->mport);
	sprintf( buf, "%d,%d,%d,%d,%d,%d%c",
	    from>>24&255,from>>16&255,from>>8&255,from&255,
	    port>>8&255,port&255, rparen );
	buf_len = strlen(buf);
#if DEBUG_CONFIG_IP_MASQ_FTP
	printk( "new PORT/PASV %X:%X\n", from, port );
#endif

	/*
	 * Calculate required delta-offset to keep TCP happy
	 */

	diff = buf_len - (data-p);

	/*
	 *	No shift.
	 */

	if (diff==0) {
		/*
		 * simple case, just replace the old PORT cmd
		 */
		memcpy(p,buf,buf_len);
		return 0;
	}

	*skb_p = ip_masq_skb_replace(skb, GFP_ATOMIC, p, data-p, buf, buf_len);
	return diff;
}

/*
 * Look at incoming ftp packets to catch the response to PASV commands
 * (server-to-client) and PORT commands (client-to-server).  When we see
 * one we build a masquerading entry for the masqueraded address, port 0
 * (unknown at the moment), the remote address and the remote port.  Mark the
 * current masquerade entry as a control channel and point the new entry at the
 * control entry.  All this work just for ftp keepalive across masquerading.
 *
 * PORT commands look like "PORT xxx,xxx,xxx,xxx,ppp,ppp".
 * The PASV reply packet should be something like
 * "227 Entering Passive Mode (xxx,xxx,xxx,xxx,ppp,ppp)".
 * xxx,xxx,xxx,xxx is the server address, ppp,ppp is the server port number.
 *
 * In the PORT case, the server is the destination machine being masqueraded,
 * the server is the source of ftp requests.  In the PASV reply case, the
 * client is the source machine being masqueraded, the server is the
 * destination for ftp requests.  It all depends on your point of view ...
 */

int
masq_ftp_in (struct ip_masq_app *mapp, struct ip_masq *ms, struct sk_buff **skb_p, struct device *dev)
{
	struct sk_buff *skb;
	struct iphdr *iph;
	struct tcphdr *th;
	char *data, *data_limit;
	__u32 to;
	__u16 port;
	struct ip_masq *n_ms;

	skb = *skb_p;
	iph = skb->h.iph;
	th = (struct tcphdr *)&(((char *)iph)[iph->ihl*4]);
	data = (char *)&th[1];
	data_limit = skb->h.raw + skb->len;

	if (htons( ms->dport ) == IP_MASQ_APP_PORT( mapp->type )) {
		/* server-to-client packet.  Check for PASV response */
		if ((ms->flags & IP_MASQ_F_FTP_PASV) == 0)
			return 0;	/* quick exit if no PASV seen */

		/* We've just seen a PASV, so this should be the response */
		do {
			if (data >= data_limit)
				return 0;
		} while (*data++ != '(');

		to = parse_ip_port( &data, &port );
		if (to == 0 || *data != ')')
			return 0;
	}
	else {
		/* client-to-server packet.  Check for PASV and PORT */
		if (skb->len >= 6 && (memcmp(data, "PASV\r\n", 6) == 0
		    || memcmp(data, "pasv\r\n", 6) == 0)) {
			ms->flags |= IP_MASQ_F_FTP_PASV;
			return 0;
		}

		while (data < data_limit && *data == ' ')
			++data;	

		if (memcmp(data,"PORT ",5) && memcmp(data,"port ",5))
			return 0;

		data += 5;
		to = parse_ip_port( &data, &port );
		if (to == 0 || (*data != '\r' && *data != '\n'))
			return 0;
	}

	/*
	 * Got an incoming PASV reponse or PORT command.
	 * Now update or create an masquerade entry for it
	 */
#if DEBUG_CONFIG_IP_MASQ_FTP
	printk( "%s %lX:%X %X:%X detected\n",
	    (*data == ')') ? "PASV response" : "PORT",
	    ntohl(ms->saddr), 0, to, port );
#endif	
	n_ms = ip_masq_out_get_2(iph->protocol,
				 ms->saddr, 0,
				 htonl(to), htons(port));
	if (n_ms) {
		/* existing masquerade, clear timer */
		ip_masq_set_expire(n_ms,0);
	}
	else {
		n_ms = ip_masq_new(dev, IPPROTO_TCP,
				   ms->saddr, 0,
				   htonl(to), htons(port),
				   IP_MASQ_F_NO_SPORT);

		if (n_ms==NULL)
			return 0;
		n_ms->control = ms;		/* keepalive from data to the control channel */
		ms->flags |= IP_MASQ_F_CONTROL;	/* this is a control channel */
	}

	/*
	 * keep for a bit longer than tcp_fin, client may not issue open
	 * to server port before tcp_fin_timeout.
	 */
	ip_masq_set_expire(n_ms, ip_masq_expire->tcp_fin_timeout*3);
	ms->flags &= ~IP_MASQ_F_FTP_PASV;
	return 0;	/* no diff required for incoming packets, thank goodness */
}

struct ip_masq_app ip_masq_ftp = {
        NULL,			/* next */
	"ftp",			/* name */
        0,                      /* type */
        0,                      /* n_attach */
        masq_ftp_init_1,        /* ip_masq_init_1 */
        masq_ftp_done_1,        /* ip_masq_done_1 */
        masq_ftp_out,           /* pkt_out */
        masq_ftp_in,            /* pkt_in */
};

/*
 * 	ip_masq_ftp initialization
 */

int ip_masq_ftp_init(void)
{
	int i, j;

	for (i=0; (i<MAX_MASQ_APP_PORTS); i++) {
		if (ports[i]) {
			if ((masq_incarnations[i] = kmalloc(sizeof(struct ip_masq_app),
							    GFP_KERNEL)) == NULL)
				return -ENOMEM;
			memcpy(masq_incarnations[i], &ip_masq_ftp, sizeof(struct ip_masq_app));
			if ((j = register_ip_masq_app(masq_incarnations[i], 
						      IPPROTO_TCP, 
						      ports[i]))) {
				return j;
			}
#if DEBUG_CONFIG_IP_MASQ_FTP
			printk("Ftp: loaded support on port[%d] = %d\n",
			       i, ports[i]);
#endif
		} else {
			/* To be safe, force the incarnation table entry to NULL */
			masq_incarnations[i] = NULL;
		}
	}
	return 0;
}

/*
 * 	ip_masq_ftp fin.
 */

int ip_masq_ftp_done(void)
{
	int i, j, k;

	k=0;
	for (i=0; (i<MAX_MASQ_APP_PORTS); i++) {
		if (masq_incarnations[i]) {
			if ((j = unregister_ip_masq_app(masq_incarnations[i]))) {
				k = j;
			} else {
				kfree(masq_incarnations[i]);
				masq_incarnations[i] = NULL;
#if DEBUG_CONFIG_IP_MASQ_FTP
				printk("Ftp: unloaded support on port[%d] = %d\n",
				       i, ports[i]);
#endif
			}
		}
	}
	return k;
}

#ifdef MODULE

int init_module(void)
{
        if (ip_masq_ftp_init() != 0)
                return -EIO;
        register_symtab(0);
        return 0;
}

void cleanup_module(void)
{
        if (ip_masq_ftp_done() != 0)
                printk("ip_masq_ftp: can't remove module");
}

#endif /* MODULE */
