/*% cc -g sphere.c -o sphere -lm
 *
 * sphere - generate a triangle mesh approximating a sphere by
 *  recursive subdivision. First approximation is an octahedron;
 *  each level of refinement increases the number of triangles by
 *  a factor of 4.
 * Level 3 (128 triangles) is a good tradeoff if gouraud
 *  shading is used to render the database.
 *
 * Usage: sphere [level] [-p] [-c]
 *      level is an integer >= 1 setting the recursion level (default 1).
 *      -p causes generation of a PPHIGS format ASCII archive
 *          instead of the default generic output format.
 *      -c causes triangles to be generated with vertices in counterclockwise
 *          order as viewed from the outside in a RHS coordinate system.
 *          The default is clockwise order.
 *
 *  The subroutines print_object() and print_triangle() should
 *  be changed to generate whatever the desired database format is.
 *
 * Jon Leech (leech@cs.unc.edu) 3/24/89
 * modified to be called by 'orb.c', to generate 'spherical heightfields'
 *  via tesselation, POV output format.
 *  John Beale (beale@jump.stanford.edu) 8/27/95
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "hcon.h"

typedef struct {
    double  x, y, z;
} point;

typedef struct {
    point     pt[3];    /* Vertices of triangle */
    double    area;     /* Unused; might be used for adaptive subdivision */
} triangle;

typedef struct {
    int       npoly;    /* # of triangles in object */
    triangle *poly;     /* Triangles */
} object;

/* Six equidistant points lying on the unit sphere */
#define XPLUS {  1,  0,  0 }    /*  X */
#define XMIN  { -1,  0,  0 }    /* -X */
#define YPLUS {  0,  1,  0 }    /*  Y */
#define YMIN  {  0, -1,  0 }    /* -Y */
#define ZPLUS {  0,  0,  1 }    /*  Z */
#define ZMIN  {  0,  0, -1 }    /* -Z */

/* Vertices of a unit octahedron */
triangle octahedron[] = {
    { { XPLUS, ZPLUS, YPLUS }, 0.0 },
    { { YPLUS, ZPLUS, XMIN  }, 0.0 },
    { { XMIN , ZPLUS, YMIN  }, 0.0 },
    { { YMIN , ZPLUS, XPLUS }, 0.0 },
    { { XPLUS, YPLUS, ZMIN  }, 0.0 },
    { { YPLUS, XMIN , ZMIN  }, 0.0 },
    { { XMIN , YMIN , ZMIN  }, 0.0 },
    { { YMIN , XPLUS, ZMIN  }, 0.0 }
};

/* A unit octahedron */
object oct = {
    sizeof(octahedron) / sizeof(octahedron[0]),
    &octahedron[0]
};

int PPHIGSflag = 0;

/* Forward declarations */
point *normalize(point *p, double fs);
point *midpoint( point *a, point *b );
void print_object(object *obj, int level, char *fname_out);
void print_triangle(triangle *t, FILE *fp);

static double pi2inv = 1.0 / (2.0 * 3.141592653589793238);

extern int xsize, ysize;    /* size of image array to map to sphere */
extern PTYPE *hf;
extern Boolean raw_out;    /* TRUE if we want to generate 'raw' triangles */

int gen_sphere(double f_scale, int maxlevel, char *fname_out)

{
    object *old,
           *new;
    int     ccwflag = 0,        /* Reverse vertex order if true */
            i,
            level;              /* Current subdivision level */

    if (ccwflag) {
        /* Reverse order of points in each triangle */
        for (i = 0; i < oct.npoly; i++) {
            point tmp;
                          tmp = oct.poly[i].pt[0];
            oct.poly[i].pt[0] = oct.poly[i].pt[2];
            oct.poly[i].pt[2] = tmp;
        }
    }

    old = &oct;  /* normalize old data to start with: */
    for (i = 0; i < oct.npoly; i++) {
         old->poly[i].pt[0] = *normalize(&old->poly[i].pt[0],f_scale);
         old->poly[i].pt[1] = *normalize(&old->poly[i].pt[1],f_scale);
         old->poly[i].pt[2] = *normalize(&old->poly[i].pt[2],f_scale);
       }

    /* Subdivide each starting triangle (maxlevel - 1) times */
    for (level = 1; level < maxlevel; level++) {
        /* Allocate a new object */
        new = (object *)malloc(sizeof(object));
        if (new == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
                level);
            exit(1);
        }
        new->npoly = old->npoly * 4;

        /* Allocate 4* the number of points in the current approximation */
        new->poly  = (triangle *)malloc(new->npoly * sizeof(triangle));
        if (new->poly == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
                level);
            exit(1);
        }

        /* Subdivide each triangle in the old approximation and normalize
         *  the new points thus generated to lie on the surface of the unit
         *  sphere.
         * Each input triangle with vertices labelled [0,1,2] as shown
         *  below will be turned into four new triangles:
         *
         *                      Make new points
         *                          a = (0+2)/2
         *                          b = (0+1)/2
         *                          c = (1+2)/2
         *        1
         *       /\             Normalize a, b, c
         *      /  \
         *    b/____\ c         Construct new triangles
         *    /\    /\              [0,b,a]
         *   /  \  /  \             [b,1,c]
         *  /____\/____\            [a,b,c]
         * 0      a     2           [a,c,2]
         */
        for (i = 0; i < old->npoly; i++) {
            triangle
                 *oldt = &old->poly[i],
                 *newt = &new->poly[i*4];
            point a, b, c;

            a = *normalize(midpoint(&oldt->pt[0], &oldt->pt[2]),f_scale);
            b = *normalize(midpoint(&oldt->pt[0], &oldt->pt[1]),f_scale);
            c = *normalize(midpoint(&oldt->pt[1], &oldt->pt[2]),f_scale);

            newt->pt[0] = oldt->pt[0];
            newt->pt[1] = b;
            newt->pt[2] = a;
            newt++;

            newt->pt[0] = b;
            newt->pt[1] = oldt->pt[1];
            newt->pt[2] = c;
            newt++;

            newt->pt[0] = a;
            newt->pt[1] = b;
            newt->pt[2] = c;
            newt++;

            newt->pt[0] = a;
            newt->pt[1] = c;
            newt->pt[2] = oldt->pt[2];
        }

        if (level > 1) {
            free(old->poly);
            free(old);
        }

        /* Continue subdividing new triangles */
        old = new;
    }

    /* Print out resulting approximation */
    print_object(old, maxlevel, fname_out); 
    return(0);
} /* end gen_sphere() */

/* Normalize a point p */
point *normalize(point *p, double f_scale)
{
    static point r;
    double mag, rad, phi, theta, r2d, scale;
    int xmap, ymap;

    r = *p;
 /*   printf("xs,ys: %d %d",xsize,ysize);

    printf("xyz: %1.3f %1.3f %1.3f ",r.x,r.y,r.z);
 */
    r2d = sqrt(r.x * r.x + r.y * r.y); /* radius in the x-y plane */
    phi = atan2(r.z,r2d);  /* range of atan2 is [-pi..pi] */
    theta = atan2(r.y, r.x);
   
    xmap = (int)((0.5 + pi2inv * theta) * (double)xsize);
    ymap = (int)((0.5 + pi2inv * phi) * (double)ysize);

    rad = 1.0 + f_scale * El(hf,xmap,ymap);

    mag = r.x * r.x + r.y * r.y + r.z * r.z;  /* r^2 of this point */
    if (mag != 0.0) {  /* definitely should not happen */
        scale = rad / sqrt(mag);
        r.x *= scale;
        r.y *= scale;
        r.z *= scale;
    }
/*    printf("xyz: %1.3f %1.3f %1.3f\n",r.x,r.y,r.z); */

    return &r;
}

/* Return the midpoint on the line between two points */
point *midpoint(a, b)
point *a, *b;
{
    static point r;

    r.x = (a->x + b->x) * 0.5;
    r.y = (a->y + b->y) * 0.5;
    r.z = (a->z + b->z) * 0.5;

    return &r;
}

/* Write out all triangles in an object */
void print_object(object *obj, int level, char *fname_out)
{
    int i;
    FILE *fp;

    fp = fopen(fname_out,"w");

    if (raw_out) fprintf(fp,"ORB\n");    /* RAW text format */
    else {
      fprintf(fp,"/* Include file for POV-Ray : orb from heightfield */\n");
      fprintf(fp,"union {\n");
    }
    /* Spit out coordinates for each triangle */
    for (i = 0; i < obj->npoly; i++)
        print_triangle(&obj->poly[i],fp);

    if (!raw_out) fprintf(fp,"} /* end union */\n");
    fclose(fp);
} /* end print_object() */

/* Output a triangle */
void print_triangle(triangle *t, FILE *fp)
{
    int i;

        /* Modify this to generate your favorite output format
         * Triangle vertices are in t->pt[0..2].{x,y,z}
         * raw and POV-Ray formats are provided.
         */
        if (raw_out) {
        for (i = 0; i < 3; i++)  {
            fprintf(fp,"%g %g %g ", t->pt[i].x, t->pt[i].y, t->pt[i].z);
            if (i != 2) fprintf(fp," ");
	  }
        } else {
        fprintf(fp,"triangle{");
        for (i = 0; i < 3; i++)  {
            fprintf(fp,"<%g,%g,%g>", t->pt[i].x, t->pt[i].y, t->pt[i].z);
            if (i != 2) fprintf(fp,",");
	  }
	fprintf(fp,"}");
        }
        fprintf(fp,"\n");

} /* end print_triangle */
