/* double.c:  convert heightfield data to various formats      
 *          read in array of height values,
 *          rescale to twice the size,
 *          write out in specified format
 *           John P. Beale 7/18/95                           
 *          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

void hf_print();                        /* print out hf array to stdout */
void rescale(PTYPE **hf, PTYPE **hf2);  /* rescale array size */
void smooth(PTYPE **hf1, PTYPE **hf2);  /* returns hf2 as smooth(hf1) */

unsigned int xsize, ysize;     /* size of hf array (input xsize+2) */
unsigned int xsize2, ysize2;   /* size of rescaled array hf2 */
PTYPE *hf;                            /* height field array */
PTYPE *hf2;                    /* rescale array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"double: A Heightfield Converter v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: double <infile> <outfile> \n\
  doubles size of infile and writes as outfile. Reads many types.\n\
  Determines filetype based on extension, eg:   hcon file1.tga file1.pot\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext1, *ext2;

  int rval;
  int inlen, outlen, otype, itype;
  size_t memspace; 


  if ((argc < 3) || (argc > 3)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);

  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
  ext1 = (char *) malloc((size_t)sizeof(char)*5);
  ext2 = (char *) malloc((size_t)sizeof(char)*5);

  strncpy(ext1,(char *)(&fname_in[inlen-4]),4);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),4);

  if ((ext1==NULL) || (ext2==NULL)) { 
    fprintf(stderr,"filenames must have type extension, eg. topo.gif\n");
    exit(1);
  }

  if       (strcmp(ext1,".gif")==0) itype = GIF;  
   else if (strcmp(ext1,".pot")==0) itype = POT;
   else if (strcmp(ext1,".tga")==0) itype = TGA;
   else if (strcmp(ext1,".pgm")==0) itype = PGM;
   else if (strcmp(ext1,".pg8")==0) itype = PG8;
   else if (strcmp(ext1,".mat")==0) itype = MAT;
   else if (strcmp(ext1,".oct")==0) itype = OCT;
   else if (strcmp(ext1,".png")==0) itype = PNG;
   else { 
     itype = PGM;       /* default type */
     fprintf(stderr,"Warning: %s has a missing or unknown file extension\n",argv[1]);
   }

  if       (strcmp(ext2,".gif")==0) otype = GIF;  
   else if (strcmp(ext2,".pot")==0) otype = POT;
   else if (strcmp(ext2,".tga")==0) otype = TGA;
   else if (strcmp(ext2,".pgm")==0) otype = PGM;
   else if (strcmp(ext2,".pg8")==0) otype = PG8;
   else if (strcmp(ext2,".mat")==0) otype = MAT;
   else if (strcmp(ext2,".oct")==0) otype = OCT;
   else if (strcmp(ext2,".png")==0) otype = PNG;
   else { 
     fprintf(stderr,"Error: %s has a missing or unknown file extension\n",argv[2]);
     exit(1);
   }

   printf("otype = %d\n",otype);
   
					/* read in heightfield */
   rval = read_hf(&hf, &xsize, &ysize, fname_in, itype);
   if (rval != 0)
     exit(rval);

   rescale((PTYPE **)&hf, (PTYPE **)&hf2);   /* rescale array to [0..1] */
   xsize = xsize2;
   ysize = ysize2;
   free(hf);          /* get rid of old image */

   memspace = (size_t) xsize*ysize*sizeof(PTYPE);

   hf = (PTYPE *)malloc(memspace);        /* space for new image */
   if (hf==NULL) {
     fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
     exit(1);
   }
   smooth((PTYPE **)&hf2, (PTYPE **)&hf); /* returns smoothed image in hf */

   printf("%s: [%d x %d] \n",fname_out,xsize,ysize);
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = write_hf(hf, xsize, ysize, fname_out, otype);
   
  return(rval);    /* return a value */

} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



/* -------------------------------------------------------------------- */
/* renorm() --  scale data in hf[xsize,ysize] to fit in the full        */
/* range 0..1  */
/* -------------------------------------------------------------------- */

void rescale(PTYPE **hf1, PTYPE **hf2)
{
 PTYPE tmp;
 int ix,iy;
 int x,y;
 size_t memspace;

 memspace = (size_t) xsize*ysize*4*sizeof(PTYPE);

 hf2[0] = (PTYPE *)malloc(memspace);
 if (hf2[0]==NULL) {
   fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
   exit(1);
 }

 xsize2 = xsize * 2;
 ysize2 = ysize * 2;

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
      tmp = El(hf1[0],ix,iy);
      x = ix * 2;
      y = iy * 2;
      El2(hf2[0],x,y) = tmp;
      El2(hf2[0],x+1,y) = tmp;
      El2(hf2[0],x+1,y+1) = tmp;
      El2(hf2[0],x,y+1) = tmp;
    }
   }

}       /* end rescale() */

/* smooth() - copy smoothed version of argument 1 into argument 2, 
 * both of which are pointers to (xsize x ysize) arrays. 
 */

void smooth(PTYPE **hf1, PTYPE **hf2)
{
int ix, iy, i;
PTYPE sum;
int xo[9] =                      /* 8-dir index offset arrays */
   { 0,-1,0,1,1,1,0,-1,-1 };
int yo[9] =
   { 0,-1,-1,-1,0,1,1,1,0 };

  El(hf2[0],0,0) = El(hf1[0],0,0);              /* corner points */
  El(hf2[0],xsize-1,0) = El(hf1[0],xsize-1,0);
  El(hf2[0],0,ysize-1) = El(hf1[0],0,ysize-1);
  El(hf2[0],xsize-1,ysize-1) = El(hf1[0],xsize-1,ysize-1);

  for (ix = 1; ix < (xsize-1); ix++) {  /* copy edge pixels except corners */
    El(hf2[0],ix,0) = (El(hf1[0],ix-1,0)+El(hf1[0],ix+1,0) )/2.0;
    El(hf2[0],ix,ysize-1) = (El(hf1[0],ix-1,ysize-1)+
	 El(hf1[0],ix+1,ysize-1) )/2.0;
  }

  for (iy = 1; iy < (ysize-1); iy++) {
    El(hf2[0],0,iy) = (El(hf1[0],0,iy-1)+El(hf1[0],0,iy+1)
			)/2.0;
    El(hf2[0],xsize-1,iy) = (El(hf1[0],xsize-1,iy-1)+
       El(hf1[0],xsize-1,iy+1))/2.0;
  }

   for (iy = 1; iy<ysize-1; iy++) { 
    for (ix = 1; ix<xsize-1; ix++) {
      sum = 0;
      for (i=2;i<9;i+=2) {
       sum += El(hf1[0],ix+xo[i],iy+yo[i]); /* adj. neighbors */
      } /* end for i... */
      El(hf2[0],ix,iy) = sum/4;
   }
  } /* end for iy.. */

}
